/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *
 */
/*
 *
 * Copyright (C) Nicolas Roussel
 * Copyright (C) 2003 Olivier Chapuis
 *
 * See the file LICENSE for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * Note: LICENSE is LGPL
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/wnc_auth.h>
#include <maf/wnc_conversion.h>
#include <maf/wnc_desktop.h>
#include <maf/wnc_source.h>
#include <maf/wnc_tcp.h>
#include <maf/wnc_window.h>

#include <maf/url.h>

#include <osg/ref_ptr>
#endif

#ifndef WIN32

#include <X11/Xmd.h>
#include <X11/Xutil.h>

extern "C"
{
#include <maf/wnc_proto.h>
}
#include <maf/wnc_auth.h>
#include <maf/url.h>
#include <maf/maferror.h>
#include <maf/wnc_conversion.h>
#include <maf/wnc_source.h>
#include <maf/wnc_tcp.h>
#include <maf/wnc_desktop.h>
#include <maf/wnc_window.h>
#endif

#ifdef WIN32
#include <winsock.h>
#include <windows.h>
#include <windef.h>
#include <winsock.h>
#include <io.h>
#include <errno.h>
#else
#include <sys/socket.h>
#include <errno.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <assert.h>
#include <sys/ioctl.h>
#endif

#include <stdio.h>
#include <math.h>

#include <stdexcept>

/*
 * NOTE: VNC supports bitsPerPixel of 8, 16 and 32 only (not 24). Raw
 * RGB is in fact raw RGBA...
 */

/*
 * When leaving a window, should we generate keyRelease events for
 * pressed modifiers (VNC client does so) ?
 *
 */

#define DEBUG_LEVEL 0

static  int
  getavail(int fd) {
#ifdef WIN32
    u_long avail;
    if (ioctlsocket(fd, FIONREAD, &avail)==-1) 
#else
    int avail ;
    if (ioctl(fd, FIONREAD, &avail)==-1) 
#endif
	 throw std::runtime_error("ioctl FIONREAD failed (getavail)") ;
    return avail ;
  }


static  int
  clearRegion(WncImage *img,
		    int x1, int y1, int x2, int y2,
		    unsigned char R, unsigned char G, unsigned char B, unsigned char A,
		    float blending = 1.0) {
    if (!blending) return 0 ;

    WncImage::Encoding e = img->getEncoding() ;
    if (!WncImage::pixelIsContiguous(e)) return 0 ;

    int w = img->getWidth() ;
    int h = img->getHeight() ;

    if (x1>w-1) x1=w-1 ; if (x2>w-1) x2=w-1 ;
    if (x1<0) x1=0 ; if (x2<0) x2=0 ;

    if (y1>h-1) y1=h-1 ; if (y2>h-1) y2=h-1 ;
    if (y1<0) y1=0 ; if (y2<0) y2=0 ;

    unsigned char *data = (unsigned char *)img->getData() ;

    int b = (int)WncImage::BytesPerPixel(e) ;
    int lineWidth = img->getWidth()*b ;

    unsigned char newvalues[4] = {R, G, B, A} ;

    float nb = 1.0-blending ;
    int i ;
    for (int y=y1; y<=y2; ++y) {
	 unsigned char *ptr = data + lineWidth*y + x1*b -1 ;
	 for (int x=x1; x<=x2; ++x)
	   switch (e) {
	   case WncImage::ABGR:
		for (i=3; i>=0; --i)
		  *(++ptr) = (unsigned char)(newvalues[i]*blending+nb*(*ptr)) ;
		break ;
	   case WncImage::RGB:
		for (i=0; i<3; ++i)
		  *(++ptr) = (unsigned char)(newvalues[i]*blending+nb*(*ptr)) ;
		break ;
	   case WncImage::RGBA:
		for (i=0; i<4; ++i)
		  *(++ptr) = (unsigned char)(newvalues[i]*blending+nb*(*ptr)) ;
		break ;
	   case WncImage::L:
		*++ptr = (int)(0.11*B + 0.59*G + 0.3*R) ;
		break ;
	   default:
		break ;
	   }
    }
    return 1 ;
  }

static int littleEndianTest = 1;

#define Swap16IfLE(s) \
    (*(char *)&littleEndianTest ? \
    ((((s) & 0xff) << 8) | (((s) >> 8) & 0xff)) : (s))

#define Swap32IfLE(l) \
    (*(char *)&littleEndianTest ? ((((l) & 0xff000000) >> 24) | \
			     (((l) & 0x00ff0000) >> 8)  | \
			     (((l) & 0x0000ff00) << 8)  | \
			     (((l) & 0x000000ff) << 24))  : (l))

#if DEBUG_LEVEL>=1
static void
PrintPixelFormat(rfbPixelFormat *format) {
  if (format->bitsPerPixel == 1) {
    fprintf(stderr,"  Single bit per pixel.\n");
    fprintf(stderr,
		  "  %s significant bit in each byte is leftmost on the screen.\n",
		  (format->bigEndian ? "Most" : "Least"));
  } else {
    fprintf(stderr,"  %d bits per pixel.\n",format->bitsPerPixel);
    if (format->bitsPerPixel != 8) {
      fprintf(stderr,"  %s significant byte first in each pixel.\n",
		    (format->bigEndian ? "Most" : "Least"));
    }
    if (format->trueColour) {
      fprintf(stderr,"  True colour: max red %d green %d blue %d",
		    format->redMax, format->greenMax, format->blueMax);
      fprintf(stderr,", shift red %d green %d blue %d\n",
		    format->redShift, format->greenShift, format->blueShift);
    } else {
      fprintf(stderr,"  Colour map (not true colour).\n");
    }
  }
}
#endif

// ----------------------------------------------------------------------
#define BUF_SIZE 8192
static char buf[BUF_SIZE];
static char *bufoutptr = buf;
static int buffered = 0;

bool
wncSource::_ReadFromRFBServer(char *out, unsigned int n)
{
  int rfbsock = _conn->getFd() ;
  
  if (n <= (unsigned int)buffered)
    {
      memcpy(out, bufoutptr, n);
      bufoutptr += n;
      buffered -= n;
      return true;
    }
  
  memcpy(out, bufoutptr, buffered);
  
  out += buffered;
  n -= buffered;
  
  bufoutptr = buf;
  buffered = 0;
  
  if (n <= BUF_SIZE)
    {
      
      while ((unsigned int)buffered < n)
	{
	  int i = read(rfbsock, buf + buffered, BUF_SIZE - buffered);
	  if (i <= 0)
	    {
	      if (i < 0)
		{
		  if (
#ifdef WIN32
		      1
#else
		      errno == EWOULDBLOCK
#endif
		      || errno == EAGAIN)
		    {
		      g_warning("Should process event?");
		      i = 0;
		    }
		  else
		    {
		      perror(": read");
		      return false;
		    }
		}
	      else
		{
		  if (1)
		    {
		      g_warning("WNC server closed connection");
		    }
		  return false;
		}
	    }
	  buffered += i;
	}
      
      memcpy(out, bufoutptr, n);
      bufoutptr += n;
      buffered -= n;
      return true;
    }
  else
    {
      while (n > 0)
	{
	  int i = read(rfbsock, out, n);
	  if (i <= 0)
	    {
	      if (i < 0)
		{
		  if (
#ifdef WIN32
		      true
#else
		      errno == EWOULDBLOCK
#endif
		      || errno == EAGAIN) 
		    {
		      g_warning("Should process event?");
		      i = 0;
		    }
		  else
		    {
		      perror(": read");
		      return false;
		    }
		}
	      else
		{
		  g_warning("WNC server closed connection");
		  return false;
		}
	    }
	  out += i;
	  n -= i;
	}
      
      return true;
    }
}

void
wncSource::_receive(char *data, unsigned int length) {
  int	fd = _conn->getFd();
  size_t	read_total = 0;

  int res=0;
  while (read_total < length) {
#ifdef WIN32
      res = recv( fd, data + read_total, length - read_total, 0 );

      if ( res == 0 || res == WSAECONNRESET ) {

          printf( "Connection Closed.\n");

          break;

      }

#else
    res = read(fd,data + read_total,length - read_total);
    if (res<0)
      return;
#endif
    read_total +=res;
  }
}

//#define RECEIVE(a,b) _ReadFromRFBServer(a,b);
#define RECEIVE(a,b) _receive(a,b);
// ------------------------------------------------------------------

wncSource::wncSource(
	XwncDesktop *wncDesktop, WncImage::Encoding encoding, const URL& url)
{
	_wncDesktop = wncDesktop;
	_rencoding = encoding==WncImage::PREFERRED ? WncImage::RGB : encoding ;
	_hostname = url.getHost() ;
	_port = 5900+atoi(url.getPort().c_str()) ;
	std::string query = url.getQuery() ;
	_password = URL::getQueryArg(query, "password") ;
	_newImage = false ;
	_conn = 0 ;
}

wncSource::~wncSource()
{
  stop() ;
}

bool
wncSource::start(void)
{
  try {
    _conn = new TcpConnection(_hostname.c_str(), _port) ;

    // --------------------------------------------------------
    // Greetings

    rfbProtocolVersionMsg pv;
    RECEIVE(pv, sz_rfbProtocolVersionMsg) ;
    pv[sz_rfbProtocolVersionMsg] = 0 ;

    int major,minor;
    if (sscanf(pv,rfbProtocolVersionFormat,&major,&minor) != 2)
      {
	throw std::runtime_error(
				 "wncSource: not a valid VNC server") ;
      }

#if DEBUG_LEVEL>=1
    g_debug("wncSource: WNC server supports protocol version %d.%d",
	    major, minor);
    g_debug("(viewer %d.%d)",
	    rfbProtocolMajorVersion, rfbProtocolMinorVersion);
#endif

    major = rfbProtocolMajorVersion ;
    minor = rfbProtocolMinorVersion ;
    sprintf(pv,rfbProtocolVersionFormat,major,minor) ;
    _conn->send(pv, sz_rfbProtocolVersionMsg) ;

    // -----------------------------------------------------------
    // Authentication

    CARD32 authScheme ;
    RECEIVE((char *)&authScheme, 4) ;
    authScheme = Swap32IfLE(authScheme);

    switch (authScheme)
      {
      case rfbConnFailed:
	{
	  CARD32 reasonLen ;
	  RECEIVE((char *)&reasonLen, 4) ;
	  reasonLen = Swap32IfLE(reasonLen) ;
	  char *reason = new char [reasonLen] ;
	  RECEIVE(reason, reasonLen) ;
	  std::string msg = "wncSource: WNC connection failed (" ;
	  msg.append(reason, reasonLen) ;
	  msg.append(")") ;
	  throw std::runtime_error(msg) ;
	}
	break ;

      case rfbNoAuth:
#if DEBUG_LEVEL>=1
	g_debug("wncSource: No authentication needed");
#endif
	break;

      case rfbVncAuth:
	{
	  CARD8 challenge[CHALLENGESIZE];
	  RECEIVE((char *)challenge, CHALLENGESIZE) ;

	  if (_password!="")
	    {
	      vncEncryptBytes(
			      challenge, (char *)_password.c_str()) ;
	    }
	  else
	    {
	      char *passwd ;
	      do
		{
		  //					passwd = getpass("WNC Password: ");
		  passwd = "0"; //getpass("WNC Password: ");
		}
	      while ((!passwd) || (strlen(passwd) == 0)) ;
	      if (strlen(passwd) > 8)
		{
		  passwd[8] = '\0';
		}
	      vncEncryptBytes(challenge, passwd);
	      // "Forget" the clear-text password
	      for (int i = strlen(passwd); i >= 0; i--)
		{
		  passwd[i] = '\0';
		}
	    }

	  _conn->send((char *)challenge, CHALLENGESIZE) ;

	  CARD32 authResult;
	  RECEIVE((char *)&authResult, 4) ;
	  authResult = Swap32IfLE(authResult);

	  switch (authResult)
	    {
	    case rfbVncAuthOK:
#if DEBUG_LEVEL>=1
	      g_debug("wncSource: WNC authentication succeeded");
#endif
	      break;
	    case rfbVncAuthFailed:
	      throw std::runtime_error("wncSource: WNC authentication failed") ;
	    case rfbVncAuthTooMany:
	      throw std::runtime_error("wncSource: WNC authentication "
				       "failed - too many tries");
	    default:
	      throw std::runtime_error("wncSource: Unknown WNC authentication "
				       "result") ;
	    }
	}
	break;

      default:
	throw std::runtime_error("Unknown authentication scheme from WNC server");
      } /* switch (authScheme) */

    // -----------------------------------------------------------

    rfbClientInitMsg ci;
    ci.shared = 1 ;
    _conn->send((char *)&ci, sz_rfbClientInitMsg) ;

    rfbServerInitMsg si;
    RECEIVE((char *)&si, sz_rfbServerInitMsg) ;

    si.framebufferWidth = Swap16IfLE(si.framebufferWidth);
    si.framebufferHeight = Swap16IfLE(si.framebufferHeight);
    si.format.redMax = Swap16IfLE(si.format.redMax);
    si.format.greenMax = Swap16IfLE(si.format.greenMax);
    si.format.blueMax = Swap16IfLE(si.format.blueMax);
    si.nameLength = Swap32IfLE(si.nameLength);

    _width = si.framebufferWidth;
    _height = si.framebufferHeight;

    char *desktopName = new char [si.nameLength + 1] ;
    RECEIVE(desktopName, si.nameLength) ;

    desktopName[si.nameLength] = 0;

#if DEBUG_LEVEL>=1
    fprintf(stderr,"Desktop name \"%s\" (%ix%i)\n",desktopName,
	    _width, _height);
    fprintf(stderr,"Connected to WNC server, using protocol "o
	    "version %d.%d\n",
	    rfbProtocolMajorVersion, rfbProtocolMinorVersion);
    fprintf(stderr,"WNC server default format:\n");
    PrintPixelFormat(&si.format);
#endif

    delete [] desktopName;

    // --- Pixel format ------------------------------------------

    rfbPixelFormat simple_format ;

    memset(&simple_format, 0, sizeof (rfbPixelFormat));
    simple_format.bitsPerPixel = 32 ;
    simple_format.depth = 8 ;
    simple_format.bigEndian = (*(char *)&littleEndianTest ? 0 : 1) ;
    simple_format.trueColour = 1 ;
    simple_format.redMax =
      simple_format.greenMax =
      simple_format.blueMax = 255 ;

    _encoding = _rencoding ;
    if (_encoding!=WncImage::RGB && _encoding!=WncImage::RGBA &&
	_encoding!=WncImage::ABGR)
      {
	_encoding = WncImage::RGB ;
      }

    if (_encoding==WncImage::ABGR) 
      {
	if (*(char *)&littleEndianTest)
	  {
	    simple_format.redShift = 24 ;
	    simple_format.greenShift = 16 ;
	    simple_format.blueShift = 8 ;
	  }
	else
	  {
	    simple_format.redShift = 0 ;
	    simple_format.greenShift = 8 ;
	    simple_format.blueShift = 16 ;
	  }
      }
    else
      {
	if (*(char *)&littleEndianTest)
	  {
	    simple_format.redShift = 0 ;
	    simple_format.greenShift = 8 ;
	    simple_format.blueShift = 16 ;
	  }
	else
	  {
	    simple_format.redShift = 24 ;
	    simple_format.greenShift = 16 ;
	    simple_format.blueShift = 8 ;
	  }
      }
  
    rfbSetPixelFormatMsg spf;
    // memset((char*)&spf, 0, sz_rfbSetPixelFormatMsg);
    spf.type = rfbSetPixelFormat;
    spf.pad1 = 0, spf.pad2 = 0; // fix valgrind warning
    spf.format = simple_format ;
    spf.format.redMax = Swap16IfLE(spf.format.redMax);
    spf.format.greenMax = Swap16IfLE(spf.format.greenMax);
    spf.format.blueMax = Swap16IfLE(spf.format.blueMax);
    _conn->send((char *)&spf, sz_rfbSetPixelFormatMsg) ;

    // --- Encoding ----------------------------------------------


    char buf[sz_rfbSetEncodingsMsg + 5 * 4]; // 5 = MAX_ENCODINGS
    memset(buf, 0, sz_rfbSetEncodingsMsg + 5 * 4);
    rfbSetEncodingsMsg *se = (rfbSetEncodingsMsg *)buf;
    CARD32 *encs = (CARD32 *)(&buf[sz_rfbSetEncodingsMsg]);
    se->type = rfbSetEncodings;
    se->nEncodings = 0;

    // Not supported by Xwnc
    // encs[se->nEncodings++] = Swap32IfLE(rfbEncodingCopyRect);
    // encs[se->nEncodings++] = Swap32IfLE(rfbEncodingHextile);
    // encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRRE);
		
		
    // broken here or in Xwnc when resizing a window ?
    //encs[se->nEncodings++] = Swap32IfLE(rfbEncodingCoRRE);

    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRaw) ;
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingPointerPos) ;
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingXCursor) ;
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingWindowShape) ;

    int len = sz_rfbSetEncodingsMsg + se->nEncodings * 4;
    se->nEncodings = Swap16IfLE(se->nEncodings);
    _conn->send(buf, len) ;

    // --- Here we go ! ------------------------------------------
    // 
    //updateRequest(false) ;
    return true ;

  } catch (std::runtime_error e) {
    g_warning("%s@%d %s", __FILE__, __LINE__, e.what());
    stop() ;
    return false ;
  }
}

void
wncSource::check()
{
  if (!_readWNCServer())
    {
      // error should exit
    }
}

bool wncSource::_framebufferUpdate(rfbServerToClientMsg msg)
{
  RECEIVE(((char *)&msg.fu) + 1, sz_rfbFramebufferUpdateMsg - 1) ;
  msg.fu.nRects = Swap16IfLE(msg.fu.nRects);

  unsigned long xWindow = Swap32IfLE(msg.fu.window);
  unsigned long isRoot = Swap32IfLE(msg.fu.topWindow);

  if (isRoot == 0)
    {
      isRoot = 1;
    }
  else
    {
      isRoot = 0;
    }

#if DEBUG_LEVEL>=2
  fprintf(stderr, "wncSource[FramebufferUpdate]: window: 0x%lx (%i,%i)\n",
	  xWindow, msg.fu.nRects, isRoot);
#endif

  osg::ref_ptr<XwncWindow> wncWin = _wncDesktop->GetWindow(xWindow);
  float wWidth = 0, wHeight = 0;

  if (wncWin.get())
    {
      wncWin->getSize(&wWidth, &wHeight);
    }

  int i;
  for (i=0; i<msg.fu.nRects; ++i)
    {

      rfbFramebufferUpdateRectHeader rect;
      RECEIVE((char *)&rect, sz_rfbFramebufferUpdateRectHeader) ;
      rect.r.x = Swap16IfLE(rect.r.x);
      rect.r.y = Swap16IfLE(rect.r.y);
      rect.r.w = Swap16IfLE(rect.r.w);
      rect.r.h = Swap16IfLE(rect.r.h);
      rect.encoding = Swap32IfLE(rect.encoding);

#if DEBUG_LEVEL>=2
      g_debug("wncSource: rfbFramebufferUpdate %d,%d %dx%d encoding %X",
	      rect.r.x, rect.r.y, rect.r.w, rect.r.h,
	      rect.encoding);
#endif

      if (rect.encoding == rfbEncodingXCursor) {
	int bpr = (rect.r.w + 7) / 8;
	int bd = bpr * rect.r.h;
	char *buf = NULL;
	rfbXCursorColors colors;

	if (rect.r.w * rect.r.h)
	  {
	    RECEIVE((char *)&colors, sz_rfbXCursorColors);

	    buf = (char *)malloc(bd * 2);
	    if (buf == NULL)
	      {
		return false;
	      }

	    RECEIVE(buf, bd * 2);
	  }
	// FIXME ignored
	// 			_wncDesktop->HandleXCursor(
	// 				rect.r.x, rect.r.y, rect.r.w, rect.r.h,
	// 				buf, colors);
	free(buf); // else leak
	continue;
      }

      if (rect.encoding == rfbEncodingPointerPos)
	{
	  _wncDesktop->handleCursorPosition(rect.r.x, rect.r.y);
	  continue;
	}


      if (rect.encoding == rfbEncodingWindowShape)
	{
	  /* rect.r contains the size of the window */
	  rfbWindowShapeHeader h;
	  RECEIVE((char *)&h, sz_rfbWindowShapeHeader) ;
	  h.nrects = Swap32IfLE(h.nrects);
	  int size =  h.nrects * 4 * sizeof(CARD16) ;
	  CARD16 *buf = NULL;
	  if (size)
	    {
	      buf = (CARD16 *)malloc(size);
	      RECEIVE((char *)buf, size) ;
	      unsigned int i;
	      for (i = 0; i < h.nrects; i++)
		{
		  buf[i*4] = Swap16IfLE(buf[i*4]);
		  buf[i*4+1] = Swap16IfLE(buf[i*4+1]);
		  buf[i*4+2] = Swap16IfLE(buf[i*4+2]);
		  buf[i*4+3] = Swap16IfLE(buf[i*4+3]);
		}
	    }
#if 0
	  _wncDesktop->handleWindowShape(wncWin.get(), buf, h.nrects);
#endif
	  continue;
	}

      if ((rect.r.h * rect.r.w) == 0)
	{
	  continue ;
	}
		
      if (wncWin.get() &&
	  (rect.r.x+rect.r.w > (int)wWidth ||
	   rect.r.y+rect.r.h > (int)wHeight))
	{
	  g_warning("wncSource [rfbFramebufferUpdate]: Rectangle too big ...");
	  // Fatal error?
	  //continue ;
	}

      switch (rect.encoding)
	{
	case rfbEncodingARGBCursor:
	  {
#ifdef HAVE_XCURSOR
	    char *buf = NULL;
	    unsigned int bufSize = rect.r.w * rect.r.h * 4;
	    
	    if (bufSize)
	      {
		buf = (char *)malloc(bufSize);
		if (buf == NULL)
		  {
		    std::cerr << "wncSource: malloc fail"
			      << std::endl;
		    return False;
		  }
		
		RECEIVE((char *)buf, bufSize);
	      }
	    _wncDesktop->HandleARGBCursor(
					  (CARD32)rect.r.x, (CARD32)rect.r.y,
					  (CARD32)rect.r.w, (CARD32)rect.r.h,
					  (XcursorPixel *)buf);
#else // HAVE_XCURSOR
		unsigned buffer_size = rect.r.w * rect.r.h * 4;
		char * buffer = new char[buffer_size];
		RECEIVE(buffer, buffer_size);
		delete [] buffer;
#endif // HAVE_XCURSOR
	    continue;
	  }
	case rfbEncodingRaw:
	  {
	    int rectImgSize = rect.r.w * rect.r.h * 4 ;
	    unsigned char *ptr = WncImage::AllocMem(rectImgSize) ;
	    RECEIVE((char *)ptr,rectImgSize) ;
	    WncImage tmpImg ;
	    if (_encoding==WncImage::RGB)
	      {
		tmpImg.setEncoding(WncImage::RGBA) ;
	      }
	    else
	      {
		tmpImg.setEncoding(_encoding) ;
	      }
	    tmpImg.setData(ptr, rectImgSize, WncImage::FREEMEM) ;
	    tmpImg.setDims(rect.r.w, rect.r.h) ;
	    convertImage(&tmpImg, _encoding) ;
	    _wncDesktop->handleImageFramebufferUpdate(
			wncWin.get(), (isRoot==0)?false:true, &tmpImg,
						      rect.r.x, rect.r.y, rect.r.w, rect.r.h);
	    _newImage = true ;
	  }
	  break; 

	case rfbEncodingCoRRE:
	  {
	    // DOES NOT WORK !!
	    int rectImgSize = rect.r.w * rect.r.h * 4 ;
	    unsigned char *ptr = WncImage::AllocMem(rectImgSize) ;
	    //RECEIVE((char *)ptr,rectImgSize) ;
	    WncImage tmpImg ;
	    if (_encoding==WncImage::RGB)
	      {
		tmpImg.setEncoding(WncImage::RGBA) ;
	      }
	    else
	      {
		tmpImg.setEncoding(_encoding) ;
	      }
	    tmpImg.setData(ptr, rectImgSize, WncImage::FREEMEM) ;
	    tmpImg.setDims(rect.r.w, rect.r.h) ;
	    rfbRREHeader hdr;
	    RECEIVE((char *)&hdr, sz_rfbRREHeader) ;
	    hdr.nSubrects = Swap32IfLE(hdr.nSubrects);
	    //fprintf(stderr, "Subrectrangle: %i\n", hdr.nSubrects);
	    CARD32 pix;
	    RECEIVE((char *)&pix, sizeof(pix)) ;
	    char *pixel = (char*)&pix ;
	    if (_encoding==WncImage::ABGR)
	      {
		clearRegion(
			    &tmpImg, 0, 0, rect.r.w-1, rect.r.h-1,
			    pixel[3], pixel[2], pixel[1], 255) ;
	      }
	    else
	      {
		clearRegion(
			    &tmpImg, 0, 0, rect.r.w-1, rect.r.h-1,
			    pixel[0], pixel[1], pixel[2], 255) ;
	      }
	    char *subPtr = (char *)malloc(
					  (unsigned int)(hdr.nSubrects*8)) ;
	    RECEIVE((char *)subPtr,
		    (unsigned int)(hdr.nSubrects*8)) ;
	    CARD8 *sptr = (CARD8 *)subPtr;
	    for (unsigned int j=0; j<hdr.nSubrects; ++j)
	      {
		pix = *(CARD32 *)sptr;
		sptr += 3 ;
		int x = *++sptr;
		int y = *++sptr;
		int w = *++sptr;
		int h = *++sptr;
		++sptr ;
		pixel = (char*)&pix ;
		int x1=x, y1=y ;
		int x2=x1+w-1, y2=y1+h-1 ;
		if (_encoding==WncImage::ABGR)
		  {
		    clearRegion(
				&tmpImg, x1,y1, x2,y2,
				pixel[3],pixel[2],pixel[1],255) ;
		  }
		else
		  {
		    clearRegion(
				&tmpImg, x1,y1, x2,y2,
				pixel[0],pixel[1],pixel[2],255) ;
		  }
	      }
	    free(subPtr);
	    _wncDesktop->handleImageFramebufferUpdate(
			wncWin.get(), (isRoot == 0) ? false : true, &tmpImg,
						      rect.r.x, rect.r.y, rect.r.w, rect.r.h);
	    _newImage = true ;
	  }
	  break;

	default:
	  g_warning("wncSource: unknown rect encoding: %lX", rect.encoding);
	  return false;
	}
    }
  updateRequest(true);
  return true;
}

bool
wncSource::_readWNCServer(void) {
  if (!_conn)
    return false;

  int fd = _conn->getFd() ;

  while (getavail(fd))
    {
      rfbServerToClientMsg msg;
      RECEIVE((char *)&msg, 1) ;

      switch (msg.type)
	{
	case rfbBell:
	  //g_warning("wncSource: ring my bell...");
	  break ;
	case rfbSetColourMapEntries:
	  {
	    RECEIVE(
		    ((char *)&msg.scme) + 1,
		    sz_rfbSetColourMapEntriesMsg - 1) ;
	    int size = msg.scme.nColours * 3 /* * CARD16 */ ;
	    char *tmp = new char [size] ;
	    RECEIVE(tmp, size) ;
	    delete tmp ;
	  }
	  break ;
	case rfbServerCutText:
	  {
	    RECEIVE(((char *)&msg.sct) + 1,
		    sz_rfbServerCutTextMsg - 1) ;
	    msg.sct.length = Swap16IfLE(msg.sct.length);
	    char *tmp = new char [msg.sct.length] ;
	    RECEIVE(tmp, msg.sct.length) ;
	    g_warning("wncSource: new text in cut buffer \"%s\"", tmp);
	    delete tmp;
	  }
	  break ;

	case rfbConfigureWindow:
	  {
	    RECEIVE(((char *)&msg.cw) + 1,
		    sz_rfbConfigureWindowMsg - 1) ;
	    msg.cw.window = Swap32IfLE(msg.cw.window);
	    msg.cw.xsgn =  Swap16IfLE(msg.cw.xsgn);
	    int x = Swap16IfLE(msg.cw.x);
	    if (msg.cw.xsgn == 0)
	      {
		x = -x;
	      }
	    msg.cw.ysgn =  Swap16IfLE(msg.cw.ysgn);
	    int y = Swap16IfLE(msg.cw.y);
	    if (msg.cw.ysgn == 0)
	      {
		y = -y;
	      }
	    msg.cw.width = Swap32IfLE(msg.cw.width);
	    msg.cw.height =	Swap32IfLE(msg.cw.height);
	    _wncDesktop->handleConfigureWindow(msg.cw.window, msg.cw.isroot, x, y,
					       (int)msg.cw.width, (int)msg.cw.height,
					       msg.cw.window_name);
	    _newImage = true;
	    updateRequest(true);
	  }
	  break;

	case rfbUnmapWindow:
	  {
	    RECEIVE(((char *)&msg.uw) + 1,
		    sz_rfbUnmapWindowMsg - 1) ;
	    msg.uw.window = Swap32IfLE(msg.uw.window);
	    _wncDesktop->handleUnmapWindow(msg.uw.window);
	  }
	  break;

	case rfbDestroyWindow:
	  {
	    RECEIVE(((char *)&msg.dw) + 1,
		    sz_rfbDestroyWindowMsg - 1) ;
	    msg.dw.window = Swap32IfLE(msg.dw.window);
	    _wncDesktop->handleDestroyWindow(msg.dw.window);
	  }
	  break;
    
	case rfbRestackWindow:
	  {
	    RECEIVE(((char *)&msg.rw) + 1,
		    sz_rfbRestackWindowMsg - 1) ;
	    msg.rw.window = Swap32IfLE(msg.rw.window);
	    msg.rw.nextWindow = Swap32IfLE(msg.rw.nextWindow);
	    msg.rw.transientFor = Swap32IfLE(msg.rw.transientFor);
	    msg.rw.flags = Swap32IfLE(msg.rw.flags);
	    _wncDesktop->handleRestackWindow(
					     msg.rw.window, msg.rw.nextWindow,
					     msg.rw.transientFor, msg.rw.flags);
	    _newImage = true;
	    updateRequest(true);
	  }
	  break;

	case rfbFramebufferUpdate:
	  {
	    bool ret = _framebufferUpdate(msg);
	    if (!ret)
	      {
		return false;
	      }
	  }
	  break;

	default:
	  g_warning("wncSource: unknown message type %X from WNC server",
		    msg.type);
	  return false;
	  break ;
	} /* switch */

    }

  return true;
}

bool
wncSource::stop(void) {
  if (_conn) {
    delete _conn ;
    _conn=0 ;
    return true ;
  }
  return false ;
}

// ------------------------------------------------------------------

void
wncSource::updateRequest(bool incremental)
{
	updateRequest(0,0,10*_width,10*_height,incremental) ;
}

void
wncSource::updateRequest(int x, int y, int w, int h, bool incremental) {
  rfbFramebufferUpdateRequestMsg fur;

  fur.type = rfbFramebufferUpdateRequest;
  fur.incremental = incremental ? 1 : 0;
  fur.x = Swap16IfLE(x);
  fur.y = Swap16IfLE(y);
  fur.w = Swap16IfLE(w);
  fur.h = Swap16IfLE(h);

  _conn->send((char *)&fur, sz_rfbFramebufferUpdateRequestMsg) ;
}

void
wncSource::keyEvent(unsigned long key, bool down_flag) {
  rfbKeyEventMsg ke;

  ke.type = rfbKeyEvent ;
  ke.down = down_flag ? 1 : 0 ;
  ke.pad = 0;
  ke.key = Swap32IfLE(key) ;
  _conn->send((char *)&ke, sz_rfbKeyEventMsg) ;
}

void
wncSource::pointerEvent(
	Window id, int x, int y, unsigned char button_mask)
{
  rfbPointerEventMsg pe;
  memset(&pe, 0, sz_rfbPointerEventMsg);

#if 0
  if (x < 0 || y < 0)
    {
      return;
    }
#endif

  pe.type = rfbPointerEvent ;
  pe.buttonMask = button_mask ;
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  pe.xsgn = 1;
  pe.ysgn = 1;
  pe.x = Swap16IfLE(x) ;
  pe.y = Swap16IfLE(y) ;
  pe.window = Swap32IfLE(id);
  _conn->send((char *)&pe, sz_rfbPointerEventMsg) ;
}

void
wncSource::pointerEvent(int x, int y, unsigned char button_mask)
{
  pointerEvent(0, x, y, button_mask);
}

void wncSource::getSize(unsigned int *width, unsigned int *height)
{
  *width = _width;
  *height = _height;
}
