/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *  Henry Precheur <henry at precheur dot org>
 *  Cedric Pinson <cpinson@freesheep.org>
 */
/*
 *
 * Copyright (C) Nicolas Roussel
 * Copyright (C) 2003 Olivier Chapuis
 *
 * See the file LICENSE for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * Note: LICENSE is LGPL
 *
 */

#define USE_NEW_WNC_WINDOW
// #define DEBUG_NEW_WNC_WINDOW

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <cassert>
#include <maf/wnc_source.h>
#include <maf/wnc_desktop.h>
#include <maf/wnc_window.h>
#include <maf/maferror.h>

#include <osg/Image>
#include <osg/Texture2D>
#include <osg/BlendFunc>
#include <osg/Material>
#include <osg/Geometry>
#include <osg/Drawable>
#include <osg/Geode>
#include <osgDB/WriteFile>

#endif

/******************************************************************************
 * Texture & Sub Texture Loading.
 ******************************************************************************/

#ifdef USE_NEW_WNC_WINDOW

#define MAX_SIZE_PIXEL 4

template <class T> T Min(T val1,T val2)
{
  if (val1<val2)
    return val1;
  return val2;
}

template <class T> T Max(T val1,T val2) 
{
  if (val1<val2)
    return val2;
  return val1;
}


void* GetSubImage(int x,int y,int width,int height,int orignalSizeW,int orignalSizeH,int bytePerPixel,unsigned char* data)
{
  assert(data);
  int rowSize=orignalSizeW*bytePerPixel;
  int rowSize2=width*bytePerPixel;
  unsigned char* ndata=new unsigned char [width*height*bytePerPixel];
  memset(ndata,-1,width*height*bytePerPixel);

  height=Min(height,orignalSizeH-y);
  width=Min(width,orignalSizeW-x);

  width*=bytePerPixel;
  x*=bytePerPixel;
  for (int i=0;i<height;i++)
    memcpy(ndata+i*rowSize2,data+rowSize*(y+i)+x,width);

  return ndata;
}



void XwncRegionWindow::Init(int x,int y,int width, int height,int textureSizeX,int textureSizeY,int yGL,const osg::Vec4& color)
{
  mGeom=0;
  mGeom=new osg::Geometry;
  osg::Vec3Array* vertexes=new osg::Vec3Array;
  osg::Vec2Array* uvs=new osg::Vec2Array;
  mCallback=new TextureSubloadCallback(&mImages);
  mImages.clear();
  vertexes->push_back(osg::Vec3(x,yGL+height,0));
  vertexes->push_back(osg::Vec3(x,yGL,0));
  vertexes->push_back(osg::Vec3(x+width,yGL,0));
  vertexes->push_back(osg::Vec3(x+width,yGL+height,0));
  uvs->push_back(osg::Vec2(0,0));
  float u,v;
   u=width*1.0/textureSizeX;
   v=height*1.0/textureSizeY;
  uvs->push_back(osg::Vec2(0,v));
  uvs->push_back(osg::Vec2(u,v));
  uvs->push_back(osg::Vec2(u,0));
  mGeom->setVertexArray(vertexes);
  mGeom->setTexCoordArray(0,uvs);


  

  osg::Vec4Array* colours = new osg::Vec4Array;
  // DEBUG if you want to debug texture block put color2 instead of color
#ifdef DEBUG_NEW_WNC_WINDOW
  float _red=(255.0*rand()/(RAND_MAX+1.0))/255.0;
  float _green=(255.0*rand()/(RAND_MAX+1.0))/255.0;
  float _blue=(255.0*rand()/(RAND_MAX+1.0))/255.0;
  osg::Vec4 color2(_red,_green,_blue,1);
  colours->push_back(color2);
#else
  colours->push_back(color);
#endif
  mGeom->setColorArray(colours);
  mGeom->setColorBinding(osg::Geometry::BIND_OVERALL);
  mGeom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::QUADS,0,4));
  

  mTextureSizeX=textureSizeX;
  mTextureSizeY=textureSizeY;
  mPosX=x;
  mPosY=y;
  mSizeX=width;
  mSizeY=height;
}


void XwncRegionWindow::DecreaseX(int size)
{
  osg::Vec3Array* vertexes=dynamic_cast<osg::Vec3Array*>(mGeom->getVertexArray());
  osg::Vec2Array* uvs=dynamic_cast<osg::Vec2Array*>(mGeom->getTexCoordArray(0));
  (*vertexes)[2][0]=size;
  (*vertexes)[3][0]=size;
  (*uvs)[2][0]=size*1.0/mTextureSizeX;
  (*uvs)[3][0]=size*1.0/mTextureSizeX;
  mSizeX=size;
}


void XwncRegionWindow::DecreaseY(int size)
{
  osg::Vec3Array* vertexes=dynamic_cast<osg::Vec3Array*>(mGeom->getVertexArray());
  osg::Vec2Array* uvs=dynamic_cast<osg::Vec2Array*>(mGeom->getTexCoordArray(0));
  int offset=mSizeY-size;
  (*vertexes)[1][1]+=offset;
  (*vertexes)[2][1]+=offset;
  (*uvs)[2][0]=size*1.0/mTextureSizeY;
  (*uvs)[3][0]=size*1.0/mTextureSizeY;
  mSizeY=size;
}


XwncRegionWindow::~XwncRegionWindow()
{
  for(std::vector<TextureSubloadCallback::SubImage*>::iterator i = mImages.begin();
      i != mImages.end();
      i++)
    delete *i;
}


void XwncRegionWindow::UpdateFromImage(osg::Image* _sub_image,int x,int y,int width,int height)
{
  osg::Image* image =_sub_image;

  int newx,newy,newsx,newsy,offsetx,offsety;
  // new coord relative to this region, and fix size of texture if needed
  newx=x-mPosX;
  newy=y-mPosY;

  newsx=width;
  newsy=height;

  offsetx=0;
  offsety=0;

  // check if corner is negative
  if (newx<0) {
    offsetx=mPosX-x;
    newsx+=newx;
    newx=0;
  }
  if (newy<0) {
    offsety=mPosY-y;
    newsy+=newy;
    newy=0;
  }

  // check if size is in region
  if (newx+newsx>mTextureSizeX) {
    newsx=mTextureSizeX-newx;
  }

  if (newy+newsy>mTextureSizeY) {
    newsy=mTextureSizeY-newy;
  }

#ifdef DEBUG_NEW_WNC_WINDOW
  g_debug("region %dx%d %dx%d data %dx%d %dx%d -> %dx%d %dx%d offset %dx%d",mPosX,mPosY,mTextureSizeX,mTextureSizeY,x,y,width,height,newx,newy,newsx,newsy,offsetx,offsety);
#endif


  // init texture if not initialised
  if (!mTexture.get()) {
    mTexture=new osg::Texture2D;
    mTexture->setInternalFormat(_sub_image->getInternalTextureFormat());
    mTexture->setTextureSize(mTextureSizeX, mTextureSizeY);
    mTexture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
    mTexture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
    mTexture->setSubloadCallback(mCallback.get());

    // set up the drawstate.
    osg::StateSet* dstate = new osg::StateSet;
#if 0
    osg::BlendFunc* tenv = new osg::BlendFunc(osg::BlendFunc::SRC_ALPHA,osg::BlendFunc::ONE_MINUS_SRC_ALPHA);
    dstate->setAttributeAndModes(tenv, osg::StateAttribute::ON);
#endif
    dstate->setMode(GL_CULL_FACE,osg::StateAttribute::OFF);
    dstate->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
    dstate->setTextureAttributeAndModes(0, mTexture.get(), osg::StateAttribute::ON);

    // set up the geoset.
    mGeom->setStateSet(dstate);

    offsetx=Max(0,(int)mPosX-(int)x);
    offsety=Max(0,(int)mPosY-(int)y);
    newsx=mTextureSizeX;
    newsy=mTextureSizeY;
  }

  // now if have a coord and size taht fit the region
  image = new osg::Image;
  int sizePixel=_sub_image->computePixelSizeInBits(_sub_image->getPixelFormat(),_sub_image->getDataType())/8;
  void* ndata=GetSubImage(offsetx,offsety,newsx,newsy,_sub_image->s(),_sub_image->t(),sizePixel,_sub_image->data());
  image->setImage(newsx,
		  newsy,
		  1,
		  _sub_image->getInternalTextureFormat(),
		  _sub_image->getPixelFormat(),
		  _sub_image->getDataType(),
		  (unsigned char*)ndata,
		  osg::Image::USE_NEW_DELETE);
  
  _sub_image=image;
  x=newx;
  y=newy;

  mImages.push_back(new TextureSubloadCallback::SubImage(_sub_image, x, y,this));
}





static int GetMaxTextureSize()
{
//  return 128;
  return osg::Texture::getExtensions(0,true)->maxTextureSize();
}



XwncGenericWindow::~XwncGenericWindow()
{
  Kill();
}


void XwncGenericWindow::GetTextureSize(int& width,int& height)
{
  if (mCols.empty()) {
    width=0;
    height=0;
    return;
  }

  int i=mCols[0].size();
  //  width=mCols[0][i-1]->mPosX + mCols[0][i-1]->mTextureSizeX;
  width=mCols[0][i-1]->mPosX + mCols[0][i-1]->mSizeX;
  i=mCols.size();
  height=mCols[i-1][0]->mPosY+mCols[i-1][0]->mSizeY;
#ifdef DEBUG_NEW_WNC_WINDOW
  g_debug("Size of Generic window %dx%d",width,height);
#endif
}

void XwncGenericWindow::ResizeWindowSmaller(int width,int height)
{
  // we only need to adjust value
  int endj=(int)mCols.size();
  for (int j=0;j<endj;j++) {
    int endi=(int)mCols[j].size();
    for (int i=0;i<endi;i++) {
      if (mCols[j][i]->mPosX>width)
	removeDrawable(mCols[j][i]->GetGeometry());
      else if (mCols[j][i]->mPosY>height)
	removeDrawable(mCols[j][i]->GetGeometry());
      else {
	if (mCols[j][i]->mPosX<width && (mCols[j][i]->mPosX+mCols[j][i]->mSizeX) > width )
	  mCols[j][i]->DecreaseX(width-mCols[j][i]->mPosX);
	if (mCols[j][i]->mPosY<height && (mCols[j][i]->mPosY+mCols[j][i]->mSizeY) > height )
	  mCols[j][i]->DecreaseY(height-mCols[j][i]->mPosY);
      }
    }
  }

}



XwncGenericWindow::RowXwncRegionWindow XwncGenericWindow::BuildRow(int starty,
								   int width, 
								   int height,
								   int textureSizeY,
								   int yGL,
								   const osg::Vec4& color)
{

  RowXwncRegionWindow row;
  int maxSize=GetMaxTextureSize();
  float w=width*1.0/maxSize;
  int nbMaxSubWindows=(int)ceil(w);
  int nbMinSubWindows=(int)floor(w);
    
  int startx=0;

  // init all subwindows with max texture size for x
  row.resize(nbMaxSubWindows);
  for (int i=0;i<nbMinSubWindows;i++) {
    row[i]=new XwncRegionWindow;
    row[i]->Init(startx,starty,maxSize,height,maxSize,textureSizeY,yGL,color);
#ifdef DEBUG_NEW_WNC_WINDOW
    g_debug("Region %d start %dx%d size %dx%d",i,startx,starty,maxSize,height);
#endif
    startx+=maxSize;
    addDrawable(row[i]->GetGeometry());
  }

  if (nbMaxSubWindows!=nbMinSubWindows) {
    int sizeWindowsRest=(int)ceil(maxSize*(w-nbMinSubWindows));
    sizeWindowsRest = osg::Image::computeNearestPowerOfTwo(sizeWindowsRest, 1.0);
    row[nbMaxSubWindows-1]=new XwncRegionWindow;
    row[nbMaxSubWindows-1]->Init(startx,starty,width-startx,height,sizeWindowsRest,textureSizeY,yGL,color);
#ifdef DEBUG_NEW_WNC_WINDOW
    g_debug("Region %d start %dx%d size %dx%d",nbMaxSubWindows,startx,starty,width-startx,height);
#endif
    addDrawable(row[nbMaxSubWindows-1]->GetGeometry());
  }
  return row;
}

void XwncGenericWindow::Kill()
{
  while (!mCols.empty()) {
    while (!mCols.back().empty()) {
      delete mCols.back().back();
      mCols.back().pop_back();
    }
    mCols.pop_back();
  }
}


void XwncGenericWindow::Init(int width,int height) 
{

  Kill();
  removeDrawable(0,getNumDrawables());

  int maxSize=GetMaxTextureSize();
  float h=height*1.0/maxSize;
  int nbMaxSubWindows=(int)ceil(h);
  int nbMinSubWindows=(int)floor(h);

  int starty=0;
  osg::Vec4 color(1,1,1,1);

  // init all subwindows with max texture size for y
  mCols.resize(nbMaxSubWindows);


  int sizeWindowsRestBase=(int)ceil(maxSize*(h-nbMinSubWindows));
  int sizeWindowsRest = (int)osg::Image::computeNearestPowerOfTwo(sizeWindowsRestBase, 1.0);

  for (int i=0;i<nbMinSubWindows;i++) {
    int yvalue=height-maxSize-i*maxSize;
#ifdef DEBUG_NEW_WNC_WINDOW
    g_debug("y value %d",yvalue);
#endif
    mCols[i]=BuildRow(starty,width,maxSize,maxSize,yvalue,color);
    starty+=maxSize;
  }

  if (nbMaxSubWindows!=nbMinSubWindows) {
    mCols[nbMaxSubWindows-1]=BuildRow(starty,width,height-starty,sizeWindowsRest,0,color);
  }


  setName("wncWindow");
}


void XwncGenericWindow::DispatchImageUpdate(osg::Image* _sub_image, int x, int y,int w, int h)
{
  int endj=(int)mCols.size();
  for (int j=0;j<endj;j++) {
    assert(!mCols[j].empty());
    if (y+h<mCols[j][0]->mPosY) // that's enough not more region to update
      return;
    
    // it's not a region to update so go to next row
    if ((mCols[j][0]->mPosY+mCols[j][0]->mTextureSizeY)<y)
      continue;
    
    int endi=(int)mCols[j].size();
    for (int i=0;i<endi;i++) {
      if (mCols[j][i]->mPosX>(x+w))
	break;
      //	continue;
      
      if ((mCols[j][i]->mPosX+mCols[j][i]->mTextureSizeX)<x)
	continue;
      //	break;

      mCols[j][i]->UpdateFromImage(_sub_image,x,y,w,h);
    }
  }
}


#endif


#ifdef USE_NEW_WNC_WINDOW
TextureSubloadCallback::SubImage::SubImage(osg::Image* image, int x, int y,XwncRegionWindow* region, bool is_complete_texture)
  : mImage(image), mX(x), mY(y), mRegion(region),mIsCompleteTexture(is_complete_texture)
#else
TextureSubloadCallback::SubImage::SubImage(osg::Image* image, int x, int y, bool is_complete_texture)
  : mImage(image), mX(x), mY(y), mIsCompleteTexture(is_complete_texture)
#endif
{
}

TextureSubloadCallback::SubImage::~SubImage()
{
  mImage = 0;
}

TextureSubloadCallback::TextureSubloadCallback(std::vector<SubImage*>* images)
{
  mImages = images;
}

TextureSubloadCallback::TextureSubloadCallback(unsigned long id,std::vector<SubImage*>* images)
{
  mImages = images;
  _frameID = id;
}

void TextureSubloadCallback::load(const osg::Texture2D& texture, osg::State& state) const
{
  SubImage* si = *(mImages->begin());

#ifdef DEBUG_NEW_WNC_WINDOW
  g_debug("TextureSubloadCallback::load %X %dx%d",this,si->mImage->s(),si->mImage->t());
#endif
  texture.applyTexImage2D_load(state, GL_TEXTURE_2D, si->mImage.get(),si->mImage->s(), si->mImage->t(), 0);

  GLenum error = glGetError();
  if (error != GL_NO_ERROR)
    g_warning("GL error in %s 0x%X", __FUNCTION__, error);
  delete si;

  
  std::vector<SubImage*>::iterator it = mImages->begin();
  mImages->erase(it);
  subload(texture, state);
}

void TextureSubloadCallback::subload(const osg::Texture2D& texture, osg::State& state) const 
{
  for (std::vector<SubImage*>::iterator it = mImages->begin(); it != mImages->end(); it++) {
    SubImage* si = *it;

    if ((*it)->mIsCompleteTexture) {
      // special case to handle reconfigured window
#ifdef DEBUG_NEW_WNC_WINDOW
      g_debug("TextureSubloadCallback::subload special case %X %dx%d",this,si->mImage->s(),si->mImage->t());
      assert(0);
#endif
      texture.applyTexImage2D_load(state, GL_TEXTURE_2D, (*it)->mImage.get(),(*it)->mImage->s(), (*it)->mImage->t(), 0);

    } else {

      int x,y;
      x=si->mX;
      y=si->mY;
#ifdef DEBUG_NEW_WNC_WINDOW
      g_debug("TextureSubloadCallback::subload texsubimage %X %dx%d (%dx%d) ",this,x,y,si->mImage->s(),si->mImage->t());
#endif
      glTexSubImage2D(GL_TEXTURE_2D, 0,
		      x,y,
		      (*it)->mImage->s(), (*it)->mImage->t(),
		      (*it)->mImage->getPixelFormat(), (*it)->mImage->getDataType(),
		      (*it)->mImage->data());

    }
     
    GLenum error = glGetError();
    if (error != GL_NO_ERROR)
      g_warning("GL error in %s 0x%X window: %lX %d %d (%dx%d) format: %d %d",
		__FUNCTION__, error,
		_frameID,
		(*it)->mX, (*it)->mY,
		(*it)->mImage->s(), (*it)->mImage->t(),
		(*it)->mImage->getPixelFormat(), texture.getInternalFormat());
    delete *it;
  }
  mImages->clear();
}

/******************************************************************************
 * Texture & Sub Texture Loading.
 ******************************************************************************/
XwncWindow::XwncWindow(std::string title, unsigned long id, wncSource *wncServer,int x, int y, unsigned int width, unsigned int height)
{
  _title = title;
  mWncServer = wncServer;
  _real_x = x;
  _real_y = y;
  x = y = 0;
  _width = width;
  _height = height;
  _tex_width = _tex_height = 0;
  _frameID = id;

#ifndef USE_NEW_WNC_WINDOW
  _texture = 0;
  _geom = new osg::Geometry;
  _vertex = new osg::Vec3Array(4);
  _texcoord = new osg::Vec2Array(4);

  g_assert(_images.begin() == _images.end());
  _callback = new TextureSubloadCallback(id, &_images);
#else
  mWindow=new XwncGenericWindow;
  addChild(mWindow.get());
  //  setBgcolor(1, 1, 1, 0.8);
  mNotInitialised=true;
#endif
  _resetTexture = false;
//   _mapped = false;
//   _neverMapped = true;
//   _changed = true;
  
  _red = _green = _blue = _alpha = _alpha_saved = 1.0;
//   _unmanaged = false;
  mIsRootWindow = false;
//   _isEwmhDesktop = false;
  _transientFor = 0;
  _hide = 0;

  _layerCorrection = 0;
  _desktopWidth = 0;
  _desktopHeight = 0;
  _eyeDist = 0;
}

XwncWindow::~XwncWindow()
{
#ifndef USE_NEW_WNC_WINDOW
  for(std::vector<TextureSubloadCallback::SubImage*>::iterator i = _images.begin();
      i != _images.end();
      i++)
    delete *i;
#endif
}

// --------------------------------------------------------------

void XwncWindow::setupVertex()
{
  unsigned int sW, sH;
  mWncServer->getSize(&sW, &sH);

  float	x = _real_x;
  float	y = sH - _real_y - _height;
  
  x += mDisplayPosition.x();
  y += mDisplayPosition.y();

#ifndef USE_NEW_WNC_WINDOW
  float	w = _width;
  float	h = _height;

  osg::Vec3	v = osg::Vec3(x, y, 0);
  (*_vertex)[0] = v + osg::Vec3(0, h, 0);
  (*_vertex)[1] = v + osg::Vec3(0, 0, 0);
  (*_vertex)[2] = v + osg::Vec3(w, 0, 0);
  (*_vertex)[3] = v + osg::Vec3(w, h, 0);

  _geom->setVertexArray(_vertex.get());
#else
  setMatrix(osg::Matrix::translate(x,y,0));
  // need here to setup vertexes for size
#endif
}

void XwncWindow::setupTexCoord()
{
#ifndef USE_NEW_WNC_WINDOW
  float	u = _width / _tex_width;
  float	v = _height / _tex_height;

  (*_texcoord)[0] = osg::Vec2(0, 0);
  (*_texcoord)[1] = osg::Vec2(0, v);
  (*_texcoord)[2] = osg::Vec2(u, v);
  (*_texcoord)[3] = osg::Vec2(u, 0);

  _geom->setTexCoordArray(0, _texcoord.get());
#else
  // need to adjust tex coord here
#endif

}

const std::string& XwncWindow::GetTitle() const
{
  return _title ;
}

// --------------------------------------------------------------
void XwncWindow::configure(int x, int y, int width, int height)
{
#ifdef DEBUG_NEW_WNC_WINDOW
  g_debug("configure %dx%d %dx%d",x,y,width,height);
#endif

#ifdef DEBUG_TEST 
  int px,py;
  px=_width;
  py=_height;
#endif
  _real_x = static_cast<float>(x);
  _real_y = static_cast<float>(y);
  _width = static_cast<float>(width);
  _height = static_cast<float>(height);

#ifdef DEBUG_TEST 
  if (px < width || py < height) {
#else
  if (_tex_width < width || _tex_height < height) {
#endif
    _resetTexture = true;
  } else if (getNumChildren()) {
    setupVertex();
    setupTexCoord();
  }

}

void XwncWindow::getSize(float *width, float *height) const
{
  *width = (float)_width ;
  *height = (float)_height ;
}

void XwncWindow::getSize(double *width, double *height) const
{
  *width = static_cast<double>(_width);
  *height = static_cast<double>(_height);
}

void XwncWindow::getSize(int *width, int *height) const
{
  *width = (int)_width ;
  *height = (int)_height ;
}

float XwncWindow::GetWidth() const
{
  return _width;
}

float XwncWindow::GetHeight() const
{
  return _height;
}

const osg::Vec3f	XwncWindow::GetSize() const
{
  return osg::Vec3f(_width, _height, 0);
}

void XwncWindow::getRealPosition(float *x, float *y)
{
  *x = _real_x;
  *y = _real_y;
}

void XwncWindow::getRealPosition(int *x, int *y)
{
  *x = (int)_real_x;
  *y = (int)_real_y;
}

void XwncWindow::getPosition(float *x, float *y, float screen_height)
{
  *x = _real_x;
  *y = screen_height - (_real_y + _height);
}

void XwncWindow::getPosition(int *x, int *y, int screen_height)
{
  *x = (int)_real_x;
  *y = (int)(screen_height - (_real_y + _height));
}

void XwncWindow::setMapped(bool mapped)
{
  _mapped = mapped;
}

bool XwncWindow::IsMapped() const
{
  return _mapped;
}

void XwncWindow::setRootWindow(void)
{
  mIsRootWindow = true;
}

bool XwncWindow::isRootWindow(void)
{
  return mIsRootWindow;
}

void XwncWindow::setTransientFor(unsigned long w)
{
  _transientFor = w;
}

unsigned long  XwncWindow::getTransientFor(void)
{
  return _transientFor;
}

void XwncWindow::hide(void)
{
  _hide = true;
}

unsigned long XwncWindow::getFrameID(void)
{
  return _frameID;
}

// --------------------------------------------------------------

void XwncWindow::setBgcolor(float red, float green, float blue, float alpha)
{
  _red = red ;
  _green = green ;
  _blue = blue ;
  _alpha = alpha ;
}

void XwncWindow::setAlpha(float alpha)
{
  _alpha = alpha ;
}

void XwncWindow::setTmpAlpha(float alpha)
{
  _alpha_saved = _alpha;
  _alpha = alpha ;
}

void XwncWindow::resetAlpha(void)
{
  _alpha = _alpha_saved;
}

void XwncWindow::getBgcolor(float *red, float *green, float *blue, float *alpha)
{
  *red = _red ;
  *green = _green ;
  *blue = _blue ;
  *alpha = _alpha ;
}

float XwncWindow::getAlpha(void)
{
  return _alpha ;
}

// --------------------------------------------------------------
// Drawing stuff

void XwncWindow::resetTexture(void)
{
  _resetTexture = true;
}

void XwncWindow::deleteTexture(void)
{
#ifndef USE_NEW_WNC_WINDOW
  _texture = 0;
#endif
}

static  void image2osgimage(osg::Image *to_img, WncImage* from_img) {
  GLenum format ;
  GLint internalformat ;
  GLint alignment = 4 ;
  GLenum type ;

  switch (from_img->getEncoding()) {
  case WncImage::L:
    format = GL_LUMINANCE ;
    internalformat = GL_LUMINANCE ;
    alignment = 1 ;
    type = GL_UNSIGNED_BYTE ;
    break ;
  case WncImage::RGB:
    format = GL_RGB ;
    internalformat = GL_RGB ;
    alignment = 1 ;
    type = GL_UNSIGNED_BYTE ;
    break ;
  case WncImage::ABGR:
#ifdef GL_ABGR_EXT
    format = GL_ABGR_EXT ;
    internalformat = GL_ABGR_EXT ;
#else
    g_warning("glSetupTextureImage: GL_ABGR_EXT undefined, using GL_RGBA instead...");
    format = GL_RGBA ;
    internalformat = GL_RGBA ;
#endif
    type = GL_UNSIGNED_BYTE ;
    break ;
  case WncImage::RGBA:
    format = GL_RGBA ;
    internalformat = GL_RGBA ;
    type = GL_UNSIGNED_BYTE ;
    break ;
  case WncImage::ARGB:
#ifdef GL_BGRA_EXT
    format = GL_BGRA_EXT ;
    type = GL_UNSIGNED_INT_8_8_8_8_REV ;
#else
    g_warning("glSetupTextureImage: GL_BGRA_EXT undefined, using GL_RGBA instead...");
    format = GL_RGBA ;
    type = GL_UNSIGNED_BYTE ;
#endif
    internalformat = GL_RGBA ;
    break ;
  default:
    g_warning("Bad image encoding for glSetupTextureImage (%s)",
	      WncImage::EncodingName(from_img->getEncoding()));
    return ;
  }


  // cpinson: now we don't need to make a copy of image we can use the buffer because we will need to make a subcopy later

  // 1 =>
  // /usr/local/src/openscenegraph/OSG_OP_OT-0.9.6-2/OpenSceneGraph-0.9.6-2/src/osg/Image.cpp
  unsigned char*	new_buffer = (unsigned char*)malloc(from_img->getSize());

  // FIXME: dirty hack to make 0xff00ff color transparent
  // this is a workaround for xwnc server bug
  g_assert(from_img->getEncoding() == WncImage::RGBA);
  for (unsigned char* p = from_img->getData();
       p < from_img->getData() + from_img->getSize();
       p += 4)
    {
      if (p[0] == 0xff &&
	  p[1] == 0x00 &&
	  p[2] == 0xff)
	p[3] = 0x00;
      else
	p[3] = 0xff;
    }

  memcpy(new_buffer, from_img->getData(), from_img->getSize());
  to_img->setImage(from_img->getWidth(), from_img->getHeight(), 1,
		   internalformat, format, type, new_buffer,
		   osg::Image::USE_MALLOC_FREE, alignment);
}

#ifndef USE_NEW_WNC_WINDOW

void XwncWindow::updateTexture(WncImage *img, int x, int y,unsigned int w, unsigned int h)
{
  osg::ref_ptr<osg::Image> _sub_image = new osg::Image;
  image2osgimage(_sub_image.get(), img);

  if (_texture == 0 || _resetTexture) {
    // flush images left
    for(std::vector<TextureSubloadCallback::SubImage*>::iterator i = _images.begin();
	i != _images.end();
	i++)
      delete *i;
    _images.clear();

    osg::Image*	image = new osg::Image;

    int width = osg::Image::computeNearestPowerOfTwo(_sub_image->s(), 1.0);
    int height = osg::Image::computeNearestPowerOfTwo(_sub_image->t(), 1.0);

    image->allocateImage(width, height, 1, _sub_image->getPixelFormat(), _sub_image->getDataType());
    image->setInternalTextureFormat(_sub_image->getInternalTextureFormat());
    image->copySubImage(0, 0, 0, _sub_image.get());
    _tex_width = width;
    _tex_height = height;

    if(_texture == 0) {
      // if the texture was not existing, there is not reason to
      // reset it
      _resetTexture = false;

      _texture = new osg::Texture2D;
      _texture->setInternalFormat(image->getInternalTextureFormat());
      _texture->setTextureSize(width, height);
      _texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
      _texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
      _texture->setSubloadCallback(_callback.get());
      _images.push_back(new TextureSubloadCallback::SubImage(image, x, y));

      // setBgcolor(1, 1, 1, 0.8); // FIXME
      // set up the drawstate.
      osg::StateSet*	dstate = new osg::StateSet;
#if 0
      osg::BlendFunc*	tenv = new osg::BlendFunc(osg::BlendFunc::SRC_ALPHA,
						  osg::BlendFunc::ONE_MINUS_SRC_ALPHA);
      dstate->setAttributeAndModes(tenv, osg::StateAttribute::ON);
#endif
      // dstate->setAttributeAndModes(color);
      dstate->setMode(GL_CULL_FACE,osg::StateAttribute::OFF);
      dstate->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
      dstate->setTextureAttributeAndModes(0, _texture.get(), osg::StateAttribute::ON);

      // set up the geoset.
      // osg::Geometry* geom = new osg::Geometry;
      _geom->setStateSet(dstate);

//       osg::Vec4Array* colours = new osg::Vec4Array(1);
//      (*colours)[0].set(_red, _green, _blue, 1.0); // _alpha);
//       _geom->setColorArray(colours);
//       _geom->setColorBinding(osg::Geometry::BIND_OVERALL);

      _geom->addPrimitiveSet(new osg::DrawArrays(osg::PrimitiveSet::QUADS,0,4));

      // set up the geode.
      osg::Geode*	geode = new osg::Geode;
      geode->setName("wncWindow");

      geode->addDrawable(_geom.get());

      addChild(geode);
    } else {
      if (_resetTexture) {
	_texture->setInternalFormat(image->getInternalTextureFormat());
	_texture->setTextureSize(width, height);
	_images.push_back(new TextureSubloadCallback::SubImage(image, x, y, true));
	_resetTexture = false;
      } else
	g_assert_not_reached();
    }
    setupVertex();
    setupTexCoord();
  } else {
    _images.push_back(new TextureSubloadCallback::SubImage(_sub_image.get(), x, y));
  }
}
#else


void XwncWindow::updateTexture(WncImage *img, int x, int y,unsigned int w, unsigned int h)
{
#ifdef DEBUG_NEW_WNC_WINDOW
  g_debug("update windows %dx%d %dx%d reset texture %d",x,y,w,h,_resetTexture);
  if (x==217 && y==27 && w==12 && h==224) {
    g_debug("my case");
  }
#endif
  osg::ref_ptr<osg::Image> _sub_image = new osg::Image;
  image2osgimage(_sub_image.get(), img);

  bool rebuild=_resetTexture || mNotInitialised;

  if (rebuild) {
    //    mWindow->Init(_sub_image->s(),_sub_image->t());
    mWindow->Init((int)_width,(int)_height);
    _tex_height=_height;
    _tex_width=_width;
    _resetTexture = false;
    g_debug("%dx%d",_sub_image->s(),_sub_image->t());
  }
  mWindow->DispatchImageUpdate(_sub_image.get(), x, y, w, h);
  mNotInitialised=false;
  setupVertex();
}

#endif


// --------------------------------------------------------------


void XwncWindow::keyEvent(unsigned long key, bool down_flag)
{
  mWncServer->keyEvent(key, down_flag) ;
}

void XwncWindow::pointerEvent(int x, int y, unsigned char button_mask)
{
  int	xrel = static_cast<int>(x - mDisplayPosition.x());
  int	yrel = static_cast<int>(y - mDisplayPosition.y());
  mWncServer->pointerEvent(getFrameID(), xrel, yrel, button_mask);
}

void XwncWindow::Slide(const osg::Vec2& offset)
{
  mDisplayPosition = offset;
  setupVertex();
}

osg::MatrixTransform* XwncWindow::staticCopy()
{
  osg::StateSet* state;

  osg::MatrixTransform* matrix = new osg::MatrixTransform(*this);
  state = getStateSet();
  if(state) matrix->setStateSet(state);

  osg::Geode* geode = new osg::Geode;
  state = mWindow->getStateSet();
  if(state) geode->setStateSet(state);
  matrix->addChild(geode);

  for(unsigned int i = 0; i < mWindow->getNumDrawables(); i++) {
    geode->addDrawable(mWindow->getDrawable(i));
  }

  return matrix;
}
