/* -*- c++ -*-
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "ugameStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#include <iostream>

#include <libxml/xmlreader.h>

#include <osgDB/ReadFile>
#include <osg/Notify>
#include <osg/Material>
#include <osg/Texture2D>
#include <osg/BlendFunc>
#include <osg/Depth>
#include <osg/NodeVisitor>
#include <osg/CullStack>

#include <osgchips/Stacks>

using namespace osgchips;

//
// Serialization helpers
//

static bool readColorFromXml(xmlTextReaderPtr reader, osg::Vec4& color) {
  bool status = false;
  xmlChar* red = xmlTextReaderGetAttribute(reader, (const xmlChar*)"red");
  if(red) {
    color.x() = atoi((const char*)red) / 255.f;
    xmlFree(red);
    status = true;
  }
  xmlChar* green = xmlTextReaderGetAttribute(reader, (const xmlChar*)"green");
  if(green) {
    color.y() = atoi((const char*)green) / 255.f;
    xmlFree(green);
    status = true;
  }
  xmlChar* blue = xmlTextReaderGetAttribute(reader, (const xmlChar*)"blue");
  if(blue) {
    color.z() = atoi((const char*)blue) / 255.f;
    xmlFree(blue);
    status = true;
  }
  xmlChar* alpha = xmlTextReaderGetAttribute(reader, (const xmlChar*)"alpha");
  if(alpha) {
    color.w() = atof((const char*)alpha);
    xmlFree(alpha);
    status = true;
  } else {
    color.w() = 1.f;
  }
  return status;
}

static bool readVec3FromXml(xmlTextReaderPtr reader, osg::Vec3& vec) {
  xmlChar* x = xmlTextReaderGetAttribute(reader, (const xmlChar*)"x");
  if(x) { vec.x() = atof((const char*)x); xmlFree(x); }
  xmlChar* y = xmlTextReaderGetAttribute(reader, (const xmlChar*)"y");
  if(y) { vec.y() = atof((const char*)y); xmlFree(y); }
  xmlChar* z = xmlTextReaderGetAttribute(reader, (const xmlChar*)"z");
  if(z) { vec.z() = atof((const char*)z); xmlFree(z); }
  return true;
}

static bool readMaterialFromXml(xmlTextReaderPtr reader, std::map<std::string, osg::Vec4>& name2material) {
  const char* fileName = xmlTextReaderCurrentDoc(reader)->name;
  xmlChar* name = xmlTextReaderGetAttribute(reader, (const xmlChar*)"name");
  if(!name) {
    osg::notify(osg::WARN) << fileName << ": materials must have a name attribute" << std::endl;
    return false;
  }

  osg::Vec4 color;
  if(readColorFromXml(reader, color))
    name2material[(const char*)name] = color;

  xmlFree(name);
  return true;
}

static bool readosgchipsSubTreeFromXml(xmlTextReaderPtr reader, ChipBank* chipBank, Stacks* stacks, osgDB::Registry* registry) {
  if(!chipBank) chipBank = ChipBank::instance();
  std::map<std::string, osg::Vec4> name2material;

  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT &&
       (!strcmp("osgchips", (const char*)xmlTextReaderConstName(reader))))
      return true;
    const char* name = (const char*)xmlTextReaderConstName(reader);
    const char* fileName = xmlTextReaderCurrentDoc(reader)->name;
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_ELEMENT) {

      if(!strcmp("material", name)) {
	if(!readMaterialFromXml(reader, name2material))
	  break;
      } else if(!strcmp("chip", name)) {
	xmlChar* value = xmlTextReaderGetAttribute(reader, (const xmlChar*)"value");
	if(!value) {
	  osg::notify(osg::WARN) << fileName << ": missing value attribute" << std::endl;
	  continue;
	}

	xmlChar* name = xmlTextReaderGetAttribute(reader, (const xmlChar*)"name");
	if(!name) {
	  xmlFree(value);
	  osg::notify(osg::WARN) << fileName << ": missing name attribute" << std::endl;
	  continue;
	}

	ChipBank::Chip* chip = new ChipBank::Chip((const char*)name, atoi((const char*)value));
	
	xmlFree(name);
	xmlFree(value);

	//
	// Create the texture from the image, if any
	//
	name = xmlTextReaderGetAttribute(reader, (const xmlChar*)"texture");
	if(name) {
	  osg::Image* image = osgDB::readImageFile((const char*)name, registry->getOptions());
	  xmlFree(name);
	  if(image) chip->setTexture(image);
	} 

	//
	// Retrieve the material from its name, if any
	//
	name = xmlTextReaderGetAttribute(reader, (const xmlChar*)"material");
	if(name) {
	  if(name2material.find((const char*)name) != name2material.end()) {
	    chip->setColor(name2material[(const char*)name]);
	  }
	  xmlFree(name);
	}

	chipBank->addChip(chip);
      } else if(!strcmp("stacks", name)) {
	while((status = xmlTextReaderRead(reader)) == 1) {
	  if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_END_ELEMENT &&
	     !strcmp("stacks", (const char*)xmlTextReaderConstName(reader)))
	    break;

	  if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_ELEMENT) {
	    if(!stacks) {
		osg::notify(osg::WARN) << fileName << ": stacks element found but ignored by caller because the stacks pointer is null " << std::endl;
		continue;
	    }

	    if(!strcmp("stack", (const char*)xmlTextReaderConstName(reader))) {
	      xmlChar* name;
	      name = xmlTextReaderGetAttribute(reader, (const xmlChar*)"chip");
	      if(!name) {
		osg::notify(osg::WARN) << fileName << ": each stack element must have a chip attribute " << std::endl;
		continue;
	      }

	      ChipBank::Chip* chip = chipBank->getChip((const char*)name);
	      xmlFree(name);
	      if(!chip) {
		osg::notify(osg::WARN) << fileName << ": a stack wants to use the unknown chip " << chip->_name << std::endl;
		continue;
	      }
	      
	      Stack* stack = new Stack();
	      stack->setChip(chip);

	      xmlChar* count = xmlTextReaderGetAttribute(reader, (const xmlChar*)"count");
	      if(count) {
		stack->setCount(atoi((const char*)count));
		xmlFree(count);
	      }

	      osg::Vec3 position;
	      if(!readVec3FromXml(reader, position)) {
		osg::notify(osg::WARN) << fileName << ": a failed to read x/y/z for a stack " << std::endl;
		continue;
	      }
	      stack->setPosition(position);

	      stacks->addStack(stack);
	    }
	  }
	}
      }
    }
  }
  return status == 0;
}

static bool readosgchipsFromXml(xmlTextReaderPtr reader, ChipBank* chipBank, Stacks* stacks, osgDB::Registry* registry) {
  int status;
  while((status = xmlTextReaderRead(reader)) == 1) {
    const char* fileName = xmlTextReaderCurrentDoc(reader)->name;
    if(xmlTextReaderNodeType(reader) == XML_READER_TYPE_ELEMENT) {

      if(!strcmp("osgchips", (const char*)xmlTextReaderConstName(reader))) {
	xmlChar* mesh_file = xmlTextReaderGetAttribute(reader, (const xmlChar*)"mesh");
	if(mesh_file) {
	  osg::ref_ptr<osg::Geode> geode = dynamic_cast<osg::Geode*>(osgDB::readNodeFile((const char*)mesh_file));
	  if(geode.valid()) {
	    osg::Geometry* mesh = dynamic_cast<osg::Geometry*>(geode->getDrawable(0));
	    if(mesh) {
	      chipBank->setVertexArray(dynamic_cast<osg::Vec3Array*>(mesh->getVertexArray()));
	      chipBank->setNormalArray(dynamic_cast<osg::Vec3Array*>(mesh->getNormalArray()));
	      chipBank->setTexCoordArray(dynamic_cast<osg::Vec2Array*>(mesh->getTexCoordArray(0)));
	      chipBank->setPrimitiveSetList(mesh->getPrimitiveSetList());
	      chipBank->setBoundingBox(mesh->getBound());
	    } else {
	      osg::notify(osg::WARN) << fileName << ": mesh " << mesh_file << " was expected to contain a single drawable" << std::endl;
	    }
	  } else {
	    osg::notify(osg::WARN) << fileName << ": failed to load mesh " << (const char*)mesh_file << std::endl;
	  }
	  xmlFree(mesh_file);
	}
	readosgchipsSubTreeFromXml(reader, chipBank, stacks, registry);
      }
    }
  }
  return status == 0;
}

//
// osgchips::ChipBank
//

ChipBank::Chip::Chip(const std::string& name, unsigned int value) :
  _name(name),
  _value(value),
  _hasColor(false)
{}

ChipBank::Chip::~Chip() {}


void ChipBank::Chip::setTexture(osg::Image* image) {
  if(image) {
    osg::Texture2D* texture = new osg::Texture2D;
    texture->setImage(image);
    texture->setWrap(osg::Texture2D::WRAP_T, osg::Texture2D::REPEAT);
    texture->setWrap(osg::Texture2D::WRAP_S, osg::Texture2D::REPEAT);
    texture->setFilter(osg::Texture2D::MIN_FILTER, osg::Texture2D::LINEAR_MIPMAP_LINEAR);
    texture->setFilter(osg::Texture2D::MAG_FILTER, osg::Texture2D::LINEAR);
    _texture = texture;
  } else {
    _texture = 0;
  }
}

ChipBank::ChipBank()
{
}

ChipBank::~ChipBank() {
  for(Value2Chip::iterator i = _value2chip.begin(); i != _value2chip.end(); i++)
    delete i->second;
}

ChipBank* ChipBank::instance() {
  static ChipBank* chipBank = new ChipBank();
  return chipBank;
}

bool ChipBank::unserialize(struct _xmlDoc* doc, osgDB::Registry* registry) {
  xmlTextReaderPtr reader = xmlReaderWalker(doc);
  if(reader == NULL) return false;
  bool status = readosgchipsFromXml(reader, this, 0, registry);
  xmlFreeTextReader(reader);
  return status;
}

bool ChipBank::unserialize(const std::string& fileName, osgDB::Registry* registry) {
  xmlTextReaderPtr reader = xmlReaderForFile(fileName.c_str(), NULL, XML_PARSE_PEDANTIC|XML_PARSE_NONET);
  if(reader == NULL) return false;
  bool status = readosgchipsFromXml(reader, this, 0, registry);
  xmlFreeTextReader(reader);
  return status;
}

//
// osgchips::Stack
//

Stack::Stack() :
  _count(1),
  _chip(0),
  _chipBank(ChipBank::instance())
{
  
  setUseDisplayList(false);
  setUseVertexBufferObjects(false);
  setMesh(_chipBank);
}

Stack::Stack(ChipBank* chipBank) :
  _count(0),
  _chip(0),
  _chipBank(chipBank)
{
  if(chipBank) {
    setUseDisplayList(false);
    setUseVertexBufferObjects(false);
    setMesh(chipBank);
  }
}

Stack::Stack(const Stack& stack, const osg::CopyOp& copyop) :
  Geometry(stack, copyop),
  _count(stack._count),
  _position(stack._position),
  _chip(stack._chip),
  _chipBank(stack._chipBank)
{
}

void Stack::setCount(unsigned int count) {
  if(_count != count) {
    _count = count;
    updateVertexArray();
    updateTexCoordArray();
    dirtyParentBound();
  }
}

void Stack::setPosition(const osg::Vec3& position) {
  if(position != _position) {
    _position = position;
    updateVertexArray();
    dirtyParentBound();
  }
}

void Stack::setChip(ChipBank::Chip* chip) {
  _chip = chip;
  if(chip && (chip->_hasColor || chip->_texture.valid())) {
    osg::StateSet* state = getOrCreateStateSet();

    if(chip->_texture.valid()) {
      state->setTextureAttributeAndModes(0, chip->_texture.get(), osg::StateAttribute::ON);
    }

    if(chip->_hasColor) {
      osg::Material* material = new osg::Material;
      material->setColorMode(osg::Material::DIFFUSE);
      material->setDiffuse(osg::Material::FRONT_AND_BACK, chip->_color);
      state->setAttributeAndModes(material);
      if(!osg::equivalent(chip->_color[3], 1.f)) {
	osg::BlendFunc* blendFunc = new osg::BlendFunc();
	blendFunc->setFunction(GL_SRC_ALPHA, GL_ONE);
	state->setAttributeAndModes(blendFunc);
	state->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);

	osg::Depth* depth = new osg::Depth();
	depth->setWriteMask(0);
	state->setAttributeAndModes(depth);
      }
    }
  }
  setCount(0);
}

void Stack::drawImplementation(osg::State& state) const {
  if(_count > 0)
    Geometry::drawImplementation(state);
}

void Stack::setMesh(ChipBank* chipBank) {
  if(chipBank->getTexCoordArray() == 0 ||
     chipBank->getVertexArray() == 0 ||
     chipBank->getNormalArray() == 0 ||
     chipBank->getPrimitiveSetList().size() == 0) {
    osg::notify(osg::WARN) << "osgchips::Stack::setMesh: chipbank has no valid mesh" << std::endl;
  } else {
    setNormalArray(chipBank->getNormalArray());
    setPrimitiveSetList(chipBank->getPrimitiveSetList());
    setTexCoordArray(0, new osg::Vec2Array(*dynamic_cast<osg::Vec2Array*>(chipBank->getTexCoordArray()),
					   osg::CopyOp(osg::CopyOp::DEEP_COPY_ARRAYS)));
    setVertexArray(new osg::Vec3Array(*dynamic_cast<osg::Vec3Array*>(chipBank->getVertexArray()),
				      osg::CopyOp(osg::CopyOp::DEEP_COPY_ARRAYS)));
  }
}

void Stack::updateVertexArray() {
  if(_count == 0 || _chipBank == 0) return;

  if(_chipBank->getVertexArray() && getVertexArray()) {
    //
    // Stretch the mesh up
    //
    osg::Vec3Array* from = _chipBank->getVertexArray();
    osg::Vec3Array* to = dynamic_cast<osg::Vec3Array*>(getVertexArray());
    for(osg::Vec3Array::iterator i = from->begin(), j = to->begin() ;
	i != from->end() && j != to->end();
	i++, j++) {
      osg::Vec3& vertex_from = *i;
      osg::Vec3& vertex_to = *j;
      vertex_to = vertex_from + _position;
      if(!osg::equivalent(vertex_from.y(), _position.y(), 0.1f))
	vertex_to.y() = _position.y() + vertex_from.y() * _count;
    }
  }
}

void Stack::updateTexCoordArray() {
  if(_count == 0) return;

  if(getTexCoordArray(0)) {
    //
    // Wrap the texture of the chip border
    //
    osg::Vec2Array* texCoord = dynamic_cast<osg::Vec2Array*>(getTexCoordArray(0));
    for(osg::Vec2Array::iterator i = texCoord->begin() ; i != texCoord->end() ; i++) {
      osg::Vec2& uv = *i;
      if(uv.x() > 0.505 && (uv.y() > 0.245))
	uv.y() = _count * 0.249;
    }
  }
}

void Stack::dirtyParentBound() {
  for(ParentList::iterator i = _parents.begin(); i != _parents.end(); ++i) {
    (*i)->dirtyBound();
  }
}

//
// osgchips::Stacks
//

class Box : public osg::Drawable {
public:
  META_Object(osgchips, Box)

  Box() {
    _box = new osg::Vec3Array(24);
  }
  Box(const Box& box, const osg::CopyOp& copyop=osg::CopyOp::SHALLOW_COPY) :
    osg::Drawable(box),
    _box(box._box)
  {
    setUseDisplayList(false);
    setUseVertexBufferObjects(false);
  }
    
  virtual void drawImplementation(osg::State&) const {
#if 0 // draw bounding box for debugging purpose
    glColor4f(1.f, 1.f, 1.f, 1.f);
    glBegin(GL_QUADS);
    for(osg::Vec3Array::const_iterator i  = _box->begin(); i != _box->end(); i++) {
      glVertex3fv((float*)i->_v);
    }
    glEnd();
#endif
  }

  virtual bool supports(osg::Drawable::AttributeFunctor&) const { return false; }
  virtual bool supports(osg::Drawable::ConstAttributeFunctor&) const { return true; }
  virtual void accept(osg::Drawable::ConstAttributeFunctor& af) const {}

  virtual bool supports(PrimitiveIndexFunctor&) const { return false; }
  virtual void accept(PrimitiveIndexFunctor&) const {}

  virtual bool supports(osg::Drawable::PrimitiveFunctor&) const { return true; }
  virtual void accept(osg::Drawable::PrimitiveFunctor& functor) const;

  osg::ref_ptr<osg::Vec3Array> _box;
};

void Box::accept(osg::Drawable::PrimitiveFunctor& functor) const {
  for(osg::Drawable::ParentList::const_iterator i = _parents.begin(); i != _parents.end(); i++) {
    (*i)->getBound();
  }

  if(_box.valid()) {
    osg::Vec3Array* box = const_cast<osg::Vec3Array*>(_box.get());
    
    functor.setVertexArray(box->getNumElements(), (osg::Vec3*)(box->getDataPointer()));
    functor.drawArrays(GL_QUADS, 0, box->getNumElements());
  }
}

Stacks::Stacks() :
  _box(new Box),
  _chipBank(ChipBank::instance())
{
  addDrawable(_box.get());
}

Stacks::Stacks(ChipBank* chipBank) :
  _box(new Box),
  _chipBank(chipBank)
{
  if(chipBank == 0)
    _chipBank = ChipBank::instance();
  addDrawable(_box.get());
}

Stacks::Stacks(const Stacks& stacks, const osg::CopyOp& copyop) :
  osg::Geode(stacks, copyop)
{
  if(getNumDrawables() < 1 || dynamic_cast<Box*>(getDrawable(0)) == 0)
    osg::notify(osg::WARN) << "osgchips::Stack::Stacks: first drawable is not of type Box" << std::endl;
  _box = dynamic_cast<Box*>(getDrawable(0));
}

Stacks::~Stacks() {}

bool Stacks::addStack(Stack* stack) {
  return addDrawable(stack);
}

bool Stacks::replaceStack(Stack* origStack, Stack* newStack) {
  return replaceStack(origStack, newStack);
}

bool Stacks::setStack(unsigned int index, Stack* stack) {
  return setDrawable(index, stack);
}

bool Stacks::computeBound() const {
  _bsphere.init();
  _bbox.init();

  const ChipBank* chipBank = _chipBank;
  for(unsigned int i = 0; i < getNumStacks(); i++) {
    const Stack* stack = getStack(i);
    if(stack && stack->getCount() > 0) {
      const osg::Vec3& position = stack->getPosition();
      const osg::BoundingBox& reference = chipBank->getBoundingBox();
      osg::BoundingBox bbox;
      float r = (reference.xMax() - reference.xMin()) / 2.f;
      float h = (reference.yMax() - reference.yMin()) * stack->getCount();
      bbox.expandBy(position + osg::Vec3(-r, 0, -r));
      bbox.expandBy(position + osg::Vec3(-r, 0, r));
      bbox.expandBy(position + osg::Vec3(r, 0, r));
      bbox.expandBy(position + osg::Vec3(r, 0, -r));
      bbox.expandBy(position + osg::Vec3(-r, h, -r));
      bbox.expandBy(position + osg::Vec3(-r, h, r));
      bbox.expandBy(position + osg::Vec3(r, h, r));
      bbox.expandBy(position + osg::Vec3(r, h, -r));
      _bbox.expandBy(bbox);
    }
  }

  if(_bbox.valid()) {
    float h = (_bbox.yMax() - _bbox.yMin());
    Box* box = const_cast<Box*>(dynamic_cast<const Box*>(_box.get()));
    osg::Vec3Array* cube = box->_box.get();
    int index = 0;
    // bottom
    (*cube)[index++].set(_bbox.xMax(), 0, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMax(), 0, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMin(), 0, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMin(), 0, _bbox.zMin());
    // top
    (*cube)[index++].set(_bbox.xMin(), h, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMin(), h, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMax(), h, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMax(), h, _bbox.zMin());
    // front
    (*cube)[index++].set(_bbox.xMin(), 0, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMax(), 0, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMax(), h, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMin(), h, _bbox.zMax());
    // back
    (*cube)[index++].set(_bbox.xMin(), h, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMax(), h, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMax(), 0, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMin(), 0, _bbox.zMin());
    // left
    (*cube)[index++].set(_bbox.xMin(), 0, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMin(), h, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMin(), h, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMin(), 0, _bbox.zMin());
    // right
    (*cube)[index++].set(_bbox.xMax(), 0, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMax(), h, _bbox.zMin());
    (*cube)[index++].set(_bbox.xMax(), h, _bbox.zMax());
    (*cube)[index++].set(_bbox.xMax(), 0, _bbox.zMax());

    _bsphere.expandBy(_bbox);
  } else {
    _bsphere.expandBy(osg::Vec3(1.f, 1.f, 1.f));
  }
  _bsphere_computed = true;
  return true;
}

bool Stacks::unserialize(struct _xmlDoc* doc, osgDB::Registry* registry) {
  xmlTextReaderPtr reader = xmlReaderWalker(doc);
  if(reader == NULL) return false;
  bool status = readosgchipsFromXml(reader, _chipBank, this, registry);
  xmlFreeTextReader(reader);
  return status;
}

bool Stacks::unserialize(const std::string& fileName, osgDB::Registry* registry) {
  xmlTextReaderPtr reader = xmlReaderForFile(fileName.c_str(), NULL, XML_PARSE_PEDANTIC|XML_PARSE_NONET);
  if(reader == NULL) return false;
  xmlDocPtr doc = xmlTextReaderCurrentDoc(reader);
  bool status = readosgchipsFromXml(reader, _chipBank, this, registry);
  xmlFreeDoc(doc);
  xmlFreeTextReader(reader);
  return status;
}


