/*
 * This program creates a pipe, and then forks a child.  The parent
 * writes a string to the child using the pipe, waits for the child to
 * exit, and then prints the child's exit code on the standard output.
 * The child reads the text from pipe and prints to the standard
 * output, and then exits.  When NO_EXEC is defined, the child does
 * the copying itself, otherwise, it exec's /bin/cat to do the job.
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>

#define PROGRAM "/bin/cat"

int main(int argc, char **argv)
{
  int filedes[2];
  if (pipe(filedes)) {
    perror("pipe");
    return 1;
  }
  pid_t pid = fork();
  if (pid == -1) {
    perror("fork");
    return 1;
  }
  if (pid == 0) {
    if (close(filedes[1]) == -1) {
      perror("close in child");
      return 1;
    }
#if defined NO_EXEC
    FILE *in = fdopen(filedes[0], "r");
    if (!in) {
      perror("fdopen in child");
      return 1;
    }
    for (;;) {
      int ch = getc(in);
      if (ch == EOF)
	break;
      else
	putchar(ch);
    }
    if (ferror(in)) {
      perror("getc");
      return 1;
    }
    else
      return 0;
#else
    if (fcntl(filedes[0], F_SETFD, FD_CLOEXEC) == -1) {
      perror("fcntl");
      return 1;
    }
    if (dup2(filedes[0], 0) ==  -1) { /* original stdin is close here */
      perror("dup2 child");
      return 1;
    }
    execl(PROGRAM, PROGRAM, 0);
    perror("execl");
    return 1;
#endif
  }
  else {
    if (close(filedes[0]) == -1) {
      perror("close in parent");
      return 1;
    }
    FILE *out = fdopen(filedes[1], "w");
    if (!out) {
      perror("fdopen in parent");
      return 1;
    }
    fprintf(out, "Hello, World!\n");
    if (fclose(out) == -1) {
      perror("fclose");
      return 1;
    }
    int status;
    if (wait(&status) == -1) {
      perror("wait");
      return 1;
    }
    printf("child exit status: %d\n", status);
    return 0;
  }
}
