/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-monomer-modif.h"
#include "polyxedit-rendering.h"


PxmModifRes
polyxedit_monomer_modif_modify_monomer (PxmMonomer *mnm,
					gint idx,
					gchar *modif,
					PxmEditCtxt *editctxt)
{
  GdkPixbuf *pixbuf = NULL;

  PxmMonomer *monomer = NULL;
  PxmPolchemdef *polchemdef = NULL;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmMonicon *monicon_old = NULL;
  PxmMonicon *monicon_new = NULL;

  g_assert (modif != NULL);

  g_assert (editctxt != NULL);
  polchemdef = editctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  /* Modifying a monomer is a two-step process: first the monomer must be 
     chemically modified, and next, if the modification is successful,
     the graphical rendering must be performed so that the modification
     can be visualized in the sequence editor.
  */

  /* Sanity checks pertaining to the size of the polymer sequence.
   */
  g_assert (editctxt->polymer->monomerGPA->len > 0);
  g_assert (idx >= 0);
  g_assert (idx <= editctxt->polymer->monomerGPA->len - 1);

  /* Make sure we can correctly access the monomer to be modified.
   */
  if (mnm == NULL)
    monomer = g_ptr_array_index (editctxt->polymer->monomerGPA, idx);
  else
    monomer = mnm;
  
  g_assert (monomer != NULL);
  
  /* Make sure the modif object is known to the polchemdef.
   */
  if (-1 == pxmchem_modif_get_index_by_name (modif, polchemdef->modifGPA))
    {
      g_critical (_("%s@%d: modif is not known to the polchemdef: '%s'\n"),
	     __FILE__, __LINE__, modif);
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* If the monomer is already modified somehow, the function
     below cleanly unmodifies it prior to modifying it with 
     'modif'.
  */
  if (FALSE == pxmchem_monomer_modify (monomer, modif))
    {
      g_critical (_("%s@%d: failed to modify monomer at index: '%d' "
	       "with modif: '%s'\n"),
	     __FILE__, __LINE__, idx, modif);
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* At this point the chemical modification went ok, so we can manage
     the rendering process. In a prudent way of doing things, we start
     by making the new rendered monicon, and if everything goes well
     we make the replacement, otherwise we leave the current monicon
     in place but issue a critical message.
  */

  /* Render a new pixbuf corresponding to the modified monomer.
   */
  pixbuf = polyxedit_rendering_pixbuf_render_modif (monomer->code,
						    modif,
						    seqeditorctxt->
						    monicon_size,
						    editctxt);
  
  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed to render monicon for monomer: '%s' "
	       "at index: '%d' with modif: '%s'. Chemical modification "
	       "was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx, modif);
	  
      /* FIXME: we should handle in a better way the situation here,
	 for example by rendering some kind of visual feedback.
       */
      return MODIF_CHEM_Y_RENDER_N;
    }

  /* Now that we have the pixbuf, we can allocate the new monicon so that
     a new canvas_item is prepared later.
  */
  monicon_new = polyxedit_monicon_new ();

  if (FALSE == 
      polyxedit_rendering_monicon_make_canvas_item (monicon_new, 
						    seqeditorctxt->
						    monicon_size,
						    pixbuf,
						    idx,
						    editctxt))
    {
      g_critical (_("%s@%d: failed to make canvas_item for monomer code: '%s' "
	       "at index: '%d', with modif: '%s'. Chemical modification "
	       "was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx, modif);
	  
      /* FIXME: we should handle in a better way the situation here,
	 for example by rendering some kind of visual feedback.
       */
      return MODIF_CHEM_Y_RENDER_N ;
    }
  
  /* Apparently we succeeded in all our attempts, so get the pointer
     to the monicon that currently renders the monomer. Remove it from
     the array of monicons and free it.
  */
  monicon_old = g_ptr_array_remove_index (seqeditorctxt->moniconGPA, idx);
  g_assert (monicon_old != NULL);
  polyxedit_monicon_free (monicon_old);


  /* At this point we have a fully qualified monicon. Depending on the
     value of idx, with respect to the size of the moniconGPA we
     either insert the monicon in the sequence array or we append it.
     We cannot use the polymer->monomerGPA->len value, because it does
     not change, while moniconGPA->len has changed as the effect of
     removing the old monicon from it. I was having crashes due using
     the polymer->monomerGPA->len value here. Which was wrong of
     course.
   */
  if (idx == seqeditorctxt->moniconGPA->len)
    g_ptr_array_add (seqeditorctxt->moniconGPA, monicon_new);
  else
    g_ptr_array_insert_val (seqeditorctxt->moniconGPA, idx, monicon_new);

  return MODIF_CHEM_Y_RENDER_Y;
}


/* It is the responsibility of the caller to set the parameters to
   values that she will use to establish if the function was 
   successful or not.
*/
void
polyxedit_monomer_modif_modify_monomer_GA (GArray *GA,
					   gchar *modif,
					   PxmEditCtxt *editctxt,
					   gchar **errors,
					   gint *full_success,
					   gint *full_failure,
					   gint *chem_failure,
					   gint *render_failure)
{
  gint iter = 0;
  gint idx = 0;

  GString *gs = NULL;
  
  PxmModifRes res = MODIF_CHEM_N_RENDER_N;
  
  GPtrArray *monomerGPA = NULL;


  /* We have a GArray (GA) of indices corresponding to indices in the
     polymer sequence where the modification (its name is in modif)
     should occur.
   */


  g_assert (GA != NULL);
  /* If no monomer is to be modified just return 0 since we do not
     modify any monomer !
   */
  if (GA->len <= 0)
    return ;
    
  g_assert (editctxt != NULL && editctxt->polymer != NULL);
  
  monomerGPA = editctxt->polymer->monomerGPA;
  g_assert (monomerGPA != NULL);
  /* If the polymer sequence is empty, no modif can occur!
   */
  if (monomerGPA->len <= 0)
    return ;
  
  g_assert (modif != NULL);
  g_assert (-1 != pxmchem_modif_get_index_by_name (modif,
						   editctxt->polchemdefctxt->
						   polchemdef->modifGPA));
  
  /* If errors is set, we want errors to be a pointer to a NULL gchar
     pointer. We will allocate the string ourselves and let the caller 
     free it after.
   */
  if (errors != NULL)
    {
      g_assert (*errors == NULL);
      
      /* Prepare the GString that will store the errors, but only if 
	 error reporting is asked.
      */
      gs = g_string_new ("");
    }
    
  /* We now should iterate in the array of indices of the monomers to
     be modified and take proper actions each time.
  */
  for (iter = 0; iter < GA->len; iter++)
    {
      idx = g_array_index (GA, gint, iter);
      
      /* Check that we are not going to access a non-existent
	 monomer... especially if the user edits the polymer sequence
	 by removing monomers from it.
      */
      if (idx > monomerGPA->len)
	{
	  g_critical (_("%s@%d: a position marked to be modified is greater "
		   "than the size of the polymer! "
		   "Do not edit the sequence!\n"),
		 __FILE__, __LINE__);
	  
	  return ;
	}
      
      /* Do the modification work proper.
       */
      res = polyxedit_monomer_modif_modify_monomer (NULL, idx, modif,
						    editctxt);
      
      /* Now, a number of possibilities need to be envisaged, and errors
	 will have to be logged. Since the first operation is to 
	 make the chemical modification, if it goes wrong the function
	 returns immediately with MODIF_CHEM_N_RENDER_N.
      */
      switch (res)
	{
	case MODIF_CHEM_Y_RENDER_Y:
	  if (full_success != NULL)
	    (*full_success)++;
      	  
	  break;

	case MODIF_CHEM_N_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not modified - no visual\n"),
	       iter + 1);

	  if (full_failure != NULL)
	    (*full_failure)++;
	  
	  break;

	case MODIF_CHEM_Y_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - modified - no visual\n"),
	       iter + 1);
	  
	  if (render_failure != NULL)
	    (*render_failure)++;
	  	  
	  break;

	case MODIF_CHEM_N_RENDER_Y: /* added for symmetry, for the moment.*/
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not modified - visual\n"),
	       iter + 1);
	  
	  if (chem_failure != NULL)
	    (*chem_failure)++;
	  	  
	  break;
	}
    }
  
  /* Now that we have finished doing the work, if the user wanted to
     get errors, then we have to give them.
   */
  if (errors != NULL)
    {
      *errors = gs->str;
      /* We know we have allocated this GString above, because errors
	 was not NULL!
      */
      g_string_free (gs, FALSE);
    }

  return;
}


PxmModifRes
polyxedit_monomer_modif_un_modify_monomer (PxmMonomer *mnm,
					   gint idx,
					   gchar *modif,
					   PxmEditCtxt *editctxt)
{
  gint res = UN_MODIF_FAILURE;

  GdkPixbuf *pixbuf = NULL;

  PxmMonomer *monomer = NULL;
  PxmPolchemdef *polchemdef = NULL;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  PxmMonicon *monicon_old = NULL;
  PxmMonicon *monicon_new = NULL;

  g_assert (editctxt != NULL);
  polchemdef = editctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  /* Un-modifying a monomer is a two-step process: first the monomer
     must be chemically un-modified (if its current modif is the same
     as the modif param, if the latter is non-NULL), and next, if the
     un-modification is successful, the graphical rendering must be
     performed so that the modification is eliminated from the monicon
     of the no-more-modified monomer.
  */

  /* Sanity checks pertaining to the size of the polymer sequence.
   */
  g_assert (editctxt->polymer->monomerGPA->len > 0);
  g_assert (idx >= 0);
  g_assert (idx <= editctxt->polymer->monomerGPA->len - 1);

  /* Make sure we can correctly access the monomer to be un-modified.
   */
  if (mnm == NULL)
    monomer = g_ptr_array_index (editctxt->polymer->monomerGPA, idx);
  else
    mnm = monomer;
  
  g_assert (monomer != NULL);

  /* Make sure the modif object is known to the polchemdef, if modif
     passed as param is non-NULL.
   */
  if (modif != NULL && 
      -1 == pxmchem_modif_get_index_by_name (modif, polchemdef->modifGPA))
    {
      g_critical (_("%s@%d: modif is not known to the polchemdef: '%s'\n"),
	     __FILE__, __LINE__, modif);
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* The call below may return any of these values:
     UN_MODIF_SUCCESS, if the unmodification was successful
     UN_MODIF_NO_MATCH, if the monomer was modified != 'modif'
     UN_MODIF_NO_MODIF, if the monomer was not modified
     UN_MODIF_FAILURE, if something went wrong.
  */
  res = pxmchem_monomer_un_modify (monomer, modif);
  

  if (res == UN_MODIF_FAILURE)
    {
      if (modif != NULL)
	{
	  g_critical (_("%s@%d: failed to un-modify monomer "
		   "at index: '%d' with modif: '%s'\n"),
		 __FILE__, __LINE__, idx, modif);
	}
      
      else
	{
	  g_critical (_("%s@%d: failed to or did not un-modify monomer "
		   "at index: '%d'\n"),
		 __FILE__, __LINE__, idx);
	}
      
      return MODIF_CHEM_N_RENDER_N;
    }

  /* The situation is tricky here: we do not want to modify the appearance
     of a monomer if its internal chemical state did not change. Which
     is why we have to check to return value precisely. In fact,
     we only go on with the graphical rendering of the monicon if
     res == UN_MODIF_SUCCESS.
  */
  if (res != UN_MODIF_SUCCESS)
    return MODIF_CHEM_N_RENDER_N;
  
  
  /* At this point the chemical un-modification went ok, so we can manage
     the rendering process. In a prudent way of doing things, we start
     by making the new rendered monicon, and if everything goes well
     we make the replacement, otherwise we leave the current monicon
     in place but issue a critical message.
  */

  /* Render a new pixbuf corresponding to the un-modified monomer.
   */
  pixbuf = polyxedit_rendering_pixbuf_render_no_modif (monomer->code,
						       seqeditorctxt->
						       monicon_size,
						       editctxt);
  
  if (pixbuf == NULL)
    {
      g_critical (_("%s@%d: failed to render monicon for un-modified "
	       "monomer: '%s' at index: '%d'. Chemical un-modification "
	       "was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx);
	  
      /* FIXME: we should handle in a better way the situation here,
	 for example by rendering some kind of visual feedback.
       */
      return MODIF_CHEM_Y_RENDER_N;
    }

  /* Now that we have the pixbuf, we can allocate the new monicon so that
     a new canvas_item is prepared later.
  */
  monicon_new = polyxedit_monicon_new ();

  if (FALSE == 
      polyxedit_rendering_monicon_make_canvas_item (monicon_new, 
						    seqeditorctxt->
						    monicon_size,
						    pixbuf,
						    idx,
						    editctxt))
    {
      g_critical (_("%s@%d: failed to make canvas_item for un-modified "
		    "monomer, code: '%s' at index: '%d'. Chemical "
		    "un-modification was performed ok, however.\n"),
	     __FILE__, __LINE__, monomer->code, idx);
	  
      /* FIXME: we should handle in a better way the situation here,
	 for example by rendering some kind of visual feedback.
       */
      return MODIF_CHEM_Y_RENDER_N ;
    }
  
  /* Apparently we succeeded in all our attempts, so get the pointer
     to the monicon that currently renders the monomer. Remove it from
     the array of monicons and free it.
  */
  monicon_old = g_ptr_array_remove_index (seqeditorctxt->moniconGPA, idx);
  g_assert (monicon_old != NULL);
  polyxedit_monicon_free (monicon_old);


  /* At this point we have a fully qualified monicon. Depending on the
     value of idx, with respect to the size of the moniconGPA we
     either insert the monicon in the sequence array or we append it.
     We cannot use the polymer->monomerGPA->len value, because it does
     not change, while moniconGPA->len has changed as the effect of
     removing the old monicon from it. I was having crashes due using
     the polymer->monomerGPA->len value here. Which was wrong of
     course.
   */
  if (idx == seqeditorctxt->moniconGPA->len)
    g_ptr_array_add (seqeditorctxt->moniconGPA, monicon_new);
  else
    g_ptr_array_insert_val (seqeditorctxt->moniconGPA, idx, monicon_new);

  return MODIF_CHEM_Y_RENDER_Y;
}


void
polyxedit_monomer_modif_un_modify_monomer_GA (GArray *GA,
					      gchar *modif,
					      PxmEditCtxt *editctxt,
					      gchar **errors,
					      gint *full_success,
					      gint *full_failure,
					      gint *chem_failure,
					      gint *render_failure)
{
  gint iter = 0;
  gint idx = 0;

  GString *gs = NULL;
  
  PxmModifRes res = MODIF_CHEM_N_RENDER_N;
  
  GPtrArray *monomerGPA = NULL;


  /* We have a GArray (GA) of indices corresponding to indices in the
     polymer sequence where the un-modification should occur. Note
     that modif may be NULL, in which case the un-modification will
     happen whatever the modification with which the target monomer is
     modified. Instead, if modif is non-NULL, the target monomer is
     un-modified only if currently modified with a modification of the
     same name as the string passed as parameter 'modif'.
   */


  g_assert (GA != NULL);
  /* If no monomer is to be un-modified just return 0 since we do not
     modify any monomer !
   */
  if (GA->len <= 0)
    return ;
    
  g_assert (editctxt != NULL && editctxt->polymer != NULL);
  
  monomerGPA = editctxt->polymer->monomerGPA;
  g_assert (monomerGPA != NULL);
  /* If the polymer sequence is empty, no un-modif can occur!
   */
  if (monomerGPA->len <= 0)
    return ;
  
  /* Sanity check: if modif is non-NULL it must be known to the
     polymer type!
  */
  if (modif != NULL)
    g_assert (-1 != pxmchem_modif_get_index_by_name (modif,
						     editctxt->polchemdefctxt->
						     polchemdef->modifGPA));
  
  /* If errors is set, we want errors to be a pointer to a NULL gchar
     pointer. We will allocate the string ourselves and let the caller 
     free it after.
   */
  if (errors != NULL)
    {
      g_assert (*errors == NULL);
      
      /* Prepare the GString that will store the errors, but only if 
	 error reporting is asked.
      */
      gs = g_string_new ("");
    }
    
  /* We now should iterate in the array of indices of the monomers to
     be un-modified and take proper actions each time.
  */
  for (iter = 0; iter < GA->len; iter++)
    {
      idx = g_array_index (GA, gint, iter);
      
      /* Check that we are not going to access a non-existent
	 monomer... especially if the user edits the polymer sequence
	 by removing monomers from it.
      */
      if (idx > monomerGPA->len)
	{
	  g_critical (_("%s@%d: a position marked to be modified is greater "
		   "than the size of the polymer! "
		   "Do not edit the sequence!\n"),
		 __FILE__, __LINE__);
	  
	  return ;
	}
      
      /* Do the un-modification work proper.
       */
      res = polyxedit_monomer_modif_un_modify_monomer (NULL, idx, modif,
						       editctxt);
      
      /* Now, a number of possibilities need to be envisaged, and errors
	 will have to be logged. Since the first operation is to 
	 make the chemical modification, if it goes wrong the function
	 returns immediately with MODIF_CHEM_N_RENDER_N.
      */
      switch (res)
	{
	case MODIF_CHEM_Y_RENDER_Y:
	  if (full_success != NULL)
	    (*full_success)++;
      	  
	  break;

	case MODIF_CHEM_N_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not un-modified - no visual\n"),
	       iter + 1);

	  if (full_failure != NULL)
	    (*full_failure)++;
	  
	  break;

	case MODIF_CHEM_Y_RENDER_N:
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - un-modified - no visual\n"),
	       iter + 1);
	  
	  if (render_failure != NULL)
	    (*render_failure)++;
	  	  
	  break;

	case MODIF_CHEM_N_RENDER_Y: /* added for symmetry, for the moment.*/
	  if (errors)
	    g_string_append_printf 
	      (gs, 
	       _("pos: '%d' - not un-modified - visual\n"),
	       iter + 1);
	  
	  if (chem_failure != NULL)
	    (*chem_failure)++;
	  	  
	  break;
	}
    }
  
  /* Now that we have finished doing the work, if the user wanted to
     get errors, then we have to give them.
   */
  if (errors != NULL)
    {
      *errors = gs->str;
      /* We know we have allocated this GString above, because errors
	 was not NULL!
      */
      g_string_free (gs, FALSE);
    }

  return;
}
