/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-fragment-options.h"
#include "polyxedit-ui-fragment-results.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxmass-window-mngmt.h"

/* columns */

enum
{
  /* fragspecif data */
  COLUMN_FGS_NAME = 0,
  COLUMN_FGS_END,
  COLUMN_FGS_ACTFORM,
  COLUMN_FGS_COMMENT,
  /* fragrule data */
  COLUMN_FGR_NAME,
  COLUMN_FGR_PREV,
  COLUMN_FGR_THIS,
  COLUMN_FGR_NEXT,
  COLUMN_FGR_ACTFORM,
  COLUMN_FGR_COMMENT,

  COLUMN_FGS_VISIBLE,
  COLUMN_FGS_EDITABLE,
  COLUMN_FGS_COL_COUNT
};



GtkWidget *
polyxedit_fragment_opt_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
  

  g_assert (editctxt != NULL);

  gui_file = 
    g_strdup_printf ("%s/polyxedit-fragment-options.glade", 
		     userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "fragment_options_wnd", 
		       PACKAGE);

  g_free (gui_file);

  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "fragment_options_wnd");
  
  if (window == NULL)
    {
      g_error (_("%s@%d: failed to create the fragmentation options window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  /* Set the polymer sequence name to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);
  
  if (editctxt->polymer->plminfo->name != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			editctxt->polymer->plminfo->name);
  else 
    gtk_entry_set_text (GTK_ENTRY (widget), _("Not set"));

  /* Set the polymer sequence context id number (its pointer) to its
     correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "identity_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),  "identity_number_entry", widget);

  /* Set the polymer id number (the pointer to editctxt to the
     relative entry).
   */
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);


  /* The fragspecs' frame must be correctly set up:
   */
  widget = glade_xml_get_widget (xml, "fragspecs_available_vbox");
  g_object_set_data (G_OBJECT (window),
		     "fragspecs_available_vbox", widget);

  if (FALSE == 
      polyxedit_fragment_opt_setup_fgs_treeview (window, widget))
    {
      g_error (_("%s@%d: failed to set up the fragspecs treeview\n"),
	     __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      g_object_unref (G_OBJECT (window));

      return NULL;
    }
  

  /* There is a GtkEntry in the window that is commonly used to 
   * display messages.
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);


  widget = glade_xml_get_widget (xml, "fragment_option_wnd_fragment_button");
  g_object_set_data (G_OBJECT (window),
		     "cleave_option_wnd_fragment_button", widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_fragment_opt_wnd_fragment_button),
		    window);


  /* We don't need the GladeXML object any more, so unref it
   * to save some memory 
   */
  g_object_unref (G_OBJECT (xml));

  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_fragment_opt_wnd_delete_event),
		    editctxt);
  
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_fragment_opt_wnd_destroy_event),
		    editctxt);


  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one fragment options' window opened for
     a given polymer seqence editing window, and we do not want that
     the second window destroys the datum name of the first window, so
     we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("fragment_options_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_fragment_opt_wnd_really_close);
  
  g_free (help);

  return window;
}



gboolean
polyxedit_fragment_opt_setup_fgs_treeview (GtkWidget *window, 
					   GtkWidget *vbox)
{
  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  
  GPtrArray *GPA = NULL;
  
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn *column;
  
  GtkWidget *sw = NULL;

  gint col_offset;


  g_assert (vbox != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");


  polchemdef = editctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  GPA = polchemdef->fragspecGPA;
  g_assert (GPA != NULL);
  

  /* Create the scrolledview that we'll pack into widget.
   */
  sw = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);  
  

  /* Create the treeview model.
   */
  model = polyxedit_fragment_opt_create_fgs_treeview_model (GPA);


  /* Set to the model a datum with a pointer to GPA, so that
   * the array of fragspecs that has been used to fill the model
   * is accessible later. Also a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "fragspec_GPA", GPA);
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible (add/remove button
   * handlers).
   */
  g_object_set_data (G_OBJECT (window), 
		     "fragspec_treeview_model", model);
  
  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later (remove item handler).
   */
  g_object_set_data (G_OBJECT (window), "fragspec_treeview", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_MULTIPLE);
  
  /* Fragspec name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGS_NAME);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, 
						 "text",
						 
						 COLUMN_FGS_NAME,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragspec end column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.0, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGS_END);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("End"),
						 renderer, 
						 "text",
						 
						 COLUMN_FGS_END,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 
						 
						 NULL);
  
  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  
  /* Fragspec actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", (gint *) 
		     COLUMN_FGS_ACTFORM);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Actform"),
						 renderer, "text",

						 COLUMN_FGS_ACTFORM,

						 "visible",
						 COLUMN_FGS_VISIBLE,

						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragspec comment column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", (gint *) 
		     COLUMN_FGS_COMMENT);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Comment"),
						 renderer, "text",

						 COLUMN_FGS_COMMENT,

						 "visible",
						 COLUMN_FGS_VISIBLE,

						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_NAME);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Name"),
						 renderer, "text",
						 
						 COLUMN_FGR_NAME,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule prev column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_PREV);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Prev"),
						 renderer, "text",
						 
						 COLUMN_FGR_PREV,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule this column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_THIS);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("This"),
						 renderer, "text",
						 
						 COLUMN_FGR_THIS,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule next column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_NEXT);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Next"),
						 renderer, "text",
						 
						 COLUMN_FGR_NEXT,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_ACTFORM);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Actform"),
						 renderer, "text",
						 
						 COLUMN_FGR_ACTFORM,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

  /* Fragrule comment column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_FGR_COMMENT);

  col_offset = 
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (treeview),
						 -1, _("Comment"),
						 renderer, "text",
						 
						 COLUMN_FGR_COMMENT,
						 
						 "visible",
						 COLUMN_FGS_VISIBLE,
						 
						 "editable",
						 COLUMN_FGS_EDITABLE, 

						 NULL);

  column = 
    gtk_tree_view_get_column (GTK_TREE_VIEW (treeview), col_offset - 1);

  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  gtk_container_add (GTK_CONTAINER (sw), treeview);
  
  gtk_widget_show_all (vbox);
  
  return TRUE;
}



GtkTreeModel *
polyxedit_fragment_opt_create_fgs_treeview_model (GPtrArray *GPA)
{
  GtkTreeStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;
  gint jter = 0;

  gchar *help = NULL;
  
  PxmFragSpec *fgs = NULL;
  PxmFragRule *fgr = NULL;
  
  model = gtk_tree_store_new (COLUMN_FGS_COL_COUNT,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN);


  /* Add data for each fragspec to the tree store.
   */
  for (iter = 0 ; iter < GPA->len ; iter++)
    {
      fgs = g_ptr_array_index (GPA, iter);
      g_assert (fgs != NULL);
      
      gtk_tree_store_append (model, &tree_iter, NULL);

      if (fgs->end == PXM_FRAG_END_LEFT)
	help = g_strdup ("LE");
      else if (fgs->end == PXM_FRAG_END_RIGHT)
	help = g_strdup ("RE");
      else if (fgs->end == PXM_FRAG_END_NONE)
	help = g_strdup ("NE");
      else
	g_critical (_("%s@%d: fragspec has an incorrect 'end' member.\n"),
	       __FILE__, __LINE__);

      gtk_tree_store_set 
	(model, &tree_iter,
	 /* fragspec data */
	 COLUMN_FGS_NAME, fgs->name,
	 COLUMN_FGS_END, help,
	 COLUMN_FGS_ACTFORM, (fgs->actform != NULL)? fgs->actform : "",
	 COLUMN_FGS_COMMENT, (fgs->comment != NULL)? fgs->comment : "",
	 /* fragrule data */
	 COLUMN_FGR_NAME, "",
	 COLUMN_FGR_PREV, "",
	 COLUMN_FGR_THIS, "",
	 COLUMN_FGR_NEXT, "",
	 COLUMN_FGR_ACTFORM, "",
	 COLUMN_FGR_COMMENT, "",
	 /* column behaviour */
	 COLUMN_FGS_VISIBLE, TRUE,
	 COLUMN_FGS_EDITABLE, FALSE,
	 -1);

      g_free (help);
      
      /* Add children, namely cleaverules of current fragspec, if any.
       */
      for (jter = 0 ; jter < fgs->fgrGPA->len ; jter++)
	{
	  GtkTreeIter tree_jter;
	  
	  fgr = g_ptr_array_index (fgs->fgrGPA, jter);
	  g_assert (fgr != NULL);
	  
	  gtk_tree_store_append (model, &tree_jter, &tree_iter);

	  gtk_tree_store_set 
	    (model, &tree_jter,
	     /* fragspec data */
	     COLUMN_FGS_NAME, "",
	     COLUMN_FGS_END, "",
	     COLUMN_FGS_ACTFORM, "",
	     COLUMN_FGS_COMMENT, "",
	     /* fragrule data */
	     COLUMN_FGR_NAME, fgr->name,
	     COLUMN_FGR_PREV, (fgr->prev != NULL)? fgr->prev : "",
	     COLUMN_FGR_THIS, (fgr->this != NULL)? fgr->this : "",
	     COLUMN_FGR_NEXT, (fgr->next != NULL)? fgr->next : "",
	     COLUMN_FGR_ACTFORM, fgr->actform,
	     COLUMN_FGR_COMMENT, (fgr->comment != NULL)? fgr->comment : "",
	     /* column behaviour */
	     COLUMN_FGS_VISIBLE, TRUE,
	     COLUMN_FGS_EDITABLE, FALSE,
	     -1);
	}
    }

  return GTK_TREE_MODEL (model);
}



void
polyxedit_fragment_opt_wnd_fragment_button (GtkWidget *widget, 
					   gpointer data)
{
  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  PxmPolymer *polymer = NULL;

  PxmFragSpec *fragspec = NULL;
  PxmFragOpt *fragopt = NULL;
  PxmCalcOpt *calcopt = NULL;
  
  GtkWidget *window = data;
  GtkWidget *display_window = NULL;
  GtkWidget *messages_entry = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkTreePath* path = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection *selection = NULL;
  

  GPtrArray *fragoptGPA = NULL; /* array of fragopt instances */
  GPtrArray *allfragsGPA = NULL; /* array of all the oligomers (fragments) */
  GPtrArray *fragsGPA = NULL; /* array of arrays ... (see above) */
  
  gboolean valid = FALSE;
  gboolean selected = FALSE;

  GString *fragopt_gs = NULL;
    
  gint iter = 0;
  gint idx = 0;
  gint len = 0;

  gchar *help = NULL;

  

  /* The mass calculation options are taken from the editctxt's
     calcopt member. The user can modify these at will by selecting
     the appropriate menu.
  */
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");

  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
  len = polymer->monomerGPA->len;
  
  polchemdef = editctxt->polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);

  /* We will use this GString to construct a string describing the 
     fragmentation options for the window registering function call below.
  */
  fragopt_gs = g_string_new ("");
  
  /* We certainly will have to display some messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window),  
				       "messages_entry");
  g_assert (messages_entry != NULL);
  

  /* Get a pointer to the currently selected fragmentation
     specifications and make an array with a duplicate of each.
  */  

  /* Get a pointer to the treeview in the window.
   */
  treeview = (GtkTreeView *) g_object_get_data (G_OBJECT (window), 
						"fragspec_treeview");
  g_assert (treeview != NULL);
  
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));

  /* And to the related model:
   */
  /* Get the fragspec treeview model from the window pointer passed
   * a data parameter.
   */
  model = (GtkTreeModel *) g_object_get_data (G_OBJECT (window), 
					      "fragspec_treeview_model");

  /* Get an iterator to the first tree-rooted fragspec.
   */
  valid = gtk_tree_model_get_iter_first (model, &treeiter);

  /* Allocate the array of fragopt instances that we'll stuff with
     newly allocated fragopt instances as selected items (if any) are
     encountered in the treeview.
  */
  fragoptGPA = g_ptr_array_new ();
  

  while (valid)
    {
      selected = gtk_tree_selection_iter_is_selected (selection, 
						      &treeiter);
      
      if (selected != TRUE)
	{
	  valid = gtk_tree_model_iter_next (model, &treeiter);

	  continue;
	}
      
      /* The row is selected, which means we have to get the fragspec
	 that corresponds to it.
      */

      /* Get the path to the selected item, which can be a fragspec row
       * or a cleaverule row, which are two very different cases that
       * we must handle. Specifically, if depth == 1, the selected row
       * is a fragspec, if depth == 2, then it is a fragrule.
       */
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (model), 
				      &treeiter);
      g_assert (path != NULL);
      
      /* We can get the index of the fragspec that is concerned, be it
	 selected or one of its fragrule children.
      */
      idx = gtk_tree_path_get_indices (path) [0];
      
      /* Finally we can free the path.
       */
      gtk_tree_path_free (path);
      
      fragspec = g_ptr_array_index (polchemdef->fragspecGPA, idx);
      g_assert (fragspec != NULL);
      
      /* Allocate the PxmFragOpt object that will be needed.
       */
      fragopt = pxmchem_fragopt_new ();
      
      /* Make sure that we set the monomer modifications to the
	 oligomers.
       */
      fragopt->mnm_chement = PXMCHEMENT_MNM_MODIF;
      
      /* Same for the polymer modifications.
       */
      fragopt->plm_chement = PXMCHEMENT_PLM_BOTH_MODIF;
      
      /* We want that each oligomer has its own sequence so that
	 we will be able to display it in the results window.
      */
      fragopt->put_sequence = TRUE;
      
      /* Duplicate in it the fragspec instance so that we have a copy
	 for us.
      */
      fragopt->fragspec = pxmchem_fragspec_dup (fragspec);

      /* Append to the fragopt_gs the current fragspec->name.
       */
      if (strlen (fragopt_gs->str) < 1)
	g_string_append_printf (fragopt_gs, 
				_("Fragmentation pattern(s): '%s' "), 
				fragopt->fragspec->name);
      else
	g_string_append_printf (fragopt_gs, 
				"; '%s' ", fragopt->fragspec->name);

            
      /* We have our copy, just append it to the array of fragopt
	 instances.
       */
      g_ptr_array_add (fragoptGPA, fragopt);
      
      /* Now continue the iteration in the treeview ...
       */
      valid = gtk_tree_model_iter_next (model, &treeiter);
      
      continue;
    }
  
  /* At this point we should have an array of fragopt instances that
     may either be emtpy, or contain one/more fragopt instances. If it
     is empty, it is just that no item was selected in the treeview,
     we return silently.
  */
  if (fragoptGPA->len <= 0)
    {
      g_ptr_array_free (fragoptGPA, TRUE);

      g_string_free (fragopt_gs, TRUE);
            
      return;
    }
  
  /* Duplicate the editctxt's calcopt (because its start_idx,
     end_idx members are going to be modified during the fragmentation
     operation):
  */
  calcopt = pxmchem_calcopt_dup (editctxt->calcopt);
    
  /* We should get the indices of the currently selected stretch of
     the polymer sequence.
  */
  if (FALSE ==
      polyxedit_seqed_wnd_get_selection_indices (editctxt,
						 &calcopt->start_idx,
						 &calcopt->end_idx))
    {
      calcopt->start_idx = 0;
      calcopt->end_idx = len - 1;
    }
  else
    {
      /* Remember that for reasons related to the dealing of the graphical 
	 rendering of the polymer sequence in the editor, the end_idx value
	 that is returned by the function call above is > the true value by
	 one unit. So we have to reduce the obtained value by one:
      */
      calcopt->end_idx--;
    }
  
  /* We seem to be ready to perform the polymer sequence
     fragmentation.
  */
  
  /* Allocate the array that will hold all the different oligomers'
     arrays (1 array of fragments per fragmentation specification).
  */
  allfragsGPA = g_ptr_array_new ();
  
  /* Now that we have the array that will hold all the fragment arrays,
     we iterate in the fragopt array and for each fragopt found we
     make a fragmentation run:
  */
  for (iter = 0; iter < fragoptGPA->len; iter++)
    {
      fragopt = g_ptr_array_index (fragoptGPA, iter);
      g_assert (fragopt != NULL);
      
      /* Allocate the array to receive the oligomers from this
	 fragmentation
       */
      fragsGPA = g_ptr_array_new ();
      
      if (-1 == 
	  pxmchem_fragment_polymer (editctxt->polymer,
				    fragsGPA,
				    polchemdef,                    
				    fragopt, 
				    calcopt))
	{
	  help = 
	    g_strdup_printf (_("Failed to fragment the polymer with "
			       "fragspec, name: '%s'\n"),
			     fragopt->fragspec->name);
	  
	  g_critical (_("%s@%d: %s"),
		 __FILE__, __LINE__, help);
	  
	  gtk_entry_set_text (GTK_ENTRY (messages_entry),
			      help);
	  
	  g_free (help);
	  
	  pxmchem_oligomer_GPA_of_GPA_free (allfragsGPA);
	  
	  pxmchem_fragopt_GPA_free (fragoptGPA);

	  pxmchem_calcopt_free (calcopt);

	  g_string_free (fragopt_gs, TRUE);
	  	  
	  return;
	}
      /* Add the newly filled array of oligomer to the array of
	 arrays.
       */
      g_ptr_array_add (allfragsGPA, fragsGPA);
    }
  

  /* By now we should have an array of arrays of oligomers that contain
     all the fragments well ordered...
  */
	  
  
  /* And now make sure we can display the data: make the window where
     the oligomers are to be displayed. The function call below will
     manage to display in the window all the oligomers that are in the
     allfragsGPA array. If something fails, just free the stuff and
     return.
  */
  display_window = polyxedit_fragment_res_wnd_setup (editctxt,
						     allfragsGPA,
						     fragoptGPA);
  if (display_window == NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (messages_entry),
			  _("Failed to display the oligomers in a window"));
      
      pxmchem_oligomer_GPA_of_GPA_free (allfragsGPA);
      
      pxmchem_fragopt_GPA_free (fragoptGPA);

      pxmchem_calcopt_free (calcopt);
      
      g_string_free (fragopt_gs, TRUE);

      return;
    }

  /* Now that we have the certainty that the oligomers' displaying
     window could be created and populated correctly with the
     oligomers, we can set the calcopt instance as datum to the
     results displaying window. This structure will be freed when the
     window will be destroyed. Note that the two GPA pointers were set
     as a datum during setting up of the displaying window above.
  */
  g_object_set_data_full (G_OBJECT (display_window),
			  "calcopt", calcopt,
			  (GDestroyNotify) pxmchem_calcopt_free);
  

  /* And now we can ask that all the informations contained in these
     structures be updated in the results' window at proper places.
  */
  polyxedit_fragment_res_wnd_update_fragmentation_data (display_window);
  
  /* Register the results' window !
   */
  /* Immediately set a automagically freeable string with the
     description of the new allocated results window so that later
     -during window management- we can describe what the window was
     for. This will be used if the user asks to find masses in the
     results and then a new window will be displayed with all the
     masses that were found in the results window, and we will want to
     display that description string.
  */
  help = g_strdup_printf (_("Fragmentation of '%p' With '%s'"), editctxt,
			  fragopt_gs->str);
  
  g_object_set_data_full (G_OBJECT (display_window),
			  "wnd_desc", help, 
			  (GDestroyNotify) free);

  if (NULL == polyxmass_winmngmt_register_window 
      ("POLYXEDIT",
       display_window,
       window,
       editctxt,
       _("Results of Fragmentation"),
       help,
       TRUE,
       polyxedit_fragment_results_wnd_make_report))
    g_critical (_("%s@%d: failed to register the fragmentation "
		  "results' window\n"),
		__FILE__, __LINE__);
  
  
  /* Now that have done our text stuff, we can free the string.
   */
  g_string_free (fragopt_gs, TRUE);


  /* Finally we can return, our work here is finished.
   */
  return;
}


/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_fragment_opt_wnd_really_close (GtkWidget *window)
{
  g_assert (window != NULL);

  /* This function is called as a GDestroyNotify callback function when
     the sequence editor window gets closed.
  */
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  gtk_widget_destroy (window);
}



gboolean
polyxedit_fragment_opt_wnd_delete_event (GtkWidget *window,
					   GdkEvent *event,
					   gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  gchar *help = NULL;
  
  g_assert (window != NULL);
  g_assert (editctxt != NULL);
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  /* This window pointer was set as a full datum to the sequence editor
     window, which means that we have to remove that pointer, without
     triggering the callback function call.
  */
  help = g_strdup_printf ("fragment_options_wnd-%p", window);
  
  window = 
    g_object_steal_data (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
			     help);
  
  g_free (help);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}


gboolean
polyxedit_fragment_opt_wnd_destroy_event (GtkWidget *window,
					  GdkEvent *event,
					  gpointer data)
{
  return FALSE;
}

