/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef LIBPOLYXMASS_CHEMGROUP_H
#define LIBPOLYXMASS_CHEMGROUP_H

#include "libpolyxmass-globals.h"
#include "libpolyxmass-prop.h"
#include "pxmchem-monomer.h"




#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
  
  
  /* typedef's
   */
  typedef struct _PxmChemgroup PxmChemgroup;
  typedef struct _PxmChemgroupRule PxmChemgroupRule;
  
  typedef enum _PxmChemgroupPolRule PxmChemgroupPolRule;
  
  
  enum _PxmChemgroupPolRule
    {
      /* This enum contains variables used to indicate how the
	 chemical group is participating to the formation of the bonds
	 that lead to the polymerization. That will help determine,
	 depending on the position of the monomer in the
	 polymer/oligomer sequence, if the chemgroup has to be taken into
	 account or not.
       */

      /* The chemical group gets trapped (that is has no more the chemgroup
	 valid) when the monomer is incorporated into a polymer
	 sequence, and that there is monomer at the left of that
	 monomer. Typically, the chemgroup is still valid if the monomer is
	 at the left end of the polymer sequence.
       */
      PXM_CHEMGROUP_GROUP_LEFT_TRAPPED = 1 << 0,
      
      /* The chemical group gets trapped (that is has no more the chemgroup
	 valid) when the monomer is incorporated into a polymer
	 sequence, and that there is monomer at the right of that
	 monomer. Typically, the chemgroup is still valid if the monomer is
	 at the right end of the polymer sequence.
       */
      PXM_CHEMGROUP_GROUP_RIGHT_TRAPPED = 1 << 1,

      /* The chemical group never gets trapped when it is incorporated
      into a polymer sequence. This is typically the case when the
      chemical group belongs to a lateral chain that never enters in
      the formation of the inter-monomer bond while the polymerization
      occurs.
      */
      PXM_CHEMGROUP_GROUP_NEVER_TRAPPED = 1 << 2, 
    };


  struct _PxmChemgroup
  {
    /* The name that the user gives to the chemical group.
     */
    gchar * name;
    
    /* Value of the chemgroup being described in this structure.
     */
    gdouble pka;

    /* A character string that must be allocated for *this object,
       because it gets freed in the free'ing function, to be used by
       the user at her discretion.
    */
    gchar *userdata;
    
    /* The variable below indicates what is the availability of the
       group depending on the way the chemical group for which the chemgroup
       is being defined might be engaged in chemical bonds...
    */
    PxmChemgroupPolRule polrule;
    
    /* The variable acid_charged tells if the acid form of the
       acido-basic pair is charged or not. For example, that variable
       would be "FALSE" for R-COOH, but would be "TRUE" for R-NH3+.
    */
    gboolean acidcharged;
    
    /* This "chemgroup rule array" ('cgrGPA') may contain objects of
       type PxmChemgroupRule indicating if the chemgroup being
       described should be dropped or not if the chemical group gets
       modified chemically. The chemical entity in such
       PxmChemgroupRule instance is described by the 'entity'
       gchar*. That could be a PxmModif name, for example. The context
       should provide enough discriminating clues know what entity is
       deal with here.
       
       For example, let's say that we are describing the chemgroup for
       the alpha-NH2 group of a protein residue (chemgroup usually
       9.6, as R-NH2 is rather basic group). We may define that when
       this group is acetylated (thus becomes H3C-CO-NH-R), its
       initial chemgroup should be dropped, as the R-NH2 group is not
       there anymore. Thus, we would add a PxmChemgroupRule object
       with data name="Acetylation" and outcome=LOST.
    */
    GPtrArray *cgrGPA;
  };
  

  #define PXM_CHEMGROUP_RULE_LOST 0
  #define PXM_CHEMGROUP_RULE_PRESERVED 1
 

  struct _PxmChemgroupRule
  {
    /* The identification of the entity that is responsible for
       changing the initial value of the Chemgroup. This might be
       something like "LE_PLM_MODIF" to describe a modification to be
       applied to the left end of the polymer sequence, or "MNM_MODIF"
       to indicate a modification to a monomer.
    */
    gchar *entity;

    /* The name member is the name of the entity object described by
       the 'entity' member above. For a "MNM_MODIF" entity, the name
       could be "Phosphorylation", while for a "LE_PLM_MODIF" entity,
       the name could be "Acetylation", for example.

    */
    gchar *name;
    
    /* Indicates if the Chemgroup for which this PxmChemgroupRule
       clause is set should be let alone, without taking it into
       consideration any more because of a chemical entity described
       with the 'entity' member variable.
     */
    gint outcome;
  };
  
   



  /* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
   */
  PxmChemgroup *
  libpolyxmass_chemgroup_new (void);

  PxmChemgroupRule *
  libpolyxmass_chemgrouprule_new (void);

  PxmProp *
  libpolyxmass_chemgroup_prop_new (PxmChemgroup *chemgroup);




  /* DATA MODIFYING FUNCTIONS
   */
  gchar *
  libpolyxmass_chemgroup_set_name (PxmChemgroup *chemgroup, gchar *name);
  
  gchar *
  libpolyxmass_chemgroup_set_userdata (PxmChemgroup *chemgroup, 
				       gchar *userdata);
  
  gchar *
  libpolyxmass_chemgrouprule_set_entity (PxmChemgroupRule *pr, 
					 gchar *entity);

  gchar *
  libpolyxmass_chemgrouprule_set_name (PxmChemgroupRule *pr, 
				       gchar *name);




  /* LOCATING FUNCTIONS
   */
  PxmChemgroupRule *
  libpolyxmass_chemgroup_get_chemgrouprule_ptr (PxmChemgroup *chemgroup,
						gchar *entity,
						gchar *name);
  
  gint
  libpolyxmass_chemgroup_ensure_single_containing_entity (PxmMonomer *monomer,
							  gchar *entity);
  
  



  /* RENDERING FUNCTIONS - FORMATTER FUNCTIONS
   */
  PxmChemgroupRule *
  libpolyxmass_chemgroup_render_xml_node_chemgrouprule (xmlDocPtr xml_doc,
					    xmlNodePtr xml_node,
					    gpointer data);
  
  PxmChemgroup *
  libpolyxmass_chemgroup_render_xml_node_chemgroup (xmlDocPtr xml_doc,
					xmlNodePtr xml_node,
					gpointer data);
  



  /* FREE'ING FUNCTIONS
   */
  gint 
  libpolyxmass_chemgroup_free (PxmChemgroup *chemgroup);

  gint 
  libpolyxmass_chemgrouprule_free (PxmChemgroupRule *pr);

  gint
  libpolyxmass_chemgrouprule_GPA_free (GPtrArray *GPA);

  gint 
  libpolyxmass_chemgroup_prop_free (PxmProp *prop);
  
  gint 
  libpolyxmass_chemgroup_prop_free_but_not_data (PxmProp *prop);
  
  










#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* LIBPOLYXMASS_CHEMGROUP_H */

  
