/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "pxmchem-polchemdef.h"
#include "pxmchem-monomer.h"
#include "pxmchem-modif.h"
#include "pxmchem-cleavespec.h"
#include "pxmchem-fragspec.h"
#include "pxmchem-polymer.h"
#include "libpolyxmass-config.h"
#include "pxmchem-formula.h"
#include "pxmchem-polchemdef-plugins.h"


/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmPolchemdef *
pxmchem_polchemdef_new (void)
{
  PxmPolchemdef *polchemdef = g_malloc0 (sizeof (PxmPolchemdef));

  polchemdef->codelen = 1;
  
  polchemdef->atomGPA = g_ptr_array_new ();
  
  polchemdef->monomerGPA = g_ptr_array_new ();
  polchemdef->modifGPA = g_ptr_array_new ();
  polchemdef->cleavespecGPA = g_ptr_array_new ();
  polchemdef->fragspecGPA = g_ptr_array_new ();

  polchemdef->ionizerule = pxmchem_ionizerule_new ();
  
  polchemdef->modified = FALSE;
  
  polchemdef->propGPA = g_ptr_array_new ();
  
  return polchemdef;
}


gboolean
pxmchem_polchemdef_set_version (PxmPolchemdef *polchemdef, gchar *version)
{
  g_assert (polchemdef != NULL && version != NULL);
  
  /* The member data may be NULL, as this function can be called
     right after pxmchem_polchemdef_new () which leaves the members
     NULL (except other non-string data that are allocated).
  */ 
  if (polchemdef->version != NULL)
    g_free (polchemdef->version);
  
  polchemdef->version = g_strdup (version);
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_type (PxmPolchemdef *polchemdef, gchar *type)
{
  g_assert (polchemdef != NULL && type != NULL);
  
  /* The member data may be NULL, as this function can be called
     right after pxmchem_polchemdef_new () which leaves the members
     NULL (except other non-string data that are allocated).
  */ 
  if (polchemdef->type != NULL)
    g_free (polchemdef->type);
  
  polchemdef->type = g_strdup (type);
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_file (PxmPolchemdef *polchemdef, gchar *file)
{
  g_assert (polchemdef != NULL && file != NULL);
  
  if (polchemdef->file != NULL)
    g_free (polchemdef->file);
  
  polchemdef->file = g_strdup (file);
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_dir (PxmPolchemdef *polchemdef, gchar *dir)
{
  g_assert (polchemdef != NULL && dir != NULL);
  
  if (polchemdef->dir != NULL)
    g_free (polchemdef->dir);
  
  polchemdef->dir = g_strdup (dir);
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_leftcap (PxmPolchemdef *polchemdef, gchar *leftcap)
{
  g_assert (polchemdef != NULL && leftcap != NULL);
  
  if (polchemdef->leftcap != NULL)
    g_free (polchemdef->leftcap);
  
  polchemdef->leftcap = g_strdup (leftcap);
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_rightcap (PxmPolchemdef *polchemdef, gchar *rightcap)
{
  g_assert (polchemdef != NULL && rightcap != NULL);
  
  if (polchemdef->rightcap != NULL)
  g_free (polchemdef->rightcap);
  
  polchemdef->rightcap = g_strdup (rightcap);
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_codelen (PxmPolchemdef *polchemdef, gint codelen)
{
  g_assert (polchemdef != NULL);
  g_assert (codelen > 0);
  
  polchemdef->codelen = codelen;
  
  return TRUE;
}


gboolean
pxmchem_polchemdef_set_delim_codes (PxmPolchemdef *polchemdef, gchar *delim_codes)
{
  g_assert (polchemdef != NULL && delim_codes != NULL);
  
  if (polchemdef->delim_codes != NULL)
    g_free (polchemdef->delim_codes);
  
  polchemdef->delim_codes = g_strdup (delim_codes);
  
  return TRUE;
}


void
pxmchem_polchemdef_set_modified (PxmPolchemdef *polchemdef, gboolean modified)
{
  g_assert (polchemdef != NULL);
  
  polchemdef->modified = modified;
}


gboolean
pxmchem_polchemdef_get_modified (PxmPolchemdef *polchemdef)
{
  g_assert (polchemdef != NULL);
  
  return polchemdef->modified;
}


gboolean
pxmchem_polchemdef_invert_modified (PxmPolchemdef *polchemdef)
{
  g_assert (polchemdef != NULL);
  
  polchemdef->modified = (!polchemdef->modified);

  return polchemdef->modified;
}

gboolean
pxmchem_polchemdef_update_delim_codes (PxmPolchemdef * polchemdef, 
				       gchar delim)
{
  g_assert (polchemdef != NULL);
  
  if (polchemdef->delim_codes != NULL)
    g_free (polchemdef->delim_codes);
  
  polchemdef->delim_codes = 
    pxmchem_polchemdef_make_delim_codes (polchemdef, delim);
  
  g_assert (polchemdef->delim_codes != NULL);
  
  return TRUE;
}





/* INTEGRITY CHECKING FUNCTIONS
 */
gboolean
pxmchem_polchemdef_validate_all (PxmPolchemdef *polchemdef, gchar **valid)
{
  gint iter = 0;
  gchar *help = NULL;
  
  GString *gs = NULL;

  PxmMonomer *monomer = NULL;
  PxmModif *modif = NULL;
  PxmCleaveSpec *cleavespec = NULL;
  PxmFragSpec *fragspec = NULL;

    
  /* Iterates in the polchemdef object and checks every bit of data for
   * consistency. If something wrong is detected, a description is 
   * appended to *valid. Else, nothing is touched.
   */
  /* Returns FALSE if errors were encountered, and messages are
   * set to *valid. If no error are encountered, TRUE is returned and 
   * valid is untouched.
   */

  /* Note that for integrity reasons, *valid MUST BE NULL to ensure 
   * that it is empty.
   */
  g_assert (valid != NULL);
  g_assert (*valid == NULL);
  
  g_assert (polchemdef != NULL);
  
  /* Allocate the GString into which the errors (if any) are to be
   * appended.
   */
  gs = g_string_new ("");

  
  /* The non-iterative data, first.
   */

  /* The polymer definition version: this is NOT IMPORTANT as we always
     will save it with the last version number.
   
     if (polchemdef->version == NULL)
     {
     g_string_append_printf (gs, 
     _("polchemdef has a NULL version\n"));
     }
     else
     {
     if (strlen (polchemdef->type) <= 0)
     g_string_append_printf (gs, _("polchemdef has an invalid version: "
     "'%s'\n"),
     polchemdef->version);
     }
  */
  
  /* The polymer definition type: this is CRUCIAL.
   */
  if (polchemdef->type == NULL)
    {
      g_string_append_printf (gs, 
			 _("polchemdef has a NULL 'type' member\n"));
    }
  else
    {
      if (strlen (polchemdef->type) <= 0)
	g_string_append_printf (gs, _("polchemdef has an invalid 'type' member: "
				      "'%s'\n"),
				 polchemdef->type);
    }
  
  /* The polymer definition leftcap, which must be non-NULL. If set,
   * it should be a valid actform. However, it can be non-set (ie of
   * zero-length).
   */
  if (polchemdef->leftcap == NULL || strlen (polchemdef->leftcap) <= 0)
    {
      g_string_append_printf (gs, 
			      _("polchemdef has a NULL or empty 'leftcap' member\n"));
    }
  else
    {
      if (FALSE == pxmchem_actform_check (polchemdef->leftcap, 
					  polchemdef->atomGPA))
	g_string_append_printf (gs, _("polchemdef has an invalid"
				      " 'leftcap' member: '%s'\n"),
				polchemdef->leftcap);
    }
  
  /* The polymer definition rightcap, which must be non-NULL. If set,
   * it should be a valid actform. However, it can be non-set (ie of
   * zero-length).
   */
  if (polchemdef->rightcap == NULL || strlen (polchemdef->rightcap) <= 0)
    {
      g_string_append_printf (gs, 
			 _("polchemdef has a NULL or empty 'rightcap' member\n"));
    }
  else
    {
      if (FALSE == pxmchem_actform_check (polchemdef->rightcap, 
					  polchemdef->atomGPA))
	g_string_append_printf (gs, _("polchemdef has an invalid"
				      " 'rightcap' member: '%s'\n"),
				polchemdef->rightcap);
    }
  
  /* The ionizerule must also be checked.
   */
  if (polchemdef->ionizerule == NULL)
    {
      g_string_append_printf (gs, 
			 _("polchemdef has a NULL 'ionizerule' member\n"));
    }
  else
    {
      if (FALSE == pxmchem_ionizerule_validate (polchemdef->ionizerule,
						polchemdef->atomGPA, &help))
	{
	  g_assert (help != NULL);
	  
	  g_string_append_printf (gs, 
			     _("the 'ionizerule' member of polchemdef has errors:\n'%s'\n"),
			     help);
	  g_free (help);
	  help = NULL;
	}
    }
  
  /* TODO
     Check the array of atoms.
  */

  /* The array of monomers.
   */
  for (iter = 0; iter < polchemdef->monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (polchemdef->monomerGPA, iter);
      g_assert (monomer != NULL);

      if (FALSE == pxmchem_monomer_unique_by_name (monomer, 
						   polchemdef->monomerGPA))
	{
	  g_string_append_printf (gs, 
			     _("monomer at index: '%d' - name: '%s'"
			       " is not unique\n"),
			     iter, monomer->name);
	}
      
      if (FALSE == pxmchem_monomer_unique_by_code (monomer, 
						   polchemdef->monomerGPA))
	{
	  g_string_append_printf (gs, 
			     _("monomer at index: '%d' - code: '%s'"
			       " is not unique\n"),
			     iter, monomer->code);
	}
      
      if (FALSE == 
	  pxmchem_monomer_validate (monomer, polchemdef->codelen, 
				    polchemdef->atomGPA, &help))
	{
	  g_assert (help != NULL);
	  
	  g_string_append_printf (gs, 
			     _("monomer at index: '%d' has errors:\n'%s'\n"),
			     iter, help);
	  g_free (help);
	  help = NULL;
	}
    }

  /* Now that the array of monomers has been checked, we can prepare
   * the string of delimited codes, so that when we will need it,
   * we'll have it uptodate.
   */
  help = pxmchem_polchemdef_make_delim_codes (polchemdef, 
					  (gchar) libpolyxmass_globals_delim);
  g_assert (help != NULL);
  pxmchem_polchemdef_set_delim_codes (polchemdef, help);
  g_free (help);
  help = NULL;


  /* The array of modifs.
   */
  for (iter = 0; iter < polchemdef->modifGPA->len; iter++)
    {
      modif = g_ptr_array_index (polchemdef->modifGPA, iter);
      g_assert (modif != NULL);

      if (FALSE == pxmchem_modif_unique_by_name (modif, 
						 polchemdef->modifGPA))
	{
	  g_string_append_printf (gs, 
			     _("modif at index %d: '%s' is not unique\n"),
			     iter, modif->name);
	}      

      if (FALSE == pxmchem_modif_validate (modif, polchemdef->atomGPA, 
					   &help))
	{
	  g_assert (help != NULL);
	  
	  g_string_append_printf (gs, 
			     _("modif at index: '%d' has errors:\n'%s'\n"),
			     iter, help);
	  g_free (help);
	  help = NULL;
	}
    }

  /* The array of cleavespecs.
   */
  for (iter = 0; iter < polchemdef->cleavespecGPA->len; iter++)
    {
      cleavespec = g_ptr_array_index (polchemdef->cleavespecGPA, iter);
      g_assert (cleavespec != NULL);

      if (FALSE == 
	  pxmchem_cleavespec_unique_by_name (cleavespec, 
					     polchemdef->cleavespecGPA))
	{
	  g_string_append_printf (gs,
			     _("cleavespec at index: '%d' - name: '%s'"
			       " is not unique\n"),
			     iter, cleavespec->name);
	}      

      if (FALSE == 
	  pxmchem_cleavespec_validate (cleavespec, polchemdef->delim_codes, 
				       polchemdef->codelen, 
				       polchemdef->atomGPA, 
				       &help))
	{
	  g_assert (help != NULL);
	  
	  g_string_append_printf (gs, 
			     _("cleavespec at index: '%d' has errors:\n'%s'\n"),
			     iter, help);
	  g_free (help);
	  help = NULL;
	}
    }

  /* The array of fragspecs.
   */
  for (iter = 0; iter < polchemdef->fragspecGPA->len; iter++)
    {
      fragspec = g_ptr_array_index (polchemdef->fragspecGPA, iter);
      g_assert (fragspec != NULL);

      if (FALSE == pxmchem_fragspec_unique_by_name (fragspec, 
						    polchemdef->fragspecGPA))
	{
	  g_string_append_printf (gs,
			     _("fragspec at index: '%d' - name: '%s'"
			       " is not unique\n"),
			     iter, fragspec->name);
	}      

      if (FALSE == 
	  pxmchem_fragspec_validate (fragspec, polchemdef, &help))
	{
	  g_assert (help != NULL);
	  
	  g_string_append_printf (gs, 
			     _("fragspec at index: '%d' has errors:\n'%s'\n"),
			     iter, help);
	  g_free (help);
	  help = NULL;
	}
    }

  /* Finally, we finished validating all the polymer pieces...
   */
  if (strlen (gs->str) > 0)
    {
      /* At least one error occurred.
       */
      *valid = gs->str;
      
      g_string_free (gs, FALSE);
      

      return FALSE;
    }
  
  g_string_free (gs, TRUE);
  
  return TRUE;
}

  



/* LOCATING FUNCTIONS
 */
PxmPolchemdef *
pxmchem_polchemdef_get_ptr_by_type (GPtrArray *GPA, gchar *type)
{
  gint iter = 0;
  PxmPolchemdef *polchemdef = NULL;

  
  g_assert (GPA != NULL);
  g_assert (type != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      polchemdef = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (polchemdef->type, type))
	return polchemdef;
    }
  
  return NULL;
}


gint
pxmchem_polchemdef_get_index_by_type (GPtrArray *GPA, gchar *type)
{
  gint iter = 0;
  PxmPolchemdef *polchemdef = NULL;

  
  g_assert (GPA != NULL);
  g_assert (type != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    {
      polchemdef = g_ptr_array_index (GPA, iter);
      
      if (0 == strcmp (polchemdef->type, type))
	return iter;
    }
  
  return -1;
}
  

gint
pxmchem_polchemdef_get_index_by_ptr (GPtrArray *GPA, PxmPolchemdef *polchemdef)
{
  gint iter = 0;
  

  g_assert (GPA != NULL && polchemdef != NULL);
  
  for (iter = 0; iter < GPA->len; iter++)
    if ((PxmPolchemdef *) g_ptr_array_index (GPA, iter) == polchemdef)
      return iter;
      
  return -1;
}





/* UTILITY FUNCTIONS
 */
gchar*
pxmchem_polchemdef_make_delim_codes (PxmPolchemdef *polchemdef, gchar delim)
{
  GString *gs = NULL;

  gchar *result = NULL;
  
  gint iter = 0;
  
  PxmMonomer *monomer = NULL;

  g_assert (polchemdef != NULL);
  g_assert (polchemdef->monomerGPA != NULL);
  g_assert (TRUE == g_ascii_isprint (delim));

  if (TRUE == g_ascii_isalnum (delim))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: the supplied char must be non-alphanumerical\n"),
	     __FILE__, __LINE__);
      return NULL;
    }
  
  gs = g_string_new ("");
  
  for (iter = 0; iter < polchemdef->monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (polchemdef->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      g_string_append_printf (gs, "%c%s", delim, monomer->code);
    }
  
  /* Close the string with a delim char:
   */
  gs = g_string_append_c (gs, delim);
  
  result = gs->str;
  
  g_string_free (gs, FALSE);
  
  return result;
}


gchar *
pxmchem_polchemdef_get_current_xml_version (void)
{
  gchar *version = g_strdup_printf ("%s", POLCHEMDEF_XML_VERSION);
  
  return version;
}




  

/* XML-format TRANSACTIONS
 */
gchar *
pxmchem_polchemdef_format_xml_string_DTD (void)
{
  gchar *result = NULL;
  
  gchar *DTD = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n"
"<!-- DTD for polymer definitions, used by the\n"            
    "'GNU polyxmass' suite of mass spectrometry applications.\n"
    "Copyright 2003, 2004, 2005 Filippo Rusconi - Licensed under the GNU GPL -->\n"
    "<!DOCTYPE polchemdefdata [\n"
    "<!ATTLIST polchemdefdata version CDATA #REQUIRED>\n"
    "<!ELEMENT polchemdefdata (type,leftcap,rightcap,codelen,ionizerule,monomers,modifs,cleavespecs,fragspecs)>\n"
    "<!ELEMENT ionizerule (actform,charge,level)>\n"
    "<!ELEMENT monomers (mnm*)>\n"
    "<!ELEMENT modifs (mdf*)>\n"
    "<!ELEMENT cleavespecs (cls*)>\n"
    "<!ELEMENT fragspecs (fgs*)>\n"
    "<!ELEMENT mnm (name,code,formula)>\n"
    "<!ELEMENT mdf (name,actform)>\n"
    "<!ELEMENT cls (name,pattern,clr*)>\n"
    "<!ELEMENT fgs (name,end,actform,comment?,fgr*)>\n"
    "<!ELEMENT clr ((le-mnm-code,le-actform)?,(re-mnm-code,re-actform)?)>\n"
    "<!ELEMENT fgr (name,actform,prev-mnm-code?,this-mnm-code?,next-mnm-code?,comment?)>\n"
    "<!ELEMENT type (#PCDATA)>\n"
    "<!ELEMENT leftcap (#PCDATA)>\n"
    "<!ELEMENT rightcap (#PCDATA)>\n"
    "<!ELEMENT codelen (#PCDATA)>\n"
    "<!ELEMENT actform (#PCDATA)>\n"
    "<!ELEMENT charge (#PCDATA)>\n"
    "<!ELEMENT level (#PCDATA)>\n"
    "<!ELEMENT name (#PCDATA)>\n"
    "<!ELEMENT code (#PCDATA)>\n"
    "<!ELEMENT formula (#PCDATA)>\n"
    "<!ELEMENT pattern (#PCDATA)>\n"
    "<!ELEMENT end (#PCDATA)>\n"
    "<!ELEMENT le-mnm-code (#PCDATA)>\n"
    "<!ELEMENT re-mnm-code (#PCDATA)>\n"
    "<!ELEMENT le-actform (#PCDATA)>\n"
    "<!ELEMENT re-actform (#PCDATA)>\n"
    "<!ELEMENT comment (#PCDATA)>\n"
    "<!ELEMENT prev-mnm-code (#PCDATA)>\n"
    "<!ELEMENT this-mnm-code (#PCDATA)>\n"
    "<!ELEMENT next-mnm-code (#PCDATA)>\n"
    "]>\n";
  
  result = g_strdup (DTD);
  
  return result;
}


gchar *
pxmchem_polchemdef_format_xml_string_polchemdefdata (PxmPolchemdef *polchemdef, 
					     gchar *indent, gint offset)
{
  /* The pointer 'polchemdef' passed as parameter will allow an iteration
   * in the polymer definition and the appending to a string of 
   * a textual representation for all the elements that constitute 
   * a polymer definition (monomers, modifs, cleavespecs...).
   */
  gint iter = 0;
  gint new_offset = 0;
  
  gchar *lead = NULL;
  gchar *help = NULL;
  
  GString *gs = NULL;
  
  PxmMonomer *monomer = NULL;
  PxmModif *modif = NULL;
  PxmCleaveSpec *cleavespec = NULL;
  PxmFragSpec *fragspec = NULL;
  
  g_assert (polchemdef != NULL);
  g_assert (indent != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);
  
  /* We are willing to create a string that begins with the DTD and
   * that next looks like this:
   *
   *<polchemdefdata version="0.1">
   *  <type>protein</type>
   *  <leftcap>+H1</leftcap>
   *  <rightcap>+O1H1</rightcap>
   *  <codelen>1</codelen>
   *  <ionizerule>
   *    <actform>+H1</actform>
   *    <charge>1</charge>
   *  </ionizerule>
   *  <monomers>
   *    <mnm>
   *      <name>Glycine</name>
   *      <code>G</code>
   *      <formula>C2H3N1O1</formula>
   *    </mnm>
   *.
   *.
   *.
   *    <mnm>
   *      <name>Proline</name>
   *      <code>P</code>
   *      <formula>C5H7N1O1</formula>
   *    </mnm>
   *  </monomers>
   *  <modifs>
   *    <mdf>
   *      <name>Phosphorylation</name>
   *      <actform>-H1+P1O3H2</actform>
   *    </mdf>
   *.
   *.
   *.
   *    <mdf>
   *      <name>SulfideBond</name>
   *      <actform>-H2</actform>
   *    </mdf>
   *  </modifs>
   *  <cleavespecs>
   *    <cls>
   *      <name>CyanogenBromide</name>
   *      <pattern>M/</pattern>
   *      <lr-rule>
   *        <re-mnm-code>M</re-mnm-code>
   *        <re-actform>-C1H2S1+O1</re-actform>
   *      </lr-rule>
   *      <lr-rule>
   *        <re-mnm-code>H</re-mnm-code>
   *        <re-actform>-H3+H3</re-actform>
   *      </lr-rule>
   *      <lr-rule>
   *        <le-mnm-code>L</le-mnm-code>
   *        <le-actform>-C3+C3</le-actform>
   *      </lr-rule>
   *    </cls>
   *.
   *.
   *.
   *    <cls>
   *      <name>GluC</name>
   *      <pattern>E/</pattern>
   *    </cls>
   *  </cleavespecs>
   *  <fragspecs>
   *    <fgs>
   *      <name>a</name>
   *      <end>LE</end>
   *      <actform>-C1O1</actform>
   *      <fgr>
   *	<name>a_fgr_1</name>
   *	<actform>-H1+H1</actform>
   *	<next-mnm-code>N</next-mnm-code>
   *	<this-mnm-code>T</this-mnm-code>
   *	<prev-mnm-code>P</prev-mnm-code>
   *      </fgr>
   *      <fgr>
   *	<name>a_fgr_2</name>
   *	<actform>-H2+H2</actform>
   *	<next-mnm-code>N</next-mnm-code>
   *	<this-mnm-code>T</this-mnm-code>
   *	<prev-mnm-code>P</prev-mnm-code>
   *      </fgr>
   *    </fgs>
   *.
   *.
   *.
   *    <fgs>
   *      <name>imm</name>
   *      <end>NE</end>
   *      <actform>-C1O1+H1</actform>
   *    </fgs>
   *  </fragspecs>
   *</polchemdefdata>
   */

  /* 
     We first have to put the Document Type definition:
  */
  help = pxmchem_polchemdef_format_xml_string_DTD ();
  

  /* Now construct the string to be sent to the xml file:
   */
  g_string_append_printf (gs, "%s", help);
  g_free (help);
  
  /* Open the <polchemdefdata> node and immediately insert the non-iterative
   * data.
   */
  lead = libpolyxmass_globals_format_string_lead (indent, offset);

  g_string_append_printf (gs, "%s<polchemdefdata version=\"%s\">\n", 
			  lead, POLCHEMDEF_XML_VERSION);
  
  g_free (lead);
  new_offset = offset + 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  
  g_assert (polchemdef->type != NULL && strlen (polchemdef->type) > 0);
  g_string_append_printf (gs, "%s<type>%s</type>\n",
		     lead, polchemdef->type);
  
  g_assert (polchemdef->leftcap != NULL && strlen (polchemdef->leftcap) > 0);
  g_string_append_printf (gs, "%s<leftcap>%s</leftcap>\n",
		     lead, polchemdef->leftcap);
  
  g_assert (polchemdef->rightcap != NULL && strlen (polchemdef->rightcap) > 0);
  g_string_append_printf (gs, "%s<rightcap>%s</rightcap>\n",
		     lead, polchemdef->rightcap);

  g_string_append_printf (gs, "%s<codelen>%d</codelen>\n",
		     lead, polchemdef->codelen);

  /* Open the <ionizerule> element which is simple ********************:
   */
  g_assert (polchemdef->ionizerule != NULL);
  help = 
    pxmchem_ionizerule_format_xml_string_ionizerule (polchemdef->ionizerule,
						     indent, new_offset);
  g_assert (help != NULL);
  g_string_append_printf (gs, "%s", help);
  g_free (help);
  
  
  /* Open the <monomers> section, which is recursive ********************:
   */
  g_string_append_printf (gs, "%s<monomers>\n", lead);

  /* The recursive <mnm> elements are indented once more.
   */
  g_free (lead);
  new_offset += 1;

  for (iter = 0; iter < polchemdef->monomerGPA->len; iter++)
    {
      monomer = g_ptr_array_index (polchemdef->monomerGPA, iter);
      help = pxmchem_monomer_format_xml_string_mnm (monomer, 
						    indent, new_offset);
      g_assert (help != NULL);
      g_string_append_printf (gs, "%s", help);
      g_free (help);
    }
  
  /* Close the <monomers> element.
   */  
  new_offset -= 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  g_string_append_printf (gs, "%s</monomers>\n", lead);

  /* Open the <modifs> section, which is recursive ********************:
   */
  g_string_append_printf (gs, "%s<modifs>\n", lead);

  /* The recursive <mdf> elements are indented once more.
   */
  g_free (lead);
  new_offset += 1;

  for (iter = 0; iter < polchemdef->modifGPA->len; iter++)
    {
      modif = g_ptr_array_index (polchemdef->modifGPA, iter);
      help = pxmchem_modif_format_xml_string_mdf (modif, 
						  indent, new_offset);
      g_assert (help != NULL);
      g_string_append_printf (gs, "%s", help);
      g_free (help);
    }
  
  /* Close the <modifs> element.
   */  
  new_offset -= 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  g_string_append_printf (gs, "%s</modifs>\n", lead);


  /* Open the <cleavespecs> section, which is recursive ********************:
   */
  g_string_append_printf (gs, "%s<cleavespecs>\n", lead);

  /* The recursive <cls> elements are indented once more.
   */
  g_free (lead);
  new_offset += 1;

  for (iter = 0; iter < polchemdef->cleavespecGPA->len; iter++)
    {
      cleavespec = g_ptr_array_index (polchemdef->cleavespecGPA, iter);
      help = pxmchem_cleavespec_format_xml_string_cls (cleavespec,
						       indent, new_offset);
      g_assert (help != NULL);
      g_string_append_printf (gs, "%s", help);
      g_free (help);
    }
  
  /* Close the <cleavespecs> element.
   */  
  new_offset -= 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  g_string_append_printf (gs, "%s</cleavespecs>\n", lead);


  /* Open the <fragspecs> section, which is recursive ********************:
   */
  g_string_append_printf (gs, "%s<fragspecs>\n", lead);

  /* The recursive <fgs> elements are indented once more.
   */
  g_free (lead);
  new_offset += 1;

  for (iter = 0; iter < polchemdef->fragspecGPA->len; iter++)
    {
      fragspec = g_ptr_array_index (polchemdef->fragspecGPA, iter);
      help = pxmchem_fragspec_format_xml_string_fgs (fragspec,
						     indent, new_offset);
      g_assert (help != NULL);
      g_string_append_printf (gs, "%s", help);
      g_free (help);
    }
  
  /* Close the <fragspecs> element.
   */  
  new_offset -= 1;
  lead = libpolyxmass_globals_format_string_lead (indent, new_offset);
  g_string_append_printf (gs, "%s</fragspecs>\n", lead);
  g_free (lead);

  /* Finally we can close the <polchemdefdata> node.
   */
  
  lead = libpolyxmass_globals_format_string_lead (indent, offset);
  g_string_append_printf (gs, "%s</polchemdefdata>\n", lead);
  g_free (lead);
  
  g_assert (gs != NULL);
  
  help = gs->str;
  
  g_string_free (gs, FALSE);
  
  return help;
}



PxmPolchemdef *
pxmchem_polchemdef_render_xml_file (gchar *file)
{
  /* We get a file name which must be a valid xml file. Then we
   * have to parse this file and construct a polymer definition that
   * is compliant with the polyxmass data conventions.
   */
  xmlDocPtr xml_doc = NULL;
  xmlNodePtr xml_node = NULL;
  
  PxmPolchemdef *polchemdef = NULL;

  gchar *help = NULL;
  
  
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return NULL;
  
  if (FALSE == g_file_test (file, G_FILE_TEST_EXISTS))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file not found: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return NULL;
    }
      
  /* The very first thing we could do is check that the file is an
   * XML file: <?xml version="1.0"?> should be the first item in the
   * file.
   */
  if (FALSE == libpolyxmass_globals_check_xml_file (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file is not valid xml format: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return NULL;
    }
  
  /* Build an XML tree from a the file.
   */
  xml_doc = xmlParseFile (file);

  if (xml_doc == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file is not valid xml format: '%s'\n"),
	     __FILE__, __LINE__, file);
      
     return NULL;
    }

  /* Check if the document is of the right kind.
   */
  xml_node = xmlDocGetRootElement (xml_doc);

  if (xml_node == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file is an empty xml file: '%s'\n"),
	     __FILE__, __LINE__, file);

      xmlFreeDoc (xml_doc);
      
      return NULL;
    }

  if (0 != strcmp ((gchar *) xml_node->name, "polchemdefdata"))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file is of wrong type, "
	     "root node is not \"polchemdefdata\": '%s'\n"),
	     __FILE__, __LINE__, file);
      
      xmlFreeDoc (xml_doc);
      
      return NULL;
    }

  /* Remember that we have to select the proper function for
     rendering that node depending on the version of the file format.
     
     We thus have to get a pointer to the proper function.
  */
  help = (gchar *) xmlGetProp (xml_node, (guchar *) "version");
  g_assert (help != NULL);
  
  pxmchem_polchemdefdata_xml_node_render_plugin = 
    pxmchem_polchemdefdata_xml_node_choose_renderer (help);
  
  if (pxmchem_polchemdefdata_xml_node_render_plugin == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to get the polchemdefdata renderer "
	       "for XML file format version: '%s'\n"),
	     __FILE__, __LINE__,
	     (help != NULL ? help : "(none)"));
      
      xmlFree (help);
      
      return NULL;
    }
  
  xmlFree (help);
  
  polchemdef = 
    pxmchem_polchemdefdata_xml_node_render_plugin (xml_doc, xml_node, NULL);
  
  
  /* Free the xml doc material.
   */
  xmlFreeDoc (xml_doc);
  
  if (polchemdef == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: failed to render a polchemdef from file: '%s'\n"),
	     __FILE__, __LINE__, file);
    }
  else
    {
      /* We can set the polchemdef->file member to the file name that we
       * have used to render this polchemdef.
       */
      pxmchem_polchemdef_set_file (polchemdef, file);
    }
  
  return polchemdef;
}
  


gchar *
pxmchem_polchemdef_get_xml_file_version (gchar *file)
{
  /* We get a file name which must be a valid xml file. Then we
   * have to parse this file and construct a polymer definition that
   * is compliant with the polyxmass data conventions.
   */
  xmlDocPtr xml_doc = NULL;
  xmlNodePtr xml_node = NULL;
  
  
  gchar *help = NULL;
  gchar *version = NULL;
  
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return NULL;
  
  if (FALSE == g_file_test (file, G_FILE_TEST_EXISTS))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file not found: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return NULL;
    }
      
  /* The very first thing we could do is check that the file is an
   * XML file: <?xml version="1.0"?> should be the first item in the
   * file.
   */
  if (FALSE == libpolyxmass_globals_check_xml_file (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file is not valid xml format: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return NULL;
    }
  
  /* Build an XML tree from a the file.
   */
  xml_doc = xmlParseFile (file);

  if (xml_doc == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file is not valid xml format: '%s'\n"),
	     __FILE__, __LINE__, file);
      
     return NULL;
    }

  /* Check if the document is of the right kind.
   */
  xml_node = xmlDocGetRootElement (xml_doc);

  if (xml_node == NULL)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	    _("%s@%d: file is an empty xml file: '%s'\n"),
	     __FILE__, __LINE__, file);

      xmlFreeDoc (xml_doc);
      
      return NULL;
    }

  if (0 != strcmp ((gchar *) xml_node->name, "polchemdefdata"))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file is of wrong type, "
	     "root node is not \"polchemdefdata\": '%s'\n"),
	     __FILE__, __LINE__, file);
      
      xmlFreeDoc (xml_doc);
      
      return NULL;
    }
  
  help = (gchar *) xmlGetProp (xml_node, (guchar *) "version");
  
  if (help == NULL)
    {
      version = g_strdup ("none");
    }
  else
    {
      version = g_strdup (help);
      xmlFree (help);
    }
    
  /* Free the xml doc material.
   */
  xmlFreeDoc (xml_doc);
  
  return version;
}
  


/* FREE'ING FUNCTIONS
 */
gboolean
pxmchem_polchemdef_free (PxmPolchemdef *polchemdef)
{
  g_assert (polchemdef != NULL);
  
  /* Free inner material first.
   */
  g_assert (polchemdef->propGPA != NULL);
  libpolyxmass_prop_GPA_free (polchemdef->propGPA);
  
  if (polchemdef->version != NULL)
    g_free (polchemdef->version);

  if (polchemdef->type != NULL)
    g_free (polchemdef->type);

  if (polchemdef->file != NULL)
    g_free (polchemdef->file);
  
  if (polchemdef->dir != NULL)
    g_free (polchemdef->dir);
  
  if (polchemdef->leftcap != NULL)
    g_free (polchemdef->leftcap);
  
  if (polchemdef->rightcap != NULL)
    g_free (polchemdef->rightcap);
  
  if (polchemdef->delim_codes != NULL)
    g_free (polchemdef->delim_codes);
  
  if (polchemdef->ionizerule != NULL)
    pxmchem_ionizerule_free (polchemdef->ionizerule);
  
  if (polchemdef->atomGPA != NULL)
    pxmchem_atom_GPA_free (polchemdef->atomGPA);
  
  if (polchemdef->monomerGPA != NULL)
    pxmchem_monomer_GPA_free (polchemdef->monomerGPA);
  
  if (polchemdef->modifGPA != NULL)
    pxmchem_modif_GPA_free (polchemdef->modifGPA);
  
  if (polchemdef->cleavespecGPA != NULL)
    pxmchem_cleavespec_GPA_free (polchemdef->cleavespecGPA);
  
  if (polchemdef->fragspecGPA != NULL)
    pxmchem_fragspec_GPA_free (polchemdef->fragspecGPA);
  
  /*
   * propGPA was the first member to be freed !
   */

  g_free (polchemdef);
  
  return TRUE;
}



/* GPtrArray-RELATED FUNCTIONS
 */
gint
pxmchem_polchemdef_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmPolchemdef *polchemdef = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      polchemdef = g_ptr_array_remove_index (GPA, 0);
      g_assert (polchemdef != NULL);
      pxmchem_polchemdef_free (polchemdef);
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);

  return count;
}



