/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <libudev.h>
#include <glib.h>
#include <glib-unix.h>

#include "powerd-internal.h"
#include "log.h"

#define ARRAY_SIZE(a) (sizeof(a) / sizeof((a)[0]))

struct udev *udev;
struct udev_monitor *udev_mon;
guint udev_mon_source_id;

enum bl_type {
    BL_TYPE_FIRMWARE,
    BL_TYPE_PLATFORM,
    BL_TYPE_RAW,
};

static const char *bl_strs[] = {
    [BL_TYPE_FIRMWARE]  = "firmware",
    [BL_TYPE_PLATFORM]  = "platform",
    [BL_TYPE_RAW]       = "raw",
};

struct bl_device {
    struct udev_device *dev;
    enum bl_type type;
    int max_brightness;
};

struct bl_device *preferred_bl;
GSList *bl_devices;

static int bl_parse_type(const char *type_str)
{
    unsigned i;

    for (i = 0; i < ARRAY_SIZE(bl_strs); i++) {
        if (!strcmp(type_str, bl_strs[i]))
            return i;
    }
    return -1;
}

static struct bl_device *new_backlight(struct udev_device *dev)
{
    struct bl_device *bl_dev;
    int type;
    long max_brightness;
    const char *data;

    powerd_debug("Processing backlight device %s",
                 udev_device_get_syspath(dev));

    data = udev_device_get_sysattr_value(dev, "type");
    if (!data) {
        powerd_error("Could not read backlight \"type\" attribute");
        return NULL;
    }
    type = bl_parse_type(data);
    if (type < 0) {
        powerd_error("Invalid backlight type \"%s\"", data);
        return NULL;
    }

    data = udev_device_get_sysattr_value(dev, "max_brightness");
    if (!data) {
        powerd_error("Could not read backlight \"max_brightness\" attribute");
        return NULL;
    }
    errno = 0;
    max_brightness = strtol(data, NULL, 10);
    if (errno != 0) {
        powerd_error("Invalid max_brightness \"%s\"", data);
        return NULL;
    }

    bl_dev = malloc(sizeof(*bl_dev));
    if (!bl_dev) {
        powerd_error("Falied to allocate memory for backlight device: %s",
                     strerror(errno));
        return NULL;
    }

    udev_device_ref(dev);
    bl_dev->dev = dev;
    bl_dev->type = type;
    bl_dev->max_brightness = (int)max_brightness;

    return bl_dev;
}

static void free_backlight(struct bl_device *bl)
{
    udev_device_unref(bl->dev);
    free(bl);
}

static gboolean remove_backlight(const char *path)
{
    GSList *item;
    struct bl_device *bl;

    for (item = bl_devices; item; item = g_slist_next(item)) {
        bl = item->data;
        if (!strcmp(path, udev_device_get_syspath(bl->dev))) {
            bl_devices = g_slist_remove(bl_devices, bl);
            free_backlight(bl);
            return TRUE;
        }
    }

    return FALSE;
}

static void update_preferred_backlight(void)
{
    GSList *item;
    struct bl_device *bl;

    preferred_bl = NULL;
    for (item = bl_devices; item; item = g_slist_next(item)) {
        bl = item->data;
        if (!preferred_bl || bl->type < preferred_bl->type)
            preferred_bl = bl;
    }

    if (preferred_bl)
        powerd_info("Using backlight at %s",
                    udev_device_get_syspath(preferred_bl->dev));
    else
        powerd_warn("No backlights present");
}

/* Returns number of backlight devices found */
static int find_backlights(void)
{
    int found = 0;
    struct udev_enumerate *enumerate;
    struct udev_list_entry *devices, *dev_list_entry;

    enumerate = udev_enumerate_new(udev);
    udev_enumerate_add_match_subsystem(enumerate, "backlight");
    udev_enumerate_scan_devices(enumerate);

    devices = udev_enumerate_get_list_entry(enumerate);
    udev_list_entry_foreach(dev_list_entry, devices) {
        struct udev_device *dev;
        const char *path;
        struct bl_device *new_bl;

        path = udev_list_entry_get_name(dev_list_entry);
        dev = udev_device_new_from_syspath(udev, path);
        new_bl = new_backlight(dev);
        if (new_bl) {
            bl_devices = g_slist_append(bl_devices, new_bl);
            found++;
        }
        udev_device_unref(dev);
    }

    udev_enumerate_unref(enumerate);
    return found;
}

static gboolean bl_monitor_handler(gint fd, GIOCondition cond, gpointer data)
{
    struct udev_device *dev;
    const char *path, *action;
    struct bl_device *new_bl;
    gboolean update = FALSE;

    dev = udev_monitor_receive_device(udev_mon);
    if (dev) {
        path = udev_device_get_syspath(dev);
        action = udev_device_get_action(dev);
        powerd_debug("Received action %s for %s", action, path);

        if (!strcmp(action, "add")) {
            new_bl = new_backlight(dev);
            if (new_bl) {
                bl_devices = g_slist_append(bl_devices, new_bl);
                update = TRUE;
            }
        } else if (!strcmp(action, "remove")) {
            update = remove_backlight(path);
        }

        udev_device_unref(dev);
    }

    if (update)
        update_preferred_backlight();

    return TRUE;
}

static int bl_monitor_init(void)
{
    int fd;

    udev_mon = udev_monitor_new_from_netlink(udev, "udev");
    if (!udev_mon)
        return -ENODEV;

    udev_monitor_filter_add_match_subsystem_devtype(udev_mon, "backlight", NULL);
    udev_monitor_enable_receiving(udev_mon);
    fd = udev_monitor_get_fd(udev_mon);
    udev_mon_source_id = g_unix_fd_add(fd, G_IO_IN, bl_monitor_handler, NULL);
    return 0;
}

static void bl_monitor_deinit(void)
{
    if (!udev_mon)
        return;

    g_source_remove(udev_mon_source_id);
    udev_monitor_unref(udev_mon);
    udev_mon = NULL;
}

int powerd_get_brightness(void)
{
    char path[128];
    char data[32];
    ssize_t ret;
    int fd;

    if (!udev || !preferred_bl)
        return -ENODEV;

    /*
     * libudev interface returned cached values for attributes.
     * When this is called we want to get the actual value from
     * sysfs, so we must read it ourselves.
     */
    ret = snprintf(path, ARRAY_SIZE(path), "%s/%s",
                   udev_device_get_syspath(preferred_bl->dev), "brightness");
    if (ret >= ARRAY_SIZE(path)) {
        powerd_warn("Buffer not big enough for path to brightness attribute");
        return -EIO;
    }

    fd = open(path, O_RDONLY);
    if (fd < 0) {
        powerd_warn("Could not open %s", path);
        return -errno;
    }

    ret = read(fd, data, ARRAY_SIZE(data) - 1);
    if (ret < 0) {
        powerd_warn("Failed to read brightness");
        ret = errno;
        goto error;
    }
    data[ret] = 0;

    close(fd);
    return atoi(data);

error:
    close(fd);
    return (int)ret;
}

int powerd_get_max_brightness(void)
{
    if (!preferred_bl)
        return -ENODEV;
    return preferred_bl->max_brightness;
}

int powerd_set_brightness(int brightness)
{
    char buf[32];
    int ret;

    if (!udev || !preferred_bl)
        return -ENODEV;

    ret = snprintf(buf, ARRAY_SIZE(buf), "%d", brightness);
    if (ret >= ARRAY_SIZE(buf))
        return -EINVAL;

    return udev_device_set_sysattr_value(preferred_bl->dev, "brightness", buf);
}

int powerd_backlight_init(void)
{
    int count;

    udev = udev_new();
    if (!udev) {
        powerd_error("udev_new() failed, backlight support is disabled");
        return -ENODEV;
    }

    /*
     * Start monitoring before scanning so we don't miss devices which
     * might appear in the interim
     */
    bl_monitor_init();

    count = find_backlights(); 
    if (count)
        update_preferred_backlight();
    else
        powerd_warn("No backlight devices found");

    return 0;
}

static void bl_destroy(gpointer data)
{
    struct bl_device *bl = data;
    free_backlight(bl);
}

void powerd_backlight_deinit(void)
{
    if (!udev)
        return;

    bl_monitor_deinit();
    g_slist_free_full(bl_devices, bl_destroy);
    udev_unref(udev);

    bl_devices = NULL;
    preferred_bl = NULL;
    udev = NULL;
}
