/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <errno.h>
#include <uuid/uuid.h>

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>

#include "powerd.h"
#include "powerd-internal.h"
#include "log.h"

struct display_request_internal {
    struct powerd_display_request req;
    const char *name;
    const char *owner;
};

static struct {
    unsigned int state[POWERD_NUM_DISPLAY_STATES];
    unsigned int flags[POWERD_NUM_DISPLAY_FLAGS];
} display_state_count;

static struct powerd_display_request internal_state = {
    .state = POWERD_DISPLAY_STATE_DONT_CARE,
    .flags = 0,
};

static GHashTable *display_request_hash;

static void update_internal_state(void)
{
    struct powerd_display_request new_state;
    int i;

    memset(&new_state, 0, sizeof(new_state));

    /* Default to off if no request for display to be on */
    if (display_state_count.state[POWERD_DISPLAY_STATE_ON] > 0)
        new_state.state = POWERD_DISPLAY_STATE_ON;
    else
        new_state.state = POWERD_DISPLAY_STATE_DONT_CARE;

    for (i = 0; i < POWERD_NUM_DISPLAY_FLAGS; i++) {
        if (display_state_count.flags[i] > 0)
            new_state.flags |= (1U << i);
    }

    if (!memcmp(&internal_state, &new_state, sizeof(new_state))) {
        powerd_debug("display state not changed");
        return;
    }

    powerd_debug("Internal state updated: state %d flags 0x%08x",
                 new_state.state, new_state.flags);
    internal_state = new_state;
    powerd_set_display_state(&internal_state);
    powerd_display_state_signal_emit(&internal_state);
}

static void __add_request(struct powerd_display_request *req)
{
    unsigned int flags;
    int i;

    display_state_count.state[req->state]++;

    flags = req->flags;
    for (i = 0; flags && i < POWERD_NUM_DISPLAY_FLAGS; flags >>= 1, i++) {
        if (flags & 1)
            display_state_count.flags[i]++;
    }
}

static void __remove_request(struct powerd_display_request *req)
{
    unsigned int flags;
    int i;

    if (display_state_count.state[req->state] > 0)
        display_state_count.state[req->state]--;

    flags = req->flags;
    for (i = 0; flags && i < POWERD_NUM_DISPLAY_FLAGS; flags >>= 1, i++) {
        if ((flags & 1) && display_state_count.flags[i] > 0)
            display_state_count.flags[i]--;
    }
}

static void add_request(struct display_request_internal *request)
{
    g_hash_table_insert(display_request_hash, request->req.cookie, request);
    powerd_dbus_name_watch_add(request->owner);
    __add_request(&request->req);
    update_internal_state();
    powerd_account_add_display_req(request->owner, request->name,
                                   &request->req);
}

static gboolean update_request(struct powerd_display_request *new_req)
{
    struct display_request_internal *ireq;

    ireq = g_hash_table_lookup(display_request_hash, new_req->cookie);
    if (!ireq) {
        powerd_debug("Display request to update not found");
        return FALSE;
    }

    __remove_request(&ireq->req);
    __add_request(new_req);

    ireq->req.state = new_req->state;
    ireq->req.flags = new_req->flags;

    update_internal_state();
    powerd_account_update_display_req(ireq->owner, ireq->name, &ireq->req);
    return TRUE;
}

static gboolean remove_request(uuid_t cookie)
{
    struct display_request_internal *ireq;
    struct powerd_display_request req;
    gboolean found = FALSE;

    /*
     * This involves two lookups into the hash, one to find the
     * request so we can retrieve the state and another to remove
     * it. GHashTable doesn't seem to provide any more efficient
     * way to do this; too bad g_hash_table_steal() doesn't return
     * a pointer to the data.
     */
    ireq = g_hash_table_lookup(display_request_hash, cookie);
    if (ireq) {
        req = ireq->req;
        /* We need to remove it from our watch hash and do accounting
         * before we remove it from the state hash or the ireq->owner
         * memory will be freed before we try to use it.
         */
        powerd_dbus_name_watch_remove(ireq->owner);
        powerd_account_clear_display_req(ireq->owner, ireq->name);
        found = g_hash_table_remove(display_request_hash, cookie);
        if (!found)
            powerd_warn("Display request found on lookup but not on remove");
    }

    if (found) {
        __remove_request(&req);
        update_internal_state();
    }

    return found;
}

static gboolean request_valid(struct powerd_display_request *request)
{
    if ((unsigned)request->state >= POWERD_NUM_DISPLAY_STATES) {
        powerd_warn("Invalid display state requested: %d", request->state);
        return FALSE;
    }

    /*
     * XXX: This will warn if we get up to 32 flags, but in that
     * case the check should just be removed.
     */
    if (request->flags & (-1U << POWERD_NUM_DISPLAY_FLAGS)) {
        powerd_warn("Invalid display flags requested: 0x%08x", request->flags);
        return FALSE;
    }

    return TRUE;
}

static int add_request_worker(gpointer data)
{
    struct display_request_internal *req = data;
    add_request(req);
    return 0;
}

/*
 * @request need not refer to persistent memory, as the data from
 * the struct will be copied into internally-managed storage.
 */
static int __powerd_add_display_request(struct powerd_display_request *request,
                                        const char *name, const char *owner)
{
    struct display_request_internal *hash_req;

    if (!request_valid(request))
        return -EINVAL;

    uuid_generate(request->cookie);
    hash_req = g_new(struct display_request_internal, 1);
    hash_req->req = *request;
    hash_req->name = g_strdup(name);
    hash_req->owner = g_strdup(owner);
    powerd_run_mainloop_sync(add_request_worker, hash_req);
    return 0;
}

/*
 * @request need not refer to persistent memory, as the data from
 * the struct will be copied into internally-managed storage.
 */
int powerd_add_display_request(struct powerd_display_request *request,
                               const char *name)
{
    if (!request || !name) {
        powerd_warn("powerd_add_display_request() called with invalid args");
        return -EINVAL;
    }
    return __powerd_add_display_request(request, name, "internal");
}

static int update_request_worker(gpointer data)
{
    struct powerd_display_request *req = data;
    return update_request(req);
}

int powerd_update_display_request(struct powerd_display_request *request)
{
    gboolean found;

    if (!request_valid(request))
        return -EINVAL;

    found = powerd_run_mainloop_sync(update_request_worker, request);
    return found ? 0 : -EINVAL;
}

static int remove_request_worker(gpointer data)
{
    unsigned char *uuid = data;
    return remove_request(uuid);
}

int powerd_remove_display_request(uuid_t cookie)
{
    gboolean found;

    found = powerd_run_mainloop_sync(remove_request_worker, cookie);
    return found ? 0 : -EINVAL;
}

/** dbus method handling **/

gboolean handle_add_display_request(PowerdSource *obj,
                                    GDBusMethodInvocation *invocation,
                                    const char *name, int state, guint32 flags)
{
    struct powerd_display_request req;
    const char *owner;
    char ext_cookie[UUID_STR_LEN];
    int ret;

    memset(&req, 0, sizeof(req));

    owner = g_dbus_method_invocation_get_sender(invocation);
    powerd_debug("%s from %s: state %d flags %#08x",
                 __func__, owner, state, flags);

    req.state = state;
    req.flags = flags;

    ret = __powerd_add_display_request(&req, name, owner);
    if (ret) {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid display request");
    } else {
        powerd_debug("%s: SUCCESS", __func__);
        uuid_unparse(req.cookie, ext_cookie);
        g_dbus_method_invocation_return_value(invocation,
                                              g_variant_new("(s)", ext_cookie));
    }

    return TRUE;
}

gboolean handle_update_display_request(PowerdSource *obj,
                                       GDBusMethodInvocation *invocation,
                                       const gchar *ext_cookie, int state,
                                       guint32 flags)
{
    struct powerd_display_request req;
    uuid_t cookie;
    int ret;

    memset(&req, 0, sizeof(req));

    powerd_debug("%s from %s: cookie: %s state %d flags %#08x",
                 __func__, g_dbus_method_invocation_get_sender(invocation),
                 ext_cookie, state, flags);

    if (uuid_parse(ext_cookie, cookie)) {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid cookie: %s",
                                              ext_cookie);
        return TRUE;
    }

    memcpy(req.cookie, cookie, sizeof(uuid_t));
    req.state = state;
    req.flags = flags;

    ret = powerd_update_display_request(&req);
    if (ret) {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid display request");
    } else {
        powerd_debug("%s: SUCCESS", __func__);
        g_dbus_method_invocation_return_value(invocation, NULL);
    }

    return TRUE;
}

gboolean handle_clear_display_request(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation,
                                      const gchar *ext_cookie)
{
    int ret;
    uuid_t cookie;

    powerd_debug("%s from %s, cookie: %s", __func__,
                 g_dbus_method_invocation_get_sender(invocation), ext_cookie);

    if (uuid_parse(ext_cookie, cookie)) {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid cookie: %s",
                                              ext_cookie);
        return TRUE;
    }

    ret = powerd_remove_display_request(cookie);
    if (ret) {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid cookie: %s", ext_cookie);
    } else {
        g_dbus_method_invocation_return_value(invocation, NULL);
    }

    return TRUE;
}

static int build_display_request_list(GVariantBuilder *builder)
{
    GHashTableIter iter;
    gpointer key, value;
    int count = 0;

    g_hash_table_iter_init(&iter, display_request_hash);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        struct display_request_internal *ireq = value;
        struct powerd_display_request *req = &ireq->req;
        g_variant_builder_add(builder, "(ssiu)", ireq->name, ireq->owner,
                              req->state, req->flags);
        count++;
    }
    return count;
}

gboolean handle_list_display_requests(PowerdSource *obj,
                                      GDBusMethodInvocation *invocation)
{
    GVariant *list, *tuple;
    GVariantBuilder *builder;
    int count;

    builder = g_variant_builder_new(G_VARIANT_TYPE("a(ssiu)"));
    count = build_display_request_list(builder);
    if (count > 0) {
        list = g_variant_builder_end(builder);
    } else {
        g_variant_builder_clear(builder);
        list = g_variant_new_array(G_VARIANT_TYPE("(ssiu)"), NULL, 0);
    }

    tuple = g_variant_new_tuple(&list, 1);
    g_dbus_method_invocation_return_value(invocation, tuple);
    g_variant_builder_unref(builder);

    return TRUE;
}

/* Destructor for hash table */
static void display_request_destroy(gpointer data)
{
    struct display_request_internal *req = data;

    g_free((gpointer)req->name);
    g_free((gpointer)req->owner);
    g_free(req);
}

void display_request_init(void)
{
    display_request_hash = g_hash_table_new_full(powerd_uuid_hash,
                                                 powerd_uuid_equal,
                                                 NULL, display_request_destroy);
}

void display_request_deinit(void)
{
    g_hash_table_destroy(display_request_hash);
}

/*
 * Callers may use this to clear all requests that the specified owner
 * is holding. This is used when the owner drops off of dbus
 */
void
clear_disp_state_by_owner(const char *owner)
{
    GHashTableIter iter;
    gpointer key, value;

    g_hash_table_iter_init(&iter, display_request_hash);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        struct display_request_internal *ireq = value;
        if (!strcmp(owner, ireq->owner)) {
            powerd_account_clear_display_req(ireq->owner, ireq->name);
            __remove_request(&ireq->req);
            powerd_dbus_name_watch_remove(ireq->owner);
            g_hash_table_iter_remove(&iter);
        }
    }
    update_internal_state();
}
