/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib.h>
#include <glib-object.h>

#include <hybris/surface_flinger/surface_flinger_compatibility_layer.h>

#include "powerd.h"
#include "powerd-internal.h"
#include "device-config.h"
#include "log.h"

/* Mir usage hint file path */
#define MIR_HINT_PATH	"/home/phablet/.display-mir"

/* Treat "don't care" display state as off */
#define DISPLAY_STATE_OFF POWERD_DISPLAY_STATE_DONT_CARE

/* Autobrightness only enabled when bright && !disabled */
#define AB_ENABLED_MASK (POWERD_DISPLAY_FLAG_BRIGHT | \
                         POWERD_DISPLAY_FLAG_DISABLE_AUTOBRIGHTNESS)
#define AB_ENABLED      POWERD_DISPLAY_FLAG_BRIGHT

static gint saved_brightness = 255;
static gint target_brightness;
static gint dim_brightness;
static gboolean running_mir = FALSE;

/* Assume screen is off to start; we will force it on */
static uuid_t internal_request_cookie;

/* Currently requested state of the display */
struct powerd_display_request internal_state = {
    .state = DISPLAY_STATE_OFF,
    .flags = 0,
};

/*
 * When using the proximity sensor, our actual state of the screen
 * may not match the requested state in internal_state. Therefore
 * this variable keeps track of the actual state of the screen.
 */
enum powerd_display_state actual_screen_state = DISPLAY_STATE_OFF;

/*
 * Screen state overrides. These are used when powerd needs to force
 * the screen off regardless of the requested display state.
 */
static unsigned screen_off_overrides;

enum fb_state {
    FB_SLEEP,
    FB_AWAKE,

    NUM_FB_STATES
};
static enum fb_state fb_state = FB_AWAKE;

static const char *fb_file_names[] = {
    [FB_SLEEP] = "/sys/power/wait_for_fb_sleep",
    [FB_AWAKE] = "/sys/power/wait_for_fb_wake"
};

static const char *display_state_strs[POWERD_NUM_DISPLAY_STATES] = {
    [POWERD_DISPLAY_STATE_DONT_CARE]    = "off",
    [POWERD_DISPLAY_STATE_ON]           = "on",
};

static const char *display_state_to_str(enum powerd_display_state state)
{
    if (state >= POWERD_NUM_DISPLAY_STATES)
        return "invalid";
    return display_state_strs[state];
}

static gboolean ab_enabled(guint32 flags)
{
    return (flags & AB_ENABLED_MASK) == AB_ENABLED;
}

gboolean display_set_power_mode(int display, char *power_mode)
{
    GError *error = NULL;
    GDBusProxy *unity_proxy = NULL;

    powerd_debug("display_set_power_mode(%s)", power_mode);

    if (!running_mir) {
        if (strcmp(power_mode, "off") == 0)
            sf_blank(display);
        else
            sf_unblank(display);
    } else {
        if (strcmp(power_mode, "on") == 0)
            powerd_hal_signal_activity();

        unity_proxy = g_dbus_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM,
                    G_DBUS_PROXY_FLAGS_NONE,
                    NULL,
                    "com.canonical.Unity.Screen",
                    "/com/canonical/Unity/Screen",
                    "com.canonical.Unity.Screen",
                    NULL,
                    &error);

        if (error != NULL) {
            powerd_warn("could not connect to Unity: %s", error->message);
            g_error_free(error);
            return FALSE;
        }

        GVariant *ret = g_dbus_proxy_call_sync(unity_proxy,
                "setScreenPowerMode",
                g_variant_new("(s)", power_mode),
                G_DBUS_CALL_FLAGS_NONE,
                -1,
                NULL,
                &error);

        if (ret == NULL) {
            powerd_warn("screen power setting failed: %s", error->message);
            g_error_free(error);
            g_object_unref(unity_proxy);
            return FALSE;
        }

        g_object_unref(unity_proxy);
    }

    return TRUE;
}

static void turn_on_display(gboolean autobrightness) {
    powerd_debug("turning on display");
    display_set_power_mode(0, "on");
    if (autobrightness)
        powerd_autobrightness_enable();
    else
        powerd_set_brightness(target_brightness);
}

gboolean powerd_display_enabled(void)
{
    return actual_screen_state == POWERD_DISPLAY_STATE_ON;
}

static int get_target_brightness(gboolean request_bright, int hw_value)
{
    /*
     * If not requesting bright, return the dim brightness, but never
     * return something higher than the current "bright" brightness.
     */
    if (request_bright == FALSE)
        return (saved_brightness < dim_brightness) ?
               saved_brightness : dim_brightness;

    /*
     * If bright is requested and our current state is also bright,
     * just return the current hardware value.
     */
    if ((internal_state.flags & POWERD_DISPLAY_FLAG_BRIGHT) && (hw_value > 0))
        return hw_value;

    /* Otherwise we're going from dim to bright, so return the saved value */
    return saved_brightness;
}

static void update_display_state(struct powerd_display_request *req)
{
    enum powerd_display_state state = req->state;
    gboolean use_ab, using_ab;
    int applied_state;
    int ret;
    int hw_brightness;

    use_ab = ab_enabled(req->flags);
    using_ab = ab_enabled(internal_state.flags);

    hw_brightness = powerd_get_brightness();
    if (hw_brightness < 0)
        hw_brightness = saved_brightness;

    if (!using_ab && hw_brightness != 0 &&
        (internal_state.flags & POWERD_DISPLAY_FLAG_BRIGHT))
        saved_brightness = hw_brightness;
    if (!use_ab)
        target_brightness = get_target_brightness(
                    !!(req->flags & POWERD_DISPLAY_FLAG_BRIGHT),
                    hw_brightness);

    applied_state = screen_off_overrides ? DISPLAY_STATE_OFF : state;

    switch (applied_state) {
    case DISPLAY_STATE_OFF:
        if (actual_screen_state == DISPLAY_STATE_OFF) {
            /* Nothing to do */
            return;
        }

        powerd_debug("turning off display");
        if (using_ab)
            powerd_autobrightness_disable();
        powerd_set_brightness(0);
        if (!display_set_power_mode(0, "off")) {
            powerd_warn("failed to set display power mode, not clearing state");
            return;
        }

        powerd_debug("Releasing internal active state request");
        ret = clear_sys_state_internal(internal_request_cookie);
        if (!ret) {
            char cookie_str[UUID_STR_LEN];
            uuid_unparse(internal_request_cookie, cookie_str);
            powerd_warn("Internal system state request cookie invalid: %s",
                        cookie_str);
        }
        break;
    case POWERD_DISPLAY_STATE_ON:
        if (actual_screen_state != POWERD_DISPLAY_STATE_ON) {
            powerd_debug("Requesting active state internally");
            ret = request_sys_state_internal("display-request",
                                             POWERD_SYS_STATE_ACTIVE,
                                             &internal_request_cookie, NULL);
            if (!ret)
                powerd_warn("Request for active state failed");

            /*
             * If the current state is suspend we need to wait for
             * notification of leaving the active state, otherwise the
             * screen may fail to turn on for devices using earlysuspend.
             * Otherwise we can turn on the screen right away.
             */
            if (fb_state == FB_AWAKE)
                turn_on_display(use_ab);
        } else {
            /* Only changing brightness */
            if (use_ab) {
                if (!using_ab)
                    powerd_autobrightness_enable();
            } else {
                if (using_ab) {
                    powerd_autobrightness_disable();
                    powerd_set_brightness(target_brightness);
                } else if ((req->flags & POWERD_DISPLAY_FLAG_BRIGHT) !=
                           (internal_state.flags & POWERD_DISPLAY_FLAG_BRIGHT)) {
                    powerd_set_brightness(target_brightness);
                }
            }
        }
        break;
    default:
        powerd_warn("Invalid display state %d", applied_state);
        return;
    }

    actual_screen_state = applied_state;
}

static void update_flags(guint32 flags)
{
    int prox_enabled, internal_prox_enabled;

    prox_enabled = (flags & POWERD_DISPLAY_FLAG_USE_PROXIMITY);
    internal_prox_enabled = (internal_state.flags & POWERD_DISPLAY_FLAG_USE_PROXIMITY);

    if (prox_enabled != internal_prox_enabled) {
        if (prox_enabled) {
            powerd_sensors_proximity_enable();
        } else {
            powerd_sensors_proximity_disable();
            powerd_display_clear_override(POWERD_OVERRIDE_REASON_PROXIMITY);
        }
    }
}

/* 
 * *** WARNING ***
 *
 * This should now only be called by display request code. Everyone
 * else must use powerd_{add,remove}_display_request()!!!
 */
void powerd_set_display_state(struct powerd_display_request *req)
{
    powerd_debug("powerd_set_display_state: %s -> %s, 0x%x -> 0x%x",
                 display_state_to_str(internal_state.state),
                 display_state_to_str(req->state),
                 internal_state.flags, req->flags);

    /* Update flags before display state to ensure proximity flag is
     * taken into account */
    update_flags(req->flags);
    update_display_state(req);

    internal_state = *req;
}

static gboolean prox_update_worker(gpointer data)
{
    unsigned long near = (unsigned long)data;
    if (near)
        powerd_display_set_override(POWERD_OVERRIDE_REASON_PROXIMITY);
    else
        powerd_display_clear_override(POWERD_OVERRIDE_REASON_PROXIMITY);
    return FALSE;
}

void powerd_proximity_event(gboolean near)
{
    unsigned long data = (unsigned long)near;
    g_timeout_add(0, prox_update_worker, (gpointer)data);
}

struct apply_override_data {
    enum powerd_override_reason reason;
    int set;
};

static int apply_override_worker(gpointer data)
{
    struct apply_override_data *ovr_data = data;
    unsigned orig_overrides = screen_off_overrides;
    unsigned mask;

    mask = 1 << ovr_data->reason;
    if (ovr_data->set)
        screen_off_overrides |= mask;
    else
        screen_off_overrides &= ~mask;

    if (orig_overrides != screen_off_overrides)
        update_display_state(&internal_state);
    return 0;
}

void powerd_display_set_override(enum powerd_override_reason reason)
{
    struct apply_override_data ovr_data;

    if (reason >= POWERD_NUM_OVERRIDE_REASONS) {
        powerd_debug("Refusing to set invalid override reason (%d)", reason);
        return;
    }

    ovr_data.reason = reason;
    ovr_data.set = 1;
    powerd_run_mainloop_sync(apply_override_worker, &ovr_data);
}

void powerd_display_clear_override(enum powerd_override_reason reason)
{
    struct apply_override_data ovr_data;

    if (reason >= POWERD_NUM_OVERRIDE_REASONS) {
        powerd_debug("Refusing to clear invalid override reason (%d)", reason);
        return;
    }

    ovr_data.reason = reason;
    ovr_data.set = 0;
    powerd_run_mainloop_sync(apply_override_worker, &ovr_data);
}


static int wait_for_file(const char *fname)
{
    int fd, ret;
    char buf;

    fd = open(fname, O_RDONLY);
    if (fd == -1)
        return -errno;

    do {
        ret = read(fd, &buf, 1);
    } while (ret == -1 && errno == EINTR);

    close(fd);
    return ret == -1 ? -errno : 0;
}

static gboolean fb_state_changed(gpointer data)
{
    fb_state = (enum fb_state)data;
    powerd_debug("fb state %s", fb_state == FB_SLEEP ? "sleep" : "awake");
    if (fb_state == FB_AWAKE && powerd_display_enabled())
        turn_on_display(ab_enabled(internal_state.flags));
    return FALSE;
}

static gpointer wait_for_fb_thread(gpointer unused)
{
    unsigned long state = fb_state;

    if (state >= NUM_FB_STATES) {
        powerd_warn("Invalid FB state");
        return NULL;
    }

    while (1) {
        state = (state == FB_SLEEP) ? FB_AWAKE : FB_SLEEP;
        if (wait_for_file(fb_file_names[state])) {
            powerd_warn("Error reading %s", fb_file_names[state]);
            return NULL;
        }
        g_timeout_add(0, fb_state_changed, (gpointer)state);
    }
    return NULL;
}

int powerd_display_init(void)
{
    GError *error;
    GValue v = G_VALUE_INIT;
    int brightness, max_brightness;
    struct stat stats;

    /*
     * Warning: much hand waving follows
     *
     * Right now there's no coherent system brightness policy, so we
     * just do the best we can. For "dim" brightness we try to read
     * the value from the Android config files, otherwise we just go
     * with 5% of maximum. For "bright" brightness, it's more
     * complicated.
     *
     * Our first preference for the bright value is whatever the screen
     * is currently set to. However, the screen could be off or dimmed
     * when powerd starts, so we fall back to 75% of maximum when that
     * happens.
     */

    if (!device_config_get("screenBrightnessDim", &v)) {
        if (G_VALUE_HOLDS_UINT(&v))
            dim_brightness = (int)g_value_get_uint(&v);
        g_value_unset(&v);
    }

    max_brightness = powerd_get_max_brightness();

    if (max_brightness <= 0) {
        powerd_warn("Could not read maximum brightness, guessing at dim/bright values");
        saved_brightness = 100;
        if (dim_brightness == 0)
            dim_brightness = 5;
    } else {
        saved_brightness = (3 * max_brightness) / 4;
        if (dim_brightness == 0)
            dim_brightness = max_brightness / 20;
    }

    brightness = powerd_get_brightness();
    if (brightness > dim_brightness)
        saved_brightness = brightness;

    target_brightness = saved_brightness;

    /* decide whether we are running on Mir or SF */
    if (stat(MIR_HINT_PATH, &stats) == 0)
        running_mir = TRUE;

    /* If kernel supports earlysuspend, start thread to monitor fb state */
    if (!access(fb_file_names[FB_SLEEP], F_OK))
        g_thread_new("powerd_fb_wake_monitor", wait_for_fb_thread, &error);

    return 0;
}
