/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>

#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <uuid/uuid.h>

#include <glib-object.h>
#include <gio/gio.h>
#include "powerd-internal.h"
#include "powerd-dbus.h"
#include "log.h"

#include "libsuspend.h"

#define NOTIFICATION_TIMEOUT_MSECS 2000

struct SysStateRequest {
    const char *name;
    const char *owner;
    uuid_t cookie;
    enum SysPowerStates state;
    int power_state_flags;
};

/*
 * Current system power state. Assume we are suspended to start. We will
 * be forced into the active state during initialization, and this
 * ensures that the system state is consistent with our internal state.
 */
static enum SysPowerStates current_system_state = POWERD_SYS_STATE_SUSPEND;

/*
 * System state requests are stored in a hash. This doesn't allow for
 * quickly finding requests corresponding to a given state, so a
 * per-state count of active requests is also maintained.
 */
static GHashTable *state_request_hash;
static gint state_request_count[POWERD_NUM_POWER_STATES - 1];

/*
 * States for "state transition" state machine.
 *
 * When no state transition is in progress we're in the idle state. When
 * a state transition is initiated we proceed through the other states
 * in order. When finished, we start at the begining if another state
 * transition is pending or return to idle otherwise.
 */
enum state_transition_state {
    STATE_TRANSITION_IDLE,      /* No transition in progress */
    STATE_TRANSITION_DEQUEUE,   /* Dequeueing next state transition */
    STATE_TRANSITION_NOTIFY,    /* Notifying clients / waiting for acks */
    STATE_TRANSITION_COMPLETE,  /* Completing the state transition */

    NUM_STATE_TRANSITIONS
};

static enum state_transition_state cur_state_transition_state = STATE_TRANSITION_IDLE;

#define INVALID_STATE -1
static GQueue queued_state_changes;
static enum SysPowerStates pending_system_state = INVALID_STATE;
static guint state_machine_source_id;
static gint update_pending;
static gboolean ack_pending;

static gboolean suspend_active = FALSE;

static const char power_request_wakelock_name[] = "powerd_power_request";
gint suspend_block_count;

static void state_transition_process(void);

enum SysPowerStates current_system_power_state(void)
{
    return current_system_state;
}

/* simple helper for debugging */
const gchar *
state_to_string(int state)
{
    switch (state) {
        case POWERD_SYS_STATE_ACTIVE: return "ACTIVE";
        case POWERD_SYS_STATE_SUSPEND: return "SUSPEND";
        default: return "UNKNOWN";
    }
}

/*
 * block_suspend and unblock_suspend are only for use when new requests
 * are received, to ensure that autosuspend doesn't suspend the system
 * before the main loop can disable it after an active state request
 * is received.
 */
static void block_suspend(void)
{
    int ret;

    if (g_atomic_int_add(&suspend_block_count, 1) == 0) {
        powerd_debug("libsuspend: acquire_wake_lock: %s", power_request_wakelock_name);
        ret = libsuspend_acquire_wake_lock(power_request_wakelock_name);
        if (ret)
            powerd_warn("Could not acquire wake lock");
    }
}

static void unblock_suspend(void)
{
    gint old_count;
    int ret;
    gboolean retry;

    do {
        old_count = g_atomic_int_get(&suspend_block_count);
        if (old_count == 0) {
            powerd_warn("Suspend not blocked, refusing to unblock");
            return;
        }
        retry = !g_atomic_int_compare_and_exchange(&suspend_block_count,
                                                   old_count, old_count - 1);
        if (!retry && old_count == 1) {
            powerd_debug("libsuspend: release_wake_lock: %s", power_request_wakelock_name);
            ret = libsuspend_release_wake_lock(power_request_wakelock_name);
            if (ret)
                powerd_warn("Could not release wake lock");
        }
    } while (retry);
}

/* Returns TRUE if the state request is valid, FALSE otherwise.
 * Nobody can request the SUSPEND state. */
static gboolean
is_valid_state_request(int state)
{
    return state > POWERD_SYS_STATE_SUSPEND && state < POWERD_NUM_POWER_STATES;
}

/*
 * Must only be called from main loop.
 *
 * internal callers should pass in NULL for the builder, its only used
 * by dbus callers. Returns a count of the total number of active
 * requests.
 */
static guint
list_sys_requests_internal(GVariantBuilder *builder)
{
    GHashTableIter iter;
    gpointer key, value;
    guint count = 0;

    g_hash_table_iter_init(&iter, state_request_hash);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        struct SysStateRequest *req = value;
        g_variant_builder_add(builder, "(ssi)", req->name, req->owner,
                              req->state);
        count++;
    }
    return count;
}

/* Dump the requests to stdout and return the list via dbus */
gboolean
handle_list_sys_requests(PowerdSource *obj, GDBusMethodInvocation *invocation)
{
    GVariant *list, *tuple = NULL;
    GVariantBuilder *builder;
    guint count = 0;

    builder = g_variant_builder_new(G_VARIANT_TYPE("a(ssi)"));
    count = list_sys_requests_internal(builder);
    if (count>0) {
        list = g_variant_builder_end(builder);
    }
    else {
        g_variant_builder_clear(builder);
        list = g_variant_new_array(G_VARIANT_TYPE("(ssi)"), NULL, 0);
    }
    tuple = g_variant_new_tuple(&list,1);
    g_dbus_method_invocation_return_value(invocation, tuple);
    g_variant_builder_unref(builder);
    return TRUE;
}

/* Should be called only on main loop */
static int request_sys_state_worker(gpointer data)
{
    struct SysStateRequest *sr = data;
    g_hash_table_insert(state_request_hash, sr->cookie, sr);
    state_request_count[sr->state - 1] += 1;
    powerd_dbus_name_watch_add(sr->owner);
    powerd_account_request_sys_state(sr->owner, sr->name);
    return 0;
}

/* Note: owner is NULL for internal usage */
gboolean
request_sys_state_internal(const char *name, int state, uuid_t *cookie,
                           const gchar *owner)
{
    struct SysStateRequest *sr = NULL;

    if (cookie == NULL) {
        powerd_error("you need to pass in memory for a cookie");
        return FALSE;
    }

    if (!is_valid_state_request(state))
    {
        powerd_warn("invalid state requested: %d",state);
        return FALSE;
    }

    sr = g_new(struct SysStateRequest, 1);
    sr->name = g_strdup(name);
    sr->state = (enum SysPowerStates)state;
    if (owner) {
        sr->owner = g_strdup(owner);
    }
    else {
        sr->owner = g_strdup("internal");
    }
    uuid_generate(sr->cookie);
    memcpy(*cookie, sr->cookie, sizeof(sr->cookie));

    powerd_run_mainloop_sync(request_sys_state_worker, sr);
    update_system_state();
    return TRUE;
}

gboolean
handle_request_sys_state(PowerdSource *obj, GDBusMethodInvocation *invocation,
			 const char *name, int state)
{
    gboolean retval;
    const char *owner;
    uuid_t cookie;
    char ext_cookie[UUID_STR_LEN];

    owner = g_dbus_method_invocation_get_sender(invocation);
    powerd_debug("handle_requestSysState from %s - %s (%d)", owner,
                 state_to_string(state), state);

    retval = request_sys_state_internal(name, state, &cookie, owner);
    if (retval == TRUE) {
        powerd_debug("handle_requestSysState - SUCCESS");
        uuid_unparse(cookie, ext_cookie);
        g_dbus_method_invocation_return_value(invocation,
                                              g_variant_new("(s)", ext_cookie));
    } else {
            g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                G_DBUS_ERROR_INVALID_ARGS, "Invalid state request");
    }
    return TRUE;
}

static void __clear_sys_request(struct SysStateRequest *req)
{
    powerd_dbus_name_watch_remove(req->owner);
    state_request_count[req->state - 1] -= 1;
    powerd_account_clear_sys_state(req->owner, req->name);
}

/* Must only be called from main loop */
static int
clear_sys_state_worker(gpointer data)
{
    unsigned char *uuid = data;
    struct SysStateRequest *sr;
    int found = FALSE;

    /*
     * This involves two lookups into the hash, one to find the
     * request so we can retrieve the state and another to remove
     * it. GHashTable doesn't seem to provide any more efficient
     * way to do this; too bad g_hash_table_steal() doesn't return
     * a pointer to the data.
     */
    sr = g_hash_table_lookup(state_request_hash, uuid);
    if (sr) {
        /* We need to remove it from our watch hash before we remove it
         * from the state hash or the sr->owner memory will be freed
         * before we try to use it.
         */
        __clear_sys_request(sr);
        found = g_hash_table_remove(state_request_hash, uuid);
        if (!found)
            powerd_warn("State request found on lookup but not on remove");
    }

    return found;
}

gboolean
clear_sys_state_internal(uuid_t cookie)
{
    char cookie_str[UUID_STR_LEN];
    int found;

    found = powerd_run_mainloop_sync(clear_sys_state_worker, cookie);
    if (!found) {
        uuid_unparse(cookie, cookie_str);
        powerd_warn("request (%s) not found", cookie_str);
        return FALSE;
    }

    update_system_state();
    return TRUE;
}

gboolean
handle_clear_sys_state(PowerdSource *obj, GDBusMethodInvocation *invocation,
                       gchar *ext_cookie)
{
    gboolean retval;
    uuid_t cookie;

    powerd_debug("handle_clearSysState from %s, cookie: %s",
        g_dbus_method_invocation_get_sender(invocation), ext_cookie);

    if (uuid_parse(ext_cookie, cookie)) {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid cookie: %s", ext_cookie);
        return TRUE;
    }

    retval = clear_sys_state_internal(cookie);
    if (retval == TRUE) {
        g_dbus_method_invocation_return_value(invocation, NULL);
    } else {
        g_dbus_method_invocation_return_error(invocation, G_DBUS_ERROR,
                                              G_DBUS_ERROR_INVALID_ARGS,
                                              "Invalid cookie: %s", ext_cookie);
    }
    return TRUE;
}

static int enter_suspend(void)
{
    powerd_debug("libsuspend: enter_suspend.");
    int ret = libsuspend_enter_suspend();
    if (!ret)
        suspend_active = TRUE;
    return ret;
}

static int exit_suspend(void)
{
    powerd_debug("libsuspend: exit_suspend.");
    int ret = libsuspend_exit_suspend();
    if (!ret) {
        suspend_active = FALSE;
    }
    return ret;
}

gboolean powerd_suspend_active(void)
{
    return suspend_active;
}

/* Must only be called from main loop */
static enum SysPowerStates max_requested_state(void)
{
    enum SysPowerStates ret = POWERD_SYS_STATE_SUSPEND;
    int i;

    for (i = POWERD_NUM_POWER_STATES - 1; i > POWERD_SYS_STATE_SUSPEND; i--) {
        if (state_request_count[i - 1] != 0) {
            ret = i;
            break;
        }
    };

    return ret;
}

static gboolean delayed_state_transition_process(gpointer unused)
{
    state_transition_process();
    return FALSE;
}

static enum state_transition_state process_state_dequeue(guint *delay)
{
    int ret;

    *delay = 0;

    pending_system_state = (enum SysPowerStates)g_queue_pop_head(&queued_state_changes);
    if (pending_system_state == current_system_state) {
        powerd_debug("pending state == current state, discarding");
        pending_system_state = INVALID_STATE;
        return STATE_TRANSITION_IDLE;
    }

    if (current_system_state == POWERD_SYS_STATE_SUSPEND) {
        powerd_debug("exiting suspend");
        ret = exit_suspend();
        if (ret)
            powerd_warn("Failed to exit suspend: %d", ret);
    }

    /*
     * XXX: If going to suspend we may want to delay a few seconds here
     * in case the user immediately tries to wake the device again after
     * an activity timeout.
     */

    /* Must call libsuspend_prepare_suspend() _before_ emitting signal */
    if (pending_system_state == POWERD_SYS_STATE_SUSPEND) {
        powerd_debug("libsuspend: prepare_suspend.");
        ret = libsuspend_prepare_suspend();
        if (ret)
            powerd_warn("Failed to prepare for suspend: %d", ret);
    }

    ack_pending = powerd_client_transition_start(pending_system_state);
    powerd_sys_state_signal_emit(pending_system_state);

    *delay = ack_pending ? NOTIFICATION_TIMEOUT_MSECS : 0;
    return STATE_TRANSITION_NOTIFY;
}

static enum state_transition_state process_state_notify(guint *delay)
{
    if (ack_pending) {
        powerd_warn("Timeout waiting for acks on state transition");
        ack_pending = FALSE;
    }

    powerd_client_transition_finish(pending_system_state);
    *delay = 0;
    return STATE_TRANSITION_COMPLETE;
}

static enum state_transition_state process_state_complete(guint *delay)
{
    int ret;
    gboolean allow_suspend;

    *delay = 0;

    /*
     * If more requests are in the queue or new requests are present
     * above our current target state, do not suspend so that those
     * requests can be processed.
     */
    allow_suspend = !(!g_queue_is_empty(&queued_state_changes) ||
                      max_requested_state() > pending_system_state);

    if (pending_system_state == POWERD_SYS_STATE_SUSPEND && allow_suspend) {
        powerd_debug("entering suspend");
        ret = enter_suspend();
        if (ret)
            powerd_warn("Failed to enter suspend: %d", ret);
    }

    powerd_debug("Transition to %s complete",
                 state_to_string(pending_system_state));
    current_system_state = pending_system_state;
    pending_system_state = INVALID_STATE;

    /*
     * If state changes are queued, go back to the dequeue state. If
     * nothing is queued and max_requested_state() > pending_system_state
     * there should be another run of update_system_state() scheduled to
     * take care of this.
     */
    return g_queue_is_empty(&queued_state_changes) ?
           STATE_TRANSITION_IDLE : STATE_TRANSITION_DEQUEUE;
}

static void state_transition_process(void)
{
    guint delay = 0;

    state_machine_source_id = 0;
    do {
        switch (cur_state_transition_state) {
        case STATE_TRANSITION_DEQUEUE:
            cur_state_transition_state = process_state_dequeue(&delay);
            break;
        case STATE_TRANSITION_NOTIFY:
            cur_state_transition_state = process_state_notify(&delay);
            break;
        case STATE_TRANSITION_COMPLETE:
            cur_state_transition_state = process_state_complete(&delay);
            break;
        default:
	    powerd_error("Invalid transition state, setting to idle");
	    cur_state_transition_state = STATE_TRANSITION_IDLE;
            break;
        }
    } while (delay == 0 &&
             cur_state_transition_state != STATE_TRANSITION_IDLE);

    if (delay != 0 && cur_state_transition_state != STATE_TRANSITION_IDLE) {
        state_machine_source_id = g_timeout_add(delay,
                                                delayed_state_transition_process,
                                                NULL);
    }
}

/* Must be called from main loop */
void power_request_transition_acked(void)
{
    if (cur_state_transition_state != STATE_TRANSITION_NOTIFY) {
        powerd_error("%s called while not waiting for acks", __func__);
        return;
    }

    powerd_debug("All acks received for state transition");
    ack_pending = FALSE;
    if (state_machine_source_id) {
        g_source_remove(state_machine_source_id);
        state_machine_source_id = g_timeout_add(0,
                                                delayed_state_transition_process,
                                                NULL);
    }
}

static void check_queued_state_changes(void)
{
    /*
     * If state change in progress we'll process the next queued state
     * change when finished.
     */
    if (cur_state_transition_state != STATE_TRANSITION_IDLE) {
        powerd_debug("state change in progress, delaying");
        return;
    }

    if (g_queue_is_empty(&queued_state_changes)) {
        powerd_debug("no pending state changes");
        return;
    }

    /* Start processing state transition */
    cur_state_transition_state = STATE_TRANSITION_DEQUEUE;
    state_transition_process();
}


/*
 * XXX: Currently the coalescing is very simplistic and may need
 * improvement.
 */
static void enqueue_state_change(enum SysPowerStates state)
{
    gboolean queue_empty;

    powerd_debug("Enqueue state change to %s", state_to_string(state));

    /*
     * If the new request is the same as the one we're currently
     * processing we can drop all pending requests (including the
     * new one).
     */
    if (pending_system_state == state) {
        powerd_debug("State == pending state, discarding");
        g_queue_clear(&queued_state_changes);
        return;
    } else if (pending_system_state == INVALID_STATE) {
        queue_empty = g_queue_is_empty(&queued_state_changes);

        /*
         * If no state changes are pending or in progress and the
         * requested state is the same as the current state, do nothing.
         */
        if (queue_empty && state == current_system_state) {
            powerd_debug("queue empty && state == current, discarding");
            return;
        }

        /*
         * If we aren't processing a request and the current request is
         * the same as the next reques in the queue, all intermediate
         * requests can be dropped. We do this by clearing the queue and
         * then adding in the new request.
         *
         * XXX: This situation is probably impossible currently.
         */
        if (!queue_empty &&
            (enum SysPowerStates)g_queue_peek_tail(&queued_state_changes) == state) {
            powerd_debug("new state == next queued state, discarding all intermediate requests");
            g_queue_clear(&queued_state_changes);
        }
    }

    g_queue_push_tail(&queued_state_changes, (gpointer)state);
    check_queued_state_changes();
}

static gboolean update_system_state_handler(gpointer unused)
{
    /*
     * We must clear update_pending just before calling max_requested_state()
     * to avoid races with adding new requests. We may end up with an
     * occassional redundant system state update, but that is harmless.
     */
    g_atomic_int_set(&update_pending, 0);
    enqueue_state_change(max_requested_state());

    /*
     * We assume that after enqueue_state_change() completes autosuspend
     * will have been disabled if any active requests are present. This
     * is currently true, but if the assumption is ever broken it will
     * expose us to potential races.
     */
    unblock_suspend();
    return FALSE;
}

void update_system_state(void)
{
    if (g_atomic_int_compare_and_exchange(&update_pending, 0, 1)) {
        block_suspend();
        g_timeout_add(0, update_system_state_handler, NULL);
    }
}

/* Destructor for the state request object */
static void sys_state_request_destroy(gpointer data)
{
    struct SysStateRequest *sr = (struct SysStateRequest *)data;
    g_free((gpointer)sr->name);
    g_free((gpointer)sr->owner);
    g_free(sr);
}

void power_request_init(void)
{
    state_request_hash = g_hash_table_new_full(powerd_uuid_hash,
                                               powerd_uuid_equal,
                                               NULL, sys_state_request_destroy);
    g_queue_init(&queued_state_changes);
}

void power_request_deinit(void)
{
    g_hash_table_destroy(state_request_hash);
    g_queue_clear(&queued_state_changes);
}

/*
 * Callers may use this to clear all requests that the specified owner
 * is holding. This is used when the owner drops off of dbus
 */
void
clear_sys_state_by_owner(const char *owner)
{
    GHashTableIter iter;
    gpointer key, value;

    g_hash_table_iter_init(&iter, state_request_hash);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        struct SysStateRequest *req = value;
        if (!strcmp(owner, req->owner)) {
            __clear_sys_request(req);
            g_hash_table_iter_remove(&iter);
        }
    }
    update_system_state();
}
