/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "cpufreq_management.h"
#include "stringutil.h"
#include "globals.h"
#include "event_management.h"
#include "cpu.h"

#include <cpufreq.h>

#define PROC_CPUINFO_FILE "/proc/cpuinfo"

using namespace Powersave::Globals;

CpufreqManagement::CpufreqManagement()
{
	if (!config_obj->current_scheme->CPUFREQ_ENABLED) {
		pDebug(DBG_INFO, "cpufrequency scaling is disabled in configuration file");
		return;
	}

	_initial_cpu_count = 0;

        if (access("/sys/devices/system/cpu/cpu0/cpufreq", F_OK) < 0) {
                return;
        }

	_control_modes = 0;

	int num_cpus = getCPUCount();

	if (num_cpus < 0) {
		pDebug(DBG_ERR, "No CPUs found in system");
		return;
	}

	if (!getCoresAffectedCpus(_cpus, num_cpus)) {
		pDebug(DBG_WARN, "Failed to evaluate one or more affected_cpu files. "
		       "Trying to parse /proc/cpuinfo...");
		if (!getCoresProcfile(_cpus, num_cpus)) {
			pDebug(DBG_ERR, "Could not figure out cpu core dependencies");
			return;
		}
	}
}

CpufreqManagement::~CpufreqManagement()
{
	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {

		delete *it;
	}

	_cpufreq_objects.clear();
}

void CpufreqManagement::init()
{
	_initial_control_mode = config_obj->current_scheme->CPUFREQ_CONTROL;

	for (list< list< int > >::iterator it = _cpus.begin();
	     it != _cpus.end(); ++it) {

		initSingleInterface(*it, config_obj->current_scheme->CPUFREQ_CONTROL);
	}
}

void CpufreqManagement::checkCPUHotplug()
{
	bool cpu_enabled = false;

	for (int i = _initial_cpu_count - 1; i > 0; --i) {
		CPU h(i);
		if (i >= config_obj->current_scheme->MAX_CPUS_ONLINE
		    && config_obj->current_scheme->MAX_CPUS_ONLINE != 0) {
			h.disable();
		} else {
			int ret = h.enable();
			if (ret > 0)
				cpu_enabled = true;
		}
	}

	if (cpu_enabled)
		adjustSpeeds();
}

bool CpufreqManagement::initSingleInterface(list< int > &cores, CPUFREQ_CONTROL_MODE control_mode)
{
	CPUFreq_Interface *interface;
	
	switch (control_mode) {
	case CPUFREQ_USERSPACE:
		interface = new CPUFreq_Userspace(cores);
		if(!interface->init()) {
			delete interface;
			pDebug(DBG_WARN, "Could not init userspace governor");
			return false;
		}
		break;
	case CPUFREQ_KERNEL:
		interface = new CPUFreq_Kernel(
			cores, config_obj->current_scheme->POLL_INTERVAL * 1000);
		
		if(!interface->init()) {
			delete interface;
			pDebug(DBG_WARN, "Kernel ondemand governor not working, "
			       "trying userspace instead");
			if (!initSingleInterface(cores, CPUFREQ_USERSPACE)) {
				pDebug(DBG_ERR, "Neither ondemand nor userspace governor working");
			}
			return false;
		}
		
		break;
	default:
		pDebug(DBG_ERR, "Wrong control mode");
			return false;
	}
	
	interface->setConfigs(config_obj->current_scheme->CPU_HIGH_LIMIT,
			      config_obj->current_scheme->JUMP_CPU_FREQ_MAX_LIMIT,
			      config_obj->current_scheme->CPUFREQ_HYSTERESIS,
			      config_obj->current_scheme->CONSIDER_NICE);
	
	if (!interface->readFrequencies()) {
		delete interface;
		pDebug(DBG_ERR, "Could not read available frequencies");
		return false;
	}

	_control_modes |= control_mode;

	_cpufreq_objects.push_back(interface);
	
	return true;
}

bool CpufreqManagement::isSupported()
{
	return !_cpufreq_objects.empty();
}

void CpufreqManagement::adjustSpeeds()
{
	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {

		if (!(*it)->online())
			continue;
		(*it)->adjustSpeed();
	}
}

int CpufreqManagement::setModes(CPUFREQ_MODE mode, EventManagement *eM)
{
	int ret = 0;

	switch (mode) {
	case _DYNAMIC:
		eM->executeEvent("processor.dynamic");
		break;
	case _PERFORMANCE:
		eM->executeEvent("processor.performance");
		break;
	case _POWERSAVE:
		eM->executeEvent("processor.powersave");
		break;
	}

	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {

		ret |= (*it)->setMode(mode);
	}

	return ret;
}


void CpufreqManagement::reinitSpeeds()
{
	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {
		(*it)->reinitSpeed();
	}
}

void CpufreqManagement::rereadFrequencies()
{
	int ret = 0;

	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {
		ret |= (*it)->readFrequencies();
	}

	if (!ret)
		adjustSpeeds();
}

int CpufreqManagement::setGovernors(const string &new_governor)
{
	int ret = 0;

	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {
		ret |= (*it)->setGovernor(new_governor);
	}

	/* if (ret != 0) -> something wrong */
	return ret;
}


CPUFREQ_MODE CpufreqManagement::getMode()
{
	return (*_cpufreq_objects.begin())->getMode();
	
}

void CpufreqManagement::setConfigs()
{
	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {

		(*it)->setConfigs(config_obj->current_scheme->CPU_HIGH_LIMIT,
				  config_obj->current_scheme->JUMP_CPU_FREQ_MAX_LIMIT,
				  config_obj->current_scheme->CPUFREQ_HYSTERESIS,
				  config_obj->current_scheme->CONSIDER_NICE);
	}
}

CPUFREQ_CONTROL_MODE CpufreqManagement::controlMode()
{
	return _initial_control_mode;
}

bool CpufreqManagement::getCoresAffectedCpus(list< list<int> > &cpu_list, int num_cpus)
{
	for (int i = 0; i < num_cpus; i++) {
		list< int > int_cpus;

		cpufreq_affected_cpus *affected_cpus = cpufreq_get_affected_cpus(i);

		while (affected_cpus != NULL) {
			int_cpus.push_back(affected_cpus->cpu);

			if (affected_cpus->next == NULL) {
				cpufreq_put_affected_cpus(affected_cpus->first);
				break;
			}
			else {
				affected_cpus = affected_cpus->next;
			}
		}

		if (!int_cpus.size()) {
			pDebug(DBG_INFO, "failed to get affected_cpus for cpu %d", i);
			continue;
		}

		cpu_list.push_back(int_cpus);
		int_cpus.clear();
	}

	cpu_list.unique();

	if (!cpu_list.size())
		return false;

	return true;
}

bool CpufreqManagement::getCoresProcfile(list< list< int > > & cpu_list, int num_cpus)
{
	for (int i = 0; i < num_cpus; i++) {
		list< int > core_list;
		if (readCpu(core_list, i)) {
			cpu_list.push_back(core_list);
		}
	}

	if (cpu_list.size() == 0) {
		return false;
	}

	cpu_list.unique();

/*	printf("added: %d\n", cpu_list.size());
	int i = 0;
	for (list< list< int > >::iterator it = cpu_list.begin();
	     it != cpu_list.end(); ++it) {
		
		printf("cpu %d, ", i);

		for (list< int >::iterator ti = it->begin();
		     ti != it->end(); ++ti) {

			printf("core: %d, ", *ti);
		}

		printf("\n");
	}
*/
	return true;
}

bool CpufreqManagement::readCpu(list< int > & core_list, int cpu)
{
	FILE *procfile;
	char line[256];
	int cpu_search = -1, physical_id_search = -1;
	int physical_id = -1;
	bool multi_core = false;

	if ((procfile = fopen(PROC_CPUINFO_FILE, "r")) == NULL){
		pDebug(DBG_DIAG, "Could not open file: %s; Error: %s",
		       PROC_CPUINFO_FILE, strerror(errno));
		return false;
	}

	/* search physical id of this CPU ****************/
	while (fgets(line, 255, procfile) != NULL){
		if (!memcmp(line, "processor", 9)){
			sscanf(line, "processor : %d", &cpu_search);
			if (cpu_search == cpu){
				while (fgets(line, 255, procfile) != NULL){
					if (!memcmp(line, "physical id", 11)){
						sscanf(line, "physical id : %d", &physical_id);
						break;
					}
				}
				break;
			}
		}
	}
	fclose(procfile);
	/* search physical id of this CPU ****************/

	/* add its siblings to this CPU ********/

	/* Could be optimized -> only initialisation -> only run once 
	         -> don't care for performance here
	*/
	if ((procfile = fopen(PROC_CPUINFO_FILE, "r")) == NULL){
		pDebug(DBG_DIAG, "Could not open file: %s; Error: %s",
		       PROC_CPUINFO_FILE, strerror(errno));
		return false;
	}
	while (fgets(line, 255, procfile) != NULL){
		if (!memcmp(line, "processor", 9)){
			sscanf(line, "processor : %d", &cpu_search);
			while (fgets(line, 255, procfile) != NULL){
				if (!memcmp(line, "physical id", 11)){
					sscanf(line, "physical id : %d", &physical_id_search);
					if (physical_id_search  == physical_id){
						pDebug (DBG_DIAG, "Multi core detected: "
							"CPU %d is part of socket %d", 
							cpu_search, physical_id);
						if (!siblingExists(core_list, cpu_search)) {
							core_list.push_back(cpu_search);
						}
						multi_core = true;
					}
					break;
				}
			}
		}
	}

	if (!multi_core)
		core_list.push_back(cpu);

	return true;
}

bool CpufreqManagement::siblingExists(list< int > core_list, int core)
{
	list< int >::iterator it = find(core_list.begin(), core_list.end(), core);

	if (it == core_list.end()) {
		return false;
	}
	return true;
}

bool CpufreqManagement::hasControlMode(CPUFREQ_CONTROL_MODE control_mode)
{
	if (control_mode & _control_modes) {
		return true;
	}

	return false;
}

int CpufreqManagement::enableCPU(int cpu)
{
	CPU c(cpu);

	if (!c.hotpluggable())
		return -1;

	int ret = c.enable();
	if (ret > 0)
		adjustSpeeds();
	return ret;
}

int CpufreqManagement::disableCPU(int cpu)
{
	// we cannot disable the first CPU, it runs the main timer
	if (cpu == 0)
		return -1;

	CPU c(cpu);

	if (!c.hotpluggable())
		return -1;

	return c.disable();
}

#ifdef CPUFREQ_MEASURE

void CpufreqManagement::startMeasures()
{
	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {

		for (int y = 0; y <= MAX_SPEEDS; y++) {
			(*it)->time_spent[y] = 0;
		}

		(*it)->t_count = 0;
		(*it)->cpu_load_sum = 0;
	}

	time(start_time);
	struct tm *tm = new(struct tm);
	tm = localtime(start_time);

	FILE *logFile = fopen(MEASURE_LOG_FILE, "a+");
	if (!logFile) {
		pDebug(DBG_ERR, "Could not open log file %s", MEASURE_LOG_FILE);
		exit(1);
	}
	fprintf(logFile, "\nStart cpufreq measures at: %d.%d.%d at %d:%d:%d\n",
		tm->tm_mday, tm->tm_mon, tm->tm_year, tm->tm_hour, tm->tm_min, tm->tm_sec);
	fclose(logFile);
	free(tm);
}

void CpufreqManagement::stopMeasures()
{
	FILE *logFile = fopen(MEASURE_LOG_FILE, "a+");
	if (!logFile) {
		pDebug(DBG_ERR, "Could not open log file %s", MEASURE_LOG_FILE);
		exit(1);
	}
	time_t *stop_time = new(time_t);
	tm *tm = new(struct tm);
	time(stop_time);
	tm = localtime(stop_time);
	unsigned long measure_time = *stop_time - *start_time;

	for (list< CPUFreq_Interface* >::iterator it = _cpufreq_objects.begin();
	     it != _cpufreq_objects.end(); ++it) {

		for (int k = 0; ((CPUFreq_Userspace*)(*it))->speeds_kHz[k] != 0; k++) {
			fprintf(logFile, "%d\t: %d minutes %d seconds\n",
				((CPUFreq_Userspace*)(*it))->speeds_kHz[k],
				((CPUFreq_Userspace*)(*it))->time_spent[k] / 60000,
				(((CPUFreq_Userspace*)(*it))->time_spent[k] / 1000) % 60);
		}
		if (((CPUFreq_Userspace*)(*it))->t_count > 0)
			fprintf(logFile, "Average CPU usage: %##f - Count: %d\n",
				(double)((CPUFreq_Userspace*)(*it))->cpu_load_sum /
				((CPUFreq_Userspace*)(*it))->t_count,
				((CPUFreq_Userspace*)(*it))->t_count);
	}
	fprintf(logFile, "\nStop cpufreq measures at: %d.%d.%d at %d:%d:%d\n",
		tm->tm_mday, tm->tm_mon, tm->tm_year, tm->tm_hour, tm->tm_min, tm->tm_sec);

	fprintf(logFile, "Total measure time: %d minutes %d seconds - Check: %d minutes %d seconds\n",
		measure_time / 60, (measure_time) % 60,
		(((CPUFreq_Userspace*)(*_cpufreq_objects.begin()))->t_count * polling_interval / 1000) / 60,
		(((CPUFreq_Userspace*)(*_cpufreq_objects.begin()))->t_count * polling_interval / 1000) % 60);
	fclose(logFile);
	free(tm);
	free(stop_time);
}
#endif



