/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *            Copyright (C) 2005 SUSE Linux Products GmbH                  *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"

#include "powersave_hal.h"
#include "powerlib_local.h"
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>

#define ACPI_BATTERY_DIR  "/proc/acpi/battery"
#define ACPI_AC_DIR "/proc/acpi/ac_adapter"

extern DBusError dbus_error;
extern LibHalContext *hal_ctx;


int getBatteriesInfo(BatteryGeneral *battery_info)
{
	int last_full_capacity_sum = 0;
	int remaining_capacity_sum = 0;

	if (battery_info == NULL) {
                errno = EINVAL;
                return -3;
        }
	
	if (!ps_hal_init()) {
		return HAL_ERROR;
	}

	battery_info->remaining_minutes = UNKNOWN;
	battery_info->remaining_percent = UNKNOWN;
	battery_info->charging_state = CHARG_STATE_UNKNOWN;

	int num_primary_batteries = numBatteries();

	/* if there are no batteries */
	if (num_primary_batteries == 0) {
		return NO_DEVICE_ERROR;
	}

	int valid_batteries = 0;
	int rate = 0;
	BatteryGeneral bg; 
	BatteryDetailed bd;
	for (int i = 0; i < num_primary_batteries; ++i) {

		if (getBatteryInfo(i, &bg) < 1) {
			continue;
		}

		valid_batteries++;

		/* calc remaining time from HAL */
		if (bg.remaining_minutes != UNKNOWN) {
			if (battery_info->remaining_minutes == UNKNOWN)
				battery_info->remaining_minutes = 0;
			battery_info->remaining_minutes += bg.remaining_minutes;
		}

		/* calc charging state */
		battery_info->charging_state |= bg.charging_state;

		if (getBatteryDetailedInfo(i, &bd) < 0) {
			continue;
		}

		/* calculate the charging rate for the multiple battery case */
		if (bg.remaining_minutes > 0 && bd.remaining_capacity != UNKNOWN) {
			int to_go = 0;
			if (bg.charging_state & CHARG_STATE_CHARGING) {
				if (bd.last_full_capacity != UNKNOWN)
					to_go = bd.last_full_capacity - bd.remaining_capacity;
			} else {
				to_go = bd.remaining_capacity;
			}
			rate += (to_go / bg.remaining_minutes);
		}

		/* FIXME: can we have one battery with valid last_full and a
			  second battery with last_full == UNKNOWN?
			  This would make those values totally bogus. */
		if (bd.last_full_capacity != UNKNOWN) {
			last_full_capacity_sum += bd.last_full_capacity;

			/* see the FIXME above, same applies here... */
			if (bd.remaining_capacity != UNKNOWN) {
				remaining_capacity_sum += bd.remaining_capacity;
			}
		}
	}

	if (rate > 0 && valid_batteries > 1) {	// with APM, this will never happen.
		if (battery_info->charging_state & CHARG_STATE_CHARGING) {
			battery_info->remaining_minutes =
				(last_full_capacity_sum - remaining_capacity_sum) / rate;
		} else {
			battery_info->remaining_minutes = remaining_capacity_sum / rate;
		}
	}

	if (last_full_capacity_sum > 0 && remaining_capacity_sum >= 0)
		battery_info->remaining_percent =
			(remaining_capacity_sum * 100.0) / last_full_capacity_sum;

	return (valid_batteries == 0) ? 0 : 1;
}

int getBatteryInfo(const int no, BatteryGeneral *bg)
{
 	char *value = NULL;
	int batt_no = -1; // will later hold the native batt.number of param no
	char found_native_battery = 0; // true if mapping could find corresponding native battery

        if (bg == NULL) {
                errno = EINVAL;
                return -3;
        }
	
	if (!ps_hal_init()) {
		return HAL_ERROR;
	}

	*bg = (struct BatteryGeneral) {
		UNKNOWN, 		//remaining_percent
		UNKNOWN, 		//remaining_minutes
		CHARG_STATE_UNKNOWN	//charging_state
	};

	char **batteries = NULL;
	int num_native_batteries = 0;
	batteries = libhal_find_device_by_capability(hal_ctx, "battery",
						     &num_native_batteries, &dbus_error);

        if (dbus_error_is_set(&dbus_error)) {
                pDebug(DBG_INFO, "Could not get list of batteries in system: %s",
                       dbus_error.message);
                ps_hal_free();
                dbus_error_free(&dbus_error);
                libhal_free_string_array(batteries);
                return HAL_ERROR;
        }
	
        /* if battery does not exist */
        if (no > num_native_batteries - 1 || no < 0) {
                libhal_free_string_array(batteries);
                return -2;
        }

	/* if there are no batteries */
        if (num_native_batteries == 0) {
                libhal_free_string_array(batteries);
                return NO_DEVICE_ERROR;
        }
	
	/* find the native batterynumber of (primary) batterynumber no*/
	for (int i = 0; i < num_native_batteries; ++i) {
		/* check if found battery is a primary source (e.g. no ups) */
		value = ps_hal_get_property_string(batteries[i], "battery.type");
		if (value != NULL) {
			if (strcmp(value, "primary") == 0) {
				batt_no++;
				if (batt_no == no) {
					batt_no = i;
					found_native_battery = 1;
					libhal_free_string(value);
					value = NULL;
					break;
				}
			}
		}
		libhal_free_string(value);
		value = NULL;
	}

	/* check if native battery could be found */
	if (!found_native_battery) {
			libhal_free_string_array(batteries);
			return NO_DEVICE_ERROR;
	}

	int present = ps_hal_get_property_bool(batteries[batt_no],
					       "battery.present");
	if (present < 0) {
		return HAL_ERROR;
	}
	else if (!present) {
		return 0;
	}

	/* get remaining percent */
	bg->remaining_percent = ps_hal_get_property_int(batteries[batt_no],
							"battery.charge_level.percentage");
	if (bg->remaining_percent < 0) {
		bg->remaining_percent = UNKNOWN;
	}

	/* get charging state */
	if (ps_hal_get_property_bool(batteries[batt_no], "battery.rechargeable.is_charging")) {
		bg->charging_state |= CHARG_STATE_CHARGING;

	        /* get remaining time */
	        int time = ps_hal_get_property_int(batteries[batt_no], "battery.remaining_time");
	        if (time >= 0) 
                        bg->remaining_minutes = time / 60;
		
	} else if (ps_hal_get_property_bool(batteries[batt_no], "battery.rechargeable.is_discharging")) {
		bg->charging_state |= CHARG_STATE_DISCHARGING;

		/* get remaining time */
                int time = ps_hal_get_property_int(batteries[batt_no], "battery.remaining_time");
                if (time >= 0)
                        bg->remaining_minutes = time / 60;
	}

	libhal_free_string_array(batteries);

	return 1;
}

int numBatteries(void)
{
	if (!ps_hal_init()) {
		/* error message already thrown in initHal() */
		return HAL_ERROR;
	}

	char **batteries;
	int num_cur_native_batteries = 0;
	static int num_native_batteries = 0; /* do some "caching" here */
	static int num_primary_batteries = 0;
	char *value = NULL;
	batteries = libhal_find_device_by_capability(hal_ctx, "battery",
						     &num_cur_native_batteries, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
                pDebug(DBG_INFO, "Could not get list of batteries in system: %s",
                       dbus_error.message);
                ps_hal_free();
                dbus_error_free(&dbus_error);
                libhal_free_string_array(batteries);
                return HAL_ERROR;
        }

	if (num_cur_native_batteries != num_native_batteries) {
		num_native_batteries = num_cur_native_batteries;
		num_primary_batteries = 0;
		
		for (int i = 0; i < num_native_batteries; ++i) {
			/* check if found battery is a primary source (e.g. no ups) */
			value = ps_hal_get_property_string(batteries[i], "battery.type");
			if (value != NULL) {
				if (!strcmp(value, "primary")) {
					num_primary_batteries++;
				}
			}
			libhal_free_string(value);
			value = NULL;
		}
	} 

	libhal_free_string_array(batteries);

	return num_primary_batteries;
}

int getBatteryDetailedInfo(const int no, BatteryDetailed * bd)
{
 	char *value = NULL;
	int batt_no = -1; /* will hold the native batt.number of primary batt.number no*/
	char found_native_battery = 0; /* will be true if mapping found corresponding native battery */
	int chkACPI;

 	if (bd == NULL) {
		errno = EINVAL;
		return -3;
	}

	if (no > MAX_BATTERIES - 1 || no < 0) {
		return -2;
	}

	if (!ps_hal_init()) {
                /* error message already thrown in initHal() */
                return HAL_ERROR;
        }

	chkACPI = checkACPI();
	if (chkACPI != ACPI && chkACPI != APM) {
                return NO_ACPI_ERROR;
        }
	
	if (chkACPI == ACPI) {
		/* check whether module has been loaded and device is available */
		int res = check_ACPI_dir(ACPI_BATTERY_DIR);
		if (res < 0)
			/* returns either NO_DEVICE_ERROR or NO_MODULE_ERROR */
			return res;
	}

	char **batteries = NULL;
	int num_native_batteries = 0;
	batteries = libhal_find_device_by_capability(hal_ctx, "battery",
						     &num_native_batteries, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_INFO, "Could not get list of batteries in system: %s",
		       dbus_error.message);
		ps_hal_free();
		dbus_error_free(&dbus_error);
		libhal_free_string_array(batteries);
		return 0;
	}

	/* if there are no batteries */
	if (num_native_batteries == 0) {
		libhal_free_string_array(batteries);
		return NO_DEVICE_ERROR;
	}

	/* if battery does not exist */
	if (no > num_native_batteries - 1 || no < 0) {
		libhal_free_string_array(batteries);
		return -2;
	}

        /* find the native batterynumber of (primary) batterynumber no*/
        for (int i = 0; i < num_native_batteries; ++i) {
                /* check if found battery is a primary source (e.g. no ups) */
                value = ps_hal_get_property_string(batteries[i], "battery.type");
                if (value != NULL) {
                        if (strcmp(value, "primary") == 0) {
                                batt_no++;
                                if (batt_no == no) {
                                        batt_no = i;
					found_native_battery = 1;
                                        libhal_free_string(value);
					value = NULL;
                                        break;
                                }
                        }
                }
                libhal_free_string(value);
		value = NULL;
        }

        /* check if native battery could be found */
        if (!found_native_battery) {
                libhal_free_string_array(batteries);
                return NO_DEVICE_ERROR;
        }
		
	
	int present = ps_hal_get_property_bool(batteries[batt_no],
					       "battery.present");

	if (present < 0)
		return -2;
	/* if not present initialise everything with unknown and return! */
	else {
		if (present) {
			bd->present = PRESENT_YES;
		} else {
			bd->present = PRESENT_NO;
			bd->capacity_state = UNKNOWN;
			bd->charging_state = CHARG_STATE_UNKNOWN;
			bd->design_capacity_low = UNKNOWN;
			bd->design_voltage = UNKNOWN;
			bd->battery_technology = UNKNOWN;
			bd->last_full_capacity = UNKNOWN;
			bd->design_capacity = UNKNOWN;
			bd->present_voltage = UNKNOWN;
			bd->power_unit = UNKNOWN;
			bd->remaining_capacity = UNKNOWN;
			bd->present_rate = UNKNOWN;
			bd->capacity_granularity_2 = UNKNOWN;
			bd->capacity_granularity_1 = UNKNOWN;
			bd->design_capacity_warning = UNKNOWN;
			strcpy(bd->OEM_info, "UNKNOWN");
			strcpy(bd->serial_number, "UNKNOWN");
			strcpy(bd->model_number, "UNKNOWN");
			strcpy(bd->battery_type, "UNKNOWN");
			bd->alarm_limit = UNKNOWN;
			pDebug(DBG_INFO, "Battery %s not present", batteries[batt_no]);
			libhal_free_string_array(batteries);
			return 0;
		}
	}

	/* capacity state */
	value = ps_hal_get_property_string(batteries[batt_no],
					   "battery.charge_level.capacity_state");
	if (value != NULL) {
		if (!strcmp(value, "ok"))
			bd->capacity_state = STATE_OK;
		else if (!strcmp(value, "critical"))
			bd->capacity_state = STATE_CRIT;
		else {
			bd->capacity_state = UNKNOWN;
		}
		libhal_free_string(value);
	}

	/* --- get charging state --- */
	bd->charging_state = ps_hal_get_property_bool(batteries[batt_no],
						      "battery.rechargeable.is_charging");

	if (bd->charging_state == 1)
		bd->charging_state = CHARG_STATE_CHARGING;
	else if (bd->charging_state == 0)
		bd->charging_state = CHARG_STATE_DISCHARGING;
	else
		bd->charging_state = CHARG_STATE_UNKNOWN;

	/* present_rate */
	bd->present_rate = ps_hal_get_property_int(batteries[batt_no], "battery.charge_level.rate");
	if (bd->present_rate <= 0) {
		bd->present_rate = UNKNOWN;
	}

	/* --- get remaining capacity --- */
	bd->remaining_capacity = ps_hal_get_property_int(batteries[batt_no],
							 "battery.charge_level.current");
	if (bd->remaining_capacity < 0) {
		bd->remaining_capacity = UNKNOWN;
	}

	/* get power unit */
	value = ps_hal_get_property_string(batteries[batt_no], "battery.charge_level.unit");
	if (value != NULL) {
		if (!strcmp(value, "mWh")) {
			bd->power_unit = WATT_H;
		} else if (!strcmp(value, "mA")) {
			bd->power_unit = AMP;
		} else {
			bd->power_unit = UNKNOWN;
		}
		libhal_free_string(value);
	}

	/* present_voltage */
	bd->present_voltage = ps_hal_get_property_int(batteries[0],
						      "battery.voltage.current");
	if (bd->present_voltage < 0) {
		bd->present_voltage = UNKNOWN;
	}

	/* design_capacity */
	bd->design_capacity = ps_hal_get_property_int(batteries[batt_no],
						      "battery.charge_level.design");
	if (bd->design_capacity < 0) {
		bd->design_capacity = UNKNOWN;
	}

	/* last_full_capacity */
	bd->last_full_capacity = ps_hal_get_property_int(batteries[batt_no],
							 "battery.charge_level.last_full");
	if (bd->design_capacity < 0) {
		bd->last_full_capacity = UNKNOWN;
	}

	/* battery_technology */
	bd->battery_technology = ps_hal_get_property_bool(batteries[batt_no],
							  "battery.is_rechargeable");
	if (bd->battery_technology) {
		bd->battery_technology = TECH_RECHARGEABLE;
	} else if (!bd->battery_technology) {
		bd->battery_technology = TECH_NON_RECHARGEABLE;
	} else {
		bd->battery_technology = UNKNOWN;
	}

	/* desing voltage */
	bd->design_voltage = ps_hal_get_property_int(batteries[batt_no], "battery.voltage.design");
	if (bd->design_voltage < 0) {
		bd->design_voltage = UNKNOWN;
	}

	/* design_capacity_warning */
	bd->design_capacity_warning = ps_hal_get_property_int(batteries[batt_no],
							      "battery.charge_level.warning");
	if (bd->design_capacity_warning < 0) {
		bd->design_capacity_warning = UNKNOWN;
	}

	/* design_capacity_low */
	bd->design_capacity_low = ps_hal_get_property_int(batteries[batt_no],
							  "battery.charge_level.low");
	if (bd->design_capacity_low < 0) {
		bd->design_capacity_low = UNKNOWN;
	}

	/* capacity_granularity_1 */
	bd->capacity_granularity_1 = ps_hal_get_property_int(batteries[batt_no],
							     "battery.charge_level.granularity_1");
	if (bd->capacity_granularity_1 < 0) {
		bd->capacity_granularity_1 = UNKNOWN;
	}

	/* capacity_granularity_1 */
	bd->capacity_granularity_2 = ps_hal_get_property_int(batteries[batt_no],
							     "battery.charge_level.granularity_2");
	if (bd->capacity_granularity_2 <= 0) {
		bd->capacity_granularity_2 = UNKNOWN;
	}

	/* alarm limit */
	bd->alarm_limit = ps_hal_get_property_int(batteries[batt_no], "battery.alarm.design");
	if (bd->alarm_limit <= 0) {
		bd->alarm_limit = UNKNOWN;
	}

	value = ps_hal_get_property_string(batteries[batt_no], "battery.model");
	if (value != NULL) {
		strcpy(bd->model_number, value);
		libhal_free_string(value);
	}
	else {
		strcpy(bd->model_number, "UNKNOWN");
	}

	value = ps_hal_get_property_string(batteries[batt_no], "battery.serial");
	if (value != NULL) {
		strcpy(bd->serial_number, value);
		libhal_free_string(value);
	}
	else {
		strcpy(bd->serial_number, "UNKNOWN");
	}

	value = ps_hal_get_property_string(batteries[batt_no], "battery.technology");
	if (value != NULL) {
		strcpy(bd->battery_type, value);
		libhal_free_string(value);
	}
	else {
		strcpy(bd->battery_type, "UNKNOWN");
	}

	value = ps_hal_get_property_string(batteries[batt_no], "battery.vendor");
	if (value != NULL) {
		strcpy(bd->OEM_info, value);
		libhal_free_string(value);
	}
	else {
		strcpy(bd->OEM_info, "UNKNOWN");
	}

	libhal_free_string_array(batteries);

	return 0;
}

int getACAdapterStatus(void)
{
	int test;
	int ret;

	if (!ps_hal_init()) {
		/* error message already thrown in initHal() */
		return HAL_ERROR;
	}

	char **ac_adapter=NULL;
	char *ac_device=NULL;
	int num=0;
	int ac_state=-1;
	int dev_ac_state=-1;

	test = checkACPI();
	/* --------- ACPI ------------- // */

	if (test == ACPI) {
		/* TODO: Only check for module here, because device
		 * availability is checked for above */
		ret = check_ACPI_dir(ACPI_AC_DIR);
		if (ret < 0)
			/* returns either NO_DEVICE_ERROR or NO_MODULE_ERROR */
			return ret;
	} else if (test == NOT_SUPPORTED) {
		pDebug(DBG_DIAG, "Neither APM nor ACPI support found");
		return AC_UNKNOWN;
	}

	ac_adapter = libhal_find_device_by_capability(hal_ctx, "ac_adapter", &num, &dbus_error);
	
	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_ERR, "Could not get ac_adapter device: %s", dbus_error.message);
		ps_hal_free();
		dbus_error_free(&dbus_error);
		libhal_free_string_array(ac_adapter);
		
		return NO_DEVICE_ERROR;
	}
	
	if (num > 0) {
		/* now could handle multiple ac-adapter (not just only taking the first):
		 * will give AC_ONLINE as soon as one of the adaperts is present
		 * will give AC_OFFLINE if at least one adapter is capable of giving status
		 * 	and those are all not-present */
		for (int i=0; i<num; i++)
		{
			ac_device = strdup(ac_adapter[i]);
			dev_ac_state = ps_hal_get_property_bool(ac_device, "ac_adapter.present");
			if (dev_ac_state == 1)
				ac_state = 1;
			else if (!dev_ac_state && ac_state == -1)
			{
				ac_state = 0;
			} 
			free(ac_device);
		}
		libhal_free_string_array(ac_adapter);
	} else {
		libhal_free_string_array(ac_adapter);
		return NO_DEVICE_ERROR;
	}
	
	if (ac_state == 1)
		return AC_ONLINE;
	else if (!ac_state)
		return AC_OFFLINE;
	else
		return AC_UNKNOWN;
}


/* only for ACPI!!!
return x < 0 on error
returns 0 if alarm has already sub-ceded(alarm will still be set)
returns positive value if alarm has been set successfully
*/

int setBatteryAlarm(int percent)
{
	int x, max_present_bats;
	/* divide_factor: factor on which all batteries are above the wanted percent (statistically) */
	int remaining_capacity_sum, last_full_capacity_sum, rem_Perc_all;
	float divide_factor = 0.0;
	remaining_capacity_sum = 0;
	last_full_capacity_sum = 0;
	int alarm_value = 0;
	char file[MAX_FILE_PATH + 1] = "";
	char dev_name[MAX_FILE_PATH + 1] = "";
	char verify[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	int ret = 1;
	FILE *fp;

	if (!ps_hal_init()) {
		/* error message already thrown in initHal() */
		return HAL_ERROR;
	}

	if (checkACPI() != ACPI) {
		return NO_ACPI_ERROR;
	}

	/* check whether module has been loaded and device is available */
	ret = check_ACPI_dir(ACPI_BATTERY_DIR);
	if (ret < 0)
		/* returns either NO_DEVICE_ERROR or NO_MODULE_ERROR */
		return ret;

	if (percent <= 0 || percent > 100) {
		pDebug(DBG_DIAG, "illegal percent value for battery alarm: %d", percent);
		return -1; //UNKNOWN
	}

	char **batteries = NULL;
	int num_native_batteries = 0;
      	int num_primary_batteries = numBatteries();
	batteries = libhal_find_device_by_capability(hal_ctx, "battery",
						     &num_native_batteries, &dbus_error);

	if (num_primary_batteries == 0) {
		libhal_free_string_array(batteries);
		return -1;
	}

	BatteryDetailed *bd[num_primary_batteries];

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_INFO, "Could not get list of batteries in system: %s",
		       dbus_error.message);
		ps_hal_free();
		dbus_error_free(&dbus_error);
		libhal_free_string_array(batteries);
		return UNKNOWN;
	}

	for (x = 0; x < num_primary_batteries; x++) {
		bd[x] = (BatteryDetailed *) malloc(sizeof(BatteryDetailed));
	}
	/* iterate present battery devices. Don't iterate on existing devices: */
	/* Imagine BAT0 -> not present, BAT1 -> present */
	for (x = 0; (getBatteryDetailedInfo(x, bd[x]) == 0) && x < num_primary_batteries; x++) {
		/* I could also get general bat info again, but this would */
		/*result in reading out battery state one more time ... */

		if (bd[x]->present == PRESENT_YES) {
			if (bd[x]->remaining_capacity > 0) {
				if (bd[x]->last_full_capacity > 0) {
					last_full_capacity_sum += bd[x]->last_full_capacity;
					remaining_capacity_sum += bd[x]->remaining_capacity;
				}
			}
		}
	}
	max_present_bats = x;
	/* calculate remaining percent of the capacity of all sumerized batteries ** */
	if (last_full_capacity_sum > 0 && remaining_capacity_sum > 0) {
		rem_Perc_all = (int)roundf(((float)remaining_capacity_sum / (float)last_full_capacity_sum) * 100.0);
		if (rem_Perc_all > 140 || rem_Perc_all < 0) {
			pDebug(DBG_DIAG, "bogus value %d%% remaining, set to 100", rem_Perc_all);
			rem_Perc_all = 100;
		} else if (rem_Perc_all > 100)
			rem_Perc_all = 100;
	} else {
		pDebug(DBG_INFO, "reading last_full and remaining_capacity failed.");
		goto ERROR;
	}
	/*****************************************************************************/
	if (percent > 0)
		divide_factor = (float)rem_Perc_all / percent;

	pDebug(DBG_INFO, "set alarms: rem_Perc_all: %d, want alarm at: %d, divisor: %f", rem_Perc_all, percent, divide_factor);
	for (x = 0; x < max_present_bats && x < num_primary_batteries; x++) {
		if (bd[x]->present != PRESENT_YES)
			continue;
		if (divide_factor < 1) {
			// we would set the alarm higher than the current charge level.
			pDebug(DBG_DIAG, "divisor: %f, smaller than 1", divide_factor);
			goto ERROR;
		}
		alarm_value = (int)roundf(bd[x]->remaining_capacity / divide_factor);

		pDebug(DBG_DIAG, "Battery %d, remaining: %d, last full: %d, set alarm to: %d",
		       x, bd[x]->remaining_capacity, bd[x]->last_full_capacity, alarm_value);

		/*******  write alarm for each battery ************************/
		if (getDirEntry(x, dev_name, sizeof(dev_name), ACPI_BATTERY_DIR) < 0) {
			pDebug(DBG_DIAG, "could not get acpi directory name for primary battery %d", x);
			goto ERROR;
		}
		
		char *path = NULL;
		/* check if we need a mapping from primary to native battery-numbering */
		if (num_primary_batteries == num_native_batteries) {
			path = ps_hal_get_property_string(batteries[x], "linux.acpi_path");
		} else {
			int batt_no = -1;
		        for (int i = 0; i < num_native_batteries; ++i) {
		                char *value;
		                /* check if found battery is a primary source (e.g. no ups) */
		                value = ps_hal_get_property_string(batteries[i], "battery.type");
		                if (value != NULL) {
		                        if (strcmp(value, "primary") == 0) {
		                                batt_no++;
		                                if (batt_no == x) {
		                                        batt_no = i;
		                                        libhal_free_string(value);
		                                        break;
		                                }
		                        }
		                }
		                libhal_free_string(value);
		        }
			path = ps_hal_get_property_string(batteries[batt_no], "linux.acpi_path");
		}

		if (path == NULL) {
			goto ERROR;
		}
 
		if (strlen(path) + strlen("/alarm") + 1 > MAX_FILE_PATH) {
			pDebug(DBG_DIAG, "Path %s/alarm too long", path);
			libhal_free_string(path);
			goto ERROR;
		} else {
			strcat(file, path);
			strcat(file, "/alarm");
			
			libhal_free_string(path);

			if (_write_line(file, "%d\n", alarm_value) < 0) {
				pDebug(DBG_INFO, "Could not write alarm value");
				goto ERROR;
			} else {
				/* don't set return value has already been set to 0 or initialised with 1 */
/* ##################### please review me ################################################ */
// can this work? and what fp are we using if this fails?
				if ((fp = fopen(file, "r")) == NULL) {
					pDebug(DBG_DIAG, "Could not read alarm settings made (%s)", strerror(errno));
				}
/* ##################### please review me ################################################ */

				if (getColonValue(fp, verify, sizeof(verify), temp, sizeof(temp)) == 1) {
					if (atoi(verify) != alarm_value) {
						pDebug(DBG_DIAG, "failure. wrote: %d, read: %s", alarm_value, verify);
						fclose(fp);
						goto ERROR;
					}
					pDebug(DBG_DEBUG, "Alarm value %d successfully written to %s", alarm_value, file);
				} else {
					pDebug(DBG_INFO, "Writing of alarm value couldn't be verified");
				}
				fclose(fp);
			}
		}
		/*******  write alarm for each battery ************************/
	}
	for (x = 0; x < num_primary_batteries; x++) {
		if (bd[x] != NULL)
			free(bd[x]);
	}
	libhal_free_string_array(batteries);
	return ret;

ERROR:
	for (x = 0; x < num_primary_batteries; x++) {
		if (bd[x] != NULL)
			free(bd[x]);
	}
	libhal_free_string_array(batteries);
	return UNKNOWN;

}

