#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <net/route.h>

#include "osdep.h"

#include <config.h>
#include <support.h>
#include <log.h>

#include <ip/ipsupport.h>

#define	_PATH_PROC_NET_ROUTE	"/proc/net/route"
#define	LEN_PROC_LINE	256
#define	MAX_IFS	10

extern int sockFd;
extern unsigned int kernelVersion;
static int numIfs=0;

static struct rtable {
    char name[IFNAMSIZ];
    struct in_addr gway;
    struct in_addr mask;
    struct in_addr dest;
} defaultGway;
    
static struct iflist {
    char name[IFNAMSIZ];
    struct in_addr l_addr;
    struct in_addr r_addr;
    short flags;
} ifList[MAX_IFS];

int
SysIpInit()
{
    struct sockaddr_in *sin;
    struct ifconf ifc;
    static struct ifreq ifs[MAX_IFS];
    int i;

    memset(&ifc, 0, sizeof(struct ifconf));
    ifc.ifc_len = sizeof(ifs);
    ifc.ifc_req = ifs;
    if (ioctl(sockFd, SIOCGIFCONF, &ifc) < 0) {
	LogError("SysIpInit SIOCGIFCONF");
	return(-1);
    }
    numIfs = ifc.ifc_len / sizeof(struct ifreq);
    if (numIfs > MAX_IFS) numIfs = MAX_IFS;
    for (i = 0; i < numIfs; i ++) {
	strcpy(ifList[i].name, ifs[i].ifr_name);
	sin = (struct sockaddr_in *)&(ifs[i].ifr_addr);
	ifList[i].l_addr.s_addr = sin->sin_addr.s_addr;
	if (ioctl(sockFd, SIOCGIFFLAGS, &ifs[i]) < 0) {
	    LogError("SysIpInit SIOCGIFFLAGS");
	    return(-1);
	}
	ifList[i].flags = ifs[i].ifr_flags;
	if (ifList[i].flags & IFF_POINTOPOINT) {
	    if (ioctl(sockFd, SIOCGIFDSTADDR, &ifs[i]) < 0) {
		LogError("SysIpInit SIOCGIFDSTADDR");
		return(-1);
	    }
	    ifList[i].r_addr.s_addr = sin->sin_addr.s_addr;
	} else {
	    if (ioctl(sockFd, SIOCGIFNETMASK, &ifs[i]) < 0) {
		LogError("SysIpInit SIOCGIFNETMASK");
		return(-1);
	    }
	    ifList[i].r_addr.s_addr = sin->sin_addr.s_addr;
	}
    }
    /* Backup default route */
    return(0);
}

void
SysIpSaveRoute()
{
    FILE *fp;
    u_int32_t dest, gway;
    char name[IFNAMSIZ], tmp[LEN_PROC_LINE];

    if (!defaultGway.gway.s_addr
	&& (fp = fopen(_PATH_PROC_NET_ROUTE, "r")) != NULL) {

	fgets(tmp, sizeof(tmp), fp);
	while (fgets(tmp, sizeof(tmp), fp)) {
	    sscanf(tmp, "%s %X %X", name, &dest, &gway);
	    if (!dest) {
		defaultGway.gway.s_addr = gway;
		strcpy(defaultGway.name, name);
		break;
	    }
	}
	fclose(fp);
    }
}

void
SysIpRestoreRoute()
{
    if (!defaultGway.gway.s_addr) return;
    SuperPrivilege(TRUE);
    SysIpRoute(IPRT_ADD, 0, defaultGway.gway.s_addr, 0,
	       defaultGway.name);
    SuperPrivilege(FALSE);
}

bool_t
SysIpAddressCheck(struct in_addr addr, char *ifname)
{
    int i;

    for (i = 0; i < numIfs; i ++) {
	if (!strcmp(ifList[i].name, ifname)) continue;
	if (ifList[i].r_addr.s_addr == addr.s_addr) {
	    fprintf(stderr, "IP address %s in use\n",
		    inet_ntoa(addr));
	    return(FALSE);
	}
    }
    return(TRUE);
}

u_int32_t
SysIpLocalAddress()
{
    u_int32_t addr=0;
    int i, j;
    bool_t found;
    static const char *ifNames[] = {IFNAME};

    SysIpInit();
    for (i = 0; i < numIfs; i ++) {
	found = FALSE;
	for (j = 0; j < (int)(sizeof(ifNames)/sizeof(ifNames[0])); j++) {
	    if (strncmp(ifList[i].name, ifNames[j], strlen(ifNames[j])) == 0) {
		found = TRUE;
		break;
	    }
	}
	if (!found) {
	    addr = ifList[i].l_addr.s_addr;
	    if (strncmp(ifList[i].name, "lo", sizeof("lo") - 1)) break;
	}
    }
    return(addr);
}

int
SysIpRoute(int cmd, u_int32_t dest, u_int32_t gway, u_int32_t mask,
	   char *ifname)
{
    struct rtentry rt;
    FILE *fp;

    memset(&rt, 0, sizeof(struct rtentry));
    ((struct sockaddr_in *)&rt.rt_dst)->sin_addr.s_addr = dest;
    ((struct sockaddr_in *)&rt.rt_gateway)->sin_addr.s_addr = gway;
    rt.rt_dst.sa_family = rt.rt_gateway.sa_family
	= rt.rt_genmask.sa_family =  AF_INET;
    rt.rt_dev = ifname;
    if (cmd & IPRT_ADD) {
	if (cmd & IPRT_HOST) rt.rt_flags = RTF_HOST;
	else {
	    /*
	     * dest==0: default gw -> mask=0
	     */
/*	    if (!dest && !gway) return(-1);*/
	    if (mask || !dest)
		((struct sockaddr_in *)&rt.rt_genmask)->sin_addr.s_addr
		    = mask;
	    ((struct sockaddr_in *)&rt.rt_dst)->sin_addr.s_addr &= mask;
	    if (gway) rt.rt_flags = RTF_GATEWAY;
	}
	rt.rt_flags |= RTF_UP;
	if (ioctl(sockFd, SIOCADDRT, &rt) < 0) {
	    LogError("SysIpRoute SIOCADDRT");
	    return(-1);
	}
    } else {
	rt.rt_flags = RTF_GATEWAY;
	if (ioctl(sockFd, SIOCDELRT, &rt) < 0 && errno != ESRCH) {
	    LogError("SysIpRoute SIOCDELRT");
	    return(-1);
	}
    }
    if (!dest && (cmd & IPRT_ADD)) {
	if ((fp = fopen(_PATH_PROC_NET_ROUTE, "r")) != NULL) {
	    u_int32_t dest0, gway0;
	    char name[IFNAMSIZ], tmp[LEN_PROC_LINE];

	    fgets(tmp, sizeof(tmp), fp);
	    while (fgets(tmp, sizeof(tmp), fp)) {
		sscanf(tmp, "%s %X %X", name, &dest0, &gway0);
		if (!dest0 && gway != gway0)
		    SysIpRoute(cmd & ~IPRT_ADD, 0, gway0, 0, name);
	    }
	    fclose(fp);
	}
    }
    return(0);
}

/*
 *
 * ipconfig local remote mask mtu
 *  v2.1-> iproute addhost route local
 *  v2.0-> none
 *
 */

int
SysIpConfig(u_int32_t l_addr, u_int32_t r_addr, u_int32_t mask,
	    u_int16_t mtu, char *ifname)
{
    struct sockaddr_in *sin;
    struct ifreq ifreq;
    char ebuf[80];

    strcpy(ifreq.ifr_name, ifname);
    sin = (struct sockaddr_in *)&(ifreq.ifr_addr);
    sin->sin_family = AF_INET;
    if (mtu != MTU_DOWNFLAG) {
	sin->sin_addr.s_addr = l_addr;
	if (ioctl(sockFd, SIOCSIFADDR, &ifreq) < 0) {
	    sprintf(ebuf, "SIOCSIFADDR(%s)", inet_ntoa(sin->sin_addr));
	    LogError(ebuf);
	    return(-1);
	}
	sin->sin_addr.s_addr = r_addr;
	if (ioctl(sockFd, SIOCSIFDSTADDR, &ifreq) < 0) {
	    sprintf(ebuf, "SIOCSIFDSTADDR(%s)", inet_ntoa(sin->sin_addr));
	    LogError(ebuf);
	    return(-1);
	}
	sin->sin_addr.s_addr = mask;
	if (ioctl(sockFd, SIOCSIFNETMASK, &ifreq) < 0) {
	    sprintf(ebuf, "SIOCSIFNETMASK(%s)", inet_ntoa(sin->sin_addr));
	    LogError(ebuf);
	    return(-1);
	}
	if (mtu > 0 && mtu < MTU_MAX) {
	    ifreq.ifr_mtu = mtu;
	    if (ioctl(sockFd, SIOCSIFMTU, &ifreq) < 0) {
		sprintf(ebuf, "SIOCSIFMTU(%s)", mtu);
		LogError(ebuf);
		return(-1);
	    }
	}
    }
    if (ioctl(sockFd, SIOCGIFFLAGS, &ifreq) < 0) {
	LogError("SysIpConfig SIOCGIFFLAGS");
	return(-1);
    }
    ifreq.ifr_flags &= ~IFF_BROADCAST;
    if (mtu == MTU_DOWNFLAG)
	ifreq.ifr_flags &= ~IFF_UP;
    else
	ifreq.ifr_flags |= IFF_UP|IFF_POINTOPOINT;
    if (ioctl(sockFd, SIOCSIFFLAGS, &ifreq) < 0) {
	LogError("SIOCSIFFLAGS");
	return(-1);
    }
    if (kernelVersion < 0x020100 && (r_addr || l_addr))
	SysIpRoute(IPRT_ADD|IPRT_HOST, r_addr, l_addr, mask, ifname);
    return(0);
}

int
SysProxyArp(u_int32_t r_addr)
{
    static struct arpreq arpr;
    struct ifreq ifr;
    int i;

    if (!r_addr) {
	if (!arpr.arp_flags) return(0);
	if (ioctl(sockFd, SIOCDARP, &arpr) < 0 && errno != ENXIO)
	    LogError(SysProxyArp, SIOCDARP);
	memset(&arpr, 0, sizeof(arpr));
	return(0);
    }
    for (i = 0; i < numIfs; i ++) {
	if (!(ifList[i].flags & IFF_POINTOPOINT)
	    && !(ifList[i].r_addr.s_addr &
		 (ifList[i].l_addr.s_addr ^ r_addr))) {
	    memset(&ifr, 0, sizeof(ifr));
	    strcpy(ifr.ifr_name, ifList[i].name);
	    if (ioctl(sockFd, SIOCGIFHWADDR, &ifr) < 0)
		LogError("SysProxyArp SIOCGIFHWADDR");
	    else break;
	}
    }
    if (i == numIfs) return(-1);
    memcpy(&arpr.arp_pa, &ifr.ifr_hwaddr, sizeof(struct sockaddr));
    strcpy(arpr.arp_dev, ifr.ifr_name);
    ((struct sockaddr_in *)&arpr.arp_pa)->sin_addr.s_addr = r_addr;
    arpr.arp_flags = ATF_PERM|ATF_PUBL;
    arpr.arp_pa.sa_family = AF_INET;
    if (ioctl(sockFd, SIOCSARP, &arpr) < 0) {
	LogError("SysProxyArp SIOCSARP");
	memset(&arpr, 0, sizeof(arpr));
	return(-1);
    }
    return(0);
}
