#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <config.h>
#include <support.h>
#include <xcio.h>
#include <sysmsg.h>

#ifdef	HAVE_SHADOW_H
#include <shadow.h>
#endif

#include "option.h"
#include "log.h"
#include "timer.h"
#include "console.h"
#include "command.h"

#include "frame.h"
#include "env.h"
#include "lcp.h"
#include "cps.h"
#include "phase.h"
#include "auth.h"

static struct passwd_s cPwd;
static struct timer_s authTimer;

#define DELIM_CHAR ':'

void
SetPasswd(char *entry, char *name, char *passwd, u_int32_t key)
{
    if (cPwd.passwd) Free(cPwd.passwd);
    if (cPwd.name) Free(cPwd.name);
    if (cPwd.entry) Free(cPwd.entry);
    cPwd.entry = entry ? Strdup(entry): NULL;
    cPwd.name = name ? Strdup(name): NULL;
    cPwd.passwd = passwd ? Strdup(passwd): NULL;
    cPwd.key = key;
}

int
XcPasswdGet(struct console_s *cp)
{
    struct xcio_s xc;

    xc.type = XCIO_LAST|XCIO_PWD_SET;
    xc.len = 0;
    xc.xid = cp->xid;

    if (cPwd.name && *cPwd.name) {
	strcpy(xc.buf + xc.len, cPwd.name);
	xc.len += strlen(cPwd.name) + 1;

	strcpy(xc.buf + xc.len, cPwd.passwd);
	xc.len += strlen(cPwd.passwd) + 1;
    }
    return(XcioWrite(cp->fd, &xc));
}

/*
 * passwd file access: functions and definitions
 */

/*
 * enrty:name:passwd:key:group:script
 */

static char *
GetEntity(char *d, char *s, int len)
{
    int n;
    bool_t shifted=FALSE;

    for (n = 0; n < len && *s; s ++) {
	if (shifted) shifted = FALSE;
	else {
	    if (*s == '\\') {
		shifted = TRUE;
		continue;
	    }
	    if (*s == DELIM_CHAR) {
		s ++;
		break;
	    }
	}
	*d = *s;
	d ++;
	n ++;
    }
    *d = '\0';
    return(s);
}

static int
WriteEntity(FILE *fp, char *s)
{
    char *buf;
    int n=0;

    if (!s || !*s) return(0);
    buf = Malloc(strlen(s) * 2);

    while (*s) {
	if (*s == DELIM_CHAR || *s == '\\') fputc('\\', fp);
	fputc(*s, fp);
	s ++;
	n ++;
    }
    Free(buf);
    return(n);
}

struct passwd_s *
GetPasswd(char *line)
{
    static struct passwd_s pw;
    u_int32_t key=0;
    char *p, buf[100];

    if (pw.entry) free(pw.entry);
    if (pw.name) free(pw.name);
    if (pw.passwd) free(pw.passwd);
    if (pw.group) free(pw.group);
    if (pw.script) free(pw.script);
    memset(&pw, 0, sizeof(pw));

    if ((p = strpbrk(line, "\r\n")) != NULL) *p = '\0';
    /* entry */
    line = GetEntity(buf, line, sizeof(buf) - 1);
    pw.entry = Strdup(buf);

    /* name */
    line = GetEntity(buf, line, sizeof(buf) - 1);
    pw.name = Strdup(buf);

    /* password */
    line = GetEntity(buf, line, sizeof(buf) - 1);
    pw.passwd = Strdup(buf);

    /* key */
    line = GetEntity(buf, line, sizeof(buf) - 1);
    sscanf(buf, "%8x", &key);
    pw.key = key;

    /* group */
    line = GetEntity(buf, line, sizeof(buf) - 1);
    pw.group = Strdup(buf);

    /* script */
    if (*line) pw.script = Strdup(line);
    return(&pw);
}

static int
PasswdFileOut(char *line, int secure, struct console_s *cp)
{
    struct passwd_s *pw;
    struct xcio_s xc;
    char *data;

    pw = GetPasswd(line);
    xc.len = 0;
    xc.type = XCIO_PWD_SET;
    xc.xid = cp->xid;

    strcpy(xc.buf + xc.len, pw->name);
    xc.len += strlen(pw->name) + 1;

    strcpy(xc.buf + xc.len, secure ? "<password>": pw->passwd);
    xc.len += strlen(secure ? "<password>": pw->passwd) + 1;

    strcpy(xc.buf + xc.len, pw->entry);
    xc.len += strlen(pw->entry) + 1;

    data = pw->group ? pw->group: "";
    strcpy(xc.buf + xc.len, data);
    xc.len += strlen(data) + 1;

    data = pw->script ? pw->script: "";
    strcpy(xc.buf + xc.len, data);
    xc.len += strlen(data) + 1;

    XcioWrite(cp->fd, &xc);
    return(0);
}

static int
PasswdFileSearch(char *line, int secure, struct passwd_s *pwp)
{
    struct passwd_s *pw;
    bool_t found=FALSE;

    if ((pw = GetPasswd(line)) == NULL) return(0);
    if (pwp->entry && !strcmp(pwp->entry, pw->entry)) {
	pwp->name = pw->name;
	found = TRUE;
    }
    if (pwp->name && !strcmp(pwp->name, pw->name)) {
	pwp->entry = pw->entry;
	found = TRUE;
    }
    if (found) {
	pwp->passwd = pw->passwd;
	pwp->script = pw->script;
	pwp->group = pw->group;
	pwp->key = pw->key;
	return(1);
    }
    return(0);
}

struct passwd_s *
GetPasswdByName(char *name)
{
    static struct passwd_s pw;
    FILE *fp;
    int secure=0;

    memset(&pw, 0, sizeof(pw));
    pw.name = name;
    if ((fp = OpenFile("passwd", NULL, &secure)) == NULL) return(NULL);
    if (LoadFile(fp, PasswdFileSearch, secure, &pw)) {
	return(&pw);
    }
    if (secure == 0) {
	secure = EXEC_SECURE;
	if ((fp = OpenFile("passwd", NULL, &secure)) == NULL)
	    return(NULL);
	if (LoadFile(fp, PasswdFileSearch, secure, &pw)) {
	    return(&pw);
	}
    }
    return(NULL);
}

struct passwd_s *
GetPasswdByEntry(char *entry)
{
    static struct passwd_s pw;
    FILE *fp;
    int secure=0;

    memset(&pw, 0, sizeof(pw));
    pw.entry = entry;
    if ((fp = OpenFile("passwd", NULL, &secure)) == NULL) return(NULL);
    if (LoadFile(fp, PasswdFileSearch, secure, &pw)) {
	return(&pw);
	return(0);
    }
    if (secure == 0) {
	secure = EXEC_SECURE;
	if ((fp = OpenFile("passwd", NULL, &secure)) == NULL)
	    return(NULL);
	if (LoadFile(fp, PasswdFileSearch, secure, &pw)) {
	    return(&pw);
	}
    }
    return(NULL);
}

static int
WriteEntry(FILE *fp, struct passwd_s *pw)
{
    char enc[100], src[100], s;
    int ns, i;

    memset(src, 0, sizeof(src));
    if (!(pw->key & ENCODE_TMASK))
	pw->key = (pw->key & ~ENCODE_TMASK) | ENCODE_BASE64;
    while (!(ns = RAND() & 0xF));
    for (i = 0; i < ns;) {
	s = (char)(RAND() & 0xFF);
	if (' ' <= s && s <= '~') src[i ++] = s;
    }
    switch (pw->key & ENCODE_TMASK) {
    case ENCODE_BASE64:
	strcpy(&src[ns], pw->passwd);
	Base64Encode(enc, src, sizeof(enc));
	pw->key = (pw->key & ~ENCODE_SMASK) | (ns << 24);
	pw->passwd = enc;
	break;
    }
    WriteEntity(fp, pw->entry);
    fputc(DELIM_CHAR, fp);
    WriteEntity(fp, pw->name);
    fputc(DELIM_CHAR, fp);
    WriteEntity(fp, pw->passwd);
    fprintf(fp, ":%0x:", pw->key);
    if (pw->group) WriteEntity(fp, pw->group);
    fputc(DELIM_CHAR, fp);
    if (pw->script) WriteEntity(fp, pw->script);
    return(fputc('\n', fp) == EOF ? -1: 0);
}

int
UpdatePasswd(char *entry, char *name, char *passwd)
{
    struct passwd_s *opw;
    FILE *ofp, *nfp;
    char line[256], line2[256], *path, *bkpath, *p;

    p = usrPPxP ? usrPPxP: sysPPxP;
    bkpath = Malloc(strlen(p) + sizeof("/passwd") + 2);
    sprintf(bkpath, "%s/passwd", p);
    path = Strdup(bkpath);
    strcat(bkpath, "~");
    unlink(bkpath);
    rename(path, bkpath);
    if ((nfp = fopen(path, "w")) == NULL) {
	rename(bkpath, path);
	Free(path);
	Free(bkpath);
	return(-1);
    }
    if ((ofp = fopen(bkpath, "r")) != NULL) {
	while (fgets(line, sizeof(line), ofp)) {
	    strcpy(line2, line);
	    opw = GetPasswd(line);
	    if (opw && entry && !strcmp(opw->entry, entry)) {
		struct passwd_s pw;

		pw.entry = entry;
		pw.name = name;
		pw.passwd = passwd;
		pw.group = opw->group;
		pw.script = opw->script;
		pw.key = opw->key;
		WriteEntry(nfp, &pw);
		entry = NULL;
	    } else fputs(line2, nfp);
	}
	fclose(ofp);
    }
    if (entry) {
	struct passwd_s pw;

	memset(&pw, 0, sizeof(pw));
	pw.entry = entry;
	pw.name = name;
	pw.passwd = passwd;
	WriteEntry(nfp, &pw);
    }
    fclose(nfp);
    chmod(path, 0600);
    chown(path, getuid(), getgid());
    Free(path);
    Free(bkpath);
    return(0);
}

void
ShowPasswd(struct console_s *cp, char *entry)
{
    FILE *fp;
    int secure=0;
    struct xcio_s xc;

    if ((fp = OpenFile("passwd", NULL, &secure)) == NULL) return;
    LoadFile(fp, PasswdFileOut, getuid() ? secure: 0, cp);
    if (secure == 0) {
	secure = EXEC_SECURE;
	if ((fp = OpenFile("passwd", NULL, &secure)) != NULL)
	    LoadFile(fp, PasswdFileOut, secure, cp);
    }
    xc.len = 0;
    xc.type = XCIO_LAST;
    xc.xid = cp->xid;
    XcioWrite(cp->fd, &xc);
    return;
}

/*
 * Authentication: generic functions
 */

extern struct lcpreg_s lcprReg;
extern struct lcpreg_s lcplReg;

void
AuthInit()
{
    if (pppOpt.a_server == AUTH_SERVER_NONE) {
	switch (lcprReg.nbo_auth_type) {
	case NBO_PROTO_PAP:
	    PapStart();
	    break;
	case 0:
	    PhaseUp();
	    break;
	}
    } else {
	switch (lcplReg.nbo_auth_type) {
	case NBO_PROTO_CHAP:
	    ChapStart();
	    break;
	}
    }
/*
    case 0:
	if (pppOpt.a_server == AUTH_SERVER_NONE) PhaseUp();
	return;
    }
*/
}

bool_t
EnvAuthProto(int argc, char *argv[], char *outs)
{
    const char *alg;
    char *p;
    int i = 0, a;
    extern u_int16_t StrtoProto();

    p = outs;
    if (!argc) {
	u_int16_t nbo_proto;

	if (pppInfo.l_stat & LSTAT_PPP) {
	    nbo_proto = lcprReg.nbo_auth_type;
	    alg = ChapAlgorithmName(lcprReg.a_algorithm);
	    p += SprintF(p, "%s", TostrProto(&nbo_proto));
	    switch(nbo_proto) {
	    case NBO_PROTO_CHAP:
		p += SprintF(p, "/%s", alg);
	    default:
		p += SprintF(p, " ");
	    }
	    return FALSE;
	}
	while (i < AUTH_MAX && pppOpt.hbo_a_order[i]) {
	    nbo_proto = htons(pppOpt.hbo_a_order[i]);
	    alg = ChapAlgorithmName(pppOpt.a_algorithm[i]);
	    if (nbo_proto) {
		p += SprintF(p, "%s", TostrProto(&nbo_proto));
		switch(nbo_proto) {
		case NBO_PROTO_CHAP:
		    p += SprintF(p, "/%s", alg);
		default:
		    p += SprintF(p, " ");
		}
	    }
	    i ++;
	}
    }
    for (a = 1; a < argc && i < AUTH_MAX; a ++) {
	pppOpt.a_algorithm[i] = 0;
	if ((p = strchr(argv[a], '/')) != NULL) {
	    *p = '\0';
	    pppOpt.a_algorithm[i] = ChapAlgorithmType(p + 1);
	}
	pppOpt.hbo_a_order[i] = StrtoProto(argv[a]);
	switch (pppOpt.hbo_a_order[i]) {
	case PROTO_CHAP:
	    if (!pppOpt.a_algorithm[i])
		pppOpt.a_algorithm[i] = 5; /* MD5 */
	}
	i ++;
    }
    if (i < AUTH_MAX) pppOpt.hbo_a_order[i] = 0;

    return TRUE;
}

char *
AuthPasswd()
{
    return(cPwd.passwd);
}

bool_t
AuthDecode(char *buf, int max)
{
    char *tmp;

    memset(buf, 0, max);
    switch (cPwd.key & ENCODE_TMASK) {
    case ENCODE_NONE:
	strncpy(buf, cPwd.passwd, max);
	return(TRUE);
    case ENCODE_BASE64:
	tmp = Malloc(max);
	Base64Decode(tmp, cPwd.passwd, max);
	strcpy(buf, tmp + ((cPwd.key & ENCODE_SMASK) >> 24));
	Free(tmp);
	return(TRUE);
    default:
	return(FALSE);
    }
}

bool_t
AuthCmp(char *key)
{
    char buf[100];

    if (!AuthDecode(buf, sizeof(buf))) {
	/*
	  AuthEncode(buf, key, sizeof(buf));
	  strcmp(buf, cPwd.passwd);
	  key = cPwd.passwd;
	  */
    }
    return(strcmp(key, buf) ? FALSE: TRUE);
}

char *
AuthName()
{
    return(cPwd.name);
}

char *
AuthEntry()
{
    return(cPwd.entry);
}

bool_t
EnvAuthPasswd(int argc, char **argv, char *outs)
{
    char *key, *name;
    struct passwd_s *pwp;

    if (!argc) {
	if (pppOpt.a_entry) strcpy(outs, pppOpt.a_entry);
	return FALSE;
    }
    if (pppOpt.a_server == AUTH_SERVER_NONE) {
	pwp = GetPasswdByEntry(argv[1]);
	if (!pwp) {
	    if (argv[0]) ConsoleMsg(MS_E_NO_AUTHENTRY, argv[1]);
	    return FALSE;
	}
	SetPasswd(pwp->entry, pwp->name, pwp->passwd, pwp->key);
    }
    pppOpt.a_entry = Strdup(argv[1]);

    return TRUE;
}

bool_t
EnvAuthServer(int argc, char **argv, char *outs)
{
    static char *authServer;
    char *p;

    if (!argc) {
	if (pppOpt.a_server != AUTH_SERVER_NONE)
	    strcpy(outs, authServer);
	return(FALSE);
    }
    if (authServer) {
	Free(authServer);
	authServer = NULL;
	pppOpt.a_server = AUTH_SERVER_NONE;
    }
    if (!strcasecmp(argv[1], "yes") || !strcasecmp(argv[1], "file")) {
	authServer = Strdup("file");
	pppOpt.a_server = AUTH_SERVER_FILE;
    } else if (!strcasecmp(argv[1], "unix")) {
	authServer = Strdup("unix");
	pppOpt.a_server = AUTH_SERVER_UNIX;
    } else if ((p = strchr(argv[1], ':')) != NULL) {
	if (!strncasecmp(argv[1], "radius", p - argv[1])
	    && RadiusSetServer(p + 1)) {
	    authServer = Strdup(argv[1]);
	    pppOpt.a_server = AUTH_SERVER_RADIUS;
	}
    }
    return(TRUE);
}

bool_t
UnixCheckPasswd(char *user, char *passwd)
{
    struct passwd *pwp;
    char *master=NULL, *new;

    if (!pppOpt.a_entry) {
	Logf(LOG_ERROR, "AUTH.PASSWD is not specified\n");
	return(FALSE);
    }
    if ((pwp = getpwnam(user)) == NULL) return(FALSE);
    if (pppOpt.a_entry[0] == '@') { /* group name */
	struct group *grp;
	char **mp;

	if ((grp = getgrnam(&pppOpt.a_entry[1])) == NULL) {
	    Logf(LOG_ERROR, "%s: unknown unix group\n",
		 &pppOpt.a_entry[1]);
	    return(FALSE);
	}
	mp = grp->gr_mem;
	while (*mp) {
	    if (!strcasecmp(*mp, user)) break;
	    mp ++;
	}
	if (!*mp && grp->gr_gid != pwp->pw_gid) return(FALSE);
    } else { /* user name */
	if (strcasecmp(user, pppOpt.a_entry)) return(FALSE);
    }
    master = pwp->pw_passwd;
#ifdef	HAVE_SHADOW_H
    if (!strcmp(master, "x")) { /* maybe shadow */
	struct spwd *swp;

	if ((swp = getspnam(user)) == NULL) return(FALSE);
	master = swp->sp_pwdp;
    }
#endif
    new = (char *)crypt(passwd, master);
    return(strcmp(new, master) ? FALSE: TRUE);
}

int
SetAuthTimer(const char *name, void (*cb)())
{
    static u_int8_t authId;

    if (name && cb) {
	authId ++;
	if (authId >= pppOpt.r_count) {
	    authId = 0;
	    PhaseDown();
	    return(-1);
	}
	authTimer.name = name;
	authTimer.callback = cb;
	authTimer.st_paused = 0;
	authTimer.rest = pppOpt.r_to;
	TimerAdd(&authTimer);
    } else {
	authTimer.st_expired = 1;
	authId = 0;
    }
    return(authId);
}

void
AuthSetup()
{
    static struct env_s envlist[]={
	{"PASSWD", {EnvAuthPasswd}, ENV_SET|ENV_PRIVATE, 0, 0, 0666},
	{"PROTO", {EnvAuthProto}, ENV_SET, 0, 0, 0666},
	{"SERVER", {EnvAuthServer}, ENV_SET, 0, 0, 0666},
	{NULL}
    };
    RegisterEnvs(envlist, "AUTH", NULL);
}
