/* longchar.c
 *
 * Copyright (C) 1992-2005 Paul Boersma
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * pb 2002/11/17 removed bug that genericized to non-Roman glyphs (thanks Mietta Lennes)
 * pb 2003/08/20 added five phonetic Greek characters
 * pb 2004/12/01 added many bitmaps
 * pb 2004/12/02 support for TIPA font (xwin column); unfortunately, several characters are missing (ringunder, diaeresisunder, tildeunder)
 * pb 2005/02/04 the IPA widths become those of the PostScript font TeX-xipa10-Praat-Regular
 * pb 2005/02/04 erev and kidneybean
 * pb 2005/03/08 ps encoding; SILIPA93 encoding for Windows and Mac
 * pb 2005/09/18 SILIPA93 widths for fontless EPS files, including bold
 */

#include "longchar.h"
#include <stdio.h>   /* For error message. */

static struct Longchar_Info Longchar_database [] = {

/* Space. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ ' ', ' ', 0, { "/space",          250, 250, 250, 250,  278, 287, 278, 287,  278, 278,  250, 250, 250, 250 },  32,  32,  32,  32, 0x0020 },

/* Letters. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'a', ' ', 0, { "/a",              444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 },  97,  97,  97,  97, 0x0061 },
{ 'b', ' ', 0, { "/b",              500, 556, 500, 500,  556, 648, 556, 611,  556, 611,  553, 611, 463, 537 },  98,  98,  98,  98, 0x0062 },
{ 'c', ' ', 0, { "/c",              444, 444, 444, 444,  444, 556, 444, 537,  500, 556,  444, 444, 407, 444 },  99,  99,  99,  99, 0x0063 },
{ 'd', ' ', 0, { "/d",              500, 556, 500, 500,  574, 667, 611, 667,  556, 611,  611, 611, 500, 556 }, 100, 100, 100, 100, 0x0064 },
{ 'e', ' ', 0, { "/e",              444, 444, 444, 444,  500, 574, 444, 519,  556, 556,  479, 500, 389, 444 }, 101, 101, 101, 101, 0x0065 },
{ 'f', ' ', 0, { "/f",              333, 333, 278, 333,  333, 389, 333, 389,  278, 333,  333, 389, 278, 333 }, 102, 102, 102, 102, 0x0066 },
{ 'g', ' ', 0, { "/g",              500, 500, 500, 500,  537, 611, 537, 611,  556, 611,  556, 556, 500, 500 }, 103, 103, 103, 103, 0x0067 },
{ 'h', ' ', 0, { "/h",              500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  582, 611, 500, 556 }, 104, 104, 104, 104, 0x0068 },
{ 'i', ' ', 0, { "/i",              278, 278, 278, 278,  315, 370, 333, 389,  222, 278,  291, 333, 278, 333 }, 105, 105, 105, 105, 0x0069 },
{ 'j', ' ', 0, { "/j",              278, 333, 278, 278,  296, 352, 315, 370,  222, 278,  234, 333, 278, 333 }, 106, 106, 106, 106, 0x006a },
{ 'k', ' ', 0, { "/k",              500, 556, 444, 500,  593, 667, 556, 648,  500, 556,  556, 611, 444, 556 }, 107, 107, 107, 107, 0x006b },
{ 'l', ' ', 0, { "/l",              278, 278, 278, 278,  315, 352, 333, 389,  222, 278,  291, 333, 278, 333 }, 108, 108, 108, 108, 0x006c },
{ 'm', ' ', 0, { "/m",              778, 833, 722, 778,  889, 963, 889, 944,  833, 889,  883, 889, 778, 833 }, 109, 109, 109, 109, 0x006d },
{ 'n', ' ', 0, { "/n",              500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  582, 611, 556, 556 }, 110, 110, 110, 110, 0x006e },
{ 'o', ' ', 0, { "/o",              500, 500, 500, 500,  500, 611, 500, 574,  556, 611,  546, 556, 444, 556 }, 111, 111, 111, 111, 0x006f },
{ 'p', ' ', 0, { "/p",              500, 556, 500, 500,  574, 667, 574, 648,  556, 611,  601, 611, 500, 556 }, 112, 112, 112, 112, 0x0070 },
{ 'q', ' ', 0, { "/q",              500, 556, 500, 500,  556, 648, 556, 630,  556, 611,  560, 611, 463, 537 }, 113, 113, 113, 113, 0x0071 },
{ 'r', ' ', 0, { "/r",              333, 444, 389, 389,  444, 519, 444, 519,  333, 389,  395, 389, 389, 389 }, 114, 114, 114, 114, 0x0072 },
{ 's', ' ', 0, { "/s",              389, 389, 389, 389,  463, 500, 444, 481,  500, 556,  424, 444, 389, 444 }, 115, 115, 115, 115, 0x0073 },
{ 't', ' ', 0, { "/t",              278, 333, 278, 278,  389, 426, 352, 407,  278, 333,  326, 333, 333, 389 }, 116, 116, 116, 116, 0x0074 },
{ 'u', ' ', 0, { "/u",              500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  603, 611, 556, 556 }, 117, 117, 117, 117, 0x0075 },
{ 'v', ' ', 0, { "/v",              500, 500, 444, 444,  537, 611, 519, 556,  500, 556,  565, 556, 500, 556 }, 118, 118, 118, 118, 0x0076 },
{ 'w', ' ', 0, { "/w",              722, 722, 667, 667,  778, 889, 778, 833,  722, 778,  834, 833, 722, 833 }, 119, 119, 119, 119, 0x0077 },
{ 'x', ' ', 0, { "/x",              500, 500, 444, 500,  537, 611, 500, 574,  500, 556,  516, 500, 500, 500 }, 120, 120, 120, 120, 0x0078 },
{ 'y', ' ', 0, { "/y",              500, 500, 444, 444,  537, 611, 500, 519,  500, 556,  556, 556, 500, 556 }, 121, 121, 121, 121, 0x0079 },
{ 'z', ' ', 0, { "/z",              444, 444, 389, 389,  481, 537, 463, 519,  500, 500,  500, 500, 444, 500 }, 122, 122, 122, 122, 0x007a },

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'A', ' ', 0, { "/A",              722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 },  65,  65,  65,  65, 0x0041 },
{ 'B', ' ', 0, { "/B",              667, 667, 611, 667,  722, 778, 722, 759,  667, 722,  611, 667, 611, 667 },  66,  66,  66,  66, 0x0042 },
{ 'C', ' ', 0, { "/C",              667, 722, 667, 667,  722, 778, 722, 759,  722, 722,  709, 722, 667, 685 },  67,  67,  67,  67, 0x0043 },
{ 'D', ' ', 0, { "/D",              722, 722, 722, 722,  778, 833, 778, 833,  722, 722,  774, 833, 778, 778 },  68,  68,  68,  68, 0x0044 },
{ 'E', ' ', 0, { "/E",              611, 667, 611, 667,  722, 759, 722, 741,  667, 667,  611, 611, 611, 611 },  69,  69,  69,  69, 0x0045 },
{ 'F', ' ', 0, { "/F",              556, 611, 611, 667,  667, 722, 667, 704,  611, 611,  556, 556, 556, 556 },  70,  70,  70,  70, 0x0046 },
{ 'G', ' ', 0, { "/G",              722, 778, 722, 722,  778, 833, 778, 815,  778, 778,  763, 833, 722, 778 },  71,  71,  71,  71, 0x0047 },
{ 'H', ' ', 0, { "/H",              722, 778, 722, 778,  833, 870, 833, 870,  722, 722,  832, 833, 778, 778 },  72,  72,  72,  72, 0x0048 },
{ 'I', ' ', 0, { "/I",              333, 389, 333, 389,  407, 444, 407, 444,  278, 278,  337, 389, 333, 389 },  73,  73,  73,  73, 0x0049 },
{ 'J', ' ', 0, { "/J",              389, 500, 444, 500,  556, 648, 611, 667,  500, 556,  333, 389, 333, 389 },  74,  74,  74,  74, 0x004a },
{ 'K', ' ', 0, { "/K",              722, 778, 667, 667,  778, 815, 741, 778,  667, 722,  726, 778, 667, 722 },  75,  75,  75,  75, 0x004b },
{ 'L', ' ', 0, { "/L",              611, 667, 556, 611,  667, 722, 667, 704,  556, 611,  611, 611, 556, 611 },  76,  76,  76,  76, 0x004c },
{ 'M', ' ', 0, { "/M",              889, 944, 833, 889,  944, 981, 944, 944,  833, 833,  946,1000, 944, 944 },  77,  77,  77,  77, 0x004d },
{ 'N', ' ', 0, { "/N",              722, 722, 667, 722,  815, 833, 815, 852,  722, 722,  831, 833, 778, 778 },  78,  78,  78,  78, 0x004e },
{ 'O', ' ', 0, { "/O",              722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 },  79,  79,  79,  79, 0x004f },
{ 'P', ' ', 0, { "/P",              556, 611, 611, 611,  667, 759, 667, 741,  667, 667,  604, 611, 611, 667 },  80,  80,  80,  80, 0x0050 },
{ 'Q', ' ', 0, { "/Q",              722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 },  81,  81,  81,  81, 0x0051 },
{ 'R', ' ', 0, { "/R",              667, 722, 611, 667,  722, 815, 741, 796,  722, 722,  668, 722, 667, 722 },  82,  82,  82,  82, 0x0052 },
{ 'S', ' ', 0, { "/S",              556, 556, 500, 556,  630, 667, 667, 685,  667, 667,  525, 611, 556, 556 },  83,  83,  83,  83, 0x0053 },
{ 'T', ' ', 0, { "/T",              611, 667, 556, 611,  667, 722, 685, 722,  611, 611,  613, 667, 611, 611 },  84,  84,  84,  84, 0x0054 },
{ 'U', ' ', 0, { "/U",              722, 722, 722, 722,  815, 833, 815, 833,  722, 722,  778, 778, 778, 778 },  85,  85,  85,  85, 0x0055 },
{ 'V', ' ', 0, { "/V",              722, 722, 611, 667,  722, 759, 704, 741,  667, 667,  722, 778, 722, 778 },  86,  86,  86,  86, 0x0056 },
{ 'W', ' ', 0, { "/W",              944,1000, 833, 889,  981, 981, 926, 944,  944, 944, 1000,1000, 944,1000 },  87,  87,  87,  87, 0x0057 },
{ 'X', ' ', 0, { "/X",              722, 722, 611, 667,  704, 722, 704, 741,  667, 667,  667, 667, 722, 722 },  88,  88,  88,  88, 0x0058 },
{ 'Y', ' ', 0, { "/Y",              722, 722, 556, 611,  704, 722, 685, 704,  667, 667,  667, 667, 667, 611 },  89,  89,  89,  89, 0x0059 },
{ 'Z', ' ', 0, { "/Z",              611, 667, 556, 611,  611, 667, 667, 704,  611, 611,  667, 667, 667, 667 },  90,  90,  90,  90, 0x005a },

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '.', ' ', 0, { "/period",         250, 250, 250, 250,  278, 278, 278, 287,  278, 278,  250, 250, 250, 250 },  46,  46,  46,  46, 0x002e },
{ ',', ' ', 0, { "/comma",          250, 250, 250, 250,  278, 278, 278, 287,  278, 278,  250, 250, 250, 250 },  44,  44,  44,  44, 0x002c },
{ ':', ' ', 0, { "/colon",          278, 333, 333, 333,  278, 278, 278, 287,  278, 333,  250, 250, 250, 250 },  58,  58,  58,  58, 0x003a },
{ ';', ' ', 0, { "/semicolon",      278, 333, 333, 333,  278, 278, 278, 287,  278, 333,  250, 250, 250, 250 },  59,  59,  59,  59, 0x003b },
{ '!', ' ', 0, { "/exclam",         333, 333, 333, 389,  296, 296, 333, 333,  278, 333,  278, 278, 333, 333 },  33,  33,  33,  33, 0x0021 },
{ '!', 'd', 0, { "/exclamdown",     333, 333, 389, 389,  296, 296, 333, 333,  333, 333,  278, 278, 333, 333 }, 161, 161, 193, 193, 0x00a1 },
{ '?', ' ', 0, { "/question",       444, 500, 500, 500,  444, 500, 444, 481,  556, 611,  444, 444, 500, 444 },  63,  63,  63,  63, 0x003f },
{ '?', 'd', 0, { "/questiondown",   444, 500, 500, 500,  444, 500, 444, 481,  611, 611,  444, 444, 500, 444 }, 191, 191, 192, 192, 0x00bf },

/* Numbers. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '0', ' ', 0, { "/zero",           500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  48,  48,  48,  48, 0x0030 },
{ '1', ' ', 0, { "/one",            500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  49,  49,  49,  49, 0x0031 },
{ '2', ' ', 0, { "/two",            500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  50,  50,  50,  50, 0x0032 },
{ '3', ' ', 0, { "/three",          500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  51,  51,  51,  51, 0x0033 },
{ '4', ' ', 0, { "/four",           500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  52,  52,  52,  52, 0x0034 },
{ '5', ' ', 0, { "/five",           500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  53,  53,  53,  53, 0x0035 },
{ '6', ' ', 0, { "/six",            500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  54,  54,  54,  54, 0x0036 },
{ '7', ' ', 0, { "/seven",          500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  55,  55,  55,  55, 0x0037 },
{ '8', ' ', 0, { "/eight",          500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  56,  56,  56,  56, 0x0038 },
{ '9', ' ', 0, { "/nine",           500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  57,  57,  57,  57, 0x0039 },

/* Parentheses, brackets, braces. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '(', ' ', 0, { "/parenleft",      333, 333, 333, 333,  333, 389, 333, 407,  333, 333,  333, 333, 333, 333 },  40,  40,  40,  40, 0x0028 },
{ ')', ' ', 0, { "/parenright",     333, 333, 333, 333,  333, 389, 333, 407,  333, 333,  333, 333, 333, 333 },  41,  41,  41,  41, 0x0029 },
{ '[', ' ', 0, { "/bracketleft",    333, 333, 389, 333,  333, 389, 333, 407,  278, 333,  333, 333, 333, 333 },  91,  91,  91,  91, 0x005b },
{ ']', ' ', 0, { "/bracketright",   333, 333, 389, 333,  333, 389, 333, 407,  278, 333,  333, 333, 333, 333 },  93,  93,  93,  93, 0x005d },
{ '{', ' ', 0, { "/braceleft",      480, 394, 400, 348,  333, 389, 333, 407,  334, 389,  333, 310, 333, 333 }, 123, 123, 123, 123, 0x007b },
{ '}', ' ', 0, { "/braceright",     480, 394, 400, 348,  333, 389, 333, 407,  334, 389,  333, 310, 333, 333 }, 125, 125, 125, 125, 0x007d },

{ '+', ' ', 0, { "/plus",           564, 570, 675, 570,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 },  43,  43,  43,  43, 0x002b },
{ '-', ' ', 0, { "/minus",          564, 570, 675, 606,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 },  45,  45, 208, 208, 0x002d }, /* Add minus to ps */
{ '-', 'h', 0, { "/hyphen",         333, 333, 333, 333,  333, 333, 333, 333,  333, 333,  333, 333, 333, 389 }, 173, 173,  45,  45, 0x00ad },
{ '-', '-', 0, { "/endash",         500, 500, 500, 500,  500, 500, 500, 500,  556, 556,  500, 500, 500, 500 },  45,  45, 208, 208, 0x2014 }, /* or endash */
{ '+', '-', 0, { "/plusminus",      564, 570, 675, 570,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 }, 177, 177, 177, 177, 0x00b1 },
{ '*', ' ', 0, { "/asterisk",       500, 500, 500, 500,  500, 500, 500, 500,  389, 389,  389, 444, 389, 444 },  42,  42,  42,  42, 0x002a },
{ '/', ' ', 0, { "/slash",          278, 278, 278, 278,  278, 278, 606, 278,  278, 278,  606, 296, 296, 315 },  47,  47,  47,  47, 0x002f },

/* Comparison. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '<', ' ', 0, { "/less",           564, 570, 675, 570,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 },  60,  60,  60,  60, 0x003c },
{ '=', ' ', 0, { "/equal",          564, 570, 675, 570,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 },  61,  61,  61,  61, 0x003d },
{ '>', ' ', 0, { "/greater",        564, 570, 675, 570,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 },  62,  62,  62,  62, 0x003e },

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '#', ' ', 0, { "/numbersign",     500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  35,  35,  35,  35, 0x0023 },
{ '$', ' ', 0, { "/dollar",         500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 },  36,  36,  36,  36, 0x0024 },
{ '%', ' ', 0, { "/percent",        833,1000, 833, 833,  833, 833, 833, 889,  889, 889,  840, 889, 889, 889 },  37,  37,  37,  37, 0x0025 },
{ '&', ' ', 0, { "/ampersand",      778, 833, 778, 778,  815, 852, 852, 889,  667, 722,  778, 833, 778, 833 },  38,  38,  38,  38, 0x0026 },
{ '@', ' ', 0, { "/at",             921, 930, 920, 832,  737, 747, 747, 747, 1015, 975,  747, 747, 747, 833 },  64,  64,  64,  64, 0x0040 },
{ 'b', 's', 0, { "/backslash",      278, 278, 278, 278,  606, 606, 606, 606,  278, 278,  606, 606, 606, 606 },  92,  92,  92,  92, 0x005c },
{ '_', ' ', 0, { "/underscore",     500, 500, 500, 500,  500, 500, 500, 500,  556, 556,  500, 500, 500, 500 },  95,  95,  95,  95, 0x005f },
{ '^', ' ', 0, { "/asciicircum",    469, 581, 422, 570,  606, 606, 606, 606,  469, 584,  606, 606, 606, 606 },  94,  94,  94,  94, 0x005e },
{ '|', ' ', 0, { "/bar",            200, 220, 275, 220,  606, 606, 606, 606,  260, 280,  606, 606, 606, 606 }, 124, 124, 124, 124, 0x007c },
{ '~', ' ', 0, { "/asciitilde",     541, 520, 541, 570,  606, 606, 606, 606,  584, 584,  606, 606, 606, 606 }, 126, 126, 126, 126, 0x007e },

/* Quotes: conversion. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '`', ' ', 0, { "/quotesinglleft", 333, 333, 333, 333,  204, 241, 204, 259,  222, 278,  278, 278, 278, 278 },  96,  96, 212, 212, 0x2018 },
{'\'', ' ', 0, { "/quotesinglright",333, 333, 333, 333,  204, 241, 204, 259,  222, 278,  278, 278, 278, 278 },  39,  39, 213, 213, 0x2019 },
{'\'', 'a', 0, { "/apostrophe",     333, 333, 333, 333,  204, 241, 204, 259,  222, 278,  278, 278, 278, 278 },  39,  39,  39,  39, 0x0027 },
{'\"', 'l', 0, { "/quotedblleft",   444, 500, 556, 500,  389, 481, 389, 481,  333, 500,  500, 500, 500, 500 },  34,  34, 210, 210, 0x201c },
{'\"', 'r', 0, { "/quotedblright",  444, 500, 556, 500,  389, 481, 389, 481,  333, 500,  500, 500, 500, 500 },  34,  34, 211, 211, 0x201d },
{'\"', ' ', 0, { "/quotedbl",       408, 555, 420, 555,  389, 333, 400, 400,  355, 474,  371, 402, 500, 500 },  34,  34,  34,  34, 0x0022 },
{ '<', '<', 0, { "/guillemotleft",  500, 500, 500, 500,  426, 500, 426, 481,  556, 556,  500, 500, 500, 500 }, 171, 171, 199, 199, 0x00ab },
{ '>', '>', 0, { "/guillemotright", 500, 500, 500, 500,  426, 500, 426, 481,  556, 556,  500, 500, 500, 500 }, 187, 187, 200, 200, 0x00bb },

/* Accented letters. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'a', '`', 0, { "/agrave",         444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 }, 224, 224, 136, 136, 0x00e0 },
{ 'a','\'', 0, { "/aacute",         444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 }, 225, 225, 135, 135, 0x00e1 },
{ 'a', '^', 0, { "/acircumflex",    444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 }, 226, 226, 137, 137, 0x00e2 },
{ 'a', '~', 0, { "/atilde",         444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 }, 227, 227, 139, 139, 0x00e3 },
{ 'a','\"', 0, { "/adieresis",      444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 }, 228, 228, 138, 138, 0x00e4 },
{ 'e', '`', 0, { "/egrave",         444, 444, 444, 444,  500, 574, 444, 519,  556, 556,  479, 500, 389, 444 }, 232, 232, 143, 143, 0x00e8 },
{ 'e','\'', 0, { "/eacute",         444, 444, 444, 444,  500, 574, 444, 519,  556, 556,  479, 500, 389, 444 }, 233, 233, 142, 142, 0x00e9 },
{ 'e', '^', 0, { "/ecircumflex",    444, 444, 444, 444,  500, 574, 444, 519,  556, 556,  479, 500, 389, 444 }, 234, 234, 144, 144, 0x00ea },
{ 'e','\"', 0, { "/edieresis",      444, 444, 444, 444,  500, 574, 444, 519,  556, 556,  479, 500, 389, 444 }, 235, 235, 145, 145, 0x00eb },
{ 'i', '`', 0, { "/igrave",         278, 278, 278, 278,  315, 370, 333, 389,  222, 278,  287, 333, 278, 333 }, 236, 236, 147, 147, 0x00ec },
{ 'i','\'', 0, { "/iacute",         278, 278, 278, 278,  315, 370, 333, 389,  222, 278,  287, 333, 278, 333 }, 237, 237, 146, 146, 0x00ed },
{ 'i', '^', 0, { "/icircumflex",    278, 278, 278, 278,  315, 370, 333, 389,  222, 278,  287, 333, 278, 333 }, 238, 238, 148, 148, 0x00ee },
{ 'i','\"', 0, { "/idieresis",      278, 278, 278, 278,  315, 370, 333, 389,  222, 278,  287, 333, 278, 333 }, 239, 239, 149, 149, 0x00ef },
{ 'n', '~', 0, { "/ntilde",         500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  582, 611, 556, 556 }, 241, 241, 150, 150, 0x00f1 },
{ 'o', '`', 0, { "/ograve",         500, 500, 500, 500,  500, 611, 500, 574,  556, 611,  546, 556, 444, 556 }, 242, 242, 152, 152, 0x00f2 },
{ 'o','\'', 0, { "/oacute",         500, 500, 500, 500,  500, 611, 500, 574,  556, 611,  546, 556, 444, 556 }, 243, 243, 151, 151, 0x00f3 },
{ 'o', '^', 0, { "/ocircumflex",    500, 500, 500, 500,  500, 611, 500, 574,  556, 611,  546, 556, 444, 556 }, 244, 244, 153, 153, 0x00f4 },
{ 'o', '~', 0, { "/otilde",         444, 500, 500, 500,  556, 611, 500, 574,  556, 611,  546, 556, 444, 556 }, 245, 245, 155, 155, 0x00f5 },
{ 'o','\"', 0, { "/odieresis",      500, 500, 500, 500,  500, 611, 500, 574,  556, 611,  546, 556, 444, 556 }, 246, 246, 154, 154, 0x00f6 },
{ 'u', '`', 0, { "/ugrave",         500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  603, 611, 556, 556 }, 249, 249, 157, 157, 0x00f9 },
{ 'u','\'', 0, { "/uacute",         500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  603, 611, 556, 556 }, 250, 250, 156, 156, 0x00fa },
{ 'u', '^', 0, { "/ucircumflex",    500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  603, 611, 556, 556 }, 251, 251, 158, 158, 0x00fb },
{ 'u','\"', 0, { "/udieresis",      500, 556, 500, 556,  611, 685, 611, 685,  556, 611,  603, 611, 556, 556 }, 252, 252, 159, 159, 0x00fc },
{ 'y','\'', 0, { "/yacute",         500, 500, 444, 444,  537, 611, 500, 519,  500, 556,  556, 556, 500, 556 }, 253, 253,   0,   0, 0x00fd },
{ 'y','\"', 0, { "/ydieresis",      500, 500, 444, 444,  537, 611, 500, 519,  500, 556,  556, 556, 500, 556 }, 255, 255, 216, 216, 0x00ff }, /* where? */

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'A', '`', 0, { "/Agrave",         722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 }, 192, 192, 203, 203, 0x00c0 },
{ 'A','\'', 0, { "/Aacute",         722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 }, 193, 193, 231, 231, 0x00c1 },
{ 'A', '^', 0, { "/Acircumflex",    722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 }, 194, 194, 229, 229, 0x00c2 },
{ 'A', '~', 0, { "/Atilde",         722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 }, 195, 195, 204, 204, 0x00c3 },
{ 'A','\"', 0, { "/Adieresis",      722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 }, 196, 196, 128, 128, 0x00c4 },
{ 'E', '`', 0, { "/Egrave",         611, 667, 611, 667,  722, 759, 722, 741,  667, 667,  611, 611, 611, 611 }, 200, 200, 233, 233, 0x00c8 },
{ 'E','\'', 0, { "/Eacute",         611, 667, 611, 667,  722, 759, 722, 741,  667, 667,  611, 611, 611, 611 }, 201, 201, 131, 131, 0x00c9 },
{ 'E', '^', 0, { "/Ecircumflex",    611, 667, 611, 667,  722, 759, 722, 741,  667, 667,  611, 611, 611, 611 }, 202, 202, 230, 230, 0x00ca },
{ 'E','\"', 0, { "/Edieresis",      611, 667, 611, 667,  722, 759, 722, 741,  667, 667,  611, 611, 611, 611 }, 203, 203, 232, 232, 0x00cb },
{ 'I', '`', 0, { "/Igrave",         333, 389, 333, 389,  407, 444, 407, 444,  278, 278,  337, 389, 333, 389 }, 204, 204, 237, 237, 0x00cc },
{ 'I','\'', 0, { "/Iacute",         333, 389, 333, 389,  407, 444, 407, 444,  278, 278,  337, 389, 333, 389 }, 205, 205, 234, 234, 0x00cd },
{ 'I', '^', 0, { "/Icircumflex",    333, 389, 333, 389,  407, 444, 407, 444,  278, 278,  337, 389, 333, 389 }, 206, 206, 235, 235, 0x00ce },
{ 'I','\"', 0, { "/Idieresis",      333, 389, 333, 389,  407, 444, 407, 444,  278, 278,  337, 389, 333, 389 }, 207, 207, 236, 236, 0x00cf },
{ 'N', '~', 0, { "/Ntilde",         722, 722, 667, 722,  815, 833, 815, 852,  722, 722,  831, 833, 778, 778 }, 209, 209, 132, 132, 0x00d1 },
{ 'O', '`', 0, { "/Ograve",         722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 }, 210, 210, 241, 241, 0x00d2 },
{ 'O','\'', 0, { "/Oacute",         722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 }, 211, 211, 238, 238, 0x00d3 },
{ 'O', '^', 0, { "/Ocircumflex",    722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 }, 212, 212, 239, 239, 0x00d4 },
{ 'O', '~', 0, { "/Otilde",         722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 }, 213, 213, 205, 205, 0x00d5 },
{ 'O','\"', 0, { "/Odieresis",      722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  786, 833, 778, 833 }, 214, 214, 133, 133, 0x00d6 },
{ 'U', '`', 0, { "/Ugrave",         722, 722, 722, 722,  815, 833, 815, 833,  722, 722,  778, 778, 778, 778 }, 217, 217, 244, 244, 0x00d9 },
{ 'U','\'', 0, { "/Uacute",         722, 722, 722, 722,  815, 833, 815, 833,  722, 722,  778, 778, 778, 778 }, 218, 218, 242, 242, 0x00da },
{ 'U', '^', 0, { "/Ucircumflex",    722, 722, 722, 722,  815, 833, 815, 833,  722, 722,  778, 778, 778, 778 }, 219, 219, 243, 243, 0x00db },
{ 'U','\"', 0, { "/Udieresis",      722, 722, 722, 722,  815, 833, 815, 833,  722, 722,  778, 778, 778, 778 }, 220, 220, 134, 134, 0x00dc },
{ 'Y','\'', 0, { "/Yacute",         722, 722, 556, 611,  704, 722, 685, 704,  667, 667,  667, 667, 667, 611 }, 221, 221,   0,   0, 0x00dd },
{ 'Y','\"', 0, { "/Ydieresis",      722, 722, 556, 611,  704, 722, 685, 704,  667, 667,  667, 667, 667, 611 },   0,   0, 217, 217,      0 }, /* where? */

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'a', 'o', 0, { "/aring",          444, 500, 500, 500,  556, 611, 574, 667,  556, 556,  500, 500, 444, 556 }, 229, 229, 140, 140, 0x00e5 },
{ 'c', ',', 0, { "/ccedilla",       444, 444, 444, 444,  444, 556, 444, 537,  500, 556,  444, 444, 407, 444 }, 231, 231, 141, 141, 0x00e7 },
{ 'a', 'e', 0, { "/ae",             667, 722, 667, 722,  796, 870, 722, 815,  889, 889,  758, 778, 638, 738 }, 230, 230, 190, 190, 0x00e6 }, /* ash */
{ 'o', '/', 0, { "/oslash",         500, 500, 500, 500,  500, 611, 500, 574,  611, 611,  556, 556, 444, 556 }, 248, 248, 191, 191, 0x00f8 },
{ 's', 's', 0, { "/germandbls",     500, 556, 500, 500,  574, 611, 556, 574,  611, 611,  556, 611, 500, 556 }, 223, 223, 167, 167, 0x00df }, /* Ringel-s. */
{ 't', 'h', 0, { "/thorn",          500, 556, 500, 500,  574, 667, 574, 648,  556, 611,  601, 611, 500, 556 }, 254, 254,   0,   0, 0x00fe },
{ 'A', 'o', 0, { "/Aring",          722, 722, 611, 667,  722, 759, 704, 741,  667, 722,  778, 778, 722, 722 }, 197, 197, 129, 129, 0x00c5 },
{ 'C', ',', 0, { "/Ccedilla",       667, 722, 667, 667,  722, 778, 722, 759,  722, 722,  709, 722, 667, 685 }, 199, 199, 130, 130, 0x00c7 },
{ 'A', 'e', 0, { "/AE",             722,1000, 889, 944, 1000, 981, 870, 889, 1000,1000,  944,1000, 941, 944 }, 198, 198, 174, 174, 0x00c6 }, /* Ash */
{ 'O', '/', 0, { "/Oslash",         722, 778, 722, 722,  778, 833, 778, 833,  778, 778,  833, 833, 778, 833 }, 216, 216, 175, 175, 0x00d8 },
{ 'T', 'h', 0, { "/Thorn",          556, 611, 611, 611,  667, 759, 667, 741,  667, 667,  604, 611, 611, 667 }, 222, 222,   0,   0, 0x00de },

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ '.', 'c', 0, { "/periodcentered", 250, 250, 250, 250,  278, 278, 278, 287,  278, 278,  250, 250, 250, 250 }, 183, 183, 225, 225, 0x00b7 },   // MIDDLE DOT
{ 'd', 'g', 0, { "/degree",         400, 400, 400, 400,  400, 400, 400, 400,  400, 400,  400, 400, 400, 400 }, 176, 176, 161, 161, 0x00b0 },   // DEGREE SIGN
{ 'c', '/', 0, { "/cent",           500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 }, 162, 162, 162, 162, 0x00a2 },   // CENT SIGN
{ 'L', 'p', 0, { "/sterling",       500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 }, 163, 163, 163, 163, 0x00a3 },   // POUND SIGN
{ 'c', 'u', 0, { "/currency",       500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 }, 164, 164, 219, 219, 0x00a4 },   // CURRENCY SIGN
{ 'e', 'u', 0, { "/euro",           500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 }, 164, 164, 219, 219, 0x20ac },   // EURO SIGN   /* = currency? */
{ 'Y', '=', 0, { "/yen",            500, 500, 500, 500,  556, 574, 556, 574,  556, 556,  500, 500, 500, 500 }, 165, 165, 180, 180, 0x00a5 },   // YEN SIGN
{ 'S', 'S', 0, { "/section",        500, 500, 500, 500,  500, 500, 500, 500,  556, 556,  500, 500, 500, 556 }, 167, 167, 164, 164, 0x00a7 },   // SECTION SIGN
{ '|', '|', 0, { "/paragraph",      453, 540, 523, 500,  606, 747, 650, 650,  537, 556,  628, 641, 500, 556 }, 182, 182, 166, 166, 0x00b6 },   // PILCROW SIGN
{ 'c', 'o', 0, { "/copyright",      760, 747, 760, 747,  737, 747, 747, 747,  737, 737,  747, 747, 747, 747 }, 169, 169, 169, 169, 0x00a9 },   // COPYRIGHT SIGN
{ 'r', 'e', 0, { "/registered",     760, 747, 760, 747,  737, 747, 747, 747,  737, 737,  747, 747, 747, 747 }, 174, 174, 168, 168, 0x00ae },   // REGISTERED SIGN
{ 'a', '_', 0, { "/ordfeminine",    276, 300, 276, 266,  334, 367, 422, 412,  370, 370,  333, 438, 333, 333 }, 170, 170, 187, 187, 0x00aa },   // FEMININE ORDINAL INDICATOR
{ 'o', '_', 0, { "/ordmasculine",   310, 330, 310, 300,  300, 367, 372, 356,  365, 365,  333, 488, 333, 333 }, 186, 186, 188, 188, 0x00ba },   // MASCULINE ORDINAL INDICATOR

{ 'F', 'I', 0, { "/fi",             556, 556, 500, 556,  611, 685, 611, 685,  500, 611,  605, 611, 528, 611 },   0,   0, 222, 222 },
{ 'F', 'L', 0, { "/fl",             556, 556, 500, 556,  611, 685, 611, 685,  500, 611,  608, 611, 545, 611 },   0,   0, 223, 223 },

/* Greek. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'a', 'l', 1, { "/alpha",          631, 0,   0,   0,    631, 0,   0,   0,    631, 0,    631, 0,   0,   0   },  97,  97,  97,  97 },
{ 'b', 'e', 1, { "/beta",           549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   },  98,  98,  98,  98 },
{ 'g', 'a', 1, { "/gamma",          411, 0,   0,   0,    411, 0,   0,   0,    411, 0,    411, 0,   0,   0   }, 103, 103, 103, 103 },
{ 'd', 'e', 1, { "/delta",          494, 0,   0,   0,    494, 0,   0,   0,    494, 0,    494, 0,   0,   0   }, 100, 100, 100, 100 },
{ 'e', 'p', 1, { "/epsilon",        439, 0,   0,   0,    439, 0,   0,   0,    439, 0,    439, 0,   0,   0   }, 101, 101, 101, 101 },
{ 'z', 'e', 1, { "/zeta",           494, 0,   0,   0,    494, 0,   0,   0,    494, 0,    494, 0,   0,   0   }, 122, 122, 122, 122 },
{ 'e', 't', 1, { "/eta",            603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 104, 104, 104, 104 },
{ 't', 'e', 1, { "/theta",          521, 0,   0,   0,    521, 0,   0,   0,    521, 0,    521, 0,   0,   0   }, 113, 113, 113, 113 }, /* like obar */
{ 't', '2', 1, { "/theta1",         631, 0,   0,   0,    631, 0,   0,   0,    631, 0,    631, 0,   0,   0   },  74,  74,  74,  74 },
{ 'i', 'o', 1, { "/iota",           329, 0,   0,   0,    329, 0,   0,   0,    329, 0,    329, 0,   0,   0   }, 105, 105, 105, 105 },
{ 'k', 'a', 1, { "/kappa",          549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 107, 107, 107, 107 },
{ 'l', 'a', 1, { "/lambda",         549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 108, 108, 108, 108 },
{ 'm', 'u', 1, { "/mu",             576, 0,   0,   0,    576, 0,   0,   0,    576, 0,    576, 0,   0,   0   }, 109, 109, 109, 109 },
{ 'n', 'u', 1, { "/nu",             521, 0,   0,   0,    521, 0,   0,   0,    521, 0,    521, 0,   0,   0   }, 110, 110, 110, 110 },
{ 'x', 'i', 1, { "/xi",             493, 0,   0,   0,    493, 0,   0,   0,    493, 0,    493, 0,   0,   0   }, 120, 120, 120, 120 },
{ 'o', 'n', 1, { "/omicron",        549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 111, 111, 111, 111 },
{ 'p', 'i', 1, { "/pi",             549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 112, 112, 112, 112 },
{ 'r', 'o', 1, { "/rho",            549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 114, 114, 114, 114 },
{ 's', 'i', 1, { "/sigma",          603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 115, 115, 115, 115 },
{ 's', '2', 1, { "/sigma1",         439, 0,   0,   0,    439, 0,   0,   0,    439, 0,    439, 0,   0,   0   },  86,  86,  86,  86 },
{ 't', 'a', 1, { "/tau",            439, 0,   0,   0,    439, 0,   0,   0,    439, 0,    439, 0,   0,   0   }, 116, 116, 116, 116 },
{ 'u', 'p', 1, { "/upsilon",        576, 0,   0,   0,    576, 0,   0,   0,    576, 0,    576, 0,   0,   0   }, 117, 117, 117, 117 },
{ 'f', 'i', 1, { "/phi",            603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 106, 106, 106, 106 },
{ 'f', '2', 1, { "/phi1",           521, 0,   0,   0,    521, 0,   0,   0,    521, 0,    521, 0,   0,   0   }, 102, 102, 102, 102 }, /* like oslash */
{ 'c', 'i', 1, { "/chi",            549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   },  99,  99,  99,  99 },
{ 'p', 's', 1, { "/psi",            686, 0,   0,   0,    686, 0,   0,   0,    686, 0,    686, 0,   0,   0   }, 121, 121, 121, 121 },
{ 'o', 'm', 1, { "/omega",          686, 0,   0,   0,    686, 0,   0,   0,    686, 0,    686, 0,   0,   0   }, 119, 119, 119, 119 },

/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'A', 'l', 1, { "/Alpha",          722, 0,   0,   0,    722, 0,   0,   0,    722, 0,    722, 0,   0,   0   },  65,  65,  65,  65 },
{ 'B', 'e', 1, { "/Beta",           667, 0,   0,   0,    667, 0,   0,   0,    667, 0,    667, 0,   0,   0   },  66,  66,  66,  66 },
{ 'G', 'a', 1, { "/Gamma",          603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   },  71,  71,  71,  71 },
{ 'D', 'e', 1, { "/Delta",          612, 0,   0,   0,    612, 0,   0,   0,    612, 0,    612, 0,   0,   0   },  68,  68,  68,  68 },
{ 'E', 'p', 1, { "/Epsilon",        611, 0,   0,   0,    611, 0,   0,   0,    611, 0,    611, 0,   0,   0   },  69,  69,  69,  69 },
{ 'Z', 'e', 1, { "/Zeta",           611, 0,   0,   0,    611, 0,   0,   0,    611, 0,    611, 0,   0,   0   },  90,  90,  90,  90 },
{ 'E', 't', 1, { "/Eta",            722, 0,   0,   0,    722, 0,   0,   0,    722, 0,    722, 0,   0,   0   },  72,  72,  72,  72 },
{ 'T', 'e', 1, { "/Theta",          741, 0,   0,   0,    741, 0,   0,   0,    741, 0,    741, 0,   0,   0   },  81,  81,  81,  81 },
{ 'I', 'o', 1, { "/Iota",           333, 0,   0,   0,    333, 0,   0,   0,    333, 0,    333, 0,   0,   0   },  73,  73,  73,  73 },
{ 'K', 'a', 1, { "/Kappa",          722, 0,   0,   0,    722, 0,   0,   0,    722, 0,    722, 0,   0,   0   },  75,  75,  75,  75 },
{ 'L', 'a', 1, { "/Lambda",         686, 0,   0,   0,    686, 0,   0,   0,    686, 0,    686, 0,   0,   0   },  76,  76,  76,  76 },
{ 'M', 'u', 1, { "/Mu",             889, 0,   0,   0,    889, 0,   0,   0,    889, 0,    889, 0,   0,   0   },  77,  77,  77,  77 },
{ 'N', 'u', 1, { "/Nu",             722, 0,   0,   0,    722, 0,   0,   0,    722, 0,    722, 0,   0,   0   },  78,  78,  78,  78 },
{ 'X', 'i', 1, { "/Xi",             645, 0,   0,   0,    645, 0,   0,   0,    645, 0,    645, 0,   0,   0   },  88,  88,  88,  88 },
{ 'O', 'n', 1, { "/Omicron",        722, 0,   0,   0,    722, 0,   0,   0,    722, 0,    722, 0,   0,   0   },  79,  79,  79,  79 },
{ 'P', 'i', 1, { "/Pi",             768, 0,   0,   0,    768, 0,   0,   0,    768, 0,    768, 0,   0,   0   },  80,  80,  80,  80 },
{ 'R', 'o', 1, { "/Rho",            556, 0,   0,   0,    556, 0,   0,   0,    556, 0,    556, 0,   0,   0   },  82,  82,  82,  82 },
{ 'S', 'i', 1, { "/Sigma",          592, 0,   0,   0,    592, 0,   0,   0,    592, 0,    592, 0,   0,   0   },  83,  83,  83,  83 },
{ 'T', 'a', 1, { "/Tau",            611, 0,   0,   0,    611, 0,   0,   0,    611, 0,    611, 0,   0,   0   },  84,  84,  84,  84 },
{ 'U', 'p', 1, { "/Upsilon",        690, 0,   0,   0,    690, 0,   0,   0,    690, 0,    690, 0,   0,   0   },  85,  85,  85,  85 },
{ 'F', 'i', 1, { "/Phi",            763, 0,   0,   0,    763, 0,   0,   0,    763, 0,    763, 0,   0,   0   },  70,  70,  70,  70 },
{ 'C', 'i', 1, { "/Chi",            722, 0,   0,   0,    722, 0,   0,   0,    722, 0,    722, 0,   0,   0   },  67,  67,  67,  67 },
{ 'P', 's', 1, { "/Psi",            795, 0,   0,   0,    795, 0,   0,   0,    795, 0,    795, 0,   0,   0   },  89,  89,  89,  89 },
{ 'O', 'm', 1, { "/Omega",          768, 0,   0,   0,    768, 0,   0,   0,    768, 0,    768, 0,   0,   0   },  87,  87,  87,  87 },

/* Symbol. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 't', 'm', 1, { "/trademarkserif", 890, 0,   0,   0,    890, 0,   0,   0,    890, 0,    890, 0,   0,   0   }, 212, 212, 212, 212 },
{ 'T', 'M', 1, { "/trademarksans",  786, 0,   0,   0,    786, 0,   0,   0,    786, 0,    786, 0,   0,   0   }, 228, 228, 228, 228 },

{ 'A', 't', 1, { "/universal",      713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   },  34,  34,  34,  34 },
{ 'E', 'r', 1, { "/existential",    549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   },  36,  36,  36,  36 },
{ '=', '~', 1, { "/congruent",      549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   },  64,  64,  64,  64 },
{ '.', '3', 1, { "/therefore",      863, 0,   0,   0,    863, 0,   0,   0,    863, 0,    863, 0,   0,   0   },  92,  92,  92,  92 },
{ 'T', 't', 1, { "/perpendicular",  658, 0,   0,   0,    658, 0,   0,   0,    658, 0,    658, 0,   0,   0   },  94,  94,  94,  94 },
{ 'u', 'n', 1, { "/underscore",     500, 0,   0,   0,    500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  95,  95,  95,  95 },
{ 'o', 'v', 1, { "/radicalex",      500, 0,   0,   0,    500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  96,  96,  96,  96 },
{ 'o', '2', 1, { "/omega1",         713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 118, 118, 118, 118 },
{'\'', 'p', 1, { "/minute",         247, 0,   0,   0,    247, 0,   0,   0,    247, 0,    247, 0,   0,   0   }, 162, 162, 162, 162 }, /* prime */
{ '<', '_', 1, { "/lessequal",      549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 163, 163, 163, 163 },
{ '/', 'd', 1, { "/fraction",       167, 0,   0,   0,    167, 0,   0,   0,    167, 0,    167, 0,   0,   0   }, 164, 164, 164, 164 },
{ 'o', 'o', 1, { "/infinity",       713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 165, 165, 165, 165 },
{ 'f', 'd', 1, { "/florin",         500, 0,   0,   0,    500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 166, 166, 166, 166 },
{ 'c', 'l', 1, { "/club",           753, 0,   0,   0,    753, 0,   0,   0,    753, 0,    753, 0,   0,   0   }, 167, 167, 167, 167 },
{ 'd', 'i', 1, { "/diamond",        753, 0,   0,   0,    753, 0,   0,   0,    753, 0,    753, 0,   0,   0   }, 168, 168, 168, 168 },
{ 'h', 'e', 1, { "/heart",          753, 0,   0,   0,    753, 0,   0,   0,    753, 0,    753, 0,   0,   0   }, 169, 169, 169, 169 },
{ 's', 'p', 1, { "/spade",          753, 0,   0,   0,    753, 0,   0,   0,    753, 0,    753, 0,   0,   0   }, 170, 170, 170, 170 },
{ '<', '>', 1, { "/arrowboth",     1042, 0,   0,   0,   1042, 0,   0,   0,   1042, 0,   1042, 0,   0,   0   }, 171, 171, 171, 171 },
{ '<', '-', 1, { "/arrowleft",      987, 0,   0,   0,    987, 0,   0,   0,    987, 0,    987, 0,   0,   0   }, 172, 172, 172, 172 },
{ '^', '|', 1, { "/arrowup",        603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 173, 173, 173, 173 },
{ '-', '>', 1, { "/arrowright",     987, 0,   0,   0,    987, 0,   0,   0,    987, 0,    987, 0,   0,   0   }, 174, 174, 174, 174 },
{ '_', '|', 1, { "/arrowdown",      603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 175, 175, 175, 175 },
{'\"', 'p', 1, { "/second",         411, 0,   0,   0,    411, 0,   0,   0,    411, 0,    411, 0,   0,   0   }, 178, 178, 178, 178 }, /* double prime */
{ '>', '_', 1, { "/greaterequal",   549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 179, 179, 179, 179 },
{ 'x', 'x', 1, { "/multiply",       549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 180, 180, 180, 180 },
{ 'o', 'c', 1, { "/proportional",   713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 181, 181, 181, 181 },
{ 'd', 'd', 1, { "/partialdiff",    494, 0,   0,   0,    494, 0,   0,   0,    494, 0,    494, 0,   0,   0   }, 182, 182, 182, 182 },
{ 'b', 'u', 1, { "/bullet",         460, 0,   0,   0,    460, 0,   0,   0,    460, 0,    460, 0,   0,   0   }, 183, 183, 183, 183 },
{ ':', '-', 1, { "/divide",         549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 184, 184, 184, 184 },
{ '=', '/', 1, { "/notequal",       549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 185, 185, 185, 185 },
{ '=', '3', 1, { "/equivalence",    549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 186, 186, 186, 186 }, /* defined as */
{ '~', '~', 1, { "/approxequal",    549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 187, 187, 187, 187 },
{ '.', '.', 1, { "/ellipsis",      1000, 0,   0,   0,   1000, 0,   0,   0,   1000, 0,   1000, 0,   0,   0   }, 188, 188, 188, 188 },
{ 'o', 'x', 1, { "/circlemultiply", 768, 0,   0,   0,    768, 0,   0,   0,    768, 0,    768, 0,   0,   0   }, 196, 196, 196, 196 },
{ 'o', '+', 1, { "/circleplus",     768, 0,   0,   0,    768, 0,   0,   0,    768, 0,    768, 0,   0,   0   }, 197, 197, 197, 197 },
{ 'O', '|', 1, { "/emptyset",       823, 0,   0,   0,    823, 0,   0,   0,    823, 0,    823, 0,   0,   0   }, 198, 198, 198, 198 },
{ 'n', 'i', 1, { "/intersection",   768, 0,   0,   0,    768, 0,   0,   0,    768, 0,    768, 0,   0,   0   }, 199, 199, 199, 199 },
{ 'u', 'u', 1, { "/union",          768, 0,   0,   0,    768, 0,   0,   0,    768, 0,    768, 0,   0,   0   }, 200, 200, 200, 200 },
{ 'c', '=', 1, { "/propersubset",   713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 204, 204, 204, 204 },
{ 'e', '=', 1, { "/element",        713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 206, 206, 206, 206 },
{ 'V', 'r', 1, { "/radical",        549, 0,   0,   0,    549, 0,   0,   0,    549, 0,    549, 0,   0,   0   }, 214, 214, 214, 214 }, /* square root */
{ 'n', 'o', 1, { "/logicalnot",     713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 216, 216, 216, 215 },
{ 'a', 'n', 1, { "/logicaland",     603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 217, 217, 217, 217 },
{ 'o', 'r', 1, { "/logicalor",      603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 218, 218, 218, 218 },
{ 'e', 'q', 1, { "/arrowdblboth",  1042, 0,   0,   0,   1042, 0,   0,   0,   1042, 0,   1042, 0,   0,   0   }, 219, 219, 219, 219 }, /* equivalence */
{ '<', '=', 1, { "/arrowdblleft",   987, 0,   0,   0,    987, 0,   0,   0,    987, 0,    987, 0,   0,   0   }, 220, 220, 220, 220 }, /* follows from */
{ '^', '#', 1, { "/arrowdblup",     603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 221, 221, 221, 221 },
{ '=', '>', 1, { "/arrowdblright",  987, 0,   0,   0,    987, 0,   0,   0,    987, 0,    987, 0,   0,   0   }, 222, 222, 222, 222 }, /* implies */
{ '_', '#', 1, { "/arrowdbldown",   603, 0,   0,   0,    603, 0,   0,   0,    603, 0,    603, 0,   0,   0   }, 223, 223, 223, 223 },
{ 's', 'u', 1, { "/summation",      713, 0,   0,   0,    713, 0,   0,   0,    713, 0,    713, 0,   0,   0   }, 229, 229, 229, 229 },
{ 'i', 'n', 1, { "/integral",       274, 0,   0,   0,    274, 0,   0,   0,    274, 0,    274, 0,   0,   0   }, 242, 242, 242, 242 },

/* Phonetic.                                                                                                   i89+ i93  i93  i89+ */
/*fir  sec  al    ps                xipa b   ipa93   b   xipa b    i    bi    xipa b     xipa b    i    bi    xwin  win  mac   ps  unicode decomp  */
{ 'd', 'h', 2, { "/eth",            500, 0,   510, 532,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  68,  68,  68,  68, 0x00f0 }, // LATIN SMALL LETTER ETH
{ 'o', 'e', 2, { "/oe",             700, 0,   751, 769,  700, 0,   0,   0,    700, 0,    700, 0,   0,   0   }, 191, 191, 191, 191, 0x0153 }, // LATIN SMALL LIGATURE OE

{ 'a', 't', 2, { "/aturn",          444, 0,   462, 520,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 140, 140, 140, 140, 0x0250 }, // LATIN SMALL LETTER TURNED A
{ 'a', 's', 2, { "/ascript",        500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  65,  65,  65,  65, 0x0251 }, // LATIN SMALL LETTER ALPHA
{ 'a', 'b', 2, { "/ascriptturn",    500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 129, 129, 129, 129, 0x0252 }, // LATIN SMALL LETTER TURNED ALPHA /* Am. pot */
{ 'a', 'y', 2, { "/ascriptturn",    500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 129, 129, 129, 129, 0x0252 }, // LATIN SMALL LETTER TURNED ALPHA /* Am. pot */
{ 'b', '^', 2, { "/bhooktop",       475, 0,   510, 580,  475, 0,   0,   0,    475, 0,    475, 0,   0,   0   }, 186, 186, 186, 186, 0x0253 }, // LATIN SMALL LETTER B WITH HOOK
{ '[', 'f', 2, { "/bracketleft",    333, 0,   346, 356,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   },  91,  91,  91,  91 },
{ ']', 'f', 2, { "/bracketright",   333, 0,   346, 356,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   },  93,  93,  93,  93 },
{ 'b', 'c', 2, { "/bcap",           513, 0,   539, 572,  513, 0,   0,   0,    513, 0,    513, 0,   0,   0   }, 245, 245, 245, 245, 0x0299 }, // LATIN LETTER SMALL CAPITAL B /* bilabial trill */
{ 'c', 't', 2, { "/cturn",          444, 0,   452, 462,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 141, 141, 141, 141, 0x0254 }, // LATIN SMALL LETTER OPEN O
{ 'c', 'c', 2, { "/ccurl",          444, 0,   462, 462,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 254, 254, 254, 254, 0x0255 }, // LATIN SMALL LETTER C WITH CURL
{ 'd', '.', 2, { "/drighttail",     500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 234, 234, 234, 234, 0x0256 }, // LATIN SMALL LETTER D WITH TAIL
{ 'd', '^', 2, { "/dhooktop",       500, 0,   523, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 235, 235, 235, 235, 0x0257 }, // LATIN SMALL LETTER D WITH HOOK
{ 'e', '-', 2, { "/erev",           444, 0,   462, 462,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 251, 130, 130, 251, 0x0258 }, // LATIN SMALL LETTER REVERSED E /* 1993 addition */
{ 's', 'w', 2, { "/schwa",          444, 0,   462, 462,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 171, 171, 171, 171, 0x0259 }, // LATIN SMALL LETTER SCHWA
{ 's', 'r', 2, { "/schwarighthook", 600, 0,   0,   0,    600, 0,   0,   0,    600, 0,    600, 0,   0,   0   }, 212,   0,   0, 212, 0x025a }, // LATIN SMALL LETTER SCHWA WITH HOOK /* Am. bird */
{ 'e', 'f', 2, { "/epsilonphonetic",444, 0,   441, 471,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   },  69,  69,  69,  69, 0x025b }, // LATIN SMALL LETTER OPEN E
{ 'e', 'r', 2, { "/epsilonrev",     444, 0,   441, 471,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 206, 206, 206, 206, 0x025c }, // LATIN SMALL LETTER REVERSED OPEN E
{ 'k', 'b', 2, { "/kidneybean",     500, 0,   471, 514,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 185, 207, 207, 185, 0x025e }, // LATIN SMALL LETTER CLOSED REVERSED OPEN E /* 1993 addition, 1996 correction */
{ 'j', '-', 2, { "/jdotlessbar",    333, 0,   289, 340,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   }, 239, 239, 239, 239, 0x025f }, // LATIN SMALL LETTER DOTLESS J WITH STROKE
{ 'g', '^', 2, { "/ghooktop",       500, 0,   520, 572,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 169, 169, 169, 169, 0x0260 }, // LATIN SMALL LETTER G WITH HOOK
{ 'g', 's', 2, { "/gscript",        475, 0,   500, 555,  475, 0,   0,   0,    475, 0,    475, 0,   0,   0   }, 103, 103, 103, 103, 0x0261 }, // LATIN SMALL LETTER SCRIPT G
{ 'g', 'c', 2, { "/gcap",           565, 0,   605, 659,  565, 0,   0,   0,    565, 0,    565, 0,   0,   0   },  71,  71,  71,  71, 0x0262 }, // LATIN LETTER SMALL CAPITAL G
{ 'g', 'f', 2, { "/gammaphonetic",  500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 196, 196, 196, 196, 0x0263 }, // LATIN SMALL LETTER GAMMA
{ 'r', 'h', 2, { "/ramshorn",       500, 0,   573, 603,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  70,  70,  70,  70, 0x0264 }, // LATIN SMALL LETTER RAMS HORN /* gambaby */
{ 'G', '^', 2, { "/gcaphooktop",    584, 0,   638, 662,  584, 0,   0,   0,    584, 0,    584, 0,   0,   0   }, 253, 253, 253, 253, 0x029b }, // LATIN LETTER SMALL CAPITAL G WITH HOOK
{ 'h', 't', 2, { "/hturn",          500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 231, 231, 231, 231, 0x0265 }, // LATIN SMALL LETTER TURNED H
{ 'h', '^', 2, { "/hhooktop",       500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 250, 250, 250, 250, 0x0266 }, // LATIN SMALL LETTER H WITH HOOK
{ 'h', 'j', 2, { "/henghooktop",    475, 0,   520, 578,  475, 0,   0,   0,    475, 0,    475, 0,   0,   0   }, 238, 238, 238, 238, 0x0267 }, // LATIN SMALL LETTER HENG WITH HOOK /* Swedish fricative */
{ 'h', 'c', 2, { "/hcap",           547, 0,   605, 659,  547, 0,   0,   0,    547, 0,    547, 0,   0,   0   },  75,  75,  75,  75, 0x029c }, //	LATIN LETTER SMALL CAPITAL H
{ 'h', '-', 2, { "/hbar",           525, 0,   520, 578,  525, 0,   0,   0,    525, 0,    525, 0,   0,   0   }, 240, 240, 240, 240 },
{ 'i', '-', 2, { "/ibar",           308, 0,   289, 289,  308, 0,   0,   0,    308, 0,    308, 0,   0,   0   }, 246, 246, 246, 246, 0x0268 }, // LATIN SMALL LETTER I WITH STROKE
{ 'i', 'c', 2, { "/icap",           280, 0,   289, 300,  280, 0,   0,   0,    280, 0,    280, 0,   0,   0   },  73,  73,  73,  73, 0x026a }, // LATIN LETTER SMALL CAPITAL I
{ 'j', 'c', 2, { "/jcurl",          318, 0,   289, 371,  318, 0,   0,   0,    318, 0,    318, 0,   0,   0   }, 198, 198, 198, 198, 0x029d }, // LATIN SMALL LETTER J WITH CROSSED TAIL
{ 'l', '~', 2, { "/ltilde",         380, 0,   0,   0,    380, 0,   0,   0,    380, 0,    380, 0,   0,   0   }, 201,   0,   0, 201, 0x026b }, // LATIN SMALL LETTER L WITH MIDDLE TILDE
{ 'l', '-', 2, { "/lbelt",          350, 0,   337, 371,  350, 0,   0,   0,    350, 0,    350, 0,   0,   0   }, 194, 194, 194, 194, 0x026c }, // LATIN SMALL LETTER L WITH BELT /* Welsh ll */
{ 'l', '.', 2, { "/lrighttail",     278, 0,   289, 298,  278, 0,   0,   0,    278, 0,    278, 0,   0,   0   }, 241, 241, 241, 241, 0x026d }, // LATIN SMALL LETTER L WITH RETROFLEX HOOK
{ 'l', 'z', 2, { "/lyogh",          506, 0,   604, 641,  506, 0,   0,   0,    506, 0,    506, 0,   0,   0   },  76,  76,  76,  76, 0x026e }, // LATIN SMALL LETTER LEZH
{ 'l', 'c', 2, { "/lcap",           455, 0,   502, 559,  455, 0,   0,   0,    455, 0,    455, 0,   0,   0   },  59,  59,  59,  59, 0x029f }, // LATIN LETTER SMALL CAPITAL L
{ 'm', 't', 2, { "/mturn",          778, 0,   809, 866,  778, 0,   0,   0,    778, 0,    778, 0,   0,   0   }, 181, 181, 181, 181, 0x026f }, // LATIN SMALL LETTER TURNED M
{ 'm', 'l', 2, { "/mturnleg",       778, 0,   809, 866,  778, 0,   0,   0,    778, 0,    778, 0,   0,   0   }, 229, 229, 229, 229, 0x0270 }, // LATIN SMALL LETTER TURNED M WITH LONG LEG
{ 'm', 'j', 2, { "/mlefttail",      753, 0,   795, 866,  753, 0,   0,   0,    753, 0,    753, 0,   0,   0   },  77,  77,  77,  77, 0x0271 }, // LATIN SMALL LETTER M WITH HOOK /* labiodental m */
{ 'n', 'g', 2, { "/eng",            475, 0,   506, 578,  475, 0,   0,   0,    475, 0,    475, 0,   0,   0   },  78,  78,  78,  78 }, /* velar nasal */
{ 'n', 'j', 2, { "/nlefttail",      500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 248, 248, 248, 248, 0x0272 }, // LATIN SMALL LETTER N WITH LEFT HOOK /* palatal n */
{ 'n', '.', 2, { "/nrighttail",     500, 0,   506, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 247, 247, 247, 247, 0x0273 }, // LATIN SMALL LETTER N WITH RETROFLEX HOOK
{ 'n', 'c', 2, { "/ncap",           547, 0,   595, 614,  547, 0,   0,   0,    547, 0,    547, 0,   0,   0   }, 178, 178, 178, 178, 0x0274 }, // LATIN LETTER SMALL CAPITAL N /* uvular nasal */
{ 'o', '-', 2, { "/obar",           500, 0,   520, 520,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  80,  80,  80,  80, 0x0275 }, //	LATIN SMALL LETTR BARRED O /* Swedish short u */
{ 'O', 'e', 2, { "/oecap",          727, 0,   745, 845,  727, 0,   0,   0,    727, 0,    727, 0,   0,   0   }, 175, 175, 175, 175, 0x0276 }, // LATIN LETTER SMALL CAPITAL OE
{ '|', '1', 2, { "/pipe",           278, 0,   221, 221,  278, 0,   0,   0,    278, 0,    278, 0,   0,   0   }, 142, 142, 142, 142, 0x01c0 },
{ '|', '2', 2, { "/pipedouble",     444, 0,   221, 221,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 146, 146, 146, 146, 0x01c1 },
{ '|', '-', 2, { "/pipedoublebar",  500, 0,   435, 435,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 156, 156, 156, 156, 0x01c2 },
{ '|', 'f', 2, { "/stroke",         278, 0,   208, 229,  278, 0,   0,   0,    278, 0,    278, 0,   0,   0   }, 150, 150, 150, 150 },
{ 'f', 'f', 2, { "/phiphonetic",    550, 0,   549, 616,  550, 0,   0,   0,    550, 0,    550, 0,   0,   0   }, 184, 184, 184, 184, 0x0278 }, // LATIN SMALL LETTER PHI
{ 'r', 't', 2, { "/rturn",          333, 0,   356, 462,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   }, 168, 168, 168, 168, 0x0279 }, // LATIN SMALL LETTER TURNED R
{ 'r', 'l', 2, { "/rturnleg",       333, 0,   356, 462,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   }, 228, 228, 228, 228, 0x027a }, // LATIN SMALL LETTER TURNED R WITH LONG LEG
{ 'r', '.', 2, { "/rturnrighttail", 333, 0,   362, 457,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   }, 211, 211, 211, 211, 0x027b }, // LATIN SMALL LETTER TURNED R WITH HOOK
{ 'f', '.', 2, { "/rrighttail",     333, 0,   356, 462,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   }, 125, 125, 125, 125, 0x027d }, // LATIN SMALL LETTER R WITH TAIL
{ 'f', 'h', 2, { "/fishhook",       333, 0,   380, 433,  333, 0,   0,   0,    333, 0,    333, 0,   0,   0   },  82,  82,  82,  82, 0x027e }, // LATIN SMALL LETTER R WITH FISHHOOK /* tap */
{ 'r', 'c', 2, { "/rcap",           541, 0,   559, 614,  541, 0,   0,   0,    541, 0,    541, 0,   0,   0   }, 123, 123, 123, 123, 0x0280 }, // LATIN LETTER SMALL CAPITAL R
{ 'r', 'i', 2, { "/rcapinv",        541, 0,   559, 613,  541, 0,   0,   0,    541, 0,    541, 0,   0,   0   }, 210, 210, 210, 210, 0x0281 }, // LATIN LETTER SMALL CAPITAL INVERTED R
{ 's', '.', 2, { "/srighttail",     389, 0,   405, 405,  389, 0,   0,   0,    389, 0,    389, 0,   0,   0   }, 167, 167, 167, 167, 0x0282 }, // LATIN SMALL LETTER S WITH HOOK
{ 's', 'h', 2, { "/esh",            328, 0,   351, 361,  328, 0,   0,   0,    328, 0,    328, 0,   0,   0   },  83,  83,  83,  83, 0x0283 }, // LATIN SMALL LETTER ESH
{ 'j', '^', 2, { "/jbarhooktop",    300, 0,   289, 360,  300, 0,   0,   0,    300, 0,    300, 0,   0,   0   }, 215, 215, 215, 215, 0x0284 }, // LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK
{ 't', '.', 2, { "/trighttail",     325, 0,   289, 345,  325, 0,   0,   0,    325, 0,    325, 0,   0,   0   }, 255, 255, 255, 255, 0x0288 }, // LATIN SMALL LETTER T WITH RETROFLEX HOOK
{ 'u', '-', 2, { "/ubar",           500, 0,   520, 578,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 172, 172, 172, 172, 0x0289 }, // LATIN SMALL LETTER U BAR /* Swedish long u */
{ 'h', 's', 2, { "/horseshoe",      550, 0,   520, 549,  550, 0,   0,   0,    550, 0,    550, 0,   0,   0   },  85,  85,  85,  85, 0x028a }, // LATIN SMALL LETTER UPSILON /* omegaturn, "upsilon" */
{ 'v', 's', 2, { "/vscript",        461, 0,   488, 533,  461, 0,   0,   0,    461, 0,    461, 0,   0,   0   },  86,  86,  86,  86, 0x028b }, // LATIN SMALL LETTER V WITH HOOK /* Dutch w */
{ 'v', 't', 2, { "/vturn",          500, 0,   520, 520,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 195, 195, 195, 195, 0x028c }, // LATIN SMALL LETTER TURNED V /* wedge */
{ 'w', 't', 2, { "/wturn",          722, 0,   751, 751,  722, 0,   0,   0,    722, 0,    722, 0,   0,   0   }, 227, 227, 227, 227, 0x028d }, // LATIN SMALL LETTER TURNED W
{ 'y', 't', 2, { "/yturn",          500, 0,   520, 520,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 180, 180, 180, 180, 0x028e }, // LATIN SMALL LETTER TURNED Y
{ 'y', 'c', 2, { "/ycap",           547, 0,   605, 612,  547, 0,   0,   0,    547, 0,    547, 0,   0,   0   },  89,  89,  89,  89, 0x028f }, // LATIN LETTER SMALL CAPITAL Y
{ 'z', '.', 2, { "/zrighttail",     444, 0,   462, 462,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   }, 189, 189, 189, 189, 0x0290 }, // LATIN SMALL LETTER Z WITH RETROFLEX HOOK
{ 'z', 'c', 2, { "/zcurl",          494, 0,   482, 521,  494, 0,   0,   0,    494, 0,    494, 0,   0,   0   }, 252, 252, 252, 252, 0x0291 }, // LATIN SMALL LETTER Z WITH CURL
{ 'z', 'h', 2, { "/yogh",           444, 0,   459, 521,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   },  90,  90,  90,  90, 0x0292 }, // LATIN SMALL LETTER EZH
{ '?', 'g', 2, { "/glottalstop",    500, 0,   440, 477,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  47,  63,  63,  47, 0x0294 }, // LATIN LETTER GLOTTAL STOP
{ '9', 'e', 2, { "/glotrev",        500, 0,   440, 477,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 214, 192, 192, 214, 0x0295 }, // LATIN LETTER PHARYNGEAL VOICED FRICATIVE
{ 'O', '.', 2, { "/bullseye",       722, 0,   799, 818,  722, 0,   0,   0,    722, 0,    722, 0,   0,   0   }, 135, 135, 135, 135, 0x0298 }, // LATIN LETTER BILABIAL CLICK
{ '?', '-', 2, { "/glotbar",        500, 0,   440, 477,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  63, 251, 251,  63, 0x02a1 }, // LATIN LETTER GLOTTAL STOP WITH STROKE /* epiglottal plosive */
{ '9', '-', 2, { "/glotrevbar",     500, 0,   440, 477,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   }, 192, 185, 185, 192, 0x02a2 }, // LATIN LETTER REVERSED GLOTTAL STOP WITH STROKE /* epiglottal fricative */
{ ':', 'f', 2, { "/lengthsign",     250, 0,   217, 235,  250, 0,   0,   0,    250, 0,    250, 0,   0,   0   }, 249, 249, 249, 249, 0x02d0 }, // MODIFIER LETTER TRIANGULAR COLON
{ 'h', 'r', 2, { "/righthook",      300, 0,   250, 265,  300, 0,   0,   0,    300, 0,    300, 0,   0,   0   }, 213, 213, 213, 213, 0x02de }, // MODIFIER LETTER RHOTIC HOOK /* or rturnsuper */
{ '`', '^', 2, { "/graveover",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  36,  36,  36,  36, 0x0300 }, // COMBINING GRAVE ACCENT
{ '\'','^', 2, { "/acuteover",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  64,  64,  64,  64, 0x0301 }, // COMBINING ACUTE ACCENT
{ '^', '^', 2, { "/circumover",       0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  94,  94,  94,  94, 0x0302 }, // COMBINING CIRCUMFLEX ACCENT
{ '~', '^', 2, { "/tildeover",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  41,  41,  41,  41, 0x0303 }, // COMBINING TILDE /* nasalized */
{ '-', '^', 2, { "/minusover",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  35,  35,  35,  35, 0x0304 }, // COMBINING MACRON /* mid tone or so */
{ ':', '^', 2, { "/diaeresisover",    0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  95,  95,  95,  95, 0x0308 }, // COMBINING DIAERESIS /* centralized */
{ '0', '^', 2, { "/ringover",         0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  42,  42,  42,  42, 0x030a }, // COMBINING RING ABOVE /* voiceless */
{ 'v', '^', 2, { "/caronover",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  38,  38,  38,  38, 0x030c }, // COMBINING CARON /* hacek */
{ 'c', 'n', 2, { "/corner",         260, 0,   299, 299,  260, 0,   0,   0,    260, 0,    260, 0,   0,   0   }, 124, 124, 124, 124, 0x031a }, // COMBINING LEFT ANGLE ABOVE? /* unreleased */
{ 'c', 'v', 2, { "/halfringleft",     0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  55,  55,  55,  55, 0x031c }, // COMBINING LEFT HALF RING BELOW /* unrounded */
{ 'T', '^', 2, { "/raising",          0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  51,  51,  51,  51, 0x031d }, // COMBINING UP TACK BELOW
{ 'T', 'v', 2, { "/lowering",         0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  52,  52,  52,  52, 0x031e }, // COMBINING DOWN TACK BELOW
{ '+', 'v', 2, { "/plusunder",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  43,  43,  43,  43, 0x031f }, // COMBINING PLUS SIGN BELOW /* fronted */
{ ':', 'v', 2, { "/diaeresisunder",   0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  45,  45,  45,  45, 0x0324 }, // COMBINING DIAERESIS BELOW /* breathy voiced */
{ '0', 'v', 2, { "/ringunder",        0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  56,  56,  56,  56, 0x0325 }, // COMBINING RING BELOW /* voiceless */
{ '|', 'v', 2, { "/strokeunder",      0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  96,  96,  96,  96, 0x0329 }, // COMBINING VERTICAL LINE BELOW /* syllabicity mark */
{ 'N', 'v', 2, { "/bridgeunder",      0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  53,  53,  53,  53, 0x032a }, // COMBINING BRIDGE BELOW /* dental */
{ '~', 'v', 2, { "/tildeunder",       0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  48,  48,  48,  48, 0x0330 }, // COMBINING TILDE BELOW /* creaky voiced */
{ '3', 'v', 2, { "/halfringright",    0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   }, 166, 166, 166, 166, 0x0339 }, // COMBINING RIGHT HALF RING BELOW /* rounded */
{ '-', 'v', 2, { "/minusunder",       0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },  61,  61,  61,  61, 0x0331 }, // COMBINING MACRON BELOW /* backed */
{ '~', '<', 2, { "/tildethrough",     0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },   0, 242, 242,   0, 0x0334 }, // COMBINING TILDE OVERLAY /* velarized l */
{ 'l', 'i', 2, { "/ligature",         0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   }, 131, 131, 131, 131 },
{ 'b', 'f', 2, { "/betaphonetic",   500, 0,   520, 597,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  66,  66,  66,  66 },
{ 't', 'f', 2, { "/thetaphonetic",  444, 0,   520, 585,  444, 0,   0,   0,    444, 0,    444, 0,   0,   0   },  84,  84,  84,  84 },
{ 'c', 'f', 2, { "/chiphonetic",    500, 0,   572, 610,  500, 0,   0,   0,    500, 0,    500, 0,   0,   0   },  88,  88,  88,  88 },

/* Dingbats. */
/*fir  sec  al    ps                tim  b    i    bi    cen  b    i    bi    hel  b     pal  b    i    bi     xwin win  mac   ps  unicode decomp  */
{ 'p', 'f', 3, { "/fingerright",   1000, 0,   0,   0,   1000, 0,   0,   0,   1000, 0,   1000, 0,   0,   0   },  43,  70,  43,  43 }, /* pointing finger */
/* Not yet bitmapped or measured. */
{ 'f', '5', 3, { "/flower5",        800, 0,   0,   0,    800, 0,   0,   0,    800, 0,    800, 0,   0,   0   },  96,  96,  96,  96 }, /* sympathy flower */

{'\0','\0', 0, { 0,                   0, 0,   0,   0,      0, 0,   0,   0,      0, 0,      0, 0,   0,   0   },   0,   0,   0,   0 }  /* Closing. */
};

static short where [95] [95];
static short inited = 0;
static struct { char first, second; } genericDigraph [256];

static void init (void) {
	Longchar_Info data;
	int i;
	for (i = 0, data = & Longchar_database [0]; data -> first != '\0'; i ++, data ++) {
		short *location = & where [data -> first - 32] [data -> second - 32];
		if (*location) {
			/* Doubly defined symbol; an error! */
			/* We may not be able to use Melder_error yet, so just write a warning to stderr. */
			fprintf (stderr, "Longchar init: symbol \"%c%c\" doubly defined.\n", data -> first, data -> second);
		}
		*location = i;
		if (data -> alphabet == Longchar_ROMAN) {
			#if defined (UNIX)
				genericDigraph [data -> xwinEncoding]. first = data -> first;
				genericDigraph [data -> xwinEncoding]. second = data -> second;
			#elif defined (macintosh)
				genericDigraph [data -> macEncoding]. first = data -> first;
				genericDigraph [data -> macEncoding]. second = data -> second;
			#elif defined (_WIN32)
				genericDigraph [data -> winEncoding]. first = data -> first;
				genericDigraph [data -> winEncoding]. second = data -> second;
			#else
				#error Choose an encoding.
			#endif
		}
	}
	inited = 1;
}

char * Longchar_nativize (const char *generic, char *native, int educateQuotes) {
	short kar;
	#ifdef macintosh
		long nquote = 0;
	#endif
	unsigned char kar1, kar2;
	if (! inited) init ();
	while ((kar = *generic++) != '\0') {
		#ifdef macintosh
			if (educateQuotes) {
				if (kar == '\"') {   /* Replace by left or right double quote. */
					*native++ = ++nquote & 1 ? 210 : 211;   /* Mac encoding */
					continue;
				} else if (kar == '`') {   /* Grave. */
					*native++ = 212;   /* Left single quote. */
					continue;
				} else if (kar == '\'') {   /* Straight apostrophe. */
					*native++ = 213;   /* Right single quote. */
					continue;
				}
			}
		#else
			(void) educateQuotes;
		#endif
		if (kar == '\\' && (kar1 = generic [0]) >= 32 && kar1 <= 126 && (kar2 = generic [1]) >= 32 && kar2 <= 126) {
			long location = where [kar1 - 32] [kar2 - 32];
			if (location == 0 || Longchar_database [location]. alphabet != Longchar_ROMAN) {
				*native++ = kar;
				*native++ = kar1;   /* Even if this is a backslash itself... */
				*native++ = kar2;   /* Even if this is a backslash itself... */
				/* These "evens" are here to ensure that Longchar_nativize does nothing on an already nativized string. */
			} else {
				#if defined (UNIX)
					*native++ = Longchar_database [location]. xwinEncoding;
				#elif defined (macintosh)
					*native++ = Longchar_database [location]. macEncoding;
				#elif defined (_WIN32)
					*native++ = Longchar_database [location]. winEncoding;
				#else
					#error Choose an encoding.
				#endif
			}	
			generic += 2;
		} else {
			*native++ = kar;
		}
	}
	*native++ = '\0';
	return native;
}

char *Longchar_genericize (const char *native, char *g) {
	unsigned char kar;
	if (! inited) init ();
	while ((kar = *native++) != '\0') {
		if (kar >= 32 && kar <= 126) {
			*g++ = kar;
		} else {
			*g++ = '\\';
			*g++ = genericDigraph [kar]. first;
			*g++ = genericDigraph [kar]. second;
		}
	}
	*g++ = '\0';
	return g;
}

Longchar_Info Longchar_getInfo (char kar1, char kar2) {
	short position;
	if (! inited) init ();
	position = kar1 < 32 || kar1 > 126 || kar2 < 32 || kar2 > 126 ?
		0 :   /* Return the 'space' character. */
		where [kar1 - 32] [kar2 - 32];
	return & Longchar_database [position];
}

Longchar_Info Longchar_getInfoFromNative (unsigned char kar) {
	return Longchar_getInfo (genericDigraph [kar]. first, genericDigraph [kar]. second);
}

short *Longchar_genericToWide (const char *generic, short *wide) {
	short kar;
	unsigned char kar1, kar2;
	if (! inited) init ();
	while ((kar = *generic++) != '\0') {
		if (kar == '\\' && (kar1 = generic [0]) >= 32 && kar1 <= 126 && (kar2 = generic [1]) >= 32 && kar2 <= 126) {
			*wide++ = (kar1 << 8) + kar2;
			generic += 2;
		} else {
			*wide++ = (kar << 8) + ' ';
		}
	}
	*wide++ = 0;
	return wide;
}

/* End of file longchar.c */
