/* Distributions.cpp
 *
 * Copyright (C) 1997-2012,2014,2015,2016 Paul Boersma
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This code is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this work. If not, see <http://www.gnu.org/licenses/>.
 */

#include "Distributions.h"

Thing_implement (Distributions, TableOfReal, 0);

void structDistributions :: v_info () {
	structDaata :: v_info ();
	MelderInfo_writeLine (U"Number of distributions: ", numberOfColumns);
	MelderInfo_writeLine (U"Number of values: ", numberOfRows);
}

autoDistributions Distributions_create (long numberOfRows, long numberOfColumns) {
	try {
		autoDistributions me = Thing_new (Distributions);
		TableOfReal_init (me.get(), numberOfRows, numberOfColumns);
		return me;
	} catch (MelderError) {
		Melder_throw (U"Distributions not created.");
	}
}

void Distributions_checkSpecifiedColumnNumberWithinRange (Distributions me, long columnNumber) {
	if (columnNumber < 1)
		Melder_throw (me, U": the specified column number is ", columnNumber, U", but should be at least 1.");
	if (columnNumber > my numberOfColumns)
		Melder_throw (me, U": the specified column number is ", columnNumber, U", but should be at most my number of columns (", my numberOfColumns, U").");
}

void Distributions_peek (Distributions me, long column, char32 **string, long *number) {
	Distributions_checkSpecifiedColumnNumberWithinRange (me, column);
	if (my numberOfRows < 1)
		Melder_throw (me, U": I have no candidates.");
	double total = 0.0;
	for (long irow = 1; irow <= my numberOfRows; irow ++) {
		total += my data [irow] [column];
	}
	if (total <= 0.0)
		Melder_throw (me, U": the total weight of column ", column, U" is not positive.");
	long irow;
	do {
		double rand = NUMrandomUniform (0, total), sum = 0.0;
		for (irow = 1; irow <= my numberOfRows; irow ++) {
			sum += my data [irow] [column];
			if (rand <= sum) break;
		}
	} while (irow > my numberOfRows);   // guard against rounding errors
	if (! my rowLabels [irow])
		Melder_throw (me, U": no string in row ", irow, U".");
	if (string)
		*string = my rowLabels [irow];
	if (number)
		*number = irow;
}

double Distributions_getProbability (Distributions me, const char32 *string, long column) {
	long row, rowOfString = 0;
	double total = 0.0;
	if (column < 1 || column > my numberOfColumns) return NUMundefined;
	for (row = 1; row <= my numberOfRows; row ++) {
		total += my data [row] [column];
		if (my rowLabels [row] && str32equ (my rowLabels [row], string))
			rowOfString = row;
	}
	if (total <= 0.0) return NUMundefined;
	if (rowOfString == 0) return 0.0;
	return my data [rowOfString] [column] / total;
}

double Distributionses_getMeanAbsoluteDifference (Distributions me, Distributions thee, long column) {
	if (column < 1 || column > my numberOfColumns || column > thy numberOfColumns ||
	    my numberOfRows != thy numberOfRows) return NUMundefined;
	double total = 0.0;
	for (long irow = 1; irow <= my numberOfRows; irow ++) {
		total += fabs (my data [irow] [column] - thy data [irow] [column]);
	}
	return total / my numberOfRows;
}

static void unicize (Distributions me) {
	/* Must have been sorted beforehand. */
	long nrow = 0, ifrom = 1;
	for (long irow = 1; irow <= my numberOfRows; irow ++) {
		if (irow == my numberOfRows || (my rowLabels [irow] == nullptr) != (my rowLabels [irow + 1] == nullptr) ||
		    (my rowLabels [irow] != nullptr && ! str32equ (my rowLabels [irow], my rowLabels [irow + 1])))
		{
			/*
			 * Detected a change.
			 */
			nrow ++;
			long ito = irow;
			/*
			 * Move row 'ifrom' to 'nrow'. May be the same row.
			 */
			if (ifrom != nrow) {
				Melder_free (my rowLabels [nrow]);
				my rowLabels [nrow] = my rowLabels [ifrom];   // surface copy
				my rowLabels [ifrom] = nullptr;   // undangle
				for (long icol = 1; icol <= my numberOfColumns; icol ++)
					my data [nrow] [icol] = my data [ifrom] [icol];
			}
			/*
			 * Purge rows from 'ifrom'+1 to 'ito'.
			 */
			for (long j = ifrom + 1; j <= ito; j ++) {
				Melder_free (my rowLabels [j]);
				for (long icol = 1; icol <= my numberOfColumns; icol ++)
					my data [nrow] [icol] += my data [j] [icol];
			}
			ifrom = ito + 1;
		}
	}
	my numberOfRows = nrow;
}

autoDistributions Distributions_addTwo (Distributions me, Distributions thee) {
	try {
		autoDistributions him = TablesOfReal_append (me, thee).static_cast_move<structDistributions>();
		TableOfReal_sortByLabel (him.get(), 0, 0);
		unicize (him.get());
		return him;
	} catch (MelderError) {
		Melder_throw (me, U" & ", thee, U": not added.");
	}
}

autoDistributions Distributions_addMany (OrderedOf<structDistributions>* me) {
	try {
		autoDistributions thee = TablesOfReal_appendMany ((OrderedOf<structTableOfReal>*) me).static_cast_move<structDistributions>();   // FIXME cast
		TableOfReal_sortByLabel (thee.get(), 0, 0);
		unicize (thee.get());
		return thee;
	} catch (MelderError) {
		Melder_throw (U"Distributions objects not added.");
	}
}

/* End of file Distributions.cpp */
