/* headers.c: output declarations of some Palm OS API functions.

   Copyright 2001 by John Marshall.
   <jmarshall@acm.org>

   This is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.  */

#include <StringMgr.h>
#include <SoundMgr.h>
#include <Font.h>
#include <Rect.h>
#include <Window.h>
#include <SysUtils.h>

#include <NewTypes.h>

#include "genbootstrap.h"

void
header (const char *fname) {
  char guard[64];
  char *s;

  StrCopy (guard, fname);
  for (s = guard; *s; s++) {
    if (*s >= 'a' && *s <= 'z')  *s += 'A' - 'a';
    if (! ((*s >= 'A' && *s <= 'Z') || (*s >= '0' && *s <= '9')))  *s = '_';
    }

  printf (
"/* DO NOT EDIT!  This file is generated by GenBootstrap; any modifications\n"
"   should be made to that program's source code instead.\n"
"\n"
"   The contents of this file are in the public domain.  */\n"
"\n");
  printf ("#ifndef %s\n", guard);
  printf ("#define %s\n", guard);
  }

void
trailer (const char *fname __attribute__ ((unused))) {
  printf ("\n");
  printf ("#endif\n");
  }

void
banner (const char *title) {
  printf ("\n/* %s */\n\n", title);
  fprintf (console, "... %s", title);
  }


void
init_codings (char *v[]) {
  int i;

  for (i = 0; i < 256; i++)
    v[i] = "";

  v['u'] = "U";
  v['8'] = "Int8";
  v['1'] = "Int16";
  v['3'] = "Int32";
  v['v'] = "void";
  v['C'] = "char";
  v['e'] = "Err";
  v['h'] = "MemHandle";
  v['p'] = "MemPtr";

  v['c'] = "const ";
  v['t'] = "typedef ";
  v['s'] = "struct ";
  v['E'] = "enum ";
  v['x'] = "extern ";

  v[' '] = " ";
  v['.'] = "  ";
  v[','] = ", ";
  v['*'] = "*";
  v['('] = "(";
  v[')'] = ")";
  v['{'] = "{";
  v['}'] = "}";
  v[';'] = ";";
  v['='] = "=";
  v['|'] = "\n";
  }

void
emit_fmt (const char *s) {
  static char *decode[256];

  if (decode[0] == NULL)
    init_codings (decode);

  while (*s)
    if (*s == '<')
      for (s++; *s != '>'; s++)
	printf ("%c", *s);
    else
      printf ("%s", decode[(unsigned char)*s++]);
  }

void
nl () {
  printf ("\n");
  }


#define stringise(x)  #x
#define xstringise(x)  stringise (x)

#define define(id)  printf ("#define " #id " " stringise (id) "\n")

#define declare2(types, id)				\
  emit_fmt (types " <" #id ">;|")

#define declare3(ret, id, params)			\
  emit_fmt (ret " <" #id "> (" params ");|")

#define trap_declare3(ret, id, params)			\
  emit_fmt (ret " <" #id "> (" params ")  "		\
	    "<TRAP (" xstringise (sysTrap##id) ")>;|")

#define declare_enumerator(id)  emit_enumerator (#id, id)
void
emit_enumerator (const char *id, int value) {
  char fmt[64];
  StrPrintF (fmt, "E{<%s> = <%d>};|", id, value);
  emit_fmt (fmt);
  }


void
basic_type (const char *sign1, const char sign2, int size, const char *mode) {
  printf ("typedef %s int %s%cInt%d __attribute__ ((mode (%s)));\n",
	  sign1, (size < 10)? " " : "", sign2, size, mode);
  }

void
basic_types (int size, const char *mode) {
  basic_type ("        ", ' ', size, mode);
  basic_type ("unsigned", 'U', size, mode);
  }


void
bootstrap (const char *fname) {
  header (fname);

  banner ("Basic types");

  basic_types (8, "QI");
  basic_types (16, "HI");
  basic_types (32, "SI");
  nl ();

  /* The Palm OS SDKs declare MemHandle as a pointer to something-other-
     than-void.  This enables some type checking, preventing some instances
     of confusing MemPtrs and MemHandles.  But what is here is good enough
     for our purposes.  */

  declare2 ("tv*", MemPtr);
  declare2 ("tv*", MemHandle);
  nl ();

  declare2 ("t8", Char);
  declare2 ("t16", Err);
  nl ();

  /* Since the argument to TRAP is always a number (rather than a macro
     evaluating to a number), we can use # directly rather than via an
     indirect STR() macro.  */

  printf ("#define TRAP(vector) __attribute__ "
	  "((__callseq__ (\"trap #15; dc.w \" #vector)))\n");

  banner ("Data Manager");

  declare2 ("tv*", DmOpenRef);
  nl ();

  define (dmModeReadOnly);
  nl ();

  trap_declare3 ("<DmOpenRef>", DmOpenDatabaseByTypeCreator, "u32,u32,u16");
  trap_declare3 ("v", DmCloseDatabase, "<DmOpenRef>");
  trap_declare3 ("h", DmGet1Resource, "u32,u16");
  trap_declare3 ("v", DmReleaseResource, "h");

  banner ("System Manager");

  emit_fmt ("ts{|");
  declare2 (".16", cmd);
  declare2 (".v*", cmdPBP);
  declare2 (".u16", launchFlags);
  /* There are more fields, but this is all the startup code needs.  */
  declare2 (".}", SysAppInfoType);
  nl ();

  define (sysAppLaunchFlagNewGlobals);
  define (sysAppLaunchFlagSubCall);
  nl ();

  trap_declare3 ("e", SysAppStartup, "<SysAppInfoType>**,v**,v**");
  trap_declare3 ("e", SysAppExit, "<SysAppInfoType>*,v*,v*");
  nl ();

  declare3 ("xu32", PilotMain, "16,v*,u16");

  banner ("Error Manager");

  trap_declare3 ("v", ErrDisplayFileLineMsg, "C*,u16,C*");
  nl ();

#define COMPOUNDIFY(stmt)  do { stmt; } while (0)

  define (COMPOUNDIFY(stmt));
  nl ();

  printf ("#define ErrFatalDisplayIf(cond, text) \\\n"
	  "  COMPOUNDIFY (if (cond)"
	  "  ErrDisplayFileLineMsg (__FILE__, __LINE__, text))\n");

  banner ("Feature Manager");

  trap_declare3 ("e", FtrUnregister, "u32,u16");
  trap_declare3 ("e", FtrGet, "u32,u16,u32*");
  trap_declare3 ("e", FtrSet, "u32,u16,u32");

  banner ("Memory Manager");

#define sysTrapMemPtrFree sysTrapMemChunkFree

  trap_declare3 ("e",	MemPtrFree,	"p");
  trap_declare3 ("p",	MemPtrNew,	"u32");
  trap_declare3 ("u32",	MemPtrSize,	"p");
  trap_declare3 ("e",	MemPtrSetOwner,	"p,u16");
  trap_declare3 ("e",	MemPtrResize,	"p,u32");
  trap_declare3 ("h",	MemHandleNew,	"u32");
  trap_declare3 ("p",	MemHandleLock,	"h");
  trap_declare3 ("e",	MemHandleUnlock,"h");
  trap_declare3 ("e",	MemMove,	"v*,cv*,u32");
  trap_declare3 ("e",	MemHandleFree,	"h");
  trap_declare3 ("u32",	MemHandleSize,	"h");
  trap_declare3 ("e",	MemHandleSetOwner,"h,u16");
  trap_declare3 ("e",	MemHandleResize,"h,u32");
  trap_declare3 ("e",	MemSet,		"v*,u32,u8");

  banner ("Sound Manager");

  declare_enumerator (sndError);
  nl ();

  trap_declare3 ("v",	SndPlaySystemSound, "u8");

  banner ("String Manager");

  trap_declare3 ("C*",	StrCopy, "C*,cC*");
  trap_declare3 ("C*",	StrCat,  "C*,cC*");
  trap_declare3 ("16",	StrLen,  "cC*");

  banner ("System Utilities");

  define (sysRandomMax);

  trailer (fname);
  }


#if SDK_VERSION < 35
enum { winUp = up };
#endif

void
bootstrap_ui (const char *fname) {
  header (fname);

  banner ("Font Manager");

  declare_enumerator (stdFont);
  nl ();

  trap_declare3 ("u8",	FntSetFont, "u8");

  banner ("Rectangles");

  emit_fmt ("ts{|");
  declare2 (".16", x);
  declare2 (".16", y);
  declare2 (".}", PointType);
  nl ();

  emit_fmt ("ts{|");
  declare2 (".<PointType>", topLeft);
  declare2 (".<PointType>", extent);
  declare2 (".}", RectangleType);

  banner ("Window Manager");

  declare_enumerator (winUp);
  nl ();

  trap_declare3 ("v",	WinScrollRectangle,
		 "c<RectangleType>*,u8,16,<RectangleType>*");
  trap_declare3 ("v",	WinEraseRectangle, "c<RectangleType>*,u16");
  trap_declare3 ("v",	WinDrawChars,      "cC*,16,16,16");

  trailer (fname);
  }
