/* Copyright (C) 1999-2000 Bernhard Trummer
 * Copyright (C) 2003      Benjamin Drieu
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: gridnode.cc,v $
 * Revision 1.6  2003/03/06 14:54:21  benj
 * Update copyrigth where it is necessary
 *
 * Revision 1.5  2003/02/18 12:56:49  bdrieu
 * - Update to make prestimel compile with g++-3.2
 * - Use Magick++ instead of Magick
 *
 * Revision 1.4  2002/03/04 13:58:59  slash
 * Bugfix for wrong detection of text-subenvironments.
 *
 * Revision 1.3  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:26  slash
 * Initial release.
 *
 */

#include "main.h"
#include "parameters.h"
#include "textnode.h"
#include "codenode.h"
#include "imagenode.h"
#include "itemizenode.h"
#include "enumeratenode.h"
#include "gridnode.h"


// The following macros are used to provide an easy handling,
// if the apropriate property isn't given in the XML-file.
#define ROWS_      (rows_ != -1 ? rows_ : 1)
#define COLUMNS_   (columns_ != -1 ? columns_ : 1)
#define STEP_      (step_ != -1 ? step_ : 0)


//---------------------------------------------------------------------------
GridNode::GridNode(xmlNodePtr node) : EnvironmentNode()
{
    rows_ = -1;
    columns_ = -1;
    step_ = -1;
    border_ = -1;
    cellpadding_ = -1;
    contract_ = -1;
    subenvironment_step_ = 0;

    // Ectract the properties of the grid-markup.
    char *rows = (char*)xmlGetProp(node, (xmlChar*)"rows");
    if (rows) {
        rows_ = atoi(rows);
    }
    char *columns = (char*)xmlGetProp(node, (xmlChar*)"columns");
    if (columns) {
        columns_ = atoi(columns);
    }
    char *step = (char*)xmlGetProp(node, (xmlChar*)"step");
    if (step) {
        if      (!strcmp(step, "yes")) step_ = 1;
        else if (!strcmp(step, "no"))  step_ = 0;
    }
    char *cellpadding = (char*)xmlGetProp(node, (xmlChar*)"cellpadding");
    if (cellpadding) {
        cellpadding_ = atoi(cellpadding);
    }
    char *border = (char*)xmlGetProp(node, (xmlChar*)"border");
    if (border) {
        border_ = atoi(border);
    }
    char *contract = (char*)xmlGetProp(node, (xmlChar*)"contract");
    if (contract) {
        if      (!strcmp(contract, "yes")) contract_ = 1;
        else if (!strcmp(contract, "no"))  contract_ = 0;
    }


    // Extract the sub-environments.
    for (node = node->xmlChildrenNode; node; node = node->next) {
        EnvironmentNode *sub_environment = NULL;
        if (node->type == XML_ELEMENT_NODE) {
            if (!strcmp((char*)node->name,"itemize")) {
                sub_environment = new ItemizeNode(node, 0, 0);
            } else if (!strcmp((char*)node->name,"enumerate")) {
                sub_environment = new EnumerateNode(node, 0, 0);
            } else if (!strcmp((char*)node->name,"image")) {
                sub_environment = new ImageNode(node);
            } else if (!strcmp((char*)node->name,"code")) {
                sub_environment = new CodeNode(node, 0);
            } else if (!strcmp((char*)node->name,"text")) {
                sub_environment = new TextNode(node, 0);
            } else if (node->type != XML_COMMENT_NODE) {
                Number_Of_Errors_++;
            }
        }

        if (sub_environment) {
            if (sub_environment->isHTMLStepped()) {
                if (sub_environment->getNumberOfHTMLSteps() > number_of_html_steps_) {
                    number_of_html_steps_ = sub_environment->getNumberOfHTMLSteps();
                }
                subenvironment_step_ = 1;
            }
            child_nodes_.push_back(sub_environment);
        }
    }

    if (child_nodes_.size() == 0) {
        Number_Of_Errors_++;
    }
}

//---------------------------------------------------------------------------
GridNode::~GridNode()
{
}


//---------------------------------------------------------------------------
void GridNode::writeHTML(int parameter) const
{
    Output_ << "<TABLE"
            << " summary=\"grid\""
            << " align=\"center\"";
    if (contract_ != 1) {
        Output_ << " width=\"90%\"";
    }
    Output_ << " cellpadding=\"" << (cellpadding_ != -1 ?
                                     cellpadding_ :
                                     Parameters_.getCellPadding()) << "\""
            << " border=\"" << (border_ != -1 ? border_ : 0) << "\""
            << ">" << std::endl;

    for (int i=0; i<ROWS_; i++) {
        Output_ << "<TR>" << std::endl;
        for (int j=0; j<COLUMNS_; j++) {
            Output_ << "<TD";
            if (contract_ != 1) {
                Output_ << " width=\"" << 100/COLUMNS_ << "%\"";
            }
            Output_ << " valign=\"top\""
                    << ">" << std::endl;

            if (i*COLUMNS_+j < (STEP_ == 0 ?
                                (signed)child_nodes_.size() : parameter)) {
                child_nodes_[i*COLUMNS_ + j]->writeHTML(STEP_ == 0 ?
                                                        parameter : -1);
            } else {
                Output_ << "&nbsp;";
            }

            Output_ << "</TD>" << std::endl;
        }
        Output_ << "</TR>" << std::endl;
    }

    Output_ << "</TABLE>" << std::endl;
}

//---------------------------------------------------------------------------
void GridNode::writeLaTeX() const
{
    if (Parameters_.isOutlineEnabled()) {
        Output_ << "\\begin{center}" << std::endl;
        Output_ << "\\begin{tabularx}{0.9\\textwidth}{";

        if (border_ == 1) {
            Output_ << "|";
        }
        for (int i=0; i<COLUMNS_; i++) {
            Output_ << "X";
            if (border_ == 1) {
                Output_ << "|";
            }
        }
        Output_ << "}" << std::endl;
        if (border_ == 1) {
            Output_ << "\\hline" << std::endl;
        }
        for (unsigned int i=0; i<child_nodes_.size(); i++) {
            Output_ << "\\parbox[t]{" << 0.85/COLUMNS_ << "\\textwidth}"
                    << "{\\raggedright ";
            child_nodes_[i]->writeLaTeX();
            Output_ << "}" << std::endl;

            if ((i+1) % COLUMNS_ == 0) {
                Output_ << "\\\\" << std::endl;
                if (border_ == 1) {
                    Output_ << "\\hline" << std::endl;
                }
            } else {
                Output_ << "&" << std::endl;
            }
        }

        Output_ << "\\end{tabularx}" << std::endl;
        Output_ << "\\end{center}" << std::endl << std::endl;

    } else {
        for (unsigned int i=0; i<child_nodes_.size(); i++) {
            // Put the subenvironments in minipages.
            Output_ << "\\begin{minipage}{" << 0.92/COLUMNS_ << "\\linewidth}"
                    << std::endl;
            child_nodes_[i]->writeLaTeX();
            Output_ << "\\end{minipage}" << std::endl;
            
            if ((i+1) % COLUMNS_ != 0) {
                Output_ << "\\hspace{0.04\\linewidth}" << std::endl;
            }
        }
    }
}

//---------------------------------------------------------------------------
void GridNode::writeXML() const
{
    Output_ << "<grid";
    if (rows_ != -1) {
        Output_ << " rows=\"" << rows_ << "\"";
    }
    if (columns_ != -1) {
        Output_ << " columns=\"" << columns_ << "\"";
    }
    if (step_ != -1) {
        Output_ << " step=\"" << (step_ == 1 ? "yes" : "no") << "\"";
    }
    if (cellpadding_ != -1) {
        Output_ << " cellpadding=\"" << cellpadding_ << "\"";
    }
    if (border_ != -1) {
        Output_ << " border=\"" << border_ << "\"";
    }
    if (contract_ != -1) {
        Output_ << " contract=\"" << (contract_ == 1 ? "yes" : "no") << "\"";
    }
    Output_ << ">" << std::endl;

    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</grid>" << std::endl;
}


//---------------------------------------------------------------------------
int GridNode::isHTMLStepped() const
{
    return (STEP_ == 1 ? 1 : subenvironment_step_);
}

//---------------------------------------------------------------------------
unsigned int GridNode::getNumberOfHTMLSteps() const
{
    return (STEP_ == 1 ? child_nodes_.size() : number_of_html_steps_);
}
