##############################################################################
#
# Print billing management system - support modules, version 4.1.2
#
# Copyright (C) 2000, 2001, 2002, 2003 Daniel Franklin
#
# This module is free software; you can redistribute it and/or modify it
# under the same terms as Perl itself.
#
# This module implements parsers to get useful stuff out of a configuration
# file. If the configuration file only contains a URL, we attempt to use
# wget to download the configuration file from the specified location.
#
##############################################################################

package Printbill::printbill_pcfg;

use POSIX;
use strict;
use Printbill::PTDB_File;

BEGIN {
	use Exporter ();
	
	@Printbill::printbill_pcfg::ISA = qw (Exporter);
	@Printbill::printbill_pcfg::EXPORT = qw (&pcfg &pcfg_printers);
	$Printbill::printbill_pcfg::VERSION = 4.1.2;
}


my $WGET = '/usr/bin/wget';

# Do a preliminary check on the config file - if it just redirects to a URL
# then suck down the config file using wget. Otherwise everything is local -
# return undefined.

sub check_remote_cfg
{
	my ($config) = @_;
	my ($newconfig, $line, $my_pid);
	
	if (defined ($config) && -r $config) {
		open (CONFIGFILE, $config) or do {
			print STDERR "$0 (printbill_pcfg::check_remote_cfg()): unable to open configuration file $config: $!\n";
			return undef;
		};
	} else {
		return undef;
	}
	
	$line = <CONFIGFILE>;
	chomp $line;
	
# Check if the first line is a URL.

	if ($line =~ /^(ht|f)tp:\/\//) {
# If so, suck the target down using wget
		if (-x $WGET) {
			`$WGET $line -Y off -O /tmp/printbillrc_$$ &> /dev/null`;

# Success? Send the new filename back.

			if ($? == 0) {
				return "/tmp/printbillrc_$$";
			} else {
				return undef;
			}
		} else {
			print " 
Sorry - couldn't find $WGET. Normally I'd put this sort of thing in a
config file, but this is the script used to read the config file and so the
path must be hard-coded (ye olde chicken and egg problem). To fix it so that
we can find wget, please edit the file printbill_pcfg.pm and change the line
at the top which says \$WGET = \"$WGET\" - or make a symlink.\n\n";
			
			return undef;
		}
	} else {
# Not remote.
		return undef;
	}
	
	close CONFIGFILE
		|| die "$0 (printbill_pcfg::check_remote_cfg()): unable to close config file $config: $!\n";
}

# Parse configuration file
#
# 1st arg: full path to configuration file
#
# 2nd arg: optional printer specification.

sub pcfg
{
	my ($config, $printer) = @_;

	if (!(defined $printer)) {
		$printer = '';
	}
	
# Remember, perl initialises variables to zero or ().

	my ($remote_config, %params, %confighash, %printerhash, $printerconfig);
	
	$remote_config = check_remote_cfg ($config);
	
	if (defined ($remote_config)) {
		tie %confighash, "Printbill::PTDB_File", $remote_config, "TRUE" or do {
			print STDERR "$0 (printbill_pcfg.pm): unable to open remotely-stored configuration file $remote_config: $!\n";
			return ();
		};
	} elsif (-r $config) {
		tie %confighash, "Printbill::PTDB_File", $config, "TRUE" or do {
			print STDERR "$0 (printbill_pcfg.pm): unable to open configuration file $config: $!\n";
			return ();
		};
	} else {
		return ();
	}
	
# At this point, we should have all default parameters of the configuration
# file loaded. Now we need to check the exceptions file, if it exists, for
# settings to override the configuration for our printer. These cannot at
# present be stored remotely.

	if (defined $printer && $printer ne "default" && $printer ne "") {
		$printerconfig = "/etc/printbill/printers/$printer";

		if (-f $printerconfig) {
			tie %printerhash, "Printbill::PTDB_File", $printerconfig, "TRUE" or do {
				print STDERR "$0 (printbill_pcfg.pm): unable to open printer-specific configuration file $printerconfig, although the file exists: $!\n";
				return ();
			};

# This allows any system-wide default to be over-ridden for a specific printer...
		
			foreach (keys %printerhash) {
				$confighash{$_} = $printerhash{$_};
			}
		
			untie %printerhash;
		}
	}
	
	%params = %confighash;

	untie %confighash;

	if (defined ($remote_config)) {
		`/bin/rm $remote_config`;
		
		if ($? >> 8) {
			print STDERR "$0 (printbill_pcfg::pcfg_printers): unable to remove remote config file $remote_config: $!\n";
		}
	}

	return %params;
}

# Get a list of available printers, and their configuration parameters.
#
# This should return a 2-dimensional hash, of the form
#
# %printers{'default'}{'param1'} = blah
# %printers{'someprinter'}{'param2'} = blah

# All configuration-file options can be over-ridden on a per-printer basis

sub pcfg_printers
{
	my ($config) = @_;

	my (%printers, %printerhash, $line, $subline, @strings, @pstr, $printer, @printer_files);
	
	%{$printers{'default'}} = pcfg ($config);
	
	if (! -d "/etc/printbill/printers") {
		return %printers;
	}
	
	opendir PRINTERS_DIR, "/etc/printbill/printers" or do {
		print STDERR "$0 (printbill_pcfg.pm): cannot open directory /etc/printbill/printers: $!\n";
		return %printers;
	};
	
	@printer_files = readdir PRINTERS_DIR;
	@printer_files = sort grep {! /^\./} @printer_files;
	
	closedir PRINTERS_DIR or do {
		print STDERR "$0 (printbill_pcfg.pm): cannot close directory /etc/printbill/printers: $!\n";
		return %printers;
	};
	
	foreach $printer (@printer_files) {
		%{$printers{$printer}} = %{$printers{'default'}};
		
		tie %printerhash, "Printbill::PTDB_File", "/etc/printbill/printers/$printer", "TRUE" or do {
			print STDERR "$0 (printbill_pcfg.pm): unable to open printer-specific configuration file /etc/printbill/printers/$printer, although the file exists: $!\n";
			return %printers;
		};

# This allows any system-wide default to be over-ridden for a specific printer...
		
		foreach (keys %printerhash) {
			$printers{$printer}{$_} = $printerhash{$_};
		}
		
		untie %printerhash;
	}
	
	return %printers;
}

END {}

1;
