/*
 * prismstumbler wireless lan analyzing tool
 *
 * Copyright (C) 2004 Florian Boor <florian.boor@kernelconcepts.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * preferences handling module
 *
 */
 
 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libintl.h>

#include "preferences.h"
 
 static const char* CFG_DEVICE = "device";
 static const char* CFG_SCAN = "scanactive";
 static const char* CFG_DRV = "driver"; 
 static const char* CFG_DELAY = "delay";
 static const char* CFG_CHANNEL = "channel";
 static const char* CFG_WPFILE = "waypointsfile";
 static const char* CFG_DUMPFILE = "packgedumpfile";
 static const char* CFG_DUMP = "dodump";
 static const char* CFG_DHCP = "dhcpcommand";

extern GtkWidget *fMain;

#define _(x) gettext(x)

extern psconfig_t cfg; /* this is current configuration, contains our defaults */


/* Returns file from file selection dialog. */
char *
ask_user_a_filename(void)
{
	char *result = NULL;
	GtkWidget *filesel;
	
	filesel = gtk_file_selection_new(_("Select configset"));
	
	if (gtk_dialog_run(GTK_DIALOG(filesel)) == GTK_RESPONSE_OK)
		result = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
	
	gtk_widget_destroy(filesel);
	
	return result;
}


static inline void 
replace_in_block(gchar*** lines, const char* var, const char* val)
{
	int i = 0;
	gboolean found = FALSE;
	if (*lines)
		while ((*lines)[i])
		{
			if ((g_strrstr (g_strchomp ((*lines)[i]), var))
				&& (!g_str_has_prefix (g_strchomp ((*lines)[i]), "#")))
			{
				/* small hack to avoid overwriting auto mode */
				if (!strcmp(var,CFG_DRV) && strstr((*lines)[i],"auto"))
					return;
				
				found = TRUE;
				g_free((*lines)[i]);
				(*lines)[i] = g_strdup_printf("%s = %s",var,val);
			}
			i++;
		}
	if (!found)
	{
		(*lines) = realloc ((*lines), (i + 2) * sizeof (gchar *));
		(*lines)[i] = g_strdup_printf("%s = %s",var,val);
		i++;
		(*lines)[i] = NULL;
	}
}
 

void 
preferences_save(psconfig_t *cfg, char* filename)
{
	gchar *content = NULL;
	gchar *tmpval;
	gchar **lines = NULL;
	gint length;
	FILE *fnew;
	gint j = 0;
	GError *err = NULL;
	gchar ival[9];

	tmpval = "";
	
	if (filename == NULL)
		filename = ask_user_a_filename();
	
	g_file_get_contents (filename, &content, &length, &err);
		
	if (content)
	{
		lines = g_strsplit (content, "\n", 2048);
		g_free (content);
	}
	
	replace_in_block(&lines,CFG_DEVICE,cfg->device);
	snprintf(ival,8,"%i",cfg->scan);
	replace_in_block(&lines,CFG_SCAN,ival);
	switch (cfg->devtype)
	{
		case DT_HOSTAP: 
			sprintf(ival,"hostap");
		break;
		case DT_PRISM: 
			sprintf(ival,"wlan-ng");
		break;
		case DT_ORINOCO: 
			sprintf(ival,"orinoco");
		break;
		case DT_SCAN:
			sprintf(ival,"scan");
		break;
		default: 
			sprintf(ival,"auto");
		break;	
	}
	replace_in_block(&lines,CFG_DRV,ival);
	snprintf(ival,8,"%i",cfg->delay);
	replace_in_block(&lines,CFG_DELAY,ival);
	snprintf(ival,8,"%i",cfg->singlechan);
	replace_in_block(&lines,CFG_CHANNEL,ival);
	replace_in_block(&lines,CFG_WPFILE,cfg->wpfile);
	replace_in_block(&lines,CFG_DUMPFILE,cfg->dumpfile);
	snprintf(ival,8,"%i",cfg->dumptofile);
	replace_in_block(&lines,CFG_DUMP,ival);
	snprintf(ival,1,"%i",cfg->dhcpcommand);
	replace_in_block(&lines,CFG_DHCP,ival);

	fnew = fopen (filename, "w");
	if (!fnew)
	{
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new (GTK_WINDOW (fMain),
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _("Could not write config file."));
		g_signal_connect_swapped (G_OBJECT (dialog), "response",
					  G_CALLBACK (gtk_widget_destroy),
					  G_OBJECT (dialog));
		gtk_widget_show (dialog);
		return;
	}

	j=0;
	while(lines[j])
	{
		fprintf (fnew, "%s\n", lines[j]);
		j++;
	}
	fclose (fnew);
	g_strfreev (lines);
}


psconfig_t 
preferences_load(char* filename)
{
	gchar *content, *tmpval;
	gchar **lines = NULL;
	gint i = 0;
	GError *err = NULL;
	psconfig_t result = cfg;
	int length;

	tmpval = "";
	
	if (filename == NULL)
		filename = ask_user_a_filename();
	if (!filename) return result;
	
	if (!g_file_get_contents (filename, &content, &length, &err))
	{
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new (GTK_WINDOW (fMain),
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _("Could not access config file."));
		g_signal_connect_swapped (G_OBJECT (dialog), "response",
					  G_CALLBACK (gtk_widget_destroy),
					  G_OBJECT (dialog));
		gtk_dialog_run (GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		return result;
	}
	if (content)
	{
		lines = g_strsplit (content, "\n", 2048);
		g_free (content);
	}
	while (lines[i])
	{
		if (!g_str_has_prefix (g_strchomp (lines[i]), "#"))
		{
 			if ((g_strrstr (g_strchomp (lines[i]), CFG_DEVICE)) 
					&& (strstr(lines[i],"=")))
					snprintf(result.device,6,"%s",
						g_strstrip (strstr(lines[i],"=")+1));
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_DRV)) 
					&& (strstr(lines[i],"=")))
			{
				if (strstr(lines[i],"hostap")) result.devtype = DT_HOSTAP;
				else if (strstr(lines[i],"wlan-ng")) result.devtype = DT_PRISM;
				else if (strstr(lines[i],"orinoco")) result.devtype = DT_ORINOCO;
				else if (strstr(lines[i],"scan")) result.devtype = DT_SCAN;
					else result.devtype = DT_NONE;
			}
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_SCAN)) 
					&& (strstr(lines[i],"=")))
					if (strstr(lines[i],"1"))
						result.scan = 1;
					else 
						result.scan = 0;
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_DELAY)) 
					&& (strstr(lines[i],"=")))
			{
				result.delay = atol(g_strchomp (strstr(lines[i],"=")+1));
				if ((result.delay < 1000) || (result.delay > 10000000))
					result.delay = 40000;
			}
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_CHANNEL)) 
					&& (strstr(lines[i],"=")))
			{
				result.singlechan = atol(g_strchomp (strstr(lines[i],"=")+1));
				if ((result.singlechan < 0) || (result.singlechan > 15))
					result.singlechan = 0;
			}
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_WPFILE)) 
					&& (strstr(lines[i],"=")))
					snprintf(result.wpfile,255,"%s",
						g_strstrip (strstr(lines[i],"=")+1));
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_DUMPFILE)) 
					&& (strstr(lines[i],"=")))
					snprintf(result.dumpfile,255,"%s",
						g_strstrip (strstr(lines[i],"=")+1));
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_DUMP)) 
					&& (strstr(lines[i],"=")))
				{
					if (strstr(lines[i],"1"))
						result.dumptofile = 1;
					else
						result.dumptofile = 0;
				}
			else if ((g_strrstr (g_strchomp (lines[i]), CFG_DHCP)) 
					&& (strstr(lines[i],"=")))
				{
					if (strstr(lines[i],"1"))
						result.dhcpcommand = 1;
					else
						result.dhcpcommand = 0;
				}
		}
		i++;
	}
	g_strfreev (lines);
	return result;
}
