/***************************************
  $Header: /home/amb/procmeter/RCS/procmeter.c 2.29 1999/03/06 14:42:03 amb Exp $

  ProcMeter - A performance metering/logging program for Linux - Version 2.5.1.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1994,95,96,97,98,99 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include "procmeter.h"


static void Usage(int verbose);
static void ParseOptions(int *argc,char** argv);
static void ParseMeters(int *argc,char** argv);

/*+ The name of the host that we are running on. +*/
char HostName[40];

/*+ The interval between updates for the graph (in seconds). +*/
static int interval=1;

#ifndef NO_X
/*+ The multiplier for the interval to log data to the log file. +*/
static int loginterval=10;

/*+ Set to true if there is only logging, no windows. +*/
static int logonly=0;
#endif

/*+ Set to true if only the help is to be printed out. +*/
static int help_only=0;

/*+ The datafiles (pipes) that we read from. +*/
static char **datfiles;

/*+ The name of the logfile if any. +*/
extern char *logfile;

#ifdef NO_X
/*+ A flag that is never set to true. +*/
int quit=0;
#else
/*+ A flag that is set to true when we are told to quit. +*/
extern int quit;

/*+ If the meters are aligned vertically. +*/
extern int vertical;
#endif


/*++++++++++++++++++++++++++++++++++++++
  The main program.

  int main Returns a value to the operating system.

  int argc The number of command line arguments.

  char** argv The actual command line arguments.
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc,char** argv)
{
 FILE *file;
 int count=0;
 time_t now,now2;

 datfiles=(char**)calloc(1+argc/2,sizeof(char*));
 HostName[0]=0;

 /* Parse the input */

 ParseOptions(&argc,argv);

 /* Initialise all of the data in the statistics file. */

 VerifyProcStatistics();

 if(datfiles)
   {
    int i=0;
    while(datfiles[i])
       VerifyOtherStatistics(datfiles[i++]);
    free(datfiles);
   }

 if(help_only)
    Usage(1);

 /* Get the hostname */

 if(HostName[0] == 0 && (file=fopen("/proc/sys/kernel/hostname","r")))
   {
    int n=fread(HostName,1,40,file);
    HostName[n-1]=0;
    fclose(file);
   }

 /* Start the output. */

#ifndef NO_X
 if(!logonly)
    StartX(&argc,argv);
 if(logfile)
#endif
    StartLog();

 /* Create the meters. */

 ParseMeters(&argc,argv);

 /* Run everything. */

 GetStatistics();

 now=time(NULL);

#ifndef NO_X
 if(logonly)
   {interval*=loginterval;loginterval=1;}
#endif

 do
   {
    int type;

   /* handle time leaps (rdate, xntpd, or laptop sleep/hybernation) */
   /* We choose to be somewhat lenient and decide that one occured if the
      expected time is more than 5 intervals ahead or behind the current time. */

    now2=time(NULL);            /* Should be now+interval if there is no jump. */

    if(now2>(now+6*interval) || now2<(now-4*interval))
       now=now2+interval;
    else
       now=now+interval;

    /* Wait for a while */

#ifndef NO_X
    if(!logonly)
       SleepX(now);
    else
#endif
       SleepNotX(now);

    /* Get the statistics */

    GetStatistics();

    for(type=0;type<NProcStats;type++)
       if(ProcStats[type].used)
          ProcStats[type].value=CurrentValue(type);

    for(type=1;type<=NOtherStats;type++)
       if(OtherStats[type].used)
          OtherStats[type].value=CurrentValue(-type);

#ifndef NO_X
    if(!logonly)
       UpdateMetersX();
    if(logfile && count%loginterval==0)
#endif
       UpdateMetersLog(now);

#ifndef NO_X
    /* Change the used status to 2 if currently 1. */

    for(type=0;type<NProcStats;type++)
       if(ProcStats[type].used&1)
         {
          ProcStats[type].used=2;
          if(logfile)
             AddRemoveMeterLog(type);
         }

    for(type=1;type<=NOtherStats;type++)
       if(OtherStats[type].used&1)
         {
          OtherStats[type].used=2;
          if(logfile)
             AddRemoveMeterLog(-type);
         }
#endif

    count++;
   }
 while(!quit);

 /* Stop the output. */

#ifndef NO_X
 if(!logonly)
    StopX();
#endif
 if(logfile)
    StopLog();

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Print the usage of the program.

  int verbose If true then print the long message.
  ++++++++++++++++++++++++++++++++++++++*/

static void Usage(int verbose)
{
 int i,c;

 if(verbose)
   {
    fprintf(stderr,"\nProcMeter Version 2.5.1\n\n");
    fprintf(stderr,"A performance metering/logging program for Linux using /proc.\n");
    fprintf(stderr,"(c) Andrew Bishop 1994,95,96,97,98,99. [amb@gedanken.demon.co.uk]\n\n");
   }

#ifdef NO_X
 fprintf(stderr,"Usage: ProcMeter [-h] [-update u]\n");
 fprintf(stderr,"                 [-log (<filename>|-)]\n");
#else
 fprintf(stderr,"Usage: ProcMeter [-h] [-update u [lu]]\n");
 fprintf(stderr,"                 [-horizontal|-vertical]\n");
 fprintf(stderr,"                 [(-log|-logonly) (<filename>|syslog|-)]\n");
#endif
 fprintf(stderr,"                 [-label <label>]\n");
 fprintf(stderr,"                 [-data <filename>]\n");
 fprintf(stderr,"                 [-force-spkt] [-force-fpkt]\n");
 fprintf(stderr,"                 [statistics-option-1] [statistics-option-2] ...\n\n");

 fprintf(stderr,"Statistic options");
 for(c=0,i=0;i<NProcStats;i++)
    if(ProcStats[i].avail)
      {
       if(!c)
          fprintf(stderr,"\nFrom /proc files -");
       if(c && !(c%5))
          fprintf(stderr,"\n                  ");
       fprintf(stderr," %-11s",ProcStats[i].name);
       c++;
      }
 for(c=0,i=1;i<=NOtherStats;i++)
    if(OtherStats[i].avail)
      {
       if(!c)
          fprintf(stderr,"\nFrom other files -");
       if(c && !(c%5))
          fprintf(stderr,"\n                  ");
       fprintf(stderr," %-11s",OtherStats[i].name);
       c++;
      }
 fprintf(stderr,"\n\n");

 if(verbose)
   {
    fprintf(stderr," -h                    : Display this help.\n");
#ifdef NO_X
    fprintf(stderr," -update u             : The time between updates in seconds (default 1).\n");
#else
    fprintf(stderr," -update u [lu]        : The time between updates in seconds (default 1).\n"
                   "                         The 2nd arg is a multiplier for log updates (def 10).\n\n");
    fprintf(stderr," -horizontal           : Display the meters horizontally.\n");
    fprintf(stderr," -vertical             : Display the meters vertically (default).\n\n");
#endif
    fprintf(stderr," -log <filename>       : log the data to the specified file.\n");
#ifndef NO_X
    fprintf(stderr," -logonly <filename>   : Like '-log' except there is no graphical display.\n");
#endif
    fprintf(stderr,"                         If 'syslog' is used then log to the system log.\n");
    fprintf(stderr,"                         If '-' is used then log to stdout.\n\n");
    fprintf(stderr," -label <label>        : Use the specified label instead of the hostname.\n\n");
    fprintf(stderr," -data <filename>      : Specifies a named pipe to read statistics from.\n\n");
    fprintf(stderr," -force-spkt           : Force the slow packet devices to be recognised.\n");
    fprintf(stderr," -force-fpkt           : Force the fast packet devices to be recognised.\n\n");
   }

 exit(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Parses the input and prints a usage message if required.

  int *argc The arguments.

  char** argv The arguments.
  ++++++++++++++++++++++++++++++++++++++*/

static void ParseOptions(int *argc,char** argv)
{
 int i,j;

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-h"))
      {
       help_only=1;
       for(i=j+1;i<*argc;i++)
          argv[i-1]=argv[i];
       *argc-=1; j--;
      }

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-update"))
      {
       int k=2;
       if(j==*argc-1)
         {fprintf(stderr,"ProcMeter: The -update option requires a time value\n");Usage(0);}
       interval=atoi(argv[j+1]);
       if(interval<=0 || interval>60)
         {fprintf(stderr,"ProcMeter: The -update option requires a sensible time value\n");Usage(0);}
# ifndef NO_X
       if(j<*argc-2)
         {
          char buf[10];
          sprintf(buf,"%d",atoi(argv[j+2]));
          if(!strcmp(argv[j+2],buf))
            {loginterval=atoi(argv[j+2]);k=3;}
         }
#endif
       for(i=j+k;i<*argc;i++)
          argv[i-k]=argv[i];
       *argc-=k; j--;
      }

#ifndef NO_X
 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-horizontal"))
      {
       vertical=0;
       for(i=j+1;i<*argc;i++)
          argv[i-1]=argv[i];
       *argc-=1; j--;
      }

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-vertical"))
      {
       vertical=1;
       for(i=j+1;i<*argc;i++)
          argv[i-1]=argv[i];
       *argc-=1; j--;
      }
#endif

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-log") || !strcmp(argv[j],"-logonly"))
      {
       if(logfile)
         {fprintf(stderr,"ProcMeter: Only one -log or -logonly option allowed\n");Usage(0);}
       if(j==*argc-1)
         {fprintf(stderr,"ProcMeter: The -log option requires a filename\n");Usage(0);}
       logfile=argv[j+1];
#ifndef NO_X
       logonly=!strcmp(argv[j],"-logonly");
#endif
       for(i=j+2;i<*argc;i++)
          argv[i-2]=argv[i];
       *argc-=2; j--;
      }

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-data"))
      {
       i=0;
       if(j==*argc-1)
         {fprintf(stderr,"ProcMeter: The -data option requires a filename\n");Usage(0);}
       while(datfiles[i])
          i++;
       datfiles[i]=argv[j+1];
       for(i=j+2;i<*argc;i++)
          argv[i-2]=argv[i];
       *argc-=2; j--;
      }

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-label"))
      {
       if(j==*argc-1)
         {fprintf(stderr,"ProcMeter: The -label option requires a label\n");Usage(0);}

       strncpy(HostName,argv[j+1],sizeof(HostName));
       HostName[sizeof(HostName)-1]=0;
       for(i=j+2;i<*argc;i++)
          argv[i-2]=argv[i];
       *argc-=2; j--;
      }

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-force-spkt"))
      {
       ProcStats[SPKT].avail=ProcStats[SPKT_RX].avail=ProcStats[SPKT_TX].avail=1;
       for(i=j+1;i<*argc;i++)
          argv[i-1]=argv[i];
       *argc-=1; j--;
      }

 for(j=1;j<*argc;j++)
    if(!strcmp(argv[j],"-force-fpkt"))
      {
       ProcStats[FPKT].avail=ProcStats[FPKT_RX].avail=ProcStats[FPKT_TX].avail=1;
       for(i=j+1;i<*argc;i++)
          argv[i-1]=argv[i];
       *argc-=1; j--;
      }
}


/*++++++++++++++++++++++++++++++++++++++
  Start of all the meters specified on the command line.

  int *argc The number of command line arguments.

  char** argv The actual command line arguments.
  ++++++++++++++++++++++++++++++++++++++*/

static void ParseMeters(int *argc,char** argv)
{
 int error=0;
 int j;
 int i;

 /* Create the meter widgets */

 for(j=1;j<*argc;j++)
   {
    for(i=0;i<NProcStats;i++)
       if(!strcmp(argv[j],ProcStats[i].name))
         {
          if(ProcStats[i].avail)
            {
             ProcStats[i].used=1;
#ifndef NO_X
             if(!logonly)
                AddRemoveMeterX(i);
#endif
            }
          else
            fprintf(stderr,"ProcMeter: Option %s is unavailable\n",argv[j]);
          break;
         }
    if(i!=NProcStats)
       continue;

    for(i=1;i<=NOtherStats;i++)
       if(!strcmp(argv[j],OtherStats[i].name))
         {
          if(OtherStats[i].avail)
            {
             OtherStats[i].used=1;
#ifndef NO_X
             if(!logonly)
                AddRemoveMeterX(-i);
#endif
            }
          else
             fprintf(stderr,"ProcMeter: Option %s is unavailable\n",argv[j]);
          break;
         }

    if(i<=NOtherStats)
       continue;

    error=fprintf(stderr,"ProcMeter: Unrecognized option %s\n",argv[j]);
   }

 if(error)
    Usage(0);
}
