/*
 * ProGuard -- shrinking, optimization, obfuscation, and preverification
 *             of Java bytecode.
 *
 * Copyright (c) 2002-2008 Eric Lafortune (eric@graphics.cornell.edu)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package proguard.classfile.editor;

import proguard.classfile.*;
import proguard.classfile.attribute.*;
import proguard.classfile.instruction.visitor.InstructionVisitor;
import proguard.classfile.instruction.*;
import proguard.classfile.util.SimplifiedVisitor;

/**
 * This InstructionVisitor adds all instructions that it visits to the given
 * target code attribute.
 *
 * @author Eric Lafortune
 */
public class InstructionAdder
extends      SimplifiedVisitor
implements   InstructionVisitor
{
    private final ConstantAdder         constantAdder;
    private final CodeAttributeComposer codeAttributeComposer;


    /**
     * Creates a new InstructionAdder that will copy classes into the given
     * target code attribute.
     */
    public InstructionAdder(ProgramClass          targetClass,
                            CodeAttributeComposer targetComposer)
    {
        constantAdder         = new ConstantAdder(targetClass);
        codeAttributeComposer = targetComposer;
    }


    // Implementations for InstructionVisitor.


    public void visitAnyInstruction(Clazz clazz, Method method, CodeAttribute codeAttribute, int offset, Instruction instruction)
    {
        // Add the instruction.
        codeAttributeComposer.appendInstruction(offset, instruction);
    }


    public void visitConstantInstruction(Clazz clazz, Method method, CodeAttribute codeAttribute, int offset, ConstantInstruction constantInstruction)
    {
        // Create a copy of the instruction.
        Instruction newConstantInstruction =
            new ConstantInstruction(constantInstruction.opcode,
                                    constantAdder.addConstant(clazz, constantInstruction.constantIndex),
                                    constantInstruction.constant).shrink();

        // Add the instruction.
        codeAttributeComposer.appendInstruction(offset, newConstantInstruction);
    }
}