/*
    BreakpointsStore.m

    Implementation of the BreakpointsStore class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "BreakpointsStore.h"

#import <Foundation/NSArray.h>
#import <Foundation/NSDictionary.h>
#import <Foundation/NSString.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSValue.h>

NSString * const BreakpointsDidChangeNotification
  = @"BreakpointsDidChangeNotification";

@implementation BreakpointsStore

static BreakpointsStore * shared = nil;

+ shared
{
  if (shared == nil)
    {
      shared = [self new];
    }

  return shared;
}

- (void) dealloc
{
  TEST_RELEASE(breakpoints);

  [super dealloc];
}

- init
{
  if ([super init])
    {
      breakpoints = [NSMutableArray new];

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) addBreakpointInFile: (NSString *) filename
                        line: (unsigned int) lineNumber
{
  [breakpoints addObject: [NSDictionary dictionaryWithObjectsAndKeys:
    filename, @"Filename",
    [NSNumber numberWithUnsignedInt: lineNumber], @"Line",
    nil]];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: BreakpointsDidChangeNotification
                  object: self];
}

- (void) addBreakpointOnSymbol: (NSString *) symbol
{
  [breakpoints addObject: [NSDictionary dictionaryWithObject: symbol
                                                      forKey: @"Symbol"]];

  [[NSNotificationCenter defaultCenter]
    postNotificationName: BreakpointsDidChangeNotification
                  object: self];
}

- (void) removeBreakpointInFile: (NSString *) filename
                           line: (unsigned int) lineNumber
{
  NSNumber * line = [NSNumber numberWithUnsignedInt: lineNumber];
  int i, n;

  for (i = 0, n = [breakpoints count]; i < n; i++)
    {
      NSDictionary * breakpoint = [breakpoints objectAtIndex: i];

      if ([[breakpoint objectForKey: @"Filename"] isEqualToString: filename] &&
          [[breakpoint objectForKey: @"Line"] isEqualToNumber: line])
        {
          [breakpoints removeObjectAtIndex: i];

          [[NSNotificationCenter defaultCenter]
            postNotificationName: BreakpointsDidChangeNotification
                          object: self];

          break;
        }
    }
}

- (void) removeBreakpointOnSymbol: (NSString *) symbol
{
  int i, n;

  for (i = 0, n = [breakpoints count]; i < n; i++)
    {
      NSDictionary * breakpoint = [breakpoints objectAtIndex: i];

      if ([[breakpoint objectForKey: @"Symbol"] isEqualToString: symbol])
        {
          [breakpoints removeObjectAtIndex: i];

          [[NSNotificationCenter defaultCenter]
            postNotificationName: BreakpointsDidChangeNotification
                          object: self];

          break;
        }
    }
}

- (NSArray *) breakpoints
{
  return [[breakpoints copy] autorelease];
}

- (NSArray *) breakpointsInFile: (NSString *) filename
{
  NSMutableArray * array = [NSMutableArray array];
  NSEnumerator * e;
  NSDictionary * breakpoint;

  e = [breakpoints objectEnumerator];
  while ((breakpoint = [e nextObject]) != nil)
    {
      if ([[breakpoint objectForKey: @"Filename"] isEqualToString: filename])
        {
          [array addObject: breakpoint];
        }
    }

  return [[array copy] autorelease];
}

@end
