/*
    Builder.m

    Implementation of the Builder class for the ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "Builder.h"

#import <Foundation/NSArray.h>
#import <Foundation/NSTask.h>
#import <Foundation/NSString.h>
#import <Foundation/NSFileHandle.h>
#import <Foundation/NSUserDefaults.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSException.h>

#import <AppKit/NSView.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTableView.h>
#import <AppKit/NSTableColumn.h>
#import <AppKit/NSPopUpButton.h>
#import <AppKit/NSScrollView.h>
#import <AppKit/NSScroller.h>
#import <AppKit/NSButton.h>
#import <AppKit/NSPanel.h>
#import <AppKit/NSMatrix.h>
#import <AppKit/NSDocumentController.h>
#import <AppKit/NSCell.h>
#import <AppKit/NSImage.h>

#import "ProjectDocument.h"
#import "ProjectType.h"
#import "SourceEditorDocument.h"

static NSDictionary *
ErrorInfoFromString (NSString * string)
{
  NSArray * components;

  // try splitting the line at ':' boundaries. There should be at least 4
  // components, structured like this:
  //
  // <file>:<line-in-file>:<message-type>:<error-description>
  components = [string componentsSeparatedByString: @":"];
  if ([components count] >= 4 &&
    [[[components objectAtIndex: 1] stringByTrimmingCharactersInSet:
    [NSCharacterSet decimalDigitCharacterSet]] length] == 0)
    {
      NSString * file, * line, * type, * description;
      unsigned int headerLength;
      NSColor * color;
      NSUserDefaults * df = [NSUserDefaults standardUserDefaults];

      file = [components objectAtIndex: 0];
      line = [components objectAtIndex: 1];
      type = [[components objectAtIndex: 2] stringByTrimmingSpaces];

      // the rest is the error description
      headerLength = [file length] + [line length] + [[components
        objectAtIndex: 2] length] + 3;
      description = [[string substringFromIndex: headerLength]
        stringByTrimmingSpaces];

      // if the type is known (is one of "error" or "warning"),
      // then colorize the description. Otherwise leave it alone.
      if ([type caseInsensitiveCompare: @"error"] == NSOrderedSame)
        {
          color = [df objectForKey: @"ErrorColor"];
          if (color == nil)
            color = [NSColor redColor];

          description = [NSString stringWithFormat: _(@"Error: %@"),
            description];
        }
      else if ([type caseInsensitiveCompare: @"warning"] == NSOrderedSame)
        {
          color = [df objectForKey: @"WarningColor"];
          if (color == nil)
            color = [NSColor yellowColor];

          description = [NSString stringWithFormat: _(@"Warning: %@"),
            description];
        }
      else
        {
          color = nil;
        }

      if (color != nil)
        {
          NSDictionary * attributes = [NSDictionary
            dictionaryWithObject: color
                          forKey: NSBackgroundColorAttributeName];

          return [NSDictionary dictionaryWithObjectsAndKeys:
            [[[NSAttributedString alloc]
              initWithString: file attributes: attributes]
              autorelease], @"File",
            [[[NSAttributedString alloc]
              initWithString: line attributes: attributes]
              autorelease], @"Line",
            [[[NSAttributedString alloc]
              initWithString: description attributes: attributes]
              autorelease], @"Description",
            nil];
        }
      else
        {
          return [NSDictionary dictionaryWithObjectsAndKeys:
            file, @"File",
            line, @"Line",
            description, @"Description",
            nil];
        }
    }
  else
    // otherwise mark this as an unknown error
    {
      return [NSDictionary dictionaryWithObject: string
                                         forKey: @"Description"];
    }
}

@implementation Builder

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  TEST_RELEASE(buildArguments);
  TEST_RELEASE(buildErrorList);
  TEST_RELEASE(lastIncompleteErrorLine);

  if (task != nil && [task isRunning])
    {
      [task interrupt];
    }
  TEST_RELEASE(task);

  [super dealloc];
}

- initWithDocument: (ProjectDocument *) aDocument
{
  if ([super initWithDocument: aDocument])
    {
      buildArguments = [NSMutableArray new];
      buildErrorList = [NSMutableArray new];
      lastIncompleteErrorLine = [NSString new];

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) awakeFromNib
{
  NSRect r;

  [super awakeFromNib];

  [buildErrors setTarget: self];
  [buildErrors setDoubleAction: @selector(openErrorFile:)];

  [buildArgs removeRow: 0];
  [buildArgs sizeToCells];
  r = [buildArgs frame];
  r.size.width = [[buildArgs superview] frame].size.width;
  [buildArgs setFrame: r];
  [buildArgs setAutoscroll: YES];
  [[[buildArgs enclosingScrollView] verticalScroller]
    setArrowsPosition: NSScrollerArrowsNone];

  [buildButton setImage: [NSImage imageNamed: @"Tab_Build"]];
}


- (void) build: (id)sender
{
  NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];

  if (stopOperation == YES)
    {
      stopOperation = NO;

      [nc removeObserver: self];
      [task interrupt];
      DESTROY(task);

      [cleanButton setEnabled: YES];
      [buildButton setTitle: _(@"Build")];
      [buildButton setImage: [NSImage imageNamed: @"Tab_Build"]];
    }
  else
    {
      NSString * makeCommand;
      NSMutableArray * arguments = [NSMutableArray array];

      stopOperation = YES;

      if ([[document projectType] prepareForBuild] == NO)
        {
          return;
        }

      // construct the build task
      task = [NSTask new];

       // set the build task's launch path
      makeCommand = [[NSUserDefaults standardUserDefaults]
        objectForKey: @"MakeCommand"];
      if (makeCommand == nil)
        {
          makeCommand = @"make";
        }
      [task setLaunchPath: makeCommand];
      [task setLaunchPath: [task validatedLaunchPath]];

       // build in the project path
      [task setCurrentDirectoryPath: [document projectDirectory]];

       // construct the build task's arguments
      switch ([buildTarget indexOfSelectedItem])
        {
        // Default target
        case 0:
          break;
        // Debug target
        case 1:
          [arguments addObject: @"debug=yes"];
          break;
        // Profile target
        case 2:
          [arguments addObject: @"profile=yes"];
          break;
        // Install target
        case 3:
          [arguments addObject: @"install"];
          break;
        }

      if ([verboseBuild state])
        {
          [arguments addObject: @"messages=yes"];
        }

      if ([warnings state])
        {
          [arguments addObject: @"warnings=yes"];
        }
      if ([allWarnings state])
        {
          [arguments addObject: @"allwarnings=yes"];
        }

      [arguments addObjectsFromArray: buildArguments];

      [task setArguments: arguments];

      [nc addObserver: self
             selector: @selector(buildCompleted:)
                 name: NSTaskDidTerminateNotification
               object: task];

       // set up the stdout and stderr pipes
      {
        NSPipe * outputPipe = [NSPipe pipe],
               * errorPipe = [NSPipe pipe];

        [task setStandardOutput: outputPipe];
        [task setStandardError: errorPipe];

        outputFileHandle = [outputPipe fileHandleForReading];
        errorFileHandle = [errorPipe fileHandleForReading];

        [nc addObserver: self
               selector: @selector(collectOutput:)
                   name: NSFileHandleDataAvailableNotification
                 object: outputFileHandle];
        [nc addObserver: self
               selector: @selector(collectErrorOutput:)
                   name: NSFileHandleDataAvailableNotification
                 object: errorFileHandle];
      }

      // clean the output lists
      [buildErrorList removeAllObjects];
      [buildErrors reloadData];
      [buildOutput setString: @""];

      [cleanButton setEnabled: NO];
      [buildButton setTitle: _(@"Stop")];
      [buildButton setImage: [NSImage imageNamed: @"Stop"]];

      // and start the build
      NS_DURING
        [task launch];
      NS_HANDLER
        NSRunAlertPanel(_(@"Failed to build"),
          _(@"Failed to run the build command \"%@\""),
          nil, nil, nil, makeCommand);

        return;
      NS_ENDHANDLER

      [outputFileHandle waitForDataInBackgroundAndNotify];
      [errorFileHandle waitForDataInBackgroundAndNotify];
    }
}


- (void) clean: (id)sender
{
  NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];

  if (stopOperation == YES)
    {
      stopOperation = NO;

      [nc removeObserver: self];
      [task interrupt];
      DESTROY(task);

      [buildButton setEnabled: YES];
      [cleanButton setTitle: _(@"Clean")];
      [cleanButton setImage: [NSImage imageNamed: @"Clean"]];
    }
  else
    {
      NSString * makeCommand;
      NSMutableArray * arguments = [NSMutableArray array];

      stopOperation = YES;

      if ([[document projectType] prepareForBuild] == NO)
        {
          return;
        }

      // construct the clean task
      task = [NSTask new];

       // set the clean task's launch path
      makeCommand = [[NSUserDefaults standardUserDefaults]
        objectForKey: @"MakeCommand"];
      if (makeCommand == nil)
        {
          makeCommand = @"make";
        }
      [task setLaunchPath: makeCommand];
      [task setLaunchPath: [task validatedLaunchPath]];

       // build in the project path
      [task setCurrentDirectoryPath: [document projectDirectory]];

      if ([distclean state])
        {
          [arguments addObject: @"distclean"];
        }
      else
        {
          [arguments addObject: @"clean"];
        }

      if ([verboseBuild state])
        {
          [arguments addObject: @"messages=yes"];
        }

      [task setArguments: arguments];

      [nc addObserver: self
             selector: @selector(cleanCompleted:)
                 name: NSTaskDidTerminateNotification
               object: task];

       // set up the stdout and stderr pipes
      {
        NSPipe * outputPipe = [NSPipe pipe],
               * errorPipe = [NSPipe pipe];

        [task setStandardOutput: outputPipe];
        [task setStandardError: errorPipe];

        outputFileHandle = [outputPipe fileHandleForReading];
        errorFileHandle = [errorPipe fileHandleForReading];

        [nc addObserver: self
               selector: @selector(collectOutput:)
                   name: NSFileHandleDataAvailableNotification
                 object: outputFileHandle];
        [nc addObserver: self
               selector: @selector(collectErrorOutput:)
                   name: NSFileHandleDataAvailableNotification
                 object: errorFileHandle];
      }

      // clean the output lists
      [buildErrorList removeAllObjects];
      [buildErrors reloadData];
      [buildOutput setString: @""];

      [buildButton setEnabled: NO];
      [cleanButton setTitle: _(@"Stop")];
      [cleanButton setImage: [NSImage imageNamed: @"Stop"]];

      // and start the build
      NS_DURING
        [task launch];
      NS_HANDLER
        NSRunAlertPanel(_(@"Failed to clean"),
          _(@"Failed to run the clean command \"%@\""),
          nil, nil, nil, makeCommand);

        return;
      NS_ENDHANDLER

      [outputFileHandle waitForDataInBackgroundAndNotify];
      [errorFileHandle waitForDataInBackgroundAndNotify];
    }
}

- (void) addBuildArgument: (id)sender
{
  NSString * arg = [buildArgsField stringValue];

  [buildArguments addObject: arg];
  [buildArgs addRow];
  [[buildArgs cellAtRow: [buildArgs numberOfRows]-1 column: 0] setTitle: arg];
  [buildArgs sizeToCells];
}


- (void) removeBuildArgument: (id)sender
{
  int selected;

  selected = [buildArgs selectedRow];
  if (selected >= 0)
    {
      [buildArguments removeObjectAtIndex: selected];
      [buildArgs removeRow: selected];

      if ([buildArgs numberOfRows] > 0)
        {
          if (selected < [buildArgs numberOfRows])
            {
              [buildArgs selectCellAtRow: selected column: 0];
            }
          else
            {
              [buildArgs selectCellAtRow: selected - 1 column: 0];
            }
        }
    }
  [buildArgs sizeToCells];
}

- (void) moveBuildArgumentUp: sender
{
  int n = [buildArgs selectedRow];

  if (n > 0)
    {
      [buildArguments insertObject: [buildArguments objectAtIndex: n]
                           atIndex: n - 1];
      [buildArguments removeObjectAtIndex: n + 1];
      [[buildArgs cellAtRow: n - 1 column: 0] setTitle:
        [buildArguments objectAtIndex: n - 1]];
      [[buildArgs cellAtRow: n column: 0] setTitle:
        [buildArguments objectAtIndex: n]];
      [buildArgs selectCellAtRow: n - 1 column: 0];
    }
}

- (void) moveBuildArgumentDown: sender
{
  int n = [buildArgs selectedRow];

  if (n >= 0 && n < (int) [buildArguments count] - 1)
    {
      NSString * oldArg;

      oldArg = [[buildArguments objectAtIndex: n + 1] retain];
      [buildArguments replaceObjectAtIndex: n + 1
                                withObject: [buildArguments objectAtIndex: n]];
      [buildArguments replaceObjectAtIndex: n withObject: oldArg];
      DESTROY(oldArg);

      [[buildArgs cellAtRow: n + 1 column: 0] setTitle:
        [buildArguments objectAtIndex: n + 1]];
      [[buildArgs cellAtRow: n column: 0] setTitle:
        [buildArguments objectAtIndex: n]];
      [buildArgs selectCellAtRow: n + 1 column: 0];
    }
}

- (void) openErrorFile: sender
{
  id filename;
  int clickedRow = [buildErrors clickedRow];
  unsigned int line;
  id lineObject;
  NSDictionary * description;
  SourceEditorDocument * editorDocument;
  NSDocumentController * dc = [NSDocumentController sharedDocumentController];

  if (clickedRow < 0)
    return;

  // extract the filename and line
  description = [buildErrorList objectAtIndex: clickedRow];
  filename = [description objectForKey: @"File"];
  if ([filename isKindOfClass: [NSAttributedString class]])
    filename = [filename string];

  if ([filename length] == 0)
    {
      NSRunAlertPanel(_(@"No file to open"),
        _(@"This error/warning message doesn't have a file "
          @"associated with it."),
        nil, nil, nil);
      return;
    }

  // FIXME: this needs to take into account that files may live
  // in different directories
  if (![filename isAbsolutePath])
    {
      filename = [[document projectDirectory]
        stringByAppendingPathComponent: filename];
    }
  lineObject = [description objectForKey: @"Line"];
  if ([lineObject isKindOfClass: [NSString class]])
    {
      line = [lineObject intValue];
    }
  else
    {
      line = [[lineObject string] intValue];
    }

  // get the document and tell it the line to view
  (editorDocument = [dc documentForFileName: filename]) ||
    (editorDocument = [dc openDocumentWithContentsOfFile: filename
                                                 display: YES]);
  if (editorDocument == nil)
    {
      NSRunAlertPanel(_(@"File not found"),
        _(@"Couldn't open the specified file."),
        nil, nil, nil);
    }
  else
    {
      [editorDocument goToLineNumber: line];
      [editorDocument showWindows];
    }
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [buildErrorList count];
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  return [[buildErrorList objectAtIndex: rowIndex]
    objectForKey: [aTableColumn identifier]];
}

- (void) collectOutput: (NSNotification *) notif
{
  NSString * string;

  string = [[[NSString alloc]
    initWithData: [outputFileHandle availableData]
        encoding: NSUTF8StringEncoding]
    autorelease];
  [buildOutput replaceCharactersInRange:
    NSMakeRange([[buildOutput string] length], 0)
                             withString: string];
  [buildOutput scrollRangeToVisible: 
    NSMakeRange([[buildOutput string] length], 0)];

  [outputFileHandle waitForDataInBackgroundAndNotify];
}

- (void) collectErrorOutput: (NSNotification *) notif
{
  NSMutableArray * incommingLines;

  incommingLines = [[[[[[NSString alloc]
    initWithData: [errorFileHandle availableData]
        encoding: NSUTF8StringEncoding]
    autorelease]
    componentsSeparatedByString: @"\n"]
    mutableCopy]
    autorelease];

  // complete the lastIncompleteErrorLine from the first line of error output
  if ([incommingLines count] > 0)
    {
      ASSIGN(lastIncompleteErrorLine, [lastIncompleteErrorLine
        stringByAppendingString: [incommingLines objectAtIndex: 0]]);

      [incommingLines removeObjectAtIndex: 0];
    }
  // now if there are still more lines to process this means that
  // the lastIncompleteLine has been completed and start processing
  if ([incommingLines count] > 0)
    {
      NSEnumerator * e;
      NSString * line;
      NSMutableArray * linesToProcess = [NSMutableArray
        arrayWithCapacity: [incommingLines count]];

      [linesToProcess addObject: lastIncompleteErrorLine];
      // the new last incomplete error line will the last line of error output
      ASSIGN(lastIncompleteErrorLine, [incommingLines lastObject]);
      [incommingLines removeLastObject];

      [linesToProcess addObjectsFromArray: incommingLines];

      // now process the completed lines
      e = [linesToProcess objectEnumerator];
      while ((line = [e nextObject]) != nil)
        {
          [buildErrorList addObject: ErrorInfoFromString(line)];
        }

      // and finally tell the error table to update itself
      [buildErrors reloadData];
      [buildErrors scrollPoint: NSMakePoint(0,
        [buildErrors frame].size.height)];
    }

  [errorFileHandle waitForDataInBackgroundAndNotify];
}

- (void) buildCompleted: (NSNotification *) notif
{
  NSString * result;

  [[NSNotificationCenter defaultCenter] removeObserver: self];

  stopOperation = NO;
  [buildButton setTitle: _(@"Build")];
  [buildButton setImage: [NSImage imageNamed: @"Tab_Build"]];
  [cleanButton setEnabled: YES];

  if ([task terminationStatus] == 0)
    {
      result = _(@" *** BUILD SUCCEEDED ***");
    }
  else
    {
      result = _(@" *** BUILD FAILED ***");
    }

  [buildOutput replaceCharactersInRange:
    NSMakeRange([[buildOutput string] length], 0)
                             withString: result];

  DESTROY(task);
}

- (void) cleanCompleted: (NSNotification *) notif
{
  NSString * result;

  [[NSNotificationCenter defaultCenter] removeObserver: self];

  stopOperation = NO;
  [cleanButton setTitle: _(@"Clean")];
  [cleanButton setImage: [NSImage imageNamed: @"Clean"]];
  [buildButton setEnabled: YES];

  if ([task terminationStatus] == 0)
    {
      result = _(@" *** CLEAN SUCCEEDED ***");
    }
  else
    {
      result = _(@" *** CLEAN FAILED ***");
    }

  [buildOutput replaceCharactersInRange:
    NSMakeRange([[buildOutput string] length], 0)
                             withString: result];

  DESTROY(task);
}

@end
