/*
    Controller.m

    Implementation of the ArgumentsInspector class for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "Controller.h"

#import <Foundation/NSBundle.h>
#import <Foundation/NSSortDescriptor.h>
#import <Foundation/NSString.h>
#import <Foundation/NSFileManager.h>
#import <Foundation/NSUserDefaults.h>
#import <Foundation/NSNotification.h>

#import <AppKit/NSOpenPanel.h>
#import <AppKit/NSView.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTextField.h>
#import <AppKit/NSDocumentController.h>
#import <AppKit/NSMenu.h>
#import <AppKit/NSMenuItem.h>
#import <AppKit/NSApplication.h>

#import "Preferences.h"
#import "ProjectCreation.h"
#import "ProjectModule.h"
#import "ProjectNameChecking.h"
#import "ProjectType.h"
#import "ProjectTypeLoader.h"
#import "ProjectWindowController.h"
#import "ProjectWizzard.h"
#import "TextFinder.h"
#import "NewFileTypeChooser.h"

@interface Controller (Private)

- (void) syncPipesMenu;

@end


@implementation Controller

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  [super dealloc];
}

- (void) awakeFromNib
{
  pipesMenu = [pipesMenu menu];
}

- (void) showPrefsPanel: (id)sender
{
  [[Preferences shared] activate];
}

- (void) newProject: sender
{
  NSString * projectPath;
  NSString * projectName;
  NSString * templatePath;
  NSDictionary * projectInfo;

  projectInfo = GetNewProjectSetup(YES);
  if (projectInfo == nil)
    {
      return;
    }

  projectPath = [projectInfo objectForKey: @"ProjectPath"];
  projectName = [projectInfo objectForKey: @"ProjectName"];
  templatePath = [projectInfo objectForKey: @"ProjectTemplate"];
  if (CreateNewProject(projectPath, projectName, templatePath) == NO)
    {
      NSRunAlertPanel(_(@"Failed to create project"),
        _(@"I couldn't create a new project at the specified path."),
        nil, nil, nil);

      return;
    }

  [[NSDocumentController sharedDocumentController]
    openDocumentWithContentsOfFile: [projectPath
    stringByAppendingPathComponent: [[projectInfo objectForKey: @"ProjectName"]
    stringByAppendingPathExtension: @"pmproj"]]
                           display: YES];
}

- (void) newSourceFile: sender
{
  NewFileTypeChooser * chooser = [NewFileTypeChooser shared];
  NSString * type;

  type = [chooser runModalWithRequiredKey: @"NSDocumentClass"
                                    value: @"SourceEditorDocument"];
  if (type != nil)
    {
      [[NSDocumentController sharedDocumentController]
        openUntitledDocumentOfType: type display: YES];
    }
}

- (void) applicationDidFinishLaunching: (NSNotification *) notif
{
  NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];

  [self syncPipesMenu];
  [nc addObserver: self
         selector: @selector(syncPipesMenu)
             name: NSUserDefaultsDidChangeNotification
           object: [NSUserDefaults standardUserDefaults]];

  /* Watches for main window changes and posts
     CurrentProjectModuleDidChangeNotification's as appropriate, so that
     inspectors can gracefully following main window changes */
  [nc addObserver: self
         selector: @selector(mainWindowChanged:)
             name: NSWindowDidBecomeMainNotification
           object: nil];
}

- (void) orderFrontFindPanel: sender
{
  [[TextFinder sharedInstance] orderFrontFindPanel: self];
}

- (void) mainWindowChanged: (NSNotification *) notif
{
  ProjectWindowController * pwc;
  NSNotificationCenter * nc = [NSNotificationCenter defaultCenter];

  pwc = [[NSApp mainWindow] windowController];
  if ([pwc respondsToSelector: @selector(currentModule)])
    {
      [nc postNotificationName: CurrentProjectModuleDidChangeNotification
                        object: self
                      userInfo: [NSDictionary
        dictionaryWithObject: [pwc currentModule] forKey: @"Module"]];
    }
  else
    {
      [nc postNotificationName: CurrentProjectModuleDidChangeNotification
                        object: self
                      userInfo: nil];
    }
}

@end

@implementation Controller (Private)

- (void) syncPipesMenu
{
  unsigned int i, n;
  NSDictionary * userPipes;
  NSArray * allItems;
  NSEnumerator * e;
  NSString * itemTitle;

  for (i = 0, n = [pipesMenu numberOfItems]; i<n; i++)
    {
      id <NSMenuItem> item = [pipesMenu itemAtIndex: i];

      if (sel_eq([item action], @selector(pipeOutput:)))
        {
          [pipesMenu removeItemAtIndex: i];
          i--;
          n--;
        }
    }

  userPipes = [[NSUserDefaults standardUserDefaults]
    objectForKey: @"UserPipes"];
  allItems = [[userPipes allKeys] sortedArrayUsingSelector:
    @selector(caseInsensitiveCompare:)];
  e = [allItems objectEnumerator];
  while ((itemTitle = [e nextObject]) != nil)
    {
      [pipesMenu insertItemWithTitle: itemTitle
                              action: @selector(pipeOutput:)
                       keyEquivalent: [[userPipes objectForKey: itemTitle]
        objectForKey: @"KeyEquivalent"]
                             atIndex: [allItems indexOfObject: itemTitle] + 1];
    }
}

@end
