/*
    Frameworks.m

    Implementation of the Frameworks class for the ProjectManager
    application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "Frameworks.h"

#import <AppKit/NSTableView.h>
#import <AppKit/NSTableColumn.h>
#import <AppKit/NSPanel.h>

#import <Foundation/NSBundle.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSDebug.h>

#import "ProjectDocument.h"

@implementation Frameworks

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  [super dealloc];
}

- initWithDocument: (ProjectDocument *) aDocument
{
  if ([super initWithDocument: aDocument])
    {
      [[NSNotificationCenter defaultCenter]
        addObserver: self
           selector: @selector(frameworksChanged)
               name: ProjectFrameworksDidChangeNotification
             object: document];

      return self;
    }
  else
    {
      return nil;
    }
}

- (void) add: sender
{
  NSArray * frameworks = [document frameworks],
          * fixedFrameworks = [document fixedFrameworks];
  NSString * newFrameworkName;
  int i;

  for (i = 1, newFrameworkName = _(@"NewFramework");
    [frameworks containsObject: newFrameworkName] ||
    [fixedFrameworks containsObject: newFrameworkName];
    newFrameworkName = [NSString stringWithFormat: _(@"NewFramework%i"), i],
    i++);

  [document addFramework: newFrameworkName];
}

- (void) remove: sender
{
  int row = [table selectedRow];
  int numFixed = [[document fixedFrameworks] count];

  if (row < 0)
    {
      return;
    }

  if (row < numFixed)
    {
      NSRunAlertPanel(_(@"Can't remove framework"),
        _(@"The framework is mandatory and thus cannot be removed."),
        nil, nil, nil);

      return;
    }

  [document removeFramework: [[document frameworks]
    objectAtIndex: row - numFixed]];
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  unsigned int frameworksCount = [[document frameworks] count],
               fixedFrameworksCount = [[document fixedFrameworks] count];

  NSDebugLog(@"Number of frameworks is: %i, fixed frameworks: %i",
    frameworksCount, fixedFrameworksCount);

  return frameworksCount + fixedFrameworksCount;
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  NSDebugLog(@"Frameworks table row: %i", rowIndex);

  if (rowIndex < (int) [[document fixedFrameworks] count])
    {
      NSDictionary * attributes = [NSDictionary
        dictionaryWithObject: [NSColor disabledControlTextColor]
                      forKey: NSForegroundColorAttributeName];

      NSDebugLog(@"Fixed frameworks count: %i", [[document fixedFrameworks]
        count]);

      return [[[NSAttributedString alloc]
        initWithString: [[document fixedFrameworks] objectAtIndex: rowIndex]
            attributes: attributes]
        autorelease];
    }
  else
    {
      NSDebugLog(@"Frameworks count: %i", [[document frameworks] count]);

      return [[document frameworks] objectAtIndex: rowIndex - [[document
        fixedFrameworks] count]];
    }
}

- (void) tableView: (NSTableView *)aTableView 
    setObjectValue: (id)anObject 
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  int nFixed = [[document fixedFrameworks] count];

  [document renameFramework: [[document frameworks]
    objectAtIndex: rowIndex - nFixed]
                     toName: anObject];
}

- (BOOL)tableView: (NSTableView *)aTableView 
shouldEditTableColumn: (NSTableColumn *)aTableColumn 
              row: (int)rowIndex
{
  // don't allow to edit names of fixed frameworks
  return (rowIndex >= (int) [[document fixedFrameworks] count]);
}

- (void) frameworksChanged
{
  [table reloadData];
}

@end
