/*
    ProjectCreation.m

    Implementation of project creation related functions for the
    ProjectManager application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "ProjectCreation.h"

#import <Foundation/NSBundle.h>
#import <Foundation/NSSortDescriptor.h>
#import <Foundation/NSString.h>
#import <Foundation/NSFileManager.h>
#import <Foundation/NSUserDefaults.h>
#import <Foundation/NSNotification.h>

#import <AppKit/NSOpenPanel.h>
#import <AppKit/NSView.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTextField.h>
#import <AppKit/NSDocumentController.h>
#import <AppKit/NSMenu.h>
#import <AppKit/NSMenuItem.h>

#import "ProjectTypeLoader.h"
#import "ProjectWizzard.h"
#import "ProjectType.h"
#import "Preferences.h"
#import "ProjectNameChecking.h"
#import "ProjectNameQueryPanel.h"

/**
 * Runs a series of panels to ask the user to define a new project.
 *
 * If argument withLocation = NO, then the user isn't asked about the
 * location of the project (and the key `ProjectPath' will be absent
 * in the return value), and the calling code is assumed to know where
 * the project is supposed to reside.
 *
 * @return A dictionary with these key-value pairs:
 * {
 *   ProjectName = projectName;
 *   ProjectPath = projectPath;
 *   ProjectType = projectTypeID;
 *   ProjectTemplate = templateLocation;
 * }
 */
NSDictionary *
GetNewProjectSetup(BOOL withLocation)
{
  NSDictionary * projectTypes;
  NSEnumerator * e;
  Class projectTypeClass;
  NSMutableArray * typeInfo;
  NSMutableArray * templateInfo;
  NSString * path = nil;
  NSString * projectType = nil;
  NSString * templateName = nil;
  ProjectWizzard * pw = [ProjectWizzard shared];
  NSString * projectName = nil;
  ProjectNameQueryPanel * pnqp = [ProjectNameQueryPanel shared];

  // prepare the typeInfo for the project type panel
  projectTypes = [[ProjectTypeLoader shared] projectTypes];
  typeInfo = [NSMutableArray arrayWithCapacity: [projectTypes count]];
  e = [projectTypes objectEnumerator];
  while ((projectTypeClass = [e nextObject]) != nil)
    {
      [typeInfo addObject: [NSDictionary dictionaryWithObjectsAndKeys:
        [projectTypeClass projectTypeID], @"ProjectTypeID",
        [projectTypeClass humanReadableProjectTypeName], @"Name",
        [projectTypeClass projectTypeDescription], @"Description",
        [projectTypeClass projectTypeIcon], @"Icon",
        nil]];
    }

  // sort the type info by name
  [typeInfo sortUsingDescriptors: [NSArray arrayWithObject:
    [[[NSSortDescriptor alloc]
    initWithKey: @"Name"
      ascending: YES
       selector: @selector(caseInsensitiveCompare:)]
    autorelease]]];

getFilename:
  if (withLocation)
    {
      NSSavePanel * sp = [NSSavePanel savePanel];

      // prepare the open panel
      [sp setTitle: _(@"Choose Project Location")];
      [sp setRequiredFileType: nil];

      // get the project location
      if ([sp runModal] != NSOKButton)
        {
          return nil;
        }

      path = [sp filename];
      if ([[NSFileManager defaultManager] fileExistsAtPath: path])
        {
          NSRunAlertPanel(_(@"File already exists"),
            _(@"A file with that name already exists, please choose\n"
              @"a different, unused one, or delete the original."),
            nil, nil, nil);

          goto getFilename;
        }
    }

getProjectName:
  switch ([pnqp runModalWithPreviousOption: withLocation])
    {
    case WizzardNext:
      projectName = [pnqp projectName];
      if ([projectName length] == 0)
        {
          if (withLocation)
            {
              if (NSRunAlertPanel(_(@"Project name same as project directory?"),
                _(@"You didn't type a project name. Should the project name\n"
                  @"be the same as the name of the project directory?"),
                _(@"Yes, next"), _(@"No, back"), nil) ==
                NSAlertAlternateReturn)
                {
                  goto getProjectName;
                }
              else
                {
                  projectName = [path lastPathComponent];
                  if (!IsValidProjectName(projectName, YES))
                    {
                      goto getProjectName;
                    }
                }
            }
          else
            {
              NSRunAlertPanel(_(@"Project name required"),
                _(@"You must type a project name!"), nil, nil, nil);

              goto getProjectName;
            }
        }
      break;
    case WizzardPrevious:
      goto getFilename;
    case WizzardCancel:
      return nil;
    }

  // get the project type
getProjectType:
  [pw setTitle: _(@"Choose Project Type")];
  [pw setNextButtonTitle: nil];
  switch ([pw runModalWithInfo: typeInfo])
    {
    case WizzardNext:
      projectType = [[typeInfo objectAtIndex: [pw selectedItem]]
        objectForKey: @"ProjectTypeID"];
      break;
    case WizzardPrevious:
      goto getProjectName;
    case WizzardCancel:
      return nil;
    }

getProjectTemplate:
  // now prepare the template info
  {
    NSDictionary * templateDescriptions = [[projectTypes
      objectForKey: projectType] projectTemplateDescriptions];
    NSString * name;

    templateInfo = [NSMutableArray arrayWithCapacity:
      [templateDescriptions count]];
    e = [[templateDescriptions allKeys] objectEnumerator];
    while ((name = [e nextObject]) != nil)
      {
        [templateInfo addObject: [NSDictionary dictionaryWithObjectsAndKeys:
          name, @"Name",
          [templateDescriptions objectForKey: name], @"Description",
          nil]];
      }
  }

  [pw setTitle: _(@"Choose Project Template")];
  [pw setNextButtonTitle: _(@"Finish")];
  switch ([pw runModalWithInfo: templateInfo])
    {
      case WizzardNext:
        templateName = [[templateInfo objectAtIndex: [pw selectedItem]]
          objectForKey: @"Name"];
        break;
      case WizzardPrevious:
        goto getProjectType;
      case WizzardCancel:
        return nil;
    }

  if (path != nil)
    {
      return [NSDictionary dictionaryWithObjectsAndKeys:
        projectName, @"ProjectName",
        path, @"ProjectPath",
        projectType, @"ProjectType",
        [[projectTypes objectForKey: projectType] pathToProjectTemplate:
          templateName], @"ProjectTemplate",
        nil];
    }
  else
    {
      return [NSDictionary dictionaryWithObjectsAndKeys:
        projectName, @"ProjectName",
        projectType, @"ProjectType",
        [[projectTypes objectForKey: projectType] pathToProjectTemplate:
          templateName], @"ProjectTemplate",
        nil];
    }

}


/**
 * Creates a new project directory at path `projectPath' for a
 * project named `projectName'. The project directory is copied
 * from a project template located at `templatePath'.
 *
 * After the template has been copied, the project directory is
 * searched for files named "$PROJECT_NAME$".<ext> and these are
 * then renamed to the proper project name. Simmilar variable
 * substitution occurs inside the ProjectInfo.plist.
 *
 * @return YES if creating the new project bundle succeeded, otherwise NO.
 */
BOOL
CreateNewProject(NSString * projectPath,
                 NSString * projectName,
                 NSString * templatePath)
{
  NSFileManager * fm = [NSFileManager defaultManager];
  NSDirectoryEnumerator * de;
  NSString * filename;

  if (!ImportProjectFile(templatePath, projectPath, projectName))
    {
      return NO;
    }

  // Now read in the project's ".pmproj" file and replace variables in it.
  {
    NSString * projectInfoPlistPath;
    NSString * projectInfoPlist;

    projectInfoPlistPath = [projectPath
      stringByAppendingPathComponent: [projectName
      stringByAppendingPathExtension: @"pmproj"]];
    if (!ImportProjectFile(projectInfoPlistPath,
                           projectInfoPlistPath,
                           projectName))
      {
        NSLog(_(@"Failed to create new project at path: %@"), projectPath);

        return NO;
      }
  }

  return YES;
}

/**
 * Creates a directory at `dirPath' and the intermediate directories.
 *
 * This function creates a directory at `dirPath' (if it doesn't exist
 * already), and any intermediate directories as necessary (simmilar
 * to mkdir -p). If a file exists at the specified `dirPath' (or any of
 * intermediate directories is in fact a file), the operation fails.
 *
 * @return YES if the directory is created (or exists already), NO otherwise.
 */
BOOL
CreateDirectoryAndIntermediateDirectories(NSString * dirPath)
{
  NSFileManager * fm = [NSFileManager defaultManager];
  BOOL isDir;

  // does it already exist?
  if ([fm fileExistsAtPath: dirPath isDirectory: &isDir])
    {
      if (isDir)
        {
          return YES;
        }
      // some file is in the way
      else
        {
          NSLog(_(@"Can't create intermediate nodes to %@: "
                  @"a file is in the way."), dirPath);

          return NO;
        }
    }
  // no, start creating the intermediate directories as necessary
  else
    {
      NSEnumerator * e = [[dirPath pathComponents] objectEnumerator];
      NSString * path, * pathComponent;

      for (path = [e nextObject]; (pathComponent = [e nextObject]) != nil;)
        {
          path = [path stringByAppendingPathComponent: pathComponent];

          if ([fm fileExistsAtPath: path isDirectory: &isDir])
            {
              if (isDir)
                {
                  continue;
                }
              else
                {
                  NSLog(_(@"Can't create intermediate nodes to %@: "
                    @"a file is in the way."), path);

                  return NO;
                }
            }
          else
            {
              if (![fm createDirectoryAtPath: path attributes: nil])
                {
                  NSLog(_(@"Couldn't create intermediate directory %@."),
                    path);

                  return NO;
                }
            }
        }
    }

  return YES;
}

static BOOL
ImportProjectDirectoryFile(NSString * sourcePath,
                           NSString * destinationPath,
                           NSString * projectName)
{
  NSFileManager * fm = [NSFileManager defaultManager];
  NSString * filepath;
  NSDirectoryEnumerator * de;

  // do we need to copy at all?
  if (![sourcePath isEqualToString: destinationPath])
    {
      if (!CreateDirectoryAndIntermediateDirectories([destinationPath
        stringByDeletingLastPathComponent]))
        {
          NSLog(_(@"Failed to create new directory's "
                  @"intermediate directory nodes."));

          return NO;
        }

      if (![fm copyPath: sourcePath toPath: destinationPath handler: nil])
        {
          NSLog(_(@"Failed to copy source directory to destination."));

          return NO;
        }
    }

  // enumerate through the files of the project and if we find
  // a file which has $PROJECT_NAME$ as part of it's name we
  // rename it to the proper project name
performRenaming:
  de = [fm enumeratorAtPath: destinationPath];
  while ((filepath = [de nextObject]) != nil)
    {
      NSString * origFilename = [filepath lastPathComponent];
      NSMutableString * filename = [[origFilename mutableCopy] autorelease];

      // substitute variables in the filename
      [filename replaceString: @"$PROJECT_NAME$" withString: projectName];
      [filename replaceString: @"$HOSTNAME$"
                   withString: [[NSProcessInfo processInfo] hostName]];
      [filename replaceString: @"$USER$" withString: NSUserName()];
      [filename replaceString: @"$DATE$"
                   withString: [[NSDate date] description]];

      // if the result is different from the original, perform
      // a move operation
      if (![filename isEqualToString: origFilename])
        {
          NSDictionary * fattrs = [de fileAttributes];
          NSString * newPath = [[filepath stringByDeletingLastPathComponent]
            stringByAppendingPathComponent: filename];

          if (![fm movePath:
            [destinationPath stringByAppendingPathComponent: filepath]
                     toPath:
            [destinationPath stringByAppendingPathComponent: newPath]
                    handler: nil])
            {
              NSLog(_(@"Failed to substitute %@ for %@."), newPath, filepath);

              return NO;
            }

          // if it's a directory we need to restart the search
          if ([[fattrs fileType] isEqualToString: NSFileTypeDirectory])
            {
              goto performRenaming;
            }
        }
    }

  return YES;
}

static BOOL
ImportProjectPlainFile(NSString * sourceFile,
                       NSString * destinationFile,
                       NSString * projectName)
{
  NSMutableString * string;

  string = [NSMutableString stringWithContentsOfFile: sourceFile];
  if (string == nil)
    {
      NSLog(_(@"Couldn't import file %@: file not readable."), sourceFile);

      return NO;
    }

  // the following variables are expanded:
  // $PROJECT_NAME$ - expands to the project's name
  // $FILENAME$ - expands to the file's name, i.e. the last path
  //        component of the path to the file
  // $FILENAME_EXT$ - expands to the file name's last path component
  //        with it's path extension trimmed
  // $FILENAME_CAPS$ - expands to the file's name with all
  //        letters uppercase. Additionally, all '.' will be replaced
  //        with '_'
  // $FILENAME_LOWER$ - expands to the file's name with all
  //        letters lowercase. Additionally, all '.' will be replaced
  //        with '_'
  // $FILEPATH$ - expands to the destination path where in the project
  //        the file will reside
  // $FILEPATH_DIR$ - expands to the destination path where in the project
  //        the file will reside with the last path component trimmed
  // $USER$ - expands to the name of the user running ProjectManager
  // $DATE$ - expands to the current date as returned by
  //        [[NSDate date] description]
  // $HOSTNAME$ - the hostname of the machine ProjectManager is running on

  [string replaceString: @"$PROJECT_NAME$" withString: projectName];

  [string replaceString: @"$FILENAME$" withString: [destinationFile
    lastPathComponent]];

  [string replaceString: @"$FILENAME_EXT$" withString: [[destinationFile
    lastPathComponent] stringByDeletingPathExtension]];
  [string replaceString: @"$FILENAME_CAPS$" withString: [[[destinationFile
    lastPathComponent] uppercaseString] stringByReplacingString: @"."
                                                     withString: @"_"]];
  [string replaceString: @"$FILENAME_LOWER$" withString: [[[destinationFile
    lastPathComponent] lowercaseString] stringByReplacingString: @"."
                                                     withString: @"_"]];
  [string replaceString: @"$FILEPATH$" withString: destinationFile];
  [string replaceString: @"$FILEPATH_DIR$" withString: [destinationFile
    stringByDeletingLastPathComponent]];

  [string replaceString: @"$USER$" withString: NSUserName()];
  [string replaceString: @"$DATE$" withString: [[NSDate date] description]];
  [string replaceString: @"$HOSTNAME$"
             withString: [[NSProcessInfo processInfo] hostName]];

  if ([string writeToFile: destinationFile atomically: NO])
    {
      return YES;
    }
  else
    {
      NSLog(_(@"Couldn't import file %@: failed to write to destination %@."),
        sourceFile, destinationFile);

      return NO;
    }
}

/**
 * Copies `sourceFile' to `destinationFile' and replaces special
 * project variables (such as $FILENAME$ or $USER$) in the file
 * while doing so. If, instead, the file is a directory, the filenames
 * of it's contents are (recursively) replaced. The argument
 * `projectName' specifies the name of the project that's importing
 * the file.
 *
 * The arguments `sourceFile' and `destinationFile' can be identical,
 * in which case only variable substitution occurs without any copying.
 *
 * @return YES if the import succeeds, otherwise NO.
 */
BOOL
ImportProjectFile(NSString * sourceFile,
                  NSString * destinationFile,
                  NSString * projectName)
{
  NSFileManager * fm = [NSFileManager defaultManager];

  if ([[[fm fileAttributesAtPath: sourceFile traverseLink: YES]
    fileType]
    isEqualToString: NSFileTypeDirectory])
    {
      return ImportProjectDirectoryFile(sourceFile,
                                        destinationFile,
                                        projectName);
    }

  // otherwise do some variable expansion on the file before copying it
  else
    {
      return ImportProjectPlainFile(sourceFile, destinationFile, projectName);
    }
}
