/*
    Subprojects.m

    Implementation of the Subprojects class for the ProjectManager
    application.

    Copyright (C) 2005  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "Subprojects.h"

#import <Foundation/NSArray.h>
#import <Foundation/NSNotification.h>

#import <AppKit/NSTableView.h>
#import <AppKit/NSTableColumn.h>

#import "ProjectDocument.h"
#import "ProjectType.h"

@implementation Subprojects

- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];

  TEST_RELEASE(sortedSubprojectNames);

  [super dealloc];
}

- (void) awakeFromNib
{
  [super awakeFromNib];

  [table setDoubleAction: @selector(openSubprojectAction:)];
}

- initWithDocument: (ProjectDocument *) aDocument
{
  if ([super initWithDocument: aDocument])
    {
      [[NSNotificationCenter defaultCenter]
        addObserver: self
           selector: @selector(subprojectsChanged)
               name: ProjectSubprojectsDidChangeNotification
             object: document];

      ASSIGN(sortedSubprojectNames, [[[document subprojects] allKeys]
        sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)]);

      [table reloadData];

      return self;
    }
  else
    {
      return nil;
    }
}

- (NSString *) selectedSubproject
{
  int row = [table selectedRow];

  if (row >= 0)
    {
      return [sortedSubprojectNames objectAtIndex: row];
    }
  else
    {
      return nil;
    }
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  return [sortedSubprojectNames count];
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  NSString * subprojectName = [sortedSubprojectNames objectAtIndex: rowIndex];

  if ([[aTableColumn identifier] isEqualToString: @"Name"])
    {
      return subprojectName;
    }
  else
    {
      return [[document subprojects] objectForKey: subprojectName];
    }
}

- (void) subprojectsChanged
{
  ASSIGN(sortedSubprojectNames, [[[document subprojects] allKeys]
    sortedArrayUsingSelector: @selector(caseInsensitiveCompare:)]);

  [table reloadData];
}

@end
