/***************************************************************************
                          gui.cpp  -  description
                             -------------------
    begin                : Wed May 2 2001
    copyright            : (C) 2001 by Holger Sattel
    email                : hsattel@rumms.uni-mannheim.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#if HAVE_CONFIG_H
   #include "config.h"
#endif /* HAVE_CONFIG_H */

#include "prokyon.h"
#include "gui.h"
#include "jobmanager.h"
#include "configuration.h"
#include "job_connecttodatabase.h"
#include "job_disconnectfromdatabase.h"
#ifdef HAVE_MIXXX
   #include "job_connecttomixxx.h"
   #include "job_disconnectfrommixxx.h"
#endif /* HAVE_MIXXX */
#include "job_querydatabase.h"
#include "job_getextradata.h"
#ifdef EMBEDDED
   #include "database_embedded_dialog.h"
#else
   #include "database_create_dialog.h"
#endif   // ifdef EMBEDDED

#include "helpviewer.h"
#ifdef HAVE_MEXTRAS
   #include "mextras_log.h"
#endif /* HAVE_MEXTRAS */
#include "database.h"
#include "pixmapcache.h"

#include <qaction.h>
#include <qtoolbutton.h>
#include <qstatusbar.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qimage.h>
#include <qtooltip.h>
#include <qvaluelist.h> 
#include <qlayout.h>
#include <qvbox.h>
#include <qprogressbar.h>
#include <qwidgetstack.h>
#include <qcolor.h>
#include <qwindowsstyle.h>
#include <qevent.h>
#include <qregexp.h>
#include <locale.h> /* required by gcc 2.95 */

// stolen form kde... see kdelibs/kdefx/kimageeffect.cpp
QImage& GUI::blend(const QColor& clr, QImage& dst, float opacity)
{
    if (dst.width() <= 0 || dst.height() <= 0)
        return dst;

    if (opacity < 0.0 || opacity > 1.0) {
        qWarning( "WARNING: KImageEffect::blend : invalid opacity. Range [0, 1]");
        return dst;
    }

    if (dst.depth() != 32)
        dst = dst.convertDepth(32);

    int pixels = dst.width() * dst.height();

    {
        int rcol, gcol, bcol;
        clr.rgb(&rcol, &gcol, &bcol);

#ifdef WORDS_BIGENDIAN   // ARGB (skip alpha)
        register unsigned char *data = (unsigned char *)dst.bits() + 1;
#else                    // BGRA
        register unsigned char *data = (unsigned char *)dst.bits();
#endif

        for (register int i=0; i<pixels; i++)
        {
#ifdef WORDS_BIGENDIAN
            *data += (unsigned char)((rcol - *data) * opacity);
            data++;
            *data += (unsigned char)((gcol - *data) * opacity);
            data++;
            *data += (unsigned char)((bcol - *data) * opacity);
            data++;
#else
            *data += (unsigned char)((bcol - *data) * opacity);
            data++;
            *data += (unsigned char)((gcol - *data) * opacity);
            data++;
            *data += (unsigned char)((rcol - *data) * opacity);
            data++;
#endif
            data++; // skip alpha
        }
    }

    return dst;
}



GUI::GUI(QWidget *parent, const char *name): QMainWindow(parent, name)
{
  setIcon(pixmapcache->get("icon.png"));
  
  action_connect = new QAction(_("Connect"), pixmapcache->getFadeIconSet("action_connect.png"), _("&Connect"), CTRL+Key_C, this, "connect");
  connect(action_connect, SIGNAL(activated()),this, SLOT(slot_connectToDB()));
  action_disconnect = new QAction(_("Disconnect"), pixmapcache->getFadeIconSet("action_disconnect.png"), _("&Disconnect"), CTRL+Key_D, this, "disconnect");
  connect(action_disconnect, SIGNAL(activated()), this, SLOT(slot_disconnectFromDB()));
  QAction *action_quit = new QAction(_("Quit"), pixmapcache->getFadeIconSet("action_quit.png"), _("&Quit"), CTRL+Key_Q, this, "quit");
  connect(action_quit, SIGNAL(activated()), this, SLOT(slot_QUIT()));
  QAction *action_pref = new QAction(_("Settings..."), pixmapcache->getFadeIconSet("action_settings.png"), _("&Settings..."), CTRL+Key_S, this, "settings");
  connect(action_pref, SIGNAL(activated()), this, SLOT(slot_PREF()));
  action_createdb = new QAction(_("Create new database..."), pixmapcache->getFadeIconSet("action_createdb.png"), _("Create &New database..."), CTRL+Key_N, this, "create");
  connect(action_createdb, SIGNAL(activated()), this, SLOT(slot_CREATEDB()));
  
  action_about = new QAction(_("About..."), pixmapcache->getFadeIconSet("idea.png"), _("A&bout..."), CTRL+Key_B, this, "about");
  connect(action_about, SIGNAL(activated()), this, SLOT(slot_ABOUT()));

  action_welcome = new QAction(_("Welcome screen..."), _("&Welcome..."), CTRL+Key_W, this, "welcome");
  connect(action_welcome, SIGNAL(activated()), this, SLOT(slot_WELCOME()));

  action_documentation = new QAction(_("Documentation..."), pixmapcache->getFadeIconSet("action_documentation.png"),_("d&Ocumentation..."), CTRL+Key_O, this, "documentation");
  connect(action_documentation, SIGNAL(activated()), this, SLOT(slot_DOCUMENTATION()));

#ifdef HAVE_MEXTRAS
  action_mextras_log = new QAction(_("Musicextras log..."), pixmapcache->getFadeIconSet("action_mextras_log.png"),_("&Musicextras log..."), CTRL+Key_M, this, "see musicextras log");
  connect(action_mextras_log, SIGNAL(activated()), this, SLOT(slot_MEXTRAS_LOG()));

  action_mextras_scan = new QAction(_("Scan all"), pixmapcache->getFadeIconSet("html.png"),_("Scan all..."), CTRL+Key_T, this, "Scan the internet for images on all tracks");
  connect(action_mextras_scan, SIGNAL(activated()), this, SLOT(slot_MEXTRAS_SCAN()));

  action_mextras_zap     = new QAction(_("&Zap cache..."), CTRL+Key_Z, this);
  connect(action_mextras_zap, SIGNAL(activated()), this, SLOT(slot_MEXTRAS_ZAP()));

  action_mextras_abort     = new QAction(_("&Abort scan..."), CTRL+Key_A, this);
  connect(action_mextras_abort, SIGNAL(activated()), this, SLOT(slot_MEXTRAS_ABORT()));
#endif /* HAVE_MEXTRAS */
 
#ifdef HAVE_MIXXX
  action_mixxx_connect = new QAction(_("Connect"), pixmapcache->getFadeIconSet("action_connect.png"), _("Connect"), 0, this, "establish connection to mixxx");
  connect(action_mixxx_connect, SIGNAL(activated()), this, SLOT(slot_connectToMixxx()));
 
  action_mixxx_disconnect = new QAction(_("Disconnect"), pixmapcache->getFadeIconSet("action_disconnect.png"), _("Disconnect"), 0, this, "disconnect from mixxx");
  connect(action_mixxx_disconnect, SIGNAL(activated()), this, SLOT(slot_disconnectFromMixxx()));
#endif /* HAVE_MIXXX */

  connect(app, SIGNAL(lastWindowClosed()), this, SLOT(slot_QUIT()));
	
  QToolBar *toolbar = new QToolBar(this, "prokyon toolbar");
	
  action_connect->addTo(toolbar);
  action_disconnect->addTo(toolbar);
  action_pref->addTo(toolbar);
  action_documentation->addTo(toolbar);
  action_quit->addTo(toolbar);
	
  action_toggleleft = new QAction(_("Toggle SelectionFrame"), pixmapcache->getFadeIconSet("action_switchleft.png"), _("Toggle &Left Frame"), CTRL+Key_L, this, "toggleleft");
  action_toggleleft->setToggleAction(true);
  action_toggleleft->setOn(true);
  connect(action_toggleleft, SIGNAL(toggled(bool)), this, SLOT(slot_toggleleft(bool)));

  action_togglebottom = new QAction(_("Toggle InfoFrame"), pixmapcache->getFadeIconSet("action_switchbottom.png"), _("Toggle &Info Frame"), CTRL+Key_I, this, "toggleinfo");
  action_togglebottom->setToggleAction(true);
  action_togglebottom->setOn(true);
  connect(action_togglebottom, SIGNAL(toggled(bool)), this, SLOT(slot_togglebottom(bool)));

  action_toggleright = new QAction(_("Toggle PlaylistFrame"), pixmapcache->getFadeIconSet("action_switchright.png"), _("Toggle Right Frame"), CTRL+Key_R, this, "toggleright");
  action_toggleright->setToggleAction(true);
  action_toggleright->setOn(true);
  connect(action_toggleright, SIGNAL(toggled(bool)), this, SLOT(slot_toggleright(bool)));

  switchbar = new QToolBar(this, "prokyon switchbar");
  action_toggleleft->addTo(switchbar);
  action_togglebottom->addTo(switchbar);
  action_toggleright->addTo(switchbar);

  QAction *action_select = new QAction(pixmapcache->getFadeIconSet("action_selectall.png"), _("Select all..."), CTRL+Key_E, this, "select all tracks");
  connect(action_select, SIGNAL(activated()), this, SLOT(slot_selectAll()));

  QAction *action_unselect = new QAction(pixmapcache->getFadeIconSet("action_unselectall.png"), _("Unselect all..."), CTRL+Key_F, this, "unselect all tracks");
  connect(action_unselect, SIGNAL(activated()), this, SLOT(slot_unselectAll()));

  commoditybar = new QToolBar(this, "prokyon commodity bar");
  action_select->addTo(commoditybar);
  action_unselect->addTo(commoditybar);

  QPopupMenu *filepopup = new QPopupMenu(this);
  action_createdb->addTo(filepopup);
  filepopup->insertSeparator();
  action_connect->addTo(filepopup);
  action_disconnect->addTo(filepopup);
  filepopup->insertSeparator();
  action_pref->addTo(filepopup);
  filepopup->insertSeparator();
  action_quit->addTo(filepopup);
  menuBar()->insertItem(_("&File"), filepopup);
	
#ifdef HAVE_MEXTRAS
  QPopupMenu *mextraspopup = new QPopupMenu(this);
  action_mextras_log->addTo(mextraspopup);
  action_mextras_scan->addTo(mextraspopup);
  action_mextras_zap->addTo(mextraspopup);
  action_mextras_abort->addTo(mextraspopup);
  menuBar()->insertItem(_("&Musicextras"), mextraspopup); 	
#endif /* HAVE_MEXTRAS */

#ifdef HAVE_MIXXX
  QPopupMenu *mixxxpopup = new QPopupMenu(this);
  action_mixxx_connect->addTo(mixxxpopup);
  action_mixxx_disconnect->addTo(mixxxpopup);
  menuBar()->insertItem(_("Mixxx"), mixxxpopup);
#endif /* HAVE_MIXXX */
  
  QPopupMenu *helppopup = new QPopupMenu(this);
  action_about->addTo(helppopup);
  action_welcome->addTo(helppopup);
  action_documentation->addTo(helppopup);
  menuBar()->insertItem(_("&Help"), helppopup); 	

  hsplit = new QSplitter(QSplitter::Horizontal, this);
  
  vsplitr = new QSplitter(QSplitter::Vertical, hsplit);
  vsplitr->setOpaqueResize( true );

  selector = new GUI_Selector(vsplitr);
  sourcer = new GUI_Sourcer(vsplitr);
  QString temps1 = _( "Right click to mount disks, \nRead files info into database, \nOr synchronize." );
  QToolTip::add( sourcer, temps1 );
  QToolTip::add( sourcer->viewport(), temps1 );
  
  vsplitm = new QSplitter(QSplitter::Vertical, hsplit);
  listing = new GUI_Listing(vsplitm);
  taglisting = new GUI_TagListing(vsplitm, listing->list()); 
  // MK begin
  // Do not set a fixed height in constructor of taglisting!
  //===================================================
  //  taglisting->adjustSize();
  //  taglisting->setFixedHeight(taglisting->height());
  //---------------------------------------------------
  // ^^^ -> moved into polish method() !! [HoSa]
  //===================================================
  // MK end

  QString temps2 = _( "View  files and tags details" );
  QToolTip::add( taglisting, temps2 );

  vsplitl = new QSplitter(QSplitter::Vertical, hsplit);

  playlisting = new GUI_Playlist(vsplitl);
  
  status1 = new QLabel(statusBar());
  statusBar()->addWidget(status1, 1, true);

  QFrame* frame1 = new QFrame( statusBar());
  QHBoxLayout* layout1 = new QHBoxLayout( frame1 ); 
  status2 = new QLabel( frame1 );
  labelpix = new QLabel( frame1 );
  layout1->addWidget( labelpix );
  layout1->addWidget( status2, 1 );
  statusBar()->addWidget( frame1, 1, true );

  status3 = new QLabel(statusBar());
  statusBar()->addWidget(status3, 1, true);
  status4 = new QLabel(statusBar());
  statusBar()->addWidget(status4, 1, true);
  status5 = new QLabel(statusBar());
  statusBar()->addWidget(status5, 1, true);

  stackStatus6 = new QWidgetStack(statusBar());
  status6 = new QLabel(statusBar());
  stackStatus6->addWidget(status6, 1);
#ifdef HAVE_MEXTRAS
  mextrasProgress = new QProgressBar(statusBar());
  mextrasProgress->setTotalSteps(0);
  mextrasProgress->setProgress(0);
  stackStatus6->addWidget(mextrasProgress, 2);
#endif /* HAVE_MEXTRAS */
  stackStatus6->raiseWidget(1);
  statusBar()->addWidget(stackStatus6, 1, true);
  
  setStatusInfo(" " + QString(_("Jobs")) + ": 0 ", 1);
  setStatusInfo(" " + QString(_("Status")) + ": " + QString(_("NOT connected")), 2);
  setStatusInfo(" " + QString(_("Artists")) + ": 0 ", 3);
  setStatusInfo(" " + QString(_("Tracks")) + ": 0 ", 4);
  setStatusInfo(" " + QString(_("Mode")) + ": " + QString(_("offline")), 5);
  setStatusInfo(" " + /* TRANSLATORS: short form of 'Images' */ QString(_("Ima.")) + ":0 " + /* TRANSLATORS: short form of 'Covers' */ QString(_("Cov.")) + ":0 " + /* TRANSLATORS: short form of 'Lyrics' */ QString(_("Lyr.")) + ":0", 6);

  
  setCentralWidget(hsplit);
  
  connectionState = false;
#ifdef HAVE_MIXXX
  mixxxConnectionState = false;
#endif /* HAVE_MIXXX */
  togglebottom_state = false;
}

void GUI::polish()
{
  QMainWindow::polish();
  if (config->getAllowGuiRestore() ) {
    action_toggleleft->setOn( config->gettoggleleft() );
    slot_toggleleft( config->gettoggleleft() );

    action_toggleright->setOn( config->gettoggleright() );
    slot_toggleright( config->gettoggleright() );

    action_togglebottom->setOn( config->gettogglebottom() );
    slot_togglebottom( config->gettogglebottom() );

    hsplit->setSizes( config->getinfo_guisize() );
    vsplitr->setSizes( config->getinfo_guisizer() );
    vsplitm->setSizes( config->getinfo_guisizem() );
#ifdef HAVE_MEXTRAS
    gui->getPlaylisting()->getExtrasSplit()->setSizes( config->getinfo_guisizel() );
#endif /* HAVE_MEXTRAS */
  }
}

QString GUI::getStatusInfo(int i) {
	switch(i) {
	case 1: return status1->text(); break;
	case 2: return status2->text(); break;
	case 3: return status3->text(); break;
	case 4: return status4->text(); break;
	case 5: return status5->text(); break;
	case 6: return status6->text(); break;
	}
}

void GUI::setStatusInfo(QString message, int i) {
	switch(i) {
	case 1: status1->setText(message); break;
	case 2: { 
	  status2->setText(message);
	  if (message.find( _("NOT connected") ) != -1 ) 
	    labelpix->setPixmap( pixmapcache->get("madsmiley.png") );
	  else 
	    labelpix->setPixmap( pixmapcache->get("smiley.png") );
	} break;
	case 3: status3->setText(message); break;
	case 4: status4->setText(message); break;
	case 5: status5->setText(message); break;
	case 6: status6->setText(message); break;
	}
}

void GUI::slot_selectAll()
{
  gui->getListing()->list()->selectAll(true);
}

void GUI::slot_unselectAll()
{
  gui->getListing()->list()->selectAll(false);
}

void GUI::slot_connectToDB()
{
  jobman->lock();
  jobman->addJob(new Job_ConnectToDatabase(config->getHOST(), config->getDATABASE(), config->getUSER(), config->getPASS()));
  jobman->unlock();
}

void GUI::slot_disconnectFromDB()
{
  jobman->lock();
  jobman->addJob(new Job_DisconnectFromDatabase());
  jobman->unlock();
}

void GUI::slot_connectToMixxx()
{
#ifdef HAVE_MIXXX
  jobman->lock();
  jobman->addJob(new Job_ConnectToMixxx(config->MixxxHost(), config->MixxxPort()));
  jobman->unlock();
#endif /* HAVE_MIXXX */
}

void GUI::slot_disconnectFromMixxx()
{
#ifdef HAVE_MIXXX
  jobman->lock();
  jobman->addJob(new Job_DisconnectFromMixxx());
  jobman->unlock();
#endif /* HAVE_MIXXX */
}

#ifdef HAVE_MIXXX
void GUI::notifyEstablishedMixxxConnection() {
  setMixxxConnectionState(true);
}

void GUI::notifyLostMixxxConnection()        {
  setMixxxConnectionState(false);
} 
#endif /* HAVE_MIXXX */

void GUI::slot_setConnectionState(bool state)
{
  if(state) {
    action_connect->setEnabled(false);
    action_createdb->setEnabled(false);
    action_disconnect->setEnabled(true);
    setStatusInfo(" " + QString(_("Status")) + ": " + QString(_("connected")), 2);
    setStatusInfo(" " + QString(_("Mode")) + ": " + QString(_("normal")), 5);
  } else {
    action_connect->setEnabled(true);
    action_createdb->setEnabled(true);
    action_disconnect->setEnabled(false);
    setStatusInfo(" " + QString(_("Status")) + ": " + QString(_("NOT connected")), 2);
    setStatusInfo(" " + QString(_("Artists")) + ": 0", 3);
    setStatusInfo(" " + QString(_("Tracks")) + ": 0", 4);
    setStatusInfo(" " + QString(_("Mode")) + ": " + QString(_("offline")), 5);
  }
  connectionState = state;
}

#ifdef HAVE_MIXXX
void GUI::setMixxxConnectionState(bool state)
{
  action_mixxx_connect->setEnabled(!state);
  action_mixxx_disconnect->setEnabled(state);
  listing->setMixxxConnectionState(state);
  playlisting->setMixxxConnectionState(state);
  selector->setMixxxConnectionState(state);
  mixxxConnectionState = state;
}

void GUI::enterMixxxMode() {
    togglebottom_state = action_togglebottom->isOn();
    if(togglebottom_state) slot_togglebottom(false);
    action_togglebottom->setEnabled(false);
}

void GUI::leaveMixxxMode() {
    if(togglebottom_state) slot_togglebottom(true);
    if(!taglisting->isEditMode()) action_togglebottom->setEnabled(true);
}
#endif /* HAVE_MIXXX */

void GUI::slot_PREF() {
    config->showConfigurationDialog();
}

void GUI::slot_QUIT() {
  close();
}

void GUI::slot_CREATEDB() {
#ifdef EMBEDDED
    Database_Embedded_Dialog *dlg = new Database_Embedded_Dialog(this);
    dlg->show();
#else
    Database_Create_Dialog *dlg = new Database_Create_Dialog(this);
    dlg->show();
#endif   // ifdef EMBEDDED
}

void GUI::slot_INITDB() {
}

void GUI::slot_toggleleft(bool b) {
    if(!b) {
	gui->getSelector()->hide();
	gui->getSourcer()->hide();
    } else {
	gui->getSelector()->show();
	gui->getSourcer()->show();
    }
}

void GUI::slot_togglebottom(bool b) {
    if(!b) {
	gui->getTagListing()->hide();
    } else {
	gui->getTagListing()->show();
    }
}

void GUI::slot_toggleright(bool b) {
    if(!b) {
	gui->getPlaylisting()->hide();
    } else {
	gui->getPlaylisting()->show();
    }
}

#include <qmessagebox.h>
void GUI::slot_ABOUT() {
  
  QMessageBox aboutbox(this, "About");
  aboutbox.setPaletteBackgroundColor ( QColor(0xbc,0xbc,0xbc) );  // correct button color rendering with next line pixmpap....
  aboutbox.setPaletteBackgroundPixmap( pixmapcache->get("prokyon_bump.png"));
  aboutbox.setPaletteForegroundColor(QColor(0,0,0));
  aboutbox.setText( ("<center><b><u>Prokyon3 " VERSION EMBEDDED_STRING ) +
	    _("</u></b><br>multithreaded music manager<br>http://prokyon3.sourceforge.net<br>"
              "using %1 %2 %3 %4 <br>"
              "<br><b><u>Developers:</u></b><br>"
              "Markus Kalkbrenner &lt;mkalkbrenner@users.sourceforge.net&gt;<br>"
              "Paul-henri Ferme &lt;paul-henri@users.sourceforge.net&gt;<br>"
              "Christian Muller &lt;cmbofh@users.sourceforge.net&gt;<br>"
              "<br><b><u>Contributors:</u></b><br>"
              "Tim Lee<br>"
              "Lars Volkhardt<br>"
              "Matthew Hughes (FLAC support)<br>"
              "Cord Horeis (Taglib support)<br>"
              "Wladimir Korolew (prokyon3 logos)<br>"
              "Tony Cebzanov<br>"
              "Ashar Voultoiz<br>"
              "<br><b><u>Packagers:</u></b><br>"
              "Gustavo Franco (Debian)<br>"
              "Maxim Heijndijk (Mandrake)<br>"
              "Christophe Vandeplas (SuSE)<br>"
              "Kai Lehman (FreeBSD)<br>"
              "<br><b><u>Translators:</u></b><br>"
              "Gustavo Franco<br>"
              "Maxim Heijndijk<br>"
              "Christophe Vandeplas<br>"
              "Luiz Augusto Pereira Fernandes<br>"
              "Szymon Stepkowski<br>"
              "<br>(c) 2002 by Holger Sattel<br>"
              "(c) 2003-2005 by Markus Kalkbrenner, Paul-henri Ferme, Christian Muller<br><br>"
              "<b><u>Special thanks to</u></b><br>Zachary P. Landau (musicextras)<br>see http://kapheine.hypa.net/kapheine.asc<br>"
              "Tue Haste Andersen (mixxx)<br>see http://mixxx.sourceforge.net</center>").arg(LIBRARY_ID3_STRING).arg(LIBRARY_OGG_STRING).arg(LIBRARY_FLAC_STRING).arg(LIBRARY_TAG_STRING));
  aboutbox.exec();
}

void GUI::slot_WELCOME() {
    QMessageBox welcomebox(gui, _("Welcome"));
    welcomebox.setPaletteBackgroundColor ( QColor(0xbc,0xbc,0xbc) );  // correct button color rendering with next line pixmpap....
    welcomebox.setPaletteBackgroundPixmap( pixmapcache->get("prokyon_bump.png"));
    welcomebox.setPaletteForegroundColor(QColor(0,0,0)); // forces black text, independly of user selected theme.
    welcomebox.setText( _( "<b><center><u>WELCOME TO PROKYON3</u></center></b><br><br>"
"Thank you for running Prokyon3 for the first time!!!<br><br>"
"Prokyon3 is a multithreaded music manager and tag editor for Unix (developed on Linux). "
"It was written in C++ using the Qt3 widget set and the MySQL database. "
"prokyon3 can access MP3, Ogg and FLAC files on harddisk, CDROM, SMB and NFS. "
"Files can be played using XMMS (default) or other players. "
"The file view is customizable and favorite artists are supported. "
"Prokyon3 also offers an editor for ID3, Ogg and FLAC tags and has been designed to support tagging for large quantities of files.<br><br>"
"Prokyon3 will now automatically create an SQL database for you "
"and connect right to it.<br> Once you are connected (Check the smiley face...), you should go to "
"\"File/Settings/Directory\" "
"and type paths to your music files. Then go to the \"Sourcer\" frame, "
"right click on your main media icon and select "
"\"synchronize (prefer Sources)\" command. You are now ready to enjoy Prokyon3.<br>"
"Prokyon3 is simple and straightforward to use. Anyway, "
"additional help will be  provided via pop-ups window when appropriate.<br><br>"
"The Prokyon3 development team....<br><br><br>"
) );
    welcomebox.exec();
}

QString GUI::getLanguageCode() {
  /****
       In the function `dcgettext' at every call the current setting of
       the highest priority environment variable is determined and used.
       Highest priority means here the following list with decreasing
       priority:

       1. `LANGUAGE'

       2. `LC_ALL'

       3. `LC_xxx', according to selected locale

       4. `LANG'
       
       note:  setlocale()  info is  overidden by $LANGUAGE....

Note: Official names for character sets  are expressed in ANSI_X3.4-1968 which is commonly called
US-ASCII or simply ASCII.
  ****/
  QString result=setlocale (LC_MESSAGES,NULL);
  //  qWarning("setlocale (LC_MESSAGES,NULL): %s", result.ascii());

  QString language(getenv("LANGUAGE"));
  if (!language.isEmpty()) {
    /****
	 While for the `LC_xxx' variables the value should consist of exactly
	 one specification of a locale the `LANGUAGE' variable's value can
	 consist of a colon separated list of locale names.
	 Here we look for the first langage supported by prokyon3
    *****/
    QStringList langlist = QStringList::split( QRegExp(","), language);
    for ( QStringList::Iterator it = langlist.begin(); it != langlist.end(); ++it ) 
      if ( QString(HAVE_LANGUAGE).contains((*it).left(2))) {
	result = (*it);
	break;
      }
  }
  //  qWarning( "return: %s", result.left(2).ascii());
  return( result.left(2));
}


void GUI::slot_DOCUMENTATION() {

  QString language = getLanguageCode().ascii();
  //  qWarning( "getLanguageCode().ascii(): %s", language.ascii());

  QString helpfile;

  if ( language =="fr" ) 
    if ( QFile::exists( PROKYON_HOME + "/html/manual_one_file/index_fr.html" ) )
      helpfile=PROKYON_HOME + "/html/manual_one_file/index_fr.html";

  if ( helpfile.isEmpty() )     // default to english
    if ( QFile::exists( PROKYON_HOME + "/html/manual_one_file/index.html" ) )
      helpfile=PROKYON_HOME + "/html/manual_one_file/index.html";

  if ( !helpfile.isEmpty() ) {
    HelpWindow *help = new HelpWindow( helpfile, PROKYON_HOME + "/html/manual_one_file/", 0, "help viewer");
    help->setCaption("Helpviewer");

    QRect qRect(QApplication::desktop()->screenGeometry());
    int iXpos=qRect.width()/2 - help->width()/2;
    int iYpos=qRect.height()/2 - help->height()/2;
    help->move(iXpos,iYpos);

    if ( QApplication::desktop()->width() > 400
	 && QApplication::desktop()->height() > 500 )
      help->show();
    else
      help->showMaximized();
  } else 
    QMessageBox::information( this, "Prokyon3",
			      _("Unable to find html documentation files.\n" 
			      "See INSTALL for instructions") );
}

void GUI::closeEvent( QCloseEvent* e ) {
  // database must stop emitting signals during closing prokyon3
  disconnect( database, 0, 0, 0 );
  // reverting possible manipulations of all toggles caused by editMode
  if(taglisting->isEditMode()) taglisting->slot_discardbutton();
#ifdef HAVE_MIXXX
  // reverting possible manipulations of togglebottom caused by mixxxQueue
  leaveMixxxMode();
#endif /* HAVE_MIXXX */
  
  config->setposx( x() );
  config->setposy( y() );
  config->setheight( height() );
  config->setwidth( width() );

  config->settoggleleft( gui->getSelector()->isVisible() );
  config->settogglebottom( gui->getTagListing()->isVisible() );
  config->settoggleright(  gui->getPlaylisting()->isVisible() );

  config->setinfo_guisize( hsplit->sizes() );
  config->setinfo_guisizer( vsplitr->sizes() );
  config->setinfo_guisizem( vsplitm->sizes() );
#ifdef HAVE_MEXTRAS
  config->setinfo_guisizel( gui->getPlaylisting()->getExtrasSplit()->sizes() );
#endif /* HAVE_MEXTRAS */

  pauseWatchers();
  
  QMainWindow::closeEvent( e );
}

void GUI::slot_MEXTRAS_LOG() {
#ifdef HAVE_MEXTRAS
  MextrasLog *log = new MextrasLog(NULL,"musicextras log window");
  log->show();
#endif /* HAVE_MEXTRAS */
}

void GUI::slot_MEXTRAS_SCAN() {
#ifdef HAVE_MEXTRAS
 jobman->addJob(new Job_QueryDatabase( MEXTRAS_QUERY_MEDIUM ));
#endif /* HAVE_MEXTRAS */
}

void GUI::slot_MEXTRAS_ZAP()
{
#ifdef HAVE_MEXTRAS
  gui->getPlaylisting()->mexlog->clear();
  jobman->addJob(new Job_GetExtraData( MEXTRAS_ZAP, MEDIUM_HARDDISK ));
#endif /* HAVE_MEXTRAS */
}

void GUI::slot_MEXTRAS_ABORT()
{
#ifdef HAVE_MEXTRAS
  jobman->terminateMusicextrasJob();
#endif /* HAVE_MEXTRAS */
}

#ifdef HAVE_MEXTRAS
void GUI::notifyMextrasTrackListing ( QList<TRACK> *list ) {
  if (list) {
    SortList * sortlist = new SortList();
    for(TRACK *track = list->first(); track != 0; track = list->next()) sortlist->append(track);
    // we sort by title name to provide a pseudo random list.
    // This will somehow avoid launching  10 musicextras jobs with identical artist and album name. 
    sortlist->sort();
    gui->getPlaylisting()->mexlog->append( QDateTime::currentDateTime().toString() );
    for(TRACK *track = sortlist->first(); track != 0; track = sortlist->next()) {
      Job_CallMusicextras *jobmusicextras = new Job_CallMusicextras(*track, (track == sortlist->getLast()) ? true : false);
      jobman->addJob(jobmusicextras);
    }
    delete sortlist;
  }
}

void GUI::addMextraJobToStatusBar ()
{
  if(stackStatus6->id(stackStatus6->visibleWidget()) == 1) {
    // progress bar not visible
    mextrasProgress->setTotalSteps(0);
    mextrasProgress->setProgress(0);
    // show progress bar
    stackStatus6->raiseWidget(2);
  }
  mextrasProgress->setTotalSteps(mextrasProgress->totalSteps() + 1);
}
 
void GUI::reportMextraJobDoneToStatusBar ()
{
  mextrasProgress->setProgress(mextrasProgress->progress() + 1);
  if(mextrasProgress->progress() == mextrasProgress->totalSteps()) {
    // hide progress bar
    stackStatus6->raiseWidget(1);
  }
}
#endif /* HAVE_MEXTRAS */

void GUI::customEvent( QCustomEvent * e )
{
    if(e->type() == MESSAGEEVENT_CODE) {
        MessageEvent* me = static_cast<MessageEvent*>( e );
        QMessageBox::information(this, _("Message"), me->message(), QMessageBox::Ok);
#ifdef HAVE_MEXTRAS
    } else if(e->type() == NEWMUSICEXTRASJOBEVENT_CODE) {
        addMextraJobToStatusBar();
    } else if(e->type() == MUSICEXTRASJOBDONEEVENT_CODE) {
        reportMextraJobDoneToStatusBar();
    } else if(e->type() == ALLMUSICEXTRASJOBSDONEEVENT_CODE) {
        stackStatus6->raiseWidget(1);
#endif /* HAVE_MEXTRAS */
    }
}

GUI::~GUI() {
}
