/*-----------------------------------------------------------------------
QMBTagger - Qt Based MusicBrainz MP3/OGG/FLAC Tagger
Copyright (C) 2003,2004 Ewen Cheslack-Postava

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-----------------------------------------------------------------------*/

#include "qmbvorbisfile.h"

#if HAVE_LIBMUSICBRAINZ
#if HAVE_OGG

#include "qmbfile.h"
#include <stdio.h>
#include <qapplication.h>
#include <qstring.h>
#include <musicbrainz/mb_c.h>
#include <vcedit.h>

QMBVorbisFile::QMBVorbisFile()
	: QMBFile()
{
	//we do nothing here.  we want to allow checking of return value,
	//so we only allow class to function through calling of Extract()
}

//get the data we want from the vorbis stream
//heavily borrowed code from vorbis doc on decoding vorbis stream
//returns:
//	0  indicates success
//	-1 indicates failed to open
//	-2 indicates invalid info
//	1 indicates success, but metadata extraction failed
int QMBVorbisFile::Extract(QString filename) {
	
	FILE* file = NULL;
	OggVorbis_File vf;
	int eof=0;
	int current_section;
	int ret = 0;

	//record the full filename so we will be able to use it for output
	fullfilename = filename;

	//set the filename in the metadata
	int lastslash = filename.findRev('/');
	metadata.SetFileName(filename.right(filename.length() - lastslash - 1));

	//open the file
	file = fopen(filename.latin1(), "r");

	//check that the file is in fact vorbis
	if(ov_open(file, &vf, NULL, 0) < 0) {
		qWarning("QMBVorbisFile::Extract(): ov_open failed: File (%s) must not be proper Ogg Vorbis file.", fullfilename.local8Bit().data());
		return -1;
	}

	//get metadata
	if (GetMetadata(&vf) == -1) {
		//getting metadata failed - very strange! indicate with return val
		qWarning("QMBVorbisFile::Extract(): GetMetadata failed: File (%s) must be corrupted.", fullfilename.local8Bit().data());
		ret = 1;
	}

	//if the trmid was read in from the file, quit now
	if (!metadata.GetTRMID().isEmpty()) {
		ov_clear(&vf);//ov_clear closes stream AND FILE, no close(file) necessary
		return 0;
	}
	
	//get the file's info
	vorbis_info *vi = ov_info(&vf, -1);
	channels = vi->channels;
	sampleRate = vi->rate;
	bits_per_sample = 16; //we decode to 16 bit samples, so we can keep this constant
	
	//get the length of the song
	songlength = ov_time_total(&vf, -1);
	//set this information in the metadata, but in milliseconds instead of seconds
	metadata.SetDuration(QString::number(songlength*1000));

	//if the info is invalid, return without doing anything
	if (channels == 0 || sampleRate == 0) {
		qWarning("QMBVorbisFile::Extract(): Channels (%d) or SampleRate (%ld) not set properly for file (%s).", channels, sampleRate, fullfilename.local8Bit().data());
		ov_clear(&vf);
		return -2;
	}
	//figure out buffer length, we want 30 seconds of data,
	//get rate and channels from vorbis file, and
	//specify 16 bit (2 byte) samples
	//= channels*30 sec*rate*2bytes
	bufferlen = 30*channels*sampleRate*bits_per_sample/8;

	//create the actual buffer
	buffer = new char[bufferlen];

	//now lets get the data from the vorbis file
	long index = 0; //where we are in the buffer
	while (!eof && index < bufferlen) {
		long ret = ov_read(&vf, &buffer[index], bufferlen-index, 0, 2, 1, &current_section);
		if (ret == 0) {
			//eof
			eof = 1;
		}
		else if (ret < 0) {
			//error in stream, what to do?
			qWarning("QMBVorbisFile::Extract(): Error in stream found while extracting data from file (%s). Ignoring...", fullfilename.local8Bit().data());
		}
		else {
			//successful read, ret contains the number of bytes read, increase pointer
			//and continue reading
			index += ret;
		}
	}

	//we're done reading, now clean up
	//ov_clear closes stream AND FILE, no close(file) necessary
	ov_clear(&vf);

	return ret;
}

int QMBVorbisFile::GetMetadata(OggVorbis_File* vf) {
	vorbis_comment* vc = ov_comment(vf, -1);

	if(vc == NULL) {
		qWarning("QMBVorbisFile::GetMetadata(): Error getting vorbis_comment on file (%s).", fullfilename.local8Bit().data());
		return -1;
	}

	for(int i = 0; i < vc->comments; i++) {
		QString thiscomment(vc->user_comments[i]);
		if (thiscomment.left(6).upper() == "TITLE=") {
			metadata.SetTrackName(thiscomment.right(thiscomment.length() - 6));
		} else if (thiscomment.left(6).upper() == "ALBUM=") {
			metadata.SetAlbumName(thiscomment.right(thiscomment.length() - 6));
		} else if (thiscomment.left(7).upper() == "ARTIST=") {
			metadata.SetArtistName(thiscomment.right(thiscomment.length() - 7));
		} else if (thiscomment.left(12).upper() == "BER=") {
			metadata.SetTrackNumber(thiscomment.right(thiscomment.length() - 12));
		} else if (thiscomment.left(20).upper() == "MUSICBRAINZ_TRACKID=") {
			metadata.SetTrackID(thiscomment.right(thiscomment.length() - 20));
		} else if (thiscomment.left(21).upper() == "MUSICBRAINZ_ARTISTID=") {
			metadata.SetArtistID(thiscomment.right(thiscomment.length() - 21));
		} else if (thiscomment.left(20).upper() == "MUSICBRAINZ_ALBUMID=") {
			metadata.SetAlbumID(thiscomment.right(thiscomment.length() - 20));
		} else if (thiscomment.left(18).upper() == "MUSICBRAINZ_TRMID=") {
			metadata.SetTRMID(thiscomment.right(thiscomment.length() - 18));
		}
	}

	return 0;
}

#endif /* HAVE_OGG */
#endif /*HAVE_LIBMUSICBRAINZ*/
