/*
 * drivers/char/sparc/rconsole.c
 *
 * Member functions of class raster do not check their argumets
 * for the speed. A better design is to have this behaviour configured
 * between rconsole.S (fast) and rconsole.c (safe). --P3  XXX
 */

#include "rconsole.h"

/*
 * lat7-1 README
 * A thunder-like voice: "A copyright? Who said \"copyright\"?"
 *
 * Once upon a time I had a home PDP-11. Its terminal was unusable
 * because its PROM was in russian. The terminal was made in Poland
 * and its case was made of wood. It was DEC VT-52 compatible. Polish
 * mother called it MERA-6052. I designed a font for the terminal and
 * called it lat7. My friend could burn the PROM one time only. I ran
 * BSD 2.9 on it hapily. But later I understood that I mistook: 'g' was
 * one line lower and 'i' was unreadable. I fixed these mistakes for
 * Linux/SPARC. Since that lat7-1 exists.
 *
 *	Signed:	September 8, 1995
 *		Pete <Zaitcev@ipmce.su>
 *
 * Update on January 14, 1999: Apparently lat7-1 was lost during my
 * move to California. Only plain lat7 survived. 
 * I recreated lat7-1 changes in lat7-2.  --zaitcev
 */
#include "lat7_2.bm"	/* lat7_1.bm */
#define LAT7_NCHARS  128
#define LAT7_HEIGHT   11
#define LAT7_WIDTH     8

static Rf_scan lat7_body[ LAT7_NCHARS*LAT7_HEIGHT ];

#if 1
/*
 * Mode one: use physical address and sun4m bypass.
 * This is resilient to MMU screwup, saves 3/4 MB of address space.
 * Cannot be used on machines with 36 bits buses.
 */
#ifndef ASI_M_BYPASS
#define ASI_M_BYPASS 0x20
#endif

static __inline__ void stfb_w(void *ptr, unsigned int data) {
	__asm__ __volatile__ ("sta %0, [%1] %2" : :
				"r" (data), "r" (ptr), "i" (ASI_M_BYPASS));
}

static __inline__ void stfb_b(void *ptr, unsigned int data) {
	__asm__ __volatile__ ("stba %0, [%1] %2" : :
				"r" (data), "r" (ptr), "i" (ASI_M_BYPASS));
}

static __inline__ unsigned int ldfb_w(void *ptr) {
	unsigned int data;
	__asm__ __volatile__ ("lda [%1] %2, %0" :
				"=r" (data) :
				"r" (ptr), "i" (ASI_M_BYPASS));
	return data;
}

static __inline__ unsigned int ldfb_b(void *ptr) {
	unsigned int data;
	__asm__ __volatile__ ("lduba [%1] %2, %0" :
				"=r" (data) :
				"r" (ptr), "i" (ASI_M_BYPASS));
	return data;
}

#else
/*
 * Mode two: traditional loads and stores at a virtual address.
 */

static __inline__ void stfb_w(void *ptr, unsigned int data) {
	*(unsigned int *)ptr = data;
}

static __inline__ void stfb_b(void *ptr, unsigned int data) {
	*(unsigned char *)ptr = (unsigned char)data;
}

static __inline__ unsigned int ldfb_w(void *ptr) {
	return *(unsigned int *)ptr;
}

static __inline__ unsigned int ldfb_b(void *ptr) {
	return *(unsigned char *)ptr;
}

#endif

static inline int swapbits(int w0)
{
  int w1 = 0;
  int i;
  for (i = 0; i < 8; i++) {
    w1 <<= 1;
    w1 = w1 | (w0 & 1);
    w0 >>= 1;
  }
  return w1;
}

void font_cons_7(struct rfont *p)
{
  int x;
  int col = 0;
  int row = 0;
  int erow = 0;
  for (x = 0; x < LAT7_NCHARS*LAT7_HEIGHT; x++ ) {
    lat7_body[ (erow * lat7_2_width/8 + col) * LAT7_HEIGHT + row ] =
                                            swapbits(lat7_2_bits[x]) & 0xFF;
    if (++col >= lat7_2_width/8) {
      col = 0;
      if (++row >= LAT7_HEIGHT) {
        row = 0;
        ++erow;
      }
    }
  }
  p->body_ = lat7_body;
  p->nchars_ = LAT7_NCHARS;
  p->width_ = LAT7_WIDTH;
  p->height_ = LAT7_HEIGHT;
}

void raster1_cons_a(struct raster *r, int height, int width, char *base)
{
  r->depth_ = 1;
  r->width_ = width;
  r->height_ = height;
  r->base_ = base;
  r->loff_ = 0;
  r->lsize_ = width/8;             /* XXX lsize_ must be in pixels. */
  r->clear_ = raster1_clear;
  r->yscroll_ = raster1_yscroll;
  r->xscroll_ = 0;
  r->render_ = raster1_render;
}

void raster8_cons_a(struct raster *r, int height, int width, char *base)
{
  r->depth_ = 8;
  r->width_ = width;
  r->height_ = height;

  r->base_ = base;

  r->loff_ = 0;
  r->lsize_ = width;
  r->clear_ = raster8_clear;
  r->yscroll_ = raster8_yscroll;
  r->xscroll_ = 0;
  r->render_ = raster8_render;
}

void raster_cons_2(
  struct raster *r, const struct raster *p,
  int y, int x,			/* Relative to the parent */
  int height, int width)
{
  r->depth_ = p->depth_;
  r->width_ = width;
  r->height_ = height;
  r->base_ = (char*)p->base_ + p->lsize_ * y;
  r->loff_ = p->loff_ + x;
  r->lsize_ = p->lsize_;
  r->clear_ = p->clear_;
  r->yscroll_ = p->yscroll_;
  r->render_ = p->render_;
}

void raster_dest(struct raster *r)
{
}

/*
 * class input {
 *   int scan_;
 *   int pix_;
 *   
 *   void setscan(int scan);
 *   int get(void) { xxxxxxxx; };
 * };
 *
 */

struct input {
  int scan_;
  int pix_;
  const char *line_;
  int cx_, length_;
  const struct rfont *f_;
  int font_width_;            /* cache of f_->width_ */
  int fl_;
};

static void input_setscan(
  struct input *t,
  const struct rfont *f,
  const char *line, int length,
  int scan)
{
  t->scan_ = scan;
  t->pix_ = 0;
  t->line_ = line;  t->length_ = length;  t->cx_ = 1;
  t->f_ = f;  t->font_width_ = f->width_;
  t->fl_ = f->body_[ (line[0] % t->f_->nchars_) * t->f_->height_ + scan ];
}

static inline int input_get(struct input *t) {
  if (t->pix_ >= t->font_width_) {
    if (t->cx_ >= t->length_) return -1;
    t->fl_ = t->f_->body_[ (t->line_[ t->cx_++ ] % t->f_->nchars_) * t->f_->height_ + t->scan_ ];
    t->pix_ = 0;
  }
  return (t->fl_ >> ((t->font_width_-1) - t->pix_++)) & 1;
}

static void render8_setscan(
  struct raster *r,
  int ybase, int xbase,
  int scan)
{
  r->outp_ = (unsigned *)((char*)r->base_ + r->lsize_*(scan + ybase) + r->loff_ + xbase);
  r->ol_ = 0;
  r->sx_ = 24 - 8 * ((unsigned)r->outp_ & 0x03);
  if (r->sx_ != 24) {
    r->outp_ = (unsigned *) ((unsigned)r->outp_ & ~0x03);
    r->ol_ = ldfb_w(r->outp_) & 0xFFFFFF00 << r->sx_;
  }
}

static inline void render8_put(struct raster *r, int pixel) {
  r->ol_ |= (pixel & 0xFF) << r->sx_;
  r->sx_ -= 8;
  if (r->sx_ < 0) {
    stfb_w(r->outp_++, r->ol_);
    r->sx_ = 24;
    r->ol_ = 0;
  }
}

static void render8_flush(struct raster *r) {
  if (r->sx_ != 24) {
    stfb_w(r->outp_, r->ol_ | (ldfb_w(r->outp_) & ~(0xFFFFFF00 << r->sx_)));
  }
}

void raster8_render(
  struct raster *r,
  int ybase,
  int xbase,
  const char* str,
  int len,
  RC_color bg,
  RC_color fg,
  const struct rfont *f)
{
  int y;
  struct input inp;

  for (y = 0; y < f->height_; y++) {
    input_setscan(&inp, f, str, len, y);
    render8_setscan(r, ybase, xbase, y);
    for (;;) {
      int pixel;
      if ((pixel = input_get(&inp)) == -1) break;
      render8_put(r, pixel? fg: bg);
    }
    render8_flush(r);
  }
}

static void render1_setscan(
  struct raster *r,
  int ybase, int xbase,
  int scan)
{
  r->outp_ = (unsigned *)((char*)r->base_ + r->lsize_*(scan + ybase) + (r->loff_ + xbase)/8);
  r->ol_ = 0;
  r->sx_ = 31 - ((((unsigned)r->outp_&0x03) << 3) | ((r->loff_ + xbase)&0x07));
  if (r->sx_ != 31) {
    r->outp_ = (unsigned *) ((unsigned)r->outp_ & ~0x03);
    r->ol_ = ldfb_w(r->outp_) & 0xFFFFFFFE << r->sx_;
  }
}

static inline void render1_put(struct raster *r, int pixel) {
  r->ol_ |= (pixel & 1) << r->sx_;
  r->sx_ -= 1;
  if (r->sx_ < 0) {
    stfb_w(r->outp_++, r->ol_);
    r->sx_ = 31;
    r->ol_ = 0;
  }
}

static void render1_flush(struct raster *r) {
  if (r->sx_ != 31) {
    stfb_w(r->outp_, r->ol_ | (ldfb_w(r->outp_) & ~(0xFFFFFFFE << r->sx_)));
  }
}

void raster1_render(
  struct raster *r,
  int ybase,
  int xbase,
  const char* str,
  int len,
  RC_color bg,
  RC_color fg,
  const struct rfont *f)
{
  int y;
  struct input inp;

  for (y = 0; y < f->height_; y++) {
    input_setscan(&inp, f, str, len, y);
    render1_setscan(r, ybase, xbase, y);
    for (;;) {
      int pixel;
      if ((pixel = input_get(&inp)) == -1) break;
      render1_put(r, pixel? fg: bg);
    }
    render1_flush(r);
  }
}

void raster1_clear(struct raster *r, int y, int x, int h, int w, RC_color c)
{
  unsigned c32;
  int iy, ix;
  unsigned *p;

  c32 = (((int) c) << 31) >> 31;
  for (iy = 0; iy < h; iy++) {

    ix = (r->lsize_*(iy + y))*8 + r->loff_+x;
    p = (unsigned*)r->base_ + ix/32;
    ix = (32 - (ix & 0x1F)) & 0x1F;

    if (ix != 0) {
      unsigned m = 0xFFFFFFFF << ix;
      if (ix > w) {
        m |= ~(0xFFFFFFFF << (ix - w));
      }
      stfb_w(p, (ldfb_w(p) & m) | (c32 & ~m));
      p++;
    }
    while (ix + 32 < w) {
      stfb_w(p++, c32);
      ix += 32;
    }
    if (ix < w) {
      unsigned m = 0xFFFFFFFF << (ix - w);
      stfb_w(p, (ldfb_w(p) & ~m) | (c32 & m));
    }
  }
}

void raster8_clear(struct raster *r, int y, int x, int h, int w, RC_color c)
{
  unsigned c4;
  int iy, ix;
  unsigned *p;

  c &= 0xFF;
  c4 = (c << 24) | (c << 16) | (c << 8) | c;
  for (iy = 0; iy < h; iy++) {
    p = (unsigned *)((char*)r->base_ + r->lsize_*(iy + y) + r->loff_+x);
    ix = (4 - ((unsigned)p & 0x03)) & 0x03;
    p = (unsigned *) ((unsigned)p & ~0x03);
    if (ix != 0) {
      unsigned m = 0xFFFFFFFF << (ix << 3);
      if (ix > w) {
        m |= ~(0xFFFFFFFF << ((ix - w) << 3));
      }
      stfb_w(p, (ldfb_w(p) & m) | (c4 & ~m));
      p++;
    }
    while (ix + 4 < w) {
      stfb_w(p++, c4);
      ix += 4;
    }
    if (ix < w) {
      unsigned m = 0xFFFFFFFF << ((ix - w) << 3);
      stfb_w(p, (ldfb_w(p) & ~m) | (c4 & m));
    }
  }
}

void raster1_yscroll(struct raster *r, int y, int x, int h, int w, int dy)
{
  int iy;
  int ix_0;
  unsigned *p1_0, *p2_0;
  int pstep;

  if (dy < 0) {
    dy = -dy;
    ix_0 = (r->lsize_*y)*8 + r->loff_+x;
    p1_0 = (unsigned *)r->base_                    + ix_0/32;
    p2_0 = (unsigned *)r->base_ + (r->lsize_*dy)/4 + ix_0/32;
    pstep = r->lsize_ / 4;
  } else {
    ix_0 = (r->lsize_*(y + h-1))*8 + r->loff_+x;
    p1_0 = (unsigned *)r->base_                    + ix_0/32;
    p2_0 = (unsigned *)r->base_ - (r->lsize_*dy)/4 + ix_0/32;
    pstep = -r->lsize_ / 4;
  }
  ix_0 = (32 - (ix_0 & 0x1F)) & 0x1F;

  for (iy = 0; iy < h - dy; iy++) {
    int ix = ix_0;
    unsigned *p1 = p1_0;
    unsigned *p2 = p2_0;

    if (ix != 0) {
      unsigned m = 0xFFFFFFFF << ix;
      if (ix > w) {
        m |= ~(0xFFFFFFFF << (ix - w));
      }
      stfb_w(p1, (ldfb_w(p1) & m) | (ldfb_w(p2) & ~m));
      p1++;
      p2++;
    }
    while (ix + 32 < w) {
      stfb_w(p1++, ldfb_w(p2++));
      ix += 32;
    }
    if (ix < w) {
      unsigned m = 0xFFFFFFFF << (ix - w);
      stfb_w(p1, (ldfb_w(p1) & ~m) | (ldfb_w(p2) & m));
    }

    p1_0 += pstep;
    p2_0 += pstep;
  }
}

void raster8_yscroll(struct raster *r, int y, int x, int h, int w, int dy)
{
  int iy;
  int ix_0;
  unsigned *p1_0, *p2_0;
  int pstep;

  if (dy < 0) {
    dy = -dy;
    p1_0 = r->base_ + r->lsize_*(y + 0)  + r->loff_+x;
    p2_0 = r->base_ + r->lsize_*(y + dy) + r->loff_+x;
    pstep = r->lsize_ / sizeof(unsigned);
  } else {
    p1_0 = r->base_ + r->lsize_*(y + h      - 1) + r->loff_+x;
    p2_0 = r->base_ + r->lsize_*(y + h - dy - 1) + r->loff_+x;
    pstep = -r->lsize_ / sizeof(unsigned);
  }
  ix_0 = (4 - ((unsigned)p1_0 & 0x03)) & 0x03;  /* Assume (lsize_ & 3) == 0 */
  p1_0 = (unsigned *) ((unsigned)p1_0 & ~0x03);
  p2_0 = (unsigned *) ((unsigned)p2_0 & ~0x03);

  for (iy = 0; iy < h - dy; iy++) {
    int ix = ix_0;
    unsigned *p1 = p1_0;
    unsigned *p2 = p2_0;

    if (ix != 0) {
      unsigned m = 0xFFFFFFFF << (ix << 3);
      if (ix > w) {
        m |= ~(0xFFFFFFFF << ((ix - w) << 3));
      }
      stfb_w(p1, (ldfb_w(p1) & m) | (ldfb_w(p2) & ~m));
      p1++;
      p2++;
    }
#if 0
    while (ix + 16 < w) {
      p1[0] = p2[0];
      p1[1] = p2[1];
      p1[2] = p2[2];
      p1[3] = p2[3];
      p1 += 4;    p2 += 4;
      ix += 16;
    }
#else
    while (ix + 16 < w) {
      unsigned a0, a1, a2, a3;
      a0 = ldfb_w(&p2[0]);  a1 = ldfb_w(&p2[1]);  a2 = ldfb_w(&p2[2]);  a3 = ldfb_w(&p2[3]);
      stfb_w(&p1[0], a0);  stfb_w(&p1[1], a1);  stfb_w(&p1[2], a2);  stfb_w(&p1[3], a3);
      p1 += 4;    p2 += 4;
      ix += 16;
    }
#endif
    while (ix + 4 < w) {
      stfb_w(p1++, ldfb_w(p2++));
      ix += 4;
    }
    if (ix < w) {
      unsigned m = 0xFFFFFFFF << ((ix - w) << 3);
      stfb_w(p1, (ldfb_w(p1) & ~m) | (ldfb_w(p2) & m));
    }

    p1_0 += pstep;
    p2_0 += pstep;
  }
}
