/* Test of getcwd() function.
   Copyright (C) 2009-2011 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.  */

#include <config.h>

#include <unistd.h>

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "pathmax.h"
#include "macros.h"

#if ! HAVE_GETPAGESIZE
# define getpagesize() 0
#endif

/* This size is chosen to be larger than PATH_MAX (4k), yet smaller than
   the 16kB pagesize on ia64 linux.  Those conditions make the code below
   trigger a bug in glibc's getcwd implementation before 2.4.90-10.  */
#define TARGET_LEN (5 * 1024)

/* Keep this test in sync with m4/getcwd-abort-bug.m4.  */
static int
test_abort_bug (void)
{
  char const *dir_name = "confdir-14B---";
  char *cwd;
  size_t initial_cwd_len;
  int fail = 0;
  size_t desired_depth;
  size_t d;

#ifdef PATH_MAX
  /* The bug is triggered when PATH_MAX < getpagesize (), so skip
     this relatively expensive and invasive test if that's not true.  */
  if (getpagesize () <= PATH_MAX)
    return 0;
#endif

  cwd = getcwd (NULL, 0);
  if (cwd == NULL)
    return 2;

  initial_cwd_len = strlen (cwd);
  free (cwd);
  desired_depth = ((TARGET_LEN - 1 - initial_cwd_len)
                   / (1 + strlen (dir_name)));
  for (d = 0; d < desired_depth; d++)
    {
      if (mkdir (dir_name, S_IRWXU) < 0 || chdir (dir_name) < 0)
        {
          if (! (errno == ERANGE || errno == ENAMETOOLONG || errno == ENOENT))
            fail = 3; /* Unable to construct deep hierarchy.  */
          break;
        }
    }

  /* If libc has the bug in question, this invocation of getcwd
     results in a failed assertion.  */
  cwd = getcwd (NULL, 0);
  if (cwd == NULL)
    fail = 4; /* getcwd didn't assert, but it failed for a long name
                 where the answer could have been learned.  */
  free (cwd);

  /* Call rmdir first, in case the above chdir failed.  */
  rmdir (dir_name);
  while (0 < d--)
    {
      if (chdir ("..") < 0)
        {
          fail = 5;
          break;
        }
      rmdir (dir_name);
    }

  return fail;
}

/* The length of this name must be 8.  */
#define DIR_NAME "confdir3"
#define DIR_NAME_LEN 8
#define DIR_NAME_SIZE (DIR_NAME_LEN + 1)

/* The length of "../".  */
#define DOTDOTSLASH_LEN 3

/* Leftover bytes in the buffer, to work around library or OS bugs.  */
#define BUF_SLOP 20

/* Keep this test in sync with m4/getcwd-path-max.m4.  */
static int
test_long_name (void)
{
#ifndef PATH_MAX
  /* The Hurd doesn't define this, so getcwd can't exhibit the bug --
     at least not on a local file system.  And if we were to start worrying
     about remote file systems, we'd have to enable the wrapper function
     all of the time, just to be safe.  That's not worth the cost.  */
  return 0;
#elif ((INT_MAX / (DIR_NAME_SIZE / DOTDOTSLASH_LEN + 1) \
        - DIR_NAME_SIZE - BUF_SLOP) \
       <= PATH_MAX)
  /* FIXME: Assuming there's a system for which this is true,
     this should be done in a compile test.  */
  return 0;
#else
  char buf[PATH_MAX * (DIR_NAME_SIZE / DOTDOTSLASH_LEN + 1)
           + DIR_NAME_SIZE + BUF_SLOP];
  char *cwd = getcwd (buf, PATH_MAX);
  size_t initial_cwd_len;
  size_t cwd_len;
  int fail = 0;
  size_t n_chdirs = 0;

  if (cwd == NULL)
    return 1;

  cwd_len = initial_cwd_len = strlen (cwd);

  while (1)
    {
      size_t dotdot_max = PATH_MAX * (DIR_NAME_SIZE / DOTDOTSLASH_LEN);
      char *c = NULL;

      cwd_len += DIR_NAME_SIZE;
      /* If mkdir or chdir fails, it could be that this system cannot create
         any file with an absolute name longer than PATH_MAX, such as cygwin.
         If so, leave fail as 0, because the current working directory can't
         be too long for getcwd if it can't even be created.  For other
         errors, be pessimistic and consider that as a failure, too.  */
      if (mkdir (DIR_NAME, S_IRWXU) < 0 || chdir (DIR_NAME) < 0)
        {
          if (! (errno == ERANGE || errno == ENAMETOOLONG || errno == ENOENT))
            fail = 2;
          break;
        }

      if (PATH_MAX <= cwd_len && cwd_len < PATH_MAX + DIR_NAME_SIZE)
        {
          c = getcwd (buf, PATH_MAX);
          if (!c && errno == ENOENT)
            {
              fail = 3;
              break;
            }
          if (c)
            {
              fail = 4;
              break;
            }
          if (! (errno == ERANGE || errno == ENAMETOOLONG))
            {
              fail = 5;
              break;
            }
        }

      if (dotdot_max <= cwd_len - initial_cwd_len)
        {
          if (dotdot_max + DIR_NAME_SIZE < cwd_len - initial_cwd_len)
            break;
          c = getcwd (buf, cwd_len + 1);
          if (!c)
            {
              if (! (errno == ERANGE || errno == ENOENT
                     || errno == ENAMETOOLONG))
                {
                  fail = 6;
                  break;
                }
              if (AT_FDCWD || errno == ERANGE || errno == ENOENT)
                {
                  fail = 7;
                  break;
                }
            }
        }

      if (c && strlen (c) != cwd_len)
        {
          fail = 8;
          break;
        }
      ++n_chdirs;
    }

  /* Leaving behind such a deep directory is not polite.
     So clean up here, right away, even though the driving
     shell script would also clean up.  */
  {
    size_t i;

    /* Try rmdir first, in case the chdir failed.  */
    rmdir (DIR_NAME);
    for (i = 0; i <= n_chdirs; i++)
      {
        if (chdir ("..") < 0)
          break;
        if (rmdir (DIR_NAME) != 0)
          break;
      }
  }

  return fail;
#endif
}

int
main (int argc, char **argv)
{
  return test_abort_bug () * 10 + test_long_name ();
}
