/* *************************************************************************
  Module:        fixed.c
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000
  Description:
                 Various routines relating to fixed settings and 
                 commands for specific printers, used in "output
                 only" mode (non readback / non dynamic). Also
                 contains routines related to setting Pup's preferences.

****************************************************************************
                 COPYRIGHT (C) 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
**************************************************************************** */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>    /* for extern int errno */
#include <gtk/gtk.h>
#include "lexgui.h"
#include "fixed.h"

/* -------------------------------------------------------------------------
        init_printer_name() 
   ------------------------------------------------------------------------- */
void init_printer_name(prefs_struct *prefs)
{
  strcpy(prefs->printer, printer_name[0]);
}
/* -------------------------------------------------------------------------
         set_query_fixed() For fixed-- packs the query with predefined
                          variables instead of dynamically querying.
   ------------------------------------------------------------------------- */
void set_query_fixed(topwin_struct *top, io_struct *fio)
{
  fio->query = pv[top->prefs.index];
  fio->j = strlen(pv[top->prefs.index]);
  /* Note top->prefs.index, which is set in set_printer_index(), is based
     on the index of a printer definition in printer_name[]. Consequently if
     there are less than this number of defined sets in *pv[], a crash will
     occur. */
}
/* -------------------------------------------------------------------------
        get_family_command_str()
   ------------------------------------------------------------------------- */
void get_family_command_str(gchar **ptr, gchar *cstring)
{
  static gchar nothing[] = "\0";
  if(!strcmp(cstring, "NOTAVAIL"))
    *ptr = nothing;
  else
    *ptr = cstring;
}
/* -------------------------------------------------------------------------
        get_family_command() Sets command to use based on command and
                             family. Command tells which array in fixed.h 
                             to use, and family tells which index in that
                             array to use.
   ------------------------------------------------------------------------- */
gchar *get_family_command(gint command, gint family, msgbox_struct *msgbox)
{
  gchar *ret_ptr;

  switch(command)
  {
    case 0: /* Menu/Settings/Configuration Page */
      get_family_command_str(&ret_ptr, pm[family]);
      break;
    case 1: /* PS Font Page */
      get_family_command_str(&ret_ptr, pps[family]);
      break;
    case 2: /* PCL Font Page */
      get_family_command_str(&ret_ptr, ppcl[family]);
      break;
    case 3: /* Demo Page */
      get_family_command_str(&ret_ptr, pdemo[family]);
      break;

    #ifdef EXAMPLE_TO_ADD_ANOTHER_COMMAND
    case 4:
      get_family_command_str(&ret_ptr, pnew_command[family]);
      break;
    #endif

    default:
      get_family_command_str(&ret_ptr, "NOTAVAIL");
  }

  if(!strcmp(ret_ptr, "\0"))
  {
    if(command == 0 && family == 2) /* settings page && HP 4000 family */
      put_msg(msgbox, "Command not available for this printer !", RED, 13);
    else
      put_msg(msgbox, "Command not available for this printer !", RED, 0);
  }
  return(ret_ptr);
}
/* -------------------------------------------------------------------------
        set_printer_family() Sets family according to index in
                             *printer_name[] (and  *pv[]). See note
                             in fixed.h under Family Specific Commands.
   ------------------------------------------------------------------------- */
void set_printer_family(prefs_struct *prefs)
{
  switch(prefs->index)
  {
    case 0: /* Lexmark Optra 40/45 Printer */
      prefs->family = 0; /* Lexmark family */
      break;
    case 1: /* Lexmark Optra E310 */
      prefs->family = 0; /* Lexmark family */
      break;
    case 2: /* HP 2100M Printer */
      prefs->family = 1; /* HP family */
      break;
    case 3: /* HP 4000 Printer */
      prefs->family = 2;
      break;
    case 4: /* HP LJ 4 Plus Printer */
      prefs->family = 2;
      break;

    #ifdef EXAMPLE_TO_ADD_ANOTHER_PRINTER
    case 5: /* Your printer */
      /* you could use one of the existing families like prefs->family = 1,
         or you could add a new family in fixed.h like in this example. */
      prefs->family = 3; /* Your family */
      break;
    #endif
    default:
      prefs->family = 2; /* Default to family with NOTAVAIL for all */
  }
}
/* -------------------------------------------------------------------------
        set_sensitize_b() Sensitizes or Insensitizes specified items.
   ------------------------------------------------------------------------- */
void set_sensitize_b(topwin_struct *top)
{
  gint i;
  for(i = 0; i < NUMPAGE; i++)
  {
    if(top->prefs.t_inhibit[i])
      gtk_widget_set_sensitive(top->bt.table[i], FALSE);
    /* we let set_sensitize() handle setting uninhibited page 
       tables to sensitive */ 
  }

  for(i = 0; i < NUMBUT; i++)
  {
    /* First we sensitize all buttons to get the ones insensitive */
    gtk_widget_set_sensitive(top->bt.button[i], TRUE);
    gtk_widget_set_sensitive(top->bt.blabel[i], TRUE);

    /* Now we insensitize the ones flagged for inhibit */
    if(top->prefs.b_inhibit[i])
    {
      gtk_widget_set_sensitive(top->bt.button[i], FALSE);
      gtk_widget_set_sensitive(top->bt.blabel[i], FALSE);
    }
  }
}
/* -------------------------------------------------------------------------
        set_inhibit() Inhibit setting the sensitivity of a top notebook's
                      table or button. Used to inhibit something when it 
                      is not applicable to the selected printer.
   ------------------------------------------------------------------------- */
void set_inhibit(topwin_struct *top)
{
  init_inhibit(&(top->prefs)); /* Set all flags to zero */
  switch(top->prefs.index)
  {
    case 0: /* Lexmark Optra 40/45 Printer */
      top->prefs.b_inhibit[14] = 1; /* Print demo button */
      break;
    case 1: /* Lexmark Optra E310 */
      top->prefs.b_inhibit[14] = 1; /* Print demo button */
      top->prefs.t_inhibit[2] = 1; /* Cartridge Maintenance page */
      break;
    case 2: /* HP 2100M Printer */
      top->prefs.t_inhibit[2] = 1; /* Cartridge Maintenance page */
      break;
    case 3: case 4: /* HP 4000 and LJ4 Plus Printers */
      top->prefs.b_inhibit[11] = 1; /* PS Font button */
      top->prefs.b_inhibit[12] = 1; /* PCL Font button */
      top->prefs.b_inhibit[13] = 1; /* Settings page button */
      top->prefs.b_inhibit[14] = 1; /* Print demo button */
      top->prefs.t_inhibit[2] = 1; /* Cartridge Maintenance page */
      break;
  }
  set_sensitize_b(top);
}
/* -------------------------------------------------------------------------
        set_printer_index()
   ------------------------------------------------------------------------- */
void set_printer_index(topwin_struct *top)
{
  gint i;

  i = 0;
  while(strcmp(printer_name[i], "end"))
  {
    if(!strcmp(printer_name[i], top->prefs.printer))
    {
      top->prefs.index = i;
      set_printer_family(&(top->prefs));
      return;
    }
    i++;
  }
  /* The above should always be found because it has already been through
     check_printer_name(), which already defaulted it if not found. */
}
/* -------------------------------------------------------------------------
        check_printer_name() Used by read_prefs() to check for valid name.
                             If not found, read_prefs() will not set
                             prefs.printer and thus it will have the
                             default printer_name[0], which was set at
                             initilization time by init_prefs().
   ------------------------------------------------------------------------- */
gint check_printer_name(gchar *name)
{
  gint i;

  i = 0;
  while(strcmp(printer_name[i], "end"))
  {
    if(!strcmp(printer_name[i], name))
      return(1);
    i++;
  }
  pop_pref_note(2);
  return(0);
}
/* -------------------------------------------------------------------------
        combo_CB()
   ------------------------------------------------------------------------- */
void combo_CB(GtkWidget *widget, topwin_struct *top)
{
  strcpy(top->prefs.printer, gtk_entry_get_text(GTK_ENTRY(widget)));
  gtk_label_set_text(GTK_LABEL(top->printerlabel), top->prefs.printer);
  set_printer_index(top);
  set_inhibit(top);
}
/* -------------------------------------------------------------------------
         read_prefs() Opens & reads prefs file if it exists. 
   ------------------------------------------------------------------------- */
void read_prefs(topwin_struct *top)
{
  gint j;
  prefs_struct *prefs;
  extern gchar *output;
  FILE *fprc;
  gchar filename[MAXLINELENGTH];
  gchar in_line[MAXLINELENGTH];
  gchar temp_line[MAXLINELENGTH];
  gchar keyword[MAXLINELENGTH];
  gchar value_str[MAXLINELENGTH];
  gchar *sptr;
  extern int errno; /* in errno.h */
  extern int open_file_errno;

  prefs = &(top->prefs);
  open_file_errno = 0;

  if(getenv("HOME"))
  {
    strcpy(filename, getenv("HOME"));
    strcat(filename, "/.puprc");
    if((fprc = fopen(filename, "r")) == NULL)
    {
      open_file_errno = errno;
      /* if file does not exist (ENOENT), just use the defaults; otherwise
         if there's a different error then report it as a warning, but still
         use the defaults. */
      if(open_file_errno != ENOENT)
        pop_pref_note(1);
    }
    else
    {
      while(fgets(in_line, MAXLINELENGTH, fprc) != NULL)
      {
        keyword[0] = 0;
        value_str[0] = 0;
        sptr = strchr(in_line, '#');
        if(sptr)
        {
          for(j = sptr - in_line; j < MAXLINELENGTH; j++)
            in_line[j] = 0;
        }
        sptr = in_line;
        while(sptr[0] && (sptr[0] == ' '))
          sptr += 1;
        if(sptr[0])
        {
          strcpy(temp_line, chopSpace(sptr, MAXLINELENGTH - 1));
          sptr = strchr(temp_line, ':');
          if(sptr)
          {
            sptr += 1;
            strncpy(keyword, temp_line, sptr - temp_line);
            keyword[sptr - temp_line] = 0;
            while(sptr[0] && (sptr[0] == ' '))
              sptr += 1;
            if(sptr[0])
              strcpy(value_str, sptr);
          }
          if(keyword[0] == 0 || value_str[0] == 0)
            continue;

          if(!strcmp(keyword, "OUTPUT:"))
          {
            strcpy(prefs->outfile, value_str);
            strcpy(output, value_str);
          }
          else if(!strcmp(keyword, "PRINTER:"))
            if(check_printer_name(value_str))
              strcpy(prefs->printer, value_str);
        }
      }
      fclose(fprc);
    }
  }
  else
    pop_pref_note(0);

  /* If above not set:
     output defaults to get_default_dev() in main()
     prefs->outfile defaults to get_default_dev() in init_prefs()
     prefs->printer defaults to printer_name[0] in init_prefs()
     The reason there is the variable "output" in addition to prefs->outfile
     is to allow the user to temporarily  change the output on Pup's main page 
     without changing the permanent pref setting. */

  set_printer_index(top); /* set index & family */
}
/* -------------------------------------------------------------------------
        prefsave_CB()
   ------------------------------------------------------------------------- */
void prefsave_CB(GtkWidget *widget, task_struct *prf)
{
  gchar filename[MAXLINELENGTH];
  FILE *fprc;
  extern int errno; /* in errno.h */
  extern int open_file_errno;

  open_file_errno = 0;

  clear_msgbox(&(prf->msgbox));
  if(getenv("HOME"))
  {
    strcpy(filename, getenv("HOME"));
    strcat(filename, "/.puprc");
    if((fprc = fopen(filename, "w")) == NULL)
    {
      open_file_errno = errno;
      put_msg(&(prf->msgbox), "Error: Can't open pref file.", RED, 12);
      return;
    }
    else
    {
      fprintf(fprc, "# Pup Preferences File.\n");
      fprintf(fprc, "# This file automatically generated by Pup. "
                    "Do not edit.\n\n");  
      fprintf(fprc, "VERSION: %s\n", VERSION);
      fprintf(fprc, "OUTPUT: %s\n", (prf->top)->prefs.outfile);
      fprintf(fprc, "PRINTER: %s\n", (prf->top)->prefs.printer);
      fclose(fprc);
      put_msg(&(prf->msgbox), "Saved file.", GREEN, 0);
    }
  }
  else
    put_msg(&(prf->msgbox), "Error: $HOME not defined.", RED, 11);
}
/* -------------------------------------------------------------------------
        prefentry_callback() Also see entry_callback() in lexroutines.c
   ------------------------------------------------------------------------- */
void prefentry_callback(GtkWidget *widget, topwin_struct *top)
{
  gchar *entrytext;

  top->pflag = 1;
  entrytext = gtk_entry_get_text(GTK_ENTRY(widget));
  strcpy(top->prefs.outfile, entrytext);
  gtk_entry_set_text(GTK_ENTRY(top->entry), entrytext);
  /* Since the above causes a "changed" callback for top->entry,
     entry_callback() will take care of changing the output. */
  top->pflag = 0;
}
/* -------------------------------------------------------------------------
        pref_event() Sets preferences.
   ------------------------------------------------------------------------- */
void pref_event(topwin_struct *top)
{
  gint i;
  static task_struct prf;
  static framebox_struct fb[3];
  GtkWidget *entry, *combo, *button;
  GList *cbitems;
  GtkWidget *pixmap1, *pixmap2, *pixmap3;
  GtkWidget *vbox, *hbox, *label;

  gtk_widget_set_sensitive(top->prefmenu, FALSE);
  set_sensitize(top, 0);
  gtk_widget_set_sensitive(top->entry, FALSE);
  top->msgbox.freeze = 1; /* fixed message */
  put_msg(&(top->msgbox), "See 'Preferences' window.", GREEN, 0);

  init_task(top, &prf);
  popup_window(&prf, "Preferences");
  gtk_widget_set_usize(prf.tasktop, 400, 410);

  set_color(&(prf.tasktop), GREY, BG, NORMAL);

  make_framebox(&fb[0], &(prf.vbox_top), 300, 60, "Default Output");

  entry = gtk_entry_new();
  gtk_widget_set_usize(entry, 150, 20);
  gtk_entry_set_text(GTK_ENTRY(entry), top->prefs.outfile);
  /* "output" in Pup's main window may have been changed by the user
      but we still set the value of this pref to top->prefs.outfile. If
      the user then changes this one, it gets changed in the main box as 
      well. This is done to allow the user to temperarily change the
      main box but still keep the preference. */
  gtk_signal_connect(GTK_OBJECT(entry), "changed",
                            GTK_SIGNAL_FUNC(prefentry_callback),
                            (gpointer)top);
  gtk_box_pack_start(GTK_BOX(fb[0].hbox), entry, FALSE, FALSE, 65);
  gtk_widget_show(entry);

  make_framebox(&fb[1], &(prf.vbox_top), 300, 60, "Printer Model");

  cbitems = NULL;
  i = 0;
  while(strcmp(printer_name[i], "end"))
  {
    cbitems = g_list_append (cbitems, printer_name[i]);
    i++;
  }

  combo = gtk_combo_new();
  gtk_widget_set_usize(combo, 150, 20);
  gtk_combo_set_popdown_strings(GTK_COMBO(combo), cbitems);
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), top->prefs.printer);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO (combo)->entry), FALSE);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO (combo)->entry), "changed",
                     GTK_SIGNAL_FUNC(combo_CB), (gpointer)top);
  gtk_box_pack_start(GTK_BOX(fb[1].hbox), combo, FALSE, FALSE, 65);
  gtk_widget_show(combo);

  make_framebox(&fb[2], &(prf.vbox_top), 300, 75, "Save Preferences");

  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(fb[2].hbox), vbox, FALSE, FALSE, 16);
  gtk_widget_show(vbox);

  label = gtk_label_new("Preferences will be saved in $HOME/.puprc");
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 10);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  button = gtk_button_new_with_label("Save");
  gtk_widget_set_usize(button, 75, 20);
  gtk_signal_connect(GTK_OBJECT (button), "clicked",
                     GTK_SIGNAL_FUNC (prefsave_CB), (gpointer)&prf);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_widget_show(button);

  make_messageBox(&(prf.vbox_top), &(prf.msgbox));

  /* Callback for the X kill */
  gtk_signal_connect(GTK_OBJECT(prf.tasktop), "delete_event",
                     GTK_SIGNAL_FUNC(deleteTWX), (gpointer)&prf);
  gtk_widget_show(prf.tasktop);

  /* must be defined after prf.tasktop is realized */
  pixmap1 = create_pix(&(prf.tasktop), 3);
  gtk_box_pack_start(GTK_BOX(fb[0].pbox), pixmap1, FALSE, FALSE, 0);

  pixmap2 = create_pix(&(prf.tasktop), 4);
  gtk_box_pack_start(GTK_BOX(fb[1].pbox), pixmap2, FALSE, FALSE, 0);

  pixmap3 = create_pix(&(prf.tasktop), 5);
  gtk_box_pack_start(GTK_BOX(fb[2].pbox), pixmap3, FALSE, FALSE, 0);
}
