require 'puppet/type'

module Puppet
class Type
    class NSSType < Puppet::Type
        class << self
            attr_reader :parentstate, :parentmodule

            # Seems a lot like this should be elsewhere,
            # but we'd have to have a different parent class for
            # netinfo types if that were the case.
            attr_accessor :netinfodir

            # Create an instance for every object that exists on the machine
            def list
                listbyname.collect do |name|
                    obj = nil
                    check = @states.collect { |st| st.name }
                    if obj = self[name]
                        obj[:check] = check
                    else
                        # unless it exists, create it as an unmanaged object
                        obj = self.create(:name => name, :check => check)
                    end

                    next unless obj # In case there was an error somewhere
                    
                    #obj.retrieve
                    obj
                end
            end

            def newstate(*args, &block)
                s = super(*args, &block)

                if s.respond_to?(:finish)
                    s.finish
                end
            end
        end
    end
end

class State
    # This is the state that all other Nameservice states descend from.  It sets
    # the standard for how one interacts with these state objects, but it leaves
    # out any implementation details.  See the 'posix' stuff for the basics
    # on how to retrieve information on most systems, but any adding of information
    # is done specially per-system (e.g., netinfo, useradd, adduser).
    class NSSState < Puppet::State
        class << self
            # Are all changes to states done in one step or do all states need
            # to be synced individually?  This differentiates between netinfo,
            # in which creation cannot be used to fill out information, and
            # things like useradd, in which creation can be done with all
            # information in one swell foop.
            def allatonce?
                #Puppet.info "Returning allatonce %s" % @allatonce
                if defined? @allatonce
                    return @allatonce
                else
                    return false
                end
            end

            # Yes, this value will autogenerate.
            def isautogen
                @isautogen = true
            end

            def noautogen
                @isautogen = false
            end

            # Can we autogenerate a value for this field?  If a required field
            # can be autogenerated then we don't require a value.
            def autogen?
                if defined? @isautogen and @isautogen
                    return true
                else
                    return false
                end
            end

            # Yes, this field is optional
            def isoptional
                @isoptional = true
            end

            # Is this field optional?  Defaults to false.
            def isoptional?
                if defined? @isoptional
                    return @isoptional
                else
                    return false
                end
            end

            # What method is used to retrieve the value from the POSIX struct?
            # Really, these method names should be stored in this class somewhere,
            # in a map or something, but then I would have to differentiate
            # between the different posix classes (e.g., user and group).  In the
            # end, we're _only_ using classes that _do_ have posix structs,
            # so we might as well store the method in the class definition,
            # rather than forcing it to be abstracted out or whatever.
            def posixmethod
                if defined? @posixmethod
                    return @posixmethod
                else
                    return self.name
                end
            end
        end

        # We use the POSIX interfaces to retrieve all information, so we don't
        # have to worry about abstracting that across the system.  Any class
        # can still override this, but it should work for the vast majority of
        # cases.
        def retrieve
            if obj = @parent.getinfo(true)
                if method = self.class.posixmethod || self.class.name
                    @is = obj.send(method)
                else
                    self.devfail "%s has no posixmethod" % self.class
                end
            else
                @is = :absent
            end
        end

        # The list of all groups the user is a member of.  Different
        # user mgmt systems will need to override this method.
        def grouplist
            groups = []

            # Reset our group list
            Etc.setgrent

            user = @parent[:name]

            # Now iterate across all of the groups, adding each one our
            # user is a member of
            while group = Etc.getgrent
                members = group.mem

                if members.include? user
                    groups << group.name
                end
            end

            # We have to close the file, so each listing is a separate
            # reading of the file.
            Etc.endgrent

            groups
        end

        # Sync the information.
        def sync
            event = nil
            # they're in sync some other way
            if self.insync?
                return nil
            end
            if @is == :absent
                self.retrieve
                if self.insync?
                    return nil
                end
            end

            unless @parent.exists?
                self.devfail "%s %s does not exist; cannot set %s" %
                    [@parent.class.name, @parent.name, self.class.name]
            end

            # this needs to be set either by the individual state
            # or its parent class
            cmd = self.modifycmd

            self.debug "Executing %s" % cmd.inspect

            output = %x{#{cmd} 2>&1}


            unless $? == 0
                self.fail "Could not modify %s on %s %s: %s" %
                    [self.class.name, @parent.class.name,
                        @parent.name, output]
            end

            if event
                return event
            else
                return "#{@parent.class.name}_modified".intern
            end
        end

        # This is only used when creating or destroying the object.
        def syncname(value)
            cmd = nil
            event = nil
            case value
            when :absent
                # we need to remove the object...
                unless @parent.exists?
                    self.info "already absent"
                    # the object already doesn't exist
                    return nil
                end

                # again, needs to be set by the ind. state or its
                # parent
                cmd = self.deletecmd
                type = "delete"
            when :present
                if @parent.exists?
                    self.info "already exists"
                    # The object already exists
                    return nil
                end

                # blah blah, define elsewhere, blah blah
                cmd = self.addcmd
                type = "create"
            end
            self.debug "Executing %s" % cmd.inspect

            output = %x{#{cmd} 2>&1}

            unless $? == 0
                raise Puppet::Error, "Could not %s %s %s: %s" %
                    [type, @parent.class.name, @parent.name, output]
            end

            # we want object creation to show up as one event, 
            # not many
            unless self.class.allatonce?
                Puppet.debug "%s is not allatonce" % @parent.class.name
                if type == "create"
                    @parent.eachstate { |state|
                        next if state.name == :ensure
                        state.sync
                        state.retrieve
                    }
                end
            end
        end
    end
end
end

# Here's where we decide what type of objects we'll be dealing with.
case Facter["operatingsystem"].value
when "Darwin":
    require 'puppet/type/nameservice/netinfo'
when "FreeBSD":
    require 'puppet/type/nameservice/pw'
else
    require 'puppet/type/nameservice/objectadd'
end


# $Id: nameservice.rb 1156 2006-05-01 01:07:47Z luke $
