/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Callback-functions from the GUI
 *
 * Copyright (C) 2003 Isak Savo
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "srv_comm.h"
#include "gui_helper.h"
#include "usr_manager.h"
#include "helper.h"
#include "prefwin.h"
#include "cfg.h"
#include "logfile.h"
#include "dirbrowser.h"
#include "helpview.h"
#include "famwrap.h"
#include "debugging.h"
popup_src_t popup_source;

void update_adv_info (void);

void exit_program (void)
{
   prog_exiting = TRUE;
   cfg_write_settings ();
#ifdef HAVE_LIBFAM
   close_logfile ();
   fam_terminate ();
#endif
   hlp_terminate ();
   gui_terminate ();
   srv_terminate ();
   gtk_main_quit ();
}

gboolean
on_main_window_delete_event            (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
   return FALSE;
}


void
on_main_window_destroy                 (GtkObject       *object,
                                        gpointer         user_data)
{
   exit_program ();
}


void
show_hide_advanced_info                (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
   GtkWidget *adv_info = lookup_widget (main_window, "tbl_advanced_info");
   
   if (gtk_toggle_button_get_active (togglebutton))
     gtk_widget_show (adv_info);
   else
     gtk_widget_hide (adv_info);
}

void
on_tree_activity_cursor_changed        (GtkTreeView     *treeview,
                                        gpointer         user_data)
{
   update_adv_info ();
}

void update_adv_info (void)
{
   static GtkWidget *tree_activities = NULL;
   static GtkTreeModel *model = NULL;
   static GtkWidget *adv_info = NULL;
   static GtkTreeSelection *sel;
   GtkTreeIter iter;
   gboolean visible = TRUE, selected;
   gint id;
   GValue val;
   PActivity *act;

   if (!tree_activities)
   {
      tree_activities = lookup_widget (main_window, "tree_activity");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_activities));
      adv_info = lookup_widget (main_window, "tbl_advanced_info");
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_activities));
   }

   memset (&val, 0, sizeof (GValue));
   g_value_init (&val, G_TYPE_BOOLEAN);
   g_object_get_property (G_OBJECT (adv_info), "visible", &val);
   visible = g_value_get_boolean (&val);
   selected = gtk_tree_selection_get_selected (sel, &model, &iter);
   if (visible)
   {
      if (selected)
      {
	 gtk_tree_model_get (model, &iter, 
			     COL_ACT_ID, &id,
			     -1);
	 act = srv_get_activity_with_id (id);
	 gui_update_info_frame (act);
      }
      else
	gui_update_info_frame (NULL);  /* Clear advanced info labels */
   }
}

gboolean timeout_update_activity (gpointer data)
{
   GList *act_list, *usr_list, *node;
   PActivity *activity;
   POnlineUser *user;
   static GdkPixbuf *icon_up = NULL;
   static GdkPixbuf *icon_down = NULL;
   static GdkPixbuf *icon_user = NULL;
   static GtkTreeView *tree_activity = NULL, *tree_users = NULL;
   static GtkTreeModel *model = NULL, *model_users = NULL;
   static GtkTreeSelection *sel = NULL, *sel_users = NULL;
   GtkTreeIter iter;
   GtkTreePath *act_path, *usr_path;
   PGuiActivityRow arow;
   PGuiUserRow urow;
   gchar *percent;
   gboolean restore_selection_act = FALSE, restore_selection_usr = FALSE, act_err;
   
   if (prog_exiting) 
   {
      g_print ("Timeout aborting\n");
      return FALSE;
   }
   if (!icon_up)
   {
      icon_up = create_pixbuf ("up.png");
      icon_down = create_pixbuf ("down.png");
      icon_user = create_pixbuf ("user.png");
   }
   if (!tree_activity)
   {
      tree_activity =  (GtkTreeView *) lookup_widget (main_window, "tree_activity");
      model = gtk_tree_view_get_model (tree_activity);
      sel = gtk_tree_view_get_selection (tree_activity);
      
      tree_users = (GtkTreeView *) lookup_widget (main_window, "tree_users");
      model_users = gtk_tree_view_get_model (tree_users);
      sel_users = gtk_tree_view_get_selection (tree_users);
   }
   restore_selection_act = gtk_tree_selection_get_selected (sel, &model, &iter);
   if (restore_selection_act)
     act_path = gtk_tree_model_get_path (model, &iter);
   else
     act_path = NULL;

   restore_selection_usr = gtk_tree_selection_get_selected (sel_users, &model_users, &iter);
   if (restore_selection_usr)
     usr_path = gtk_tree_model_get_path (model_users, &iter);
   else
     usr_path = NULL;
							    
   gui_clear_activity_list ();
   gui_clear_users_list ();
   
   act_list = srv_get_activities (&act_err);
   if (act_err)
   {
      /* Error reading scoreboard, abort! */
      gui_display_msgdialog (_("Unable to read FTP-Server data"),
			     _("PureAdmin was unable to retrieve ftpserver data. This is most likely due to insufficient permissions.\nTry restarting PureAdmin with root privileges."), MSGDLG_TYPE_ERROR, main_window);
      return FALSE;
   }
   node = g_list_first (act_list);
   while (node)
   {
      activity = (PActivity *) node->data;
      if (activity->state == FTPWHO_STATE_DOWNLOAD)
	arow.icon = icon_down;
      else if (activity->state == FTPWHO_STATE_UPLOAD)
	arow.icon = icon_up;
      else
      {
	 /* Only append actual actions (DL/UL) */
	 node = g_list_next (node);
	 continue;
      }
      arow.id = activity->id;
      
      if (activity->download_total_size > 0)
	percent = g_strdup_printf ("%.2f", 100 * (double) activity->download_current_size / (double) activity->download_total_size);
      else
	percent = g_strdup ("??");
      arow.text = g_strdup_printf ("%s (%s):'%s' - (%s%%)", activity->username, activity->remote_addr, 
				   activity->filename, percent);
      gui_add_to_activities (arow);
      node = g_list_next (node);
      g_free (arow.text);
      g_free (percent);
   }
   
   if (restore_selection_act)
   {
      gtk_tree_selection_select_path (sel, act_path);
      g_signal_emit_by_name ((gpointer) tree_activity, "cursor_changed", tree_activity, NULL);
   }
   
   /* Update users list */
   usr_list = srv_get_connected_users (g_list_first (act_list));
   node = usr_list;
   urow.icon = icon_user;
   while (node)
   {
      user = (POnlineUser *) node->data;
      urow.user = g_locale_to_utf8 (user->username, -1, NULL, NULL, NULL);
      urow.host = g_strdup (user->remote_addr);
      urow.num_connections = user->num_connections;
      gui_add_to_online_users (urow);

      g_free (node->data);
      g_free (urow.user);
      g_free (urow.host);
      node = g_list_next (node);
   }
   g_list_free (usr_list);
   
   if (restore_selection_usr)
     gtk_tree_selection_select_path (sel_users, usr_path);

   gui_update_menu_sensivity ();

   /* Always return TRUE */
   return TRUE;
}

void gui_init_dlg_show_users (GtkWidget *dlg)
{
   GtkTreeView *tree;
   GtkListStore *model;
   GtkCellRenderer *renderer;
   GtkTreeViewColumn *column;
   GtkWidget *menu;
   GtkWidget *opt_menu;
   const gchar *titles[N_USRMAN_COLUMNS] = { _("-"), _("Username"), _("Real Name") };
   GList *IDs;

   /* Set-up treeview */
   tree = GTK_TREE_VIEW (lookup_widget (dlg, "treeview_dlg_users"));
   model  = gtk_list_store_new (N_USRMAN_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING);
   
   gtk_tree_view_set_model (GTK_TREE_VIEW (tree), GTK_TREE_MODEL (model));
   //   g_object_unref (G_OBJECT (model));
   gtk_tree_selection_set_mode (gtk_tree_view_get_selection (GTK_TREE_VIEW (tree)),
				GTK_SELECTION_SINGLE);

   
   renderer = gtk_cell_renderer_pixbuf_new();
   column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
						      "pixbuf", COL_USRMAN_ICON,
						      NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW(tree), column);
   
   renderer = gtk_cell_renderer_text_new();
   column = gtk_tree_view_column_new_with_attributes (titles[1], renderer,
						      "text", COL_USRMAN_LOGIN,
						      NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW(tree), column);

   renderer = gtk_cell_renderer_text_new();
   column = gtk_tree_view_column_new_with_attributes (titles[2], renderer,
						      "text", COL_USRMAN_REALNAME,
						      NULL);
   gtk_tree_view_append_column (GTK_TREE_VIEW(tree), column);
   
   opt_menu = lookup_widget (dlg, "usr_dlg_optmenu_uid");
   IDs = misc_get_user_ids ();
   menu = gui_create_menu_from_glist (IDs);
   gtk_option_menu_remove_menu (GTK_OPTION_MENU (opt_menu));
   gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_menu), menu);
   g_object_set_data (G_OBJECT (opt_menu), "items", IDs);

   opt_menu = lookup_widget (dlg, "usr_dlg_optmenu_gid");
   IDs = misc_get_group_ids ();
   menu = gui_create_menu_from_glist (IDs);
   gtk_option_menu_remove_menu (GTK_OPTION_MENU (opt_menu));
   gtk_option_menu_set_menu (GTK_OPTION_MENU (opt_menu), menu);
   g_object_set_data (G_OBJECT (opt_menu), "items", IDs);
   
   g_object_set_data (G_OBJECT (dlg), "editing", GINT_TO_POINTER (FALSE));
}

void
on_btn_user_manager_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
   if (!usr_manager_check_prerequisites ())
     return;
   if (!dlg_usrman)
   {
      dlg_usrman = create_dlg_show_users ();
      gui_init_dlg_show_users (dlg_usrman);
   }
   gui_fill_userman_list ();
   usr_select_first_user ();
   gtk_widget_show (dlg_usrman);
   gtk_window_present (GTK_WINDOW (dlg_usrman));
}

void
on_treeview_dlg_users_cursor_changed   (GtkTreeView     *treeview,
                                        gpointer         user_data)
{
   static GtkTreeModel *model = NULL;
   static GtkTreeSelection *sel = NULL;
   static GtkWidget *btn_edit, *btn_remove;
   GtkTreeIter iter;
   gchar *username;
   
   if (!model || !sel)
   {
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
   }
   if (!btn_edit || !btn_remove)
   {
      btn_edit = lookup_widget (dlg_usrman, "usr_dlg_btn_edituser");
      btn_remove = lookup_widget (dlg_usrman, "usr_dlg_btn_removeuser");
   }
   if (!gtk_tree_selection_get_selected (sel, &model, &iter))
   {
      gtk_widget_set_sensitive (btn_edit, FALSE);
      gtk_widget_set_sensitive (btn_remove, FALSE);
      return;
   }
   
   gtk_tree_model_get (model, &iter, 
		       COL_USRMAN_LOGIN, &username,
		       -1);
   usr_widgets_set_info (username);
   gtk_widget_set_sensitive (btn_edit, TRUE);
   gtk_widget_set_sensitive (btn_remove, TRUE);
}


void
on_usr_dlg_btn_edituser_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *btn_remove, *btn_add, *tree;
   GtkWidget *entry;

   tree = lookup_widget (dlg_usrman, "treeview_dlg_users");
   btn_remove = lookup_widget (dlg_usrman, "usr_dlg_btn_removeuser");
   btn_add = lookup_widget (dlg_usrman, "usr_dlg_btn_adduser");

   usr_widgets_set_active_state (TRUE);

   gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
   gtk_widget_set_sensitive (tree, FALSE);
   gtk_widget_set_sensitive (btn_remove, FALSE);
   gtk_widget_set_sensitive (btn_add, FALSE);
   
   entry = lookup_widget (dlg_usrman, "usr_dlg_entry_realname");
   gtk_widget_grab_focus (entry);
   
   g_object_set_data (G_OBJECT (dlg_usrman), "editing", GINT_TO_POINTER (TRUE));
}


void
on_usr_dlg_btn_removeuser_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
   static GtkTreeView *tree;
   static GtkTreeModel *model = NULL;
   static GtkTreeSelection *sel = NULL;
   GtkTreePath *path;
   GtkTreeIter iter;
   gchar *username, *msg;
   gint ret, i;
   
   if (!tree || !model || !sel)
   {
      tree = (GtkTreeView *) lookup_widget (dlg_usrman, "treeview_dlg_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree));
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
   }

   if (!gtk_tree_selection_get_selected (sel, &model, &iter))
     return;
   path = gtk_tree_model_get_path (model, &iter);
   gtk_tree_model_get (model, &iter, 
		       COL_USRMAN_LOGIN, &username,
		       -1);
   msg = g_strdup_printf (_("Remove User '%s'?"), username);
   if (gui_display_confirmdialog (msg, 
				  _("This will permanently remove the user from pureftpd, preventing him or her to login to your server.\nAre you sure you want to remove this user?"), 
				  _("Yes, _Remove User"), NULL) != GTK_RESPONSE_YES)
   {
      g_free (msg);
      return;
   }
   g_free (msg);
   ret = usr_remove_user (username);
   switch (ret)
   {
      case E_PERMISSION_DENIED:
	msg = g_strdup_printf (_("You do not have permission to remove the user %s from the password-file. This could be because another process is accessing the file(s)."), username);
	gui_display_msgdialog (_("Error removing user."), msg, MSGDLG_TYPE_WARNING, dlg_usrman);
	g_free (msg);
	break;
      case E_CMD_NOT_FOUND:
	msg = g_strdup_printf (_("The user '%s' could not be removed because the pure-pw command could not be found. Make sure that it is installed and that the configuration points to the correct file."), username);
	gui_display_msgdialog (_("Error removing user."), msg, MSGDLG_TYPE_WARNING, dlg_usrman);
	g_free (msg);
	break;
      case E_GENERAL_ERROR:
	msg = g_strdup_printf (_("The user '%s' could not be removed."), username);
	gui_display_msgdialog (_("Error removing user."), msg, MSGDLG_TYPE_WARNING, dlg_usrman);
	g_free (msg);
	break;
      case E_USER_NOT_FOUND:
	msg = g_strdup_printf (_("The user '%s' could not be removed because it doesn't exist. If you are sure it exist, please contact the author of this program with a description of your problem."), username);
	gui_display_msgdialog (_("Error removing user."), msg, MSGDLG_TYPE_WARNING, dlg_usrman);
	g_free (msg);
	break;
      case E_NO_ERROR:
	i = gui_fill_userman_list ();
	/* Try to restore the selection or move it to the row above if that exists */
	while (i && gtk_tree_selection_path_is_selected (sel, path) == FALSE)
	{
	   gtk_tree_selection_select_path (sel, path);
	   i = gtk_tree_path_prev (path);
	}
	/* Trigger widget updates FIXME: Is this necessary? */
	g_signal_emit_by_name ((gpointer) tree, "cursor_changed", tree, NULL);
   }
}

void
on_usr_dlg_btn_save_changes_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *btn_remove, *btn_add, *btn_edit;
   static GtkTreeView *tree;
   static GtkTreeModel *model = NULL;
   static GtkTreeSelection *sel = NULL;
   GtkTreePath *path;
   GtkTreeIter iter;
   gint ret;

   if (!tree)
   {
      tree = (GtkTreeView *) lookup_widget (dlg_usrman, "treeview_dlg_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree));
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree));
   }
   if (!gtk_tree_selection_get_selected (sel, &model, &iter)) /* This should never happen, but we need the iter anyway */
     return; 
   path = gtk_tree_model_get_path (model, &iter);
   
   btn_remove = lookup_widget (dlg_usrman, "usr_dlg_btn_removeuser");
   btn_add = lookup_widget (dlg_usrman, "usr_dlg_btn_adduser");
   btn_edit = lookup_widget (dlg_usrman, "usr_dlg_btn_edituser");

   ret = usr_save_user_info (TRUE);

   switch (ret)
   {
      case E_PERMISSION_DENIED:
	gui_display_msgdialog (_("Error saving user."),
			       _("You do not have permission to save the user-information. This could be because another process is accessing the required file(s).\nYou may try again and if that doesn't work, please contact the author with a description of your problem."),
			       MSGDLG_TYPE_WARNING, dlg_usrman);
	break;
      case E_CMD_NOT_FOUND:
	gui_display_msgdialog (_("Error saving user."),
			       _("The pure-pw command could not be found. Make sure that the command is installed and that the configuration points to the correct file.\n"),
			       MSGDLG_TYPE_WARNING, dlg_usrman);
	break;
      case E_NO_ERROR:
	usr_widgets_set_active_state (FALSE);
	
	gtk_widget_set_sensitive (GTK_WIDGET (button), FALSE);
	
	gtk_widget_set_sensitive (GTK_WIDGET (tree), TRUE);
	gtk_widget_set_sensitive (btn_remove, TRUE);
	gtk_widget_set_sensitive (btn_add, TRUE);
	gtk_widget_set_sensitive (btn_edit, TRUE);
	
	gui_fill_userman_list ();
	gtk_tree_selection_select_path (sel, path);
	
	g_object_set_data (G_OBJECT (dlg_usrman), "editing", GINT_TO_POINTER (FALSE));
	break;
   }
}

void
on_usr_dlg_btn_adduser_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *dlg, *entry;
   gint response;
   gchar *user = NULL, *pw, *encrypted = NULL;
   gboolean add = FALSE, keep_asking = TRUE;
   
   
   dlg = create_dlg_ask_new_username ();
   while (keep_asking)
   {
      keep_asking = FALSE;
      add = FALSE;
      response = gtk_dialog_run (GTK_DIALOG (dlg));
      //      gtk_widget_hide (dlg);
      switch (response)
      {
	 case GTK_RESPONSE_OK:
	   entry = lookup_widget (dlg, "entry_username");
	   user = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
	   entry = lookup_widget (dlg, "entry_pw1");
	   pw = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
	   encrypted = g_strdup (misc_crypt_passwd (pw));
	   add = TRUE;
	   if (!misc_validate_username (user))
	   {
	      gui_display_msgdialog (_("Invalid Username"), 
				     _("The username you have entered contains invalid characters. Only standard english letters (a-z) and the characters '@', '-', '_', '.', ',' ':' ' ' (space) and ''' (single quote) are permitted."), 
				     MSGDLG_TYPE_ERROR, dlg);
	      keep_asking = TRUE;
	   }
	   else if (usr_user_exists (user))
	   {
	      gui_display_msgdialog (_("User already exists"), 
				     _("The username you have entered already exists in PureFTPd. You must select a different name for your new login."),
				     MSGDLG_TYPE_ERROR, dlg);
	      keep_asking = TRUE;
	   }
	   break;
	 case GTK_RESPONSE_CANCEL:
	   pur_log_dbg ("Aborting creation!\n");
	   break;
      }
   }
   gtk_widget_destroy (dlg);
   if (!add)
     return;
   response = usr_add_user (user, encrypted);
   switch (response)
   {
      case E_PERMISSION_DENIED:
	gui_display_msgdialog (_("Add User Failed"), _("Could not add the user because you have insufficient permissions to do so. This could be bacause some other process is accessing the required file(s)."), MSGDLG_TYPE_WARNING, dlg);
	break;
      case E_CMD_NOT_FOUND:
	gui_display_msgdialog (_("Add User Failed"), _("The pure-pw command could not be found. Make sure that the command is installed and that the configuration points to the correct file.\n"), MSGDLG_TYPE_WARNING, dlg);
	break;
      case E_NO_ERROR:
	gui_fill_userman_list ();
	gui_userman_select_user (user);
	break;
   }
   
}


void
on_usr_dlg_btn_change_pw_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *dlg, *entry;
   const gchar *pw;
   gchar *encrypted;
   gint ret;

   dlg = create_dlg_set_password ();
   
   ret = gtk_dialog_run (GTK_DIALOG (dlg));
   if (ret == GTK_RESPONSE_OK)
   {
      entry = lookup_widget (dlg, "passwd1");
      pw = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
      
      encrypted = g_strdup (misc_crypt_passwd (pw));
      g_free (g_object_get_data (G_OBJECT (button), "usr_password"));
      g_object_set_data (G_OBJECT (button), "usr_password", encrypted);
      
   }
   gtk_widget_destroy (dlg);
}


void
on_usr_dlg_chk_chroot_toggled          (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
   GtkWidget *entry;
   entry = lookup_widget (GTK_WIDGET (togglebutton), "usr_dlg_entry_fakeroot");
   
   if (GTK_WIDGET_SENSITIVE (GTK_WIDGET (togglebutton)) == FALSE)
     return; /* Don't set sensitive state if togglebutton is insensitive */
   gtk_widget_set_sensitive (entry, 
			     gtk_toggle_button_get_active (togglebutton));
}

void
on_usr_dlg_btn_cancel_changes_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *tree, *btn_remove, *btn_add, *btn_edit;
   
   tree = lookup_widget (dlg_usrman, "treeview_dlg_users");
   btn_remove = lookup_widget (dlg_usrman, "usr_dlg_btn_removeuser");
   btn_edit = lookup_widget (dlg_usrman, "usr_dlg_btn_edituser");
   btn_add = lookup_widget (dlg_usrman, "usr_dlg_btn_adduser");
   
   gtk_widget_set_sensitive (GTK_WIDGET (tree), TRUE);
   gtk_widget_set_sensitive (btn_remove, TRUE);
   gtk_widget_set_sensitive (btn_add, TRUE);
   gtk_widget_set_sensitive (btn_edit, TRUE);
   usr_widgets_set_active_state (FALSE);
   
   g_object_set_data (G_OBJECT (dlg_usrman), "editing", GINT_TO_POINTER (FALSE));
   g_signal_emit_by_name (G_OBJECT (tree), "cursor_changed", GTK_TREE_VIEW (tree), NULL);
}

void
close_usrman_dialog                    (GtkButton       *button,
                                        gpointer         user_data)
{
   gboolean editing;
   gint dlg_ret;
   GtkWidget *save_btn = lookup_widget (dlg_usrman, "usr_dlg_btn_save_changes");
   editing = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (dlg_usrman), "editing"));

   if (editing)
   {
      dlg_ret = gui_display_confirmdialog (_("You have unsaved info."), 
					   _("You are currently editing the information of an user.\n"
					     "Do you wish to save this information first?"),
					   _("Yes, _Save Info"), _("_Cancel"));
      if (dlg_ret == GTK_RESPONSE_YES)
	g_signal_emit_by_name (G_OBJECT (save_btn), "clicked", GTK_BUTTON (save_btn), NULL);
      else
	return;
   }
   gtk_widget_hide (dlg_usrman);
}


void
menu_import_users                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   
}


void
menu_quit                              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   exit_program ();
}


void
menu_preferences                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   static GtkWidget *prefwin = NULL;
   
   if (!prefwin)
   {
      prefwin = create_preferences ();
      pref_init_prefwindow (prefwin);
   }
   
   gtk_dialog_run (GTK_DIALOG (prefwin));
   gtk_widget_hide (prefwin);
   cfg_write_settings ();
}


void
menu_srv_startstop                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   int ret;
   gchar *msg, *title;
   gboolean running = srv_vars.pureftpd_running;

   if (*srv_vars.startupscript == '\0')
   {
      gui_display_msgdialog (_("Startupscript Not Found"),
			     _("PureAdmin could not locate the script used to start and stop the server.\n"
			       "This file should be located in /etc/rc.d/init.d/ on RedHat (and similar) and "
			       "/etc/rc.d/ on Slackware (and similar)\n"
			       "Starting and stopping is disabled."),
			     MSGDLG_TYPE_ERROR, main_window);
      return;
   }
   
   if (running)
     ret = misc_stop_server ();
   else
     ret = misc_start_server ();

   if (ret == FALSE)
   {
      if (running)
      {
	 title = g_strdup (_("Failure to start server"));
	 msg = g_strdup_printf (_("PureAdmin was unable to start the server because the execution of \"%s\" failed.\n"
				  "Error reported was: %s"), srv_vars.startupscript, strerror (errno));
      }
      else
      {
	 title = g_strdup (_("Failure to stop server"));
	 msg = g_strdup_printf (_("PureAdmin was unable to stop the server because the execution of \"%s\" failed.\n"
				  "Error reported was: %s"), srv_vars.startupscript, strerror (errno));
      }
      
      gui_display_msgdialog (title, msg,
			     MSGDLG_TYPE_ERROR, main_window);
      g_free (title);
      g_free (msg);
      return;
   }
   if (running)
     srv_vars.pureftpd_running = FALSE;
   else
     srv_vars.pureftpd_running = TRUE;

   gui_update_server_status ();
}


void
menu_close_connections                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   GList *act_list;
   PActivity *activity;
   gint ret;

   act_list = srv_get_activities_in_memory ();
   while (act_list)
   {
      activity = (PActivity *) act_list->data;
      g_print ("Closing connection with pid: %d...", activity->pid);
      ret = misc_close_connection (activity->pid);
      if (ret)
	g_print ("success\n");
      else
	g_print ("failure\n");
      act_list = act_list->next;
   }
}



void
menu_user_manager                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   if (!usr_manager_check_prerequisites ())
     return;
   if (!dlg_usrman)
   {
      dlg_usrman = create_dlg_show_users ();
      gui_init_dlg_show_users (dlg_usrman);
   }
   gui_fill_userman_list ();
   gtk_widget_show (dlg_usrman);
   gtk_window_present (GTK_WINDOW (dlg_usrman));
}


void
menu_config_server                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
menu_about                             (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   GtkWidget *about, *title;
   about = create_dlg_about ();
   
   title = lookup_widget (about, "lbl_title");
   /* Make the About-text reflect the actual version */
   gtk_label_set_text (GTK_LABEL (title),
		       "<span weight=\"bold\" size=\"x-large\">" 
		       "PureAdmin v" VERSION 
		       "</span>");
   gtk_label_set_use_markup (GTK_LABEL (title), TRUE);

   gtk_dialog_run (GTK_DIALOG (about));
   gtk_widget_destroy (about);
}


void
menu_help                              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   hlp_show_help (HLP_ITEM_INDEX);
}

void
on_usr_dlg_btn_help_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
   hlp_show_help (HLP_ITEM_USER_MANAGER);
}

gboolean
on_usrman_delete_event                 (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
   return FALSE;
}


/* This is used from the add-new-user dialog */
void
on_new_user_entry_change               (GtkEditable     *editable,
                                        gpointer         data)
{
   GtkWidget *entry_uname = NULL;
   GtkWidget *entry_pw1 = NULL;
   GtkWidget *entry_pw2 = NULL;
   GtkWidget *btn_add = NULL;
   const gchar *uname, *pw1, *pw2;

   entry_uname = lookup_widget (GTK_WIDGET (editable), "entry_username");
   entry_pw1 = lookup_widget (GTK_WIDGET (editable), "entry_pw1");
   entry_pw2 = lookup_widget (GTK_WIDGET (editable), "entry_pw2");
   btn_add = lookup_widget (GTK_WIDGET (editable), "btn_add");
   
   uname = gtk_entry_get_text (GTK_ENTRY (entry_uname));

   pw1 = gtk_entry_get_text (GTK_ENTRY (entry_pw1));
   pw2 = gtk_entry_get_text (GTK_ENTRY (entry_pw2));
   
   if (strlen (uname) > 0 && strlen (pw1) > 0 && misc_str_is_equal (pw1, pw2))
   {
      gtk_widget_set_sensitive (btn_add, TRUE);
      GTK_WIDGET_SET_FLAGS (btn_add, GTK_CAN_DEFAULT);
      GTK_WIDGET_SET_FLAGS (btn_add, GTK_HAS_DEFAULT);
   }
   else
     gtk_widget_set_sensitive (btn_add, FALSE);
}


/* This is used in the dialog that changes the password for a user */
void
on_pwentry_changed                     (GtkEditable     *editable,
                                        gpointer         user_data)
{
   GtkWidget *pw1 = NULL;
   GtkWidget *pw2 = NULL;
   GtkWidget *btn = NULL;
   const gchar *txt1, *txt2;

   pw1 = lookup_widget (GTK_WIDGET (editable), "passwd1");
   pw2 = lookup_widget (GTK_WIDGET (editable), "passwd2");
   btn = lookup_widget (GTK_WIDGET (editable), "btn_set");
      
   txt1 = gtk_entry_get_text (GTK_ENTRY (pw1));
   txt2 = gtk_entry_get_text (GTK_ENTRY (pw2));
   
   if (strlen (txt1) > 0 && misc_str_is_equal (txt1, txt2))
     gtk_widget_set_sensitive (btn, TRUE);
   else
     gtk_widget_set_sensitive (btn, FALSE);
}


void
optmenu_id_changed                     (GtkOptionMenu   *optionmenu,
                                        gpointer         user_data)
{
   static GtkWidget *lbl_gid = NULL, *lbl_uid = NULL;
   static GtkWidget *opt_menu_uid = NULL, *opt_menu_gid = NULL;
   
   if (!opt_menu_uid)
   {
      lbl_uid = lookup_widget (dlg_usrman, "lbl_invalid_uid");
      lbl_gid = lookup_widget (dlg_usrman, "lbl_invalid_gid");
      opt_menu_uid = lookup_widget (dlg_usrman, "usr_dlg_optmenu_uid");
      opt_menu_gid = lookup_widget (dlg_usrman, "usr_dlg_optmenu_gid");
   }
   if ((gint) optionmenu == (gint) opt_menu_uid)
     gtk_widget_hide (lbl_uid);
   else
     gtk_widget_hide (lbl_gid);


}


void
on_btn_logview_clear_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
   log_clear_logview ();
}


void
on_btn_logview_reread_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
   close_logfile ();
   init_logfile ();
}


void
on_btn_browse_homedir_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *dbr, *entry;
   const gchar *old;
   gchar *dir;
   gint response;

   entry = lookup_widget (GTK_WIDGET (button), "usr_dlg_entry_home");
   old = gtk_entry_get_text (GTK_ENTRY (entry));
   dbr = dirbrowser_new (NULL);
   dirbrowser_set_directory (DIRBROWSER (dbr), old);
   response = gtk_dialog_run (GTK_DIALOG (dbr));
   if (response == GTK_RESPONSE_OK)
   {
      dir = dirbrowser_get_directory (DIRBROWSER (dbr));
      gtk_entry_set_text (GTK_ENTRY (entry), dir);
      g_free (dir);
   }
   gtk_widget_destroy (dbr);
}


void
on_btn_browse_fakeroot_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
   GtkWidget *dbr, *entry;
   const gchar *old;
   gchar *dir;
   gint response;

   entry = lookup_widget (GTK_WIDGET (button), "usr_dlg_entry_fakeroot");
   old = gtk_entry_get_text (GTK_ENTRY (entry));
   dbr = dirbrowser_new (NULL);
   dirbrowser_set_directory (DIRBROWSER (dbr), old);
   response = gtk_dialog_run (GTK_DIALOG (dbr));
   if (response == GTK_RESPONSE_OK)
   {
      dir = dirbrowser_get_directory (DIRBROWSER (dbr));
      gtk_entry_set_text (GTK_ENTRY (entry), dir);
      g_free (dir);
   }
   gtk_widget_destroy (dbr);
}

gboolean
on_tree_activity_popup_menu            (GtkWidget       *widget,
                                        gpointer         user_data)
{
   popup_source = POPUP_FROM_ACTTREE;
   gui_display_activity_popup (popup_source);
   return TRUE;
}


gboolean
on_tree_activity_button_press_event    (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
   GdkEventButton *event_button;

   if (event->type == GDK_BUTTON_PRESS)
   {
      event_button = (GdkEventButton *) event;
      if (event_button->button == 3)
      {
	 popup_source = POPUP_FROM_ACTTREE;
	 gui_display_activity_popup (popup_source);
	 return TRUE;
      }
   }
   return FALSE;
}


gboolean
on_tree_users_popup_menu               (GtkWidget       *widget,
                                        gpointer         user_data)
{
   popup_source = POPUP_FROM_USRTREE;
   gui_display_activity_popup (popup_source);
   return FALSE;
}


gboolean
on_tree_users_button_press_event       (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
   GdkEventButton *event_button;

   if (event->type == GDK_BUTTON_PRESS)
   {
      event_button = (GdkEventButton *) event;
      if (event_button->button == 3)
      {
	 popup_source = POPUP_FROM_USRTREE;
	 gui_display_activity_popup (popup_source);
	 return TRUE;
      }
   }
   return FALSE;
}


void
on_popup_close_tx                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   gint id;
   PActivity *act;

   if (popup_source == POPUP_FROM_ACTTREE)
   {
      id = gui_tree_act_get_selected_id ();
      if (id < 0)
      {
	 pur_log_wrn ("Unable to fetch information about selected activity");
	 return;
      }
      act = srv_get_activity_with_id (id);
      pur_log_dbg ("Closing connection for user %s (pid: %d)", act->username, act->pid);
      misc_close_connection (act->pid);
   }
   else
   {
      /* This should _never_ happen! */
      pur_log_err ("Close transfer for user NOT possible!");
   }
}


void
on_popup_close_all_conns               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   gchar *user, *host;
   gint id, success;
   PActivity *act;
   GList *conns;

   if (popup_source == POPUP_FROM_ACTTREE)
   {
      id = gui_tree_act_get_selected_id ();
      if (id < 0)
      {
	 pur_log_wrn ("Unable to fetch information about selected activity\n");
	 return;
      }
      act = srv_get_activity_with_id (id);
      pur_log_dbg ("Closing all connection for user %s (pid: %d)\n", act->username, act->pid);
      conns = srv_get_pids_for_user_host (act->username, act->remote_addr);
      while (conns)
      {
	 pur_log_dbg ("Killing process %d\n", GPOINTER_TO_INT (conns->data));
	 success = misc_close_connection (GPOINTER_TO_INT (conns->data));
	 if (!success)
	   pur_log_err ("Failed to kill process %d\n", GPOINTER_TO_INT (conns->data));
	 conns = g_list_next (conns);
      }
      g_list_free (conns);
   }
   else
   {
      if (!gui_tree_usr_get_selected_user (&user, &host))
      {
	 pur_log_wrn ("Unable to fetch information about selected user");
	 return;
      }
      pur_log_dbg ("Closing all connection for user %s (host: %s)\n", user, host);
      conns = srv_get_pids_for_user_host (user, host);
      while (conns)
      {
	 pur_log_dbg ("Killing process %d\n", GPOINTER_TO_INT (conns->data));
	 success = misc_close_connection (GPOINTER_TO_INT (conns->data));
	 if (!success)
	   pur_log_wrn ("Failed to kill process %d\n", GPOINTER_TO_INT (conns->data));
	 conns = g_list_next (conns);
      }
      g_list_free (conns);
   }
}


void
on_popup_edit_user                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   gint id;
   gchar *user, *host;
   PActivity *act;
   if (popup_source == POPUP_FROM_ACTTREE)
   {
      id = gui_tree_act_get_selected_id ();
      if (id < 0)
      {
	 pur_log_wrn ("Unable to fetch information about selected activity\n");
	 return;
      }
      act = srv_get_activity_with_id (id);
      user = act->username;
   }
   else
   {
      if (!gui_tree_usr_get_selected_user (&user, &host))
      {
	 pur_log_wrn ("Unable to fetch information about selected user\n");
	 return;
      }
   }
   pur_log_dbg ("Editing userinfo for %s\n", user);
   
   usr_manager_check_prerequisites ();
   if (!dlg_usrman)
   {
      dlg_usrman = create_dlg_show_users ();
      gui_init_dlg_show_users (dlg_usrman);
   }
   gui_fill_userman_list ();
   gui_userman_select_user (user);   
   gtk_widget_show (dlg_usrman);
   gtk_window_present (GTK_WINDOW (dlg_usrman));
}


void
on_view_debug_console_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   dbg_present_dbg_console ();
}


gboolean
on_dlg_debug_delete_event              (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
   gtk_widget_hide (widget);
   return TRUE;
}


gboolean
on_dlg_debug_destroy_event             (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
   pur_log_dbg ("Debugwindow destroy event: Returning false!");
   return FALSE;
}

