/* PureAdmin
 * Copyright (C) 2003 Isak Savo
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * Misc helperfunctions related to the GUI
 *
 * Copyright (C) 2003 Isak Savo
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <time.h>
#include "helper.h"
#include "gui_helper.h"
#include "support.h"
#include "interface.h"
#include "srv_comm.h"
#include "usr_manager.h"

gchar *_number_separate (gint num);
gchar *_sec_to_time (gulong sec);
gchar *_gen_speed_string (gulong speed);


void gui_terminate (void)
{
   
}

/* Statusbar functions, FIXME? Should contex-id be taken into account?? */
void statusbar_push (const gchar *msg)
{
   static GtkWidget *statusbar = NULL;
   gchar *utf_msg;
  
   if (!statusbar)
     statusbar = lookup_widget (main_window, "main_statusbar");
   if (!g_utf8_validate (msg, -1, NULL))
   {
      utf_msg = g_locale_to_utf8 (msg, -1, NULL, NULL, NULL);
      gtk_statusbar_push (GTK_STATUSBAR (statusbar), 0, utf_msg);
      g_free (utf_msg);
   }
   else
     gtk_statusbar_push (GTK_STATUSBAR (statusbar), 0, msg);
}

void statusbar_pop (void)
{
   static GtkWidget *statusbar = NULL;
   
   if (!statusbar)
     statusbar = lookup_widget (main_window, "main_statusbar");
   
   gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 0);
}

void gui_update_server_status (void)
{
   GtkWidget *mnu_label, *mnu_item;
   gchar *msg, *lbl;
   
   if (srv_vars.pureftpd_running)
   {
      srv_vars.pureftpd_pid = misc_get_pureftpd_pid ();
      msg = g_strdup_printf (_("PureFTPd running (pid %d)"), srv_vars.pureftpd_pid);
      lbl = g_strdup (_("_Stop Server"));
   }
   else
   {
      msg = g_strdup (_("PureFTPd not running"));
      lbl = g_strdup (_("_Start Server"));
   }
      

   statusbar_pop ();
   statusbar_push (msg);
   g_free (msg);

   mnu_item = lookup_widget (main_window, "server_startstop");
   mnu_label = gtk_bin_get_child (GTK_BIN (mnu_item));
   gtk_label_set_text_with_mnemonic (GTK_LABEL (mnu_label), lbl);
   g_free (lbl);
   if (*srv_vars.startupscript == '\0')
     gtk_widget_set_sensitive (mnu_item, FALSE);
   else
     gtk_widget_set_sensitive (mnu_item, TRUE);
}

/* Updates the sensitivity of the menu-items depending on the state of the program */
void gui_update_menu_sensivity (void)
{
   GtkWidget *mnu_item;

   mnu_item = lookup_widget (main_window, "close_all_connections");
   if (srv_get_activities_in_memory () == NULL)
     gtk_widget_set_sensitive (mnu_item, FALSE);
   else
     gtk_widget_set_sensitive (mnu_item, TRUE);
}

/* Creates and displays the popup-menu for the activities */
void gui_display_activity_popup (popup_src_t source)
{
   static GtkWidget *mnu = NULL;
   GtkWidget *item;

   if (!mnu)
     mnu = create_menu_act_popup ();

   /* Not all menu-items should be clickable (depending on where the user popped-up the menu) */
   item = lookup_widget (mnu, "mnu_close_tx");
   if (source == POPUP_FROM_USRTREE)
     gtk_widget_set_sensitive (item, FALSE);
   else if (source == POPUP_FROM_ACTTREE)
     gtk_widget_set_sensitive (item, TRUE);

   gtk_menu_popup (GTK_MENU(mnu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time ());
}

/* Appends a row to the activities list... */
void gui_add_to_activities (PGuiActivityRow line)
{
   GtkTreeIter iter;
   static GtkWidget *tree_activities = NULL;
   static GtkTreeModel *model = NULL;
   
   if (!tree_activities)
   {
      tree_activities = lookup_widget (main_window, "tree_activity");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_activities));
   }
   gtk_list_store_append (GTK_LIST_STORE (model), &iter);
   gtk_list_store_set (GTK_LIST_STORE (model), &iter,
		       COL_ACT_ICON, line.icon,
		       COL_ACT_TEXT, line.text,
		       COL_ACT_ID, line.id,
		       -1);
}

void gui_add_to_online_users (PGuiUserRow line)
{
   GtkTreeIter iter;
   static GtkWidget *tree_users = NULL;
   static GtkTreeModel *model = NULL;
   
   if (!tree_users)
   {
      tree_users = lookup_widget (main_window, "tree_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_users));
   }
   gtk_list_store_append (GTK_LIST_STORE (model), &iter);
   gtk_list_store_set (GTK_LIST_STORE (model), &iter,
		       COL_USR_ICON, line.icon,
		       COL_USR_USER, line.user,
		       COL_USR_HOST, line.host,
		       COL_USR_NUM_CONNECTIONS, line.num_connections,
		       -1);
   
}

void gui_add_to_usrman_list (PGuiUsermanRow line)
{
   GtkTreeIter iter;
   static GtkWidget *tree_usrman = NULL;
   static GtkTreeModel *model = NULL;

   if (!tree_usrman)
   {
      tree_usrman = lookup_widget (dlg_usrman, "treeview_dlg_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_usrman));
   }
   gtk_list_store_append (GTK_LIST_STORE (model), &iter);
   gtk_list_store_set (GTK_LIST_STORE (model), &iter,
		       COL_USRMAN_ICON, line.icon,
		       COL_USRMAN_LOGIN, line.login,
		       COL_USRMAN_REALNAME, line.realname,
		       -1);
}

/* FIXME: Needed? */
gchar *gui_gen_progress_string (gint total, gint current)
{
   if (total > 100 * MB)
   {
      /* Valuse in Mb */
      if (current < 10 * MB)
	return g_strdup_printf (_("%.1f Mb (%d Mb total)"), (double) current / MB, total / MB);
      else
	return g_strdup_printf (_("%d Mb (%d Mb total)"), current / MB, total / MB);
   }
   else if (total > 100 * KB)
     /* Values in Kb */
     return g_strdup_printf ("%d Kb (%d Kb total)", current / KB, total / KB);
   else
     /* Values in bytes */
     return g_strdup_printf ("%d b (%d b total)", current, total);
}

void gui_clear_activity_list (void)
{
   static GtkWidget *tree_activities = NULL;
   static GtkTreeModel *model = NULL;
   
   if (!tree_activities)
   {
      tree_activities = lookup_widget (main_window, "tree_activity");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_activities));
   }
   gtk_list_store_clear (GTK_LIST_STORE (model));
}

void gui_clear_users_list (void)
{
   static GtkWidget *tree_users = NULL;
   static GtkTreeModel *model = NULL;
   
   if (!tree_users)
   {
      tree_users = lookup_widget (main_window, "tree_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_users));
   }
   gtk_list_store_clear (GTK_LIST_STORE (model));
}

void gui_clear_usersman_list (void)
{
   static GtkWidget *tree_userman = NULL;
   static GtkTreeModel *model = NULL;
   
   if (!tree_userman)
   {
      tree_userman = lookup_widget (dlg_usrman, "treeview_dlg_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_userman));
   }
   gtk_list_store_clear (GTK_LIST_STORE (model));
}

gint gui_fill_userman_list (void)
{
   GList *user_list = NULL, *node;
   gint ret = 0;
   static GdkPixbuf *icon_usr = NULL;
   PGuiUsermanRow line;
   PAvailableUser *usr;
   
   if (!icon_usr)
     icon_usr = create_pixbuf ("user.png");
   line.icon = icon_usr;
   gui_clear_usersman_list ();
   user_list = usr_get_available_users ();
   node = user_list;
   while (node)
   {
      ret++;
      usr = (PAvailableUser *) node->data;
      line.login =  usr->login;
      line.realname = usr->realname;
      gui_add_to_usrman_list (line);
      g_free (line.login);
      g_free (line.realname);
/*       g_free (usr->login); */
/*       g_free (usr->realname); */
      node = g_list_next (node);
   }
   g_list_free (user_list);

   return ret;
}

void gui_userman_select_user (const gchar *user)
{
   static GtkWidget *tree_userman = NULL;
   static GtkTreeModel *model = NULL;
   static GtkTreeSelection *sel = NULL;
   GtkTreeIter iter;
   gboolean more, found = FALSE;
   gchar *fetched_user;

   if (!tree_userman)
   {
      tree_userman = lookup_widget (dlg_usrman, "treeview_dlg_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_userman));
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_userman));
   }
   more = gtk_tree_model_get_iter_first (model, &iter);
   while (more)
   {
      gtk_tree_model_get (model, &iter,
			  COL_USR_USER, &fetched_user,
			  -1);
      if (misc_str_is_equal (fetched_user, user))
      {
	 found = TRUE;
	 break;
      }
      more = gtk_tree_model_iter_next (model, &iter);
   }
   if (found)
   {
      gtk_tree_selection_select_iter (sel, &iter);
      g_signal_emit_by_name ((gpointer) tree_userman, "cursor-changed", tree_userman, NULL);
   }
}

void gui_update_info_frame (PActivity *a)
{
   static GtkWidget *w_name = NULL;
   static GtkWidget *w_host = NULL;
   static GtkWidget *w_action = NULL;
   static GtkWidget *w_progress = NULL;
   static GtkWidget *w_elapsed = NULL;
   static GtkWidget *w_eta = NULL;
   gchar *tmp, *num_cur_str, *num_tot_str, *speed_str, *percent;
#define DATE_STR_LEN 50

   tmp = num_cur_str = num_tot_str = speed_str = percent = NULL;

   if (!w_name)
   {
      w_name = lookup_widget (main_window, "lbl_username");
      w_host = lookup_widget (main_window, "lbl_host");
      w_action = lookup_widget (main_window, "lbl_action");
      w_progress = lookup_widget (main_window, "lbl_progress");
      w_elapsed = lookup_widget (main_window, "lbl_elapsed_time");
      w_eta = lookup_widget (main_window, "lbl_remaining_time");
   }
   if (!a)
   {
      /* Nothing sent, clear labels */
      gtk_label_set_text (GTK_LABEL (w_name), "");
      gtk_label_set_text (GTK_LABEL (w_host), "");
      gtk_label_set_text (GTK_LABEL (w_action), "");
      gtk_label_set_text (GTK_LABEL (w_progress), "");
      gtk_label_set_text (GTK_LABEL (w_elapsed), "");
      gtk_label_set_text (GTK_LABEL (w_eta), "");
      return;
   }
   gtk_label_set_text (GTK_LABEL (w_name), a->username);
   gtk_label_set_text (GTK_LABEL (w_host), a->remote_addr);

   tmp = g_strdup_printf ("%s '%s'", a->state == FTPWHO_STATE_DOWNLOAD ? _("Downloading") : _("Uploading"), a->filename);
   gtk_label_set_text (GTK_LABEL (w_action), tmp);
   g_free (tmp);
   
   
   num_cur_str = _number_separate (a->download_current_size);
   speed_str   = _gen_speed_string (a->speed);
   if (a->download_total_size > 0)
   {
      percent = g_strdup_printf ("%.2f", (double) 100 * (double) a->download_current_size / (double) a->download_total_size);
      num_tot_str = _number_separate (a->download_total_size);
   }
   else
   {
      num_tot_str = g_strdup ("???");
      percent = g_strdup ("??");
   }
   tmp = g_strdup_printf (_("%s of %s bytes (%s%%) %s"), num_cur_str, num_tot_str,
			  percent, speed_str);
   gtk_label_set_text (GTK_LABEL (w_progress), tmp);
   
   g_free (percent);
   g_free (tmp);
   g_free (num_cur_str);
   g_free (num_tot_str);
   g_free (speed_str);

   tmp = _sec_to_time (a->online_since);
   gtk_label_set_text (GTK_LABEL (w_elapsed), tmp);
   if (a->download_total_size - a->download_current_size == 0 || a->speed == 0)
     tmp = g_strdup ("-");
   else
     tmp = _sec_to_time ((a->download_total_size - a->download_current_size) / a->speed);
   gtk_label_set_text (GTK_LABEL (w_eta), tmp);
   
   g_free (tmp);
}

gchar *_sec_to_time (gulong sec)
{
   gchar *retval;
   gint hr, min;

   hr = sec / 3600;
   sec -= hr * 3600;
   min = sec / 60;
   sec -= min * 60;
     
   retval = g_strdup_printf ("%02d:%02d:%02d", hr, min, (gint) sec);
   return retval;
}
   
gchar *_number_separate (gint num)
{
   GString *str;
   gchar *retval;
   gboolean more_prev = FALSE;
   
   str = g_string_new ("");
   if (num > 1000000000)
   {
      g_string_append_printf (str, "%03d,", num / 1000000000);
      num = num % 1000000000;
      more_prev = TRUE;
   }
   if (num > 1000000)
   {
      if (more_prev)
	g_string_append_printf (str, "%03d,", num / 1000000);
      else
	g_string_append_printf (str, "%d,", num / 1000000);
      num = num % 1000000;
      more_prev = TRUE;
   }
   if (num > 1000)
   {
      if (more_prev)
	g_string_append_printf (str, "%03d,", num / 1000);
      else
	g_string_append_printf (str, "%d,", num / 1000);
      num = num % 1000;
      more_prev = TRUE;
   }
   g_string_append_printf (str, "%03d", num);
   retval = str->str;
   g_string_free (str, FALSE);
   return retval;
}

gchar *_gen_speed_string (gulong speed)
{
   gchar *retval;

   if (speed > 0UL) 
   {
      if (speed < 1024UL) {
	 retval = g_strdup_printf("%4luK/s", speed);
      } else {
	 speed /= 1024UL;
	 if (speed < 1024UL) {
	    retval = g_strdup_printf("%4luM/s", speed);
	 } else {
	    speed /= 1024UL;
	    if (speed < 1024UL) {
	       retval = g_strdup_printf("%4luG/s", speed);
	    } else {
	       retval = g_strdup ("");
	    }                    
	 }
      }    
   }
   else
   {
      retval = g_strdup ("00 b/s (stalled)");
   }
   return retval;
}

GtkWidget *gui_create_menu_from_glist (GList *l)
{
   GList *node;
   GtkWidget *menu, *menu_item;
   
   menu = gtk_menu_new ();
   node = g_list_first (l);
   while (node)
   {
      menu_item = gtk_menu_item_new_with_label ((gchar *) node->data);
      gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);
      node = g_list_next (node);
   }
   gtk_widget_show_all (menu);
   return menu;
}

gint gui_tree_act_get_selected_id (void)
{
   static GtkWidget *tree_activities = NULL;
   static GtkTreeModel *model = NULL;
   static GtkTreeSelection *sel;
   GtkTreeIter iter;
   gboolean selected;
   gint id;

   if (!tree_activities)
   {
      tree_activities = lookup_widget (main_window, "tree_activity");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_activities));
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_activities));
   }
   selected = gtk_tree_selection_get_selected (sel, &model, &iter);
   if (!selected) /* Nothing is selected! */
     return -1;
   gtk_tree_model_get (model, &iter, 
		       COL_ACT_ID, &id,
		       -1);
   return id;
}

gboolean gui_tree_usr_get_selected_user (gchar **user, gchar **host)
{
   static GtkWidget *tree_users = NULL;
   static GtkTreeModel *model = NULL;
   static GtkTreeSelection *sel;
   GtkTreeIter iter;
   gboolean selected;

   if (!tree_users)
   {
      tree_users = lookup_widget (main_window, "tree_users");
      model = gtk_tree_view_get_model (GTK_TREE_VIEW (tree_users));
      sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_users));
   }
   selected = gtk_tree_selection_get_selected (sel, &model, &iter);
   if (!selected) /* Nothing is selected! */
     return FALSE;
   gtk_tree_model_get (model, &iter, 
		       COL_USR_USER, user,
		       COL_USR_HOST, host,
		       -1);
   if (host && user)
     return TRUE;
   else
     return FALSE;
}

/* Convenient dialog creation functions */
void gui_display_msgdialog_checkbox
(const gchar *primary, const gchar *secondary, const gchar *icon_type, GtkWidget *parent, gboolean *show_again)
{
   GtkWidget *dlg_alert;
   GtkWidget *dialog_vbox;
   GtkWidget *hbox, *vbox;
   GtkWidget *icon;
   GtkWidget *lbl;
   GtkWidget *okbutton, *checkbox;
   gchar *text;

   dlg_alert = gtk_dialog_new ();
   gtk_container_set_border_width (GTK_CONTAINER (dlg_alert), 6);
   gtk_window_set_modal (GTK_WINDOW (dlg_alert), TRUE);
   gtk_window_set_resizable (GTK_WINDOW (dlg_alert), FALSE);
   gtk_dialog_set_has_separator (GTK_DIALOG (dlg_alert), FALSE);
   if (parent)
     gtk_window_set_transient_for (GTK_WINDOW (dlg_alert), GTK_WINDOW (parent));
   gtk_window_set_title (GTK_WINDOW (dlg_alert), "");
   dialog_vbox = GTK_DIALOG (dlg_alert)->vbox;
   gtk_widget_show (dialog_vbox);
   
   hbox = gtk_hbox_new (FALSE, 12);
   gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

   icon = gtk_image_new_from_stock (icon_type, GTK_ICON_SIZE_DIALOG);
   gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, 0);
   gtk_misc_set_alignment (GTK_MISC (icon), 0.5, 0);
   
   vbox = gtk_vbox_new (FALSE, 6);
   gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
   
   text = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s", primary, secondary);
   lbl = gtk_label_new (text);
   g_free (text);
   gtk_box_pack_start (GTK_BOX (vbox), lbl, FALSE, FALSE, 0);
   gtk_label_set_use_markup (GTK_LABEL (lbl), TRUE);
   gtk_label_set_justify (GTK_LABEL (lbl), GTK_JUSTIFY_LEFT);
   gtk_label_set_line_wrap (GTK_LABEL (lbl), TRUE);
   gtk_misc_set_alignment (GTK_MISC (lbl), 0.5, 0);
   
   checkbox = gtk_check_button_new_with_mnemonic (_("_Don't show this dialog again"));
   gtk_box_pack_start (GTK_BOX (vbox), checkbox, FALSE, FALSE, 0);

   gtk_button_box_set_layout (GTK_BUTTON_BOX (GTK_DIALOG (dlg_alert)->action_area), GTK_BUTTONBOX_END);
   
   okbutton = gtk_button_new_from_stock ("gtk-ok");
   gtk_dialog_add_action_widget (GTK_DIALOG (dlg_alert), okbutton, GTK_RESPONSE_OK);
   GTK_WIDGET_SET_FLAGS (okbutton, GTK_CAN_DEFAULT);
   gtk_widget_show_all (dlg_alert);
   
   gtk_dialog_run (GTK_DIALOG (dlg_alert));
   if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbox)))
     *show_again = FALSE;
   else
     *show_again = TRUE;
   gtk_widget_destroy (dlg_alert);
}

     
void gui_display_msgdialog (const gchar *primary, const gchar *secondary, const gchar *icon_type, GtkWidget *parent)
{
   GtkWidget *dlg_alert;
   GtkWidget *dialog_vbox;
   GtkWidget *hbox;
   GtkWidget *icon;
   GtkWidget *lbl;
   GtkWidget *okbutton;
   gchar *text;

   dlg_alert = gtk_dialog_new ();
   gtk_container_set_border_width (GTK_CONTAINER (dlg_alert), 6);
   gtk_window_set_modal (GTK_WINDOW (dlg_alert), TRUE);
   gtk_window_set_resizable (GTK_WINDOW (dlg_alert), FALSE);
   gtk_dialog_set_has_separator (GTK_DIALOG (dlg_alert), FALSE);
   if (parent)
     gtk_window_set_transient_for (GTK_WINDOW (dlg_alert), GTK_WINDOW (parent));
   gtk_window_set_title (GTK_WINDOW (dlg_alert), "");
   dialog_vbox = GTK_DIALOG (dlg_alert)->vbox;
   gtk_widget_show (dialog_vbox);
   
   hbox = gtk_hbox_new (FALSE, 12);
   gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, TRUE, TRUE, 0);
   gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

   icon = gtk_image_new_from_stock (icon_type, GTK_ICON_SIZE_DIALOG);
   gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, 0);
   gtk_misc_set_alignment (GTK_MISC (icon), 0.5, 0);
   
   text = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s", primary, secondary);
   lbl = gtk_label_new (text);
   g_free (text);
   gtk_box_pack_start (GTK_BOX (hbox), lbl, FALSE, FALSE, 0);
   gtk_label_set_use_markup (GTK_LABEL (lbl), TRUE);
   gtk_label_set_justify (GTK_LABEL (lbl), GTK_JUSTIFY_LEFT);
   gtk_label_set_line_wrap (GTK_LABEL (lbl), TRUE);
   gtk_misc_set_alignment (GTK_MISC (lbl), 0.5, 0);
   
   gtk_button_box_set_layout (GTK_BUTTON_BOX (GTK_DIALOG (dlg_alert)->action_area), GTK_BUTTONBOX_END);
   
   okbutton = gtk_button_new_from_stock ("gtk-ok");
   gtk_dialog_add_action_widget (GTK_DIALOG (dlg_alert), okbutton, GTK_RESPONSE_OK);
   GTK_WIDGET_SET_FLAGS (okbutton, GTK_CAN_DEFAULT);
   gtk_widget_show_all (dlg_alert);
   
   gtk_dialog_run (GTK_DIALOG (dlg_alert));
   gtk_widget_destroy (dlg_alert);
}

/* Creates and runs a HIGified confirmation dialog asking the user to confirm something.
 * Args: 
 *	primary: The 'header' text, keep short and descriptive
 *	secondary: The longer description of the action to confirm
 *	txt_btn_yes: Text to use instead of 'Yes' (if NULL, then stockbutton YES is used)
 *	txt_btn_no: Text to use instead of 'No' (if NULL, then stockbutton NO is used)
 * Returns:
 *	the GTK_RESPONSE_ID as returned by gtk_dialog_run()
 */
gint gui_display_confirmdialog (const gchar *primary, const gchar *secondary, const gchar *txt_btn_yes, const gchar *txt_btn_no)
{
  GtkWidget *dlg_confirm_yes_no;
  GtkWidget *dlg_vbox;
  GtkWidget *hbox;
  GtkWidget *img;
  GtkWidget *lbl;
  GtkWidget *dialog_action_area;
  GtkWidget *btn_no;
  GtkWidget *btn_yes;
  GtkWidget *alignment;
  GtkWidget *btn_hbox;
  gchar *text;
  gint ret;

  dlg_confirm_yes_no = gtk_dialog_new ();
  gtk_container_set_border_width (GTK_CONTAINER (dlg_confirm_yes_no), 6);
  gtk_window_set_modal (GTK_WINDOW (dlg_confirm_yes_no), TRUE);
  gtk_window_set_resizable (GTK_WINDOW (dlg_confirm_yes_no), FALSE);
  gtk_dialog_set_has_separator (GTK_DIALOG (dlg_confirm_yes_no), FALSE);

  dlg_vbox = GTK_DIALOG (dlg_confirm_yes_no)->vbox;
  gtk_widget_show (dlg_vbox);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (dlg_vbox), hbox, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

  img = gtk_image_new_from_stock ("gtk-dialog-warning", GTK_ICON_SIZE_DIALOG);
  gtk_widget_show (img);
  gtk_box_pack_start (GTK_BOX (hbox), img, FALSE, FALSE, 0);
  gtk_misc_set_alignment (GTK_MISC (img), 0, 0.5);

  text = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s", primary, secondary);
  lbl = gtk_label_new (text);
  g_free (text);
  gtk_widget_show (lbl);
  gtk_box_pack_start (GTK_BOX (hbox), lbl, FALSE, FALSE, 0);
  gtk_label_set_use_markup (GTK_LABEL (lbl), TRUE);
  gtk_label_set_justify (GTK_LABEL (lbl), GTK_JUSTIFY_LEFT);
  gtk_label_set_line_wrap (GTK_LABEL (lbl), TRUE);
  gtk_misc_set_alignment (GTK_MISC (lbl), 0, 0);

  dialog_action_area = GTK_DIALOG (dlg_confirm_yes_no)->action_area;
  gtk_widget_show (dialog_action_area);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_END);

  if (txt_btn_no == NULL)
    btn_no = gtk_button_new_from_stock ("gtk-no");
  else
  {
     btn_no = gtk_button_new ();

     alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
     gtk_widget_show (alignment);
     gtk_container_add (GTK_CONTAINER (btn_no), alignment);

     btn_hbox = gtk_hbox_new (FALSE, 2);
     gtk_widget_show (btn_hbox);
     gtk_container_add (GTK_CONTAINER (alignment), btn_hbox);
     
     img = gtk_image_new_from_stock ("gtk-no", GTK_ICON_SIZE_BUTTON);
     gtk_widget_show (img);
     gtk_box_pack_start (GTK_BOX (btn_hbox), img, FALSE, FALSE, 0);

     lbl = gtk_label_new_with_mnemonic (txt_btn_no);
     gtk_widget_show (lbl);
     gtk_box_pack_start (GTK_BOX (btn_hbox), lbl, FALSE, FALSE, 0);
     gtk_label_set_justify (GTK_LABEL (lbl), GTK_JUSTIFY_LEFT);
  }
     
  gtk_widget_show (btn_no);
  gtk_dialog_add_action_widget (GTK_DIALOG (dlg_confirm_yes_no), btn_no, GTK_RESPONSE_NO);
  GTK_WIDGET_SET_FLAGS (btn_no, GTK_CAN_DEFAULT);

  if (txt_btn_yes == NULL)
    btn_yes = gtk_button_new_from_stock ("gtk-yes");
  else
  {
     btn_yes = gtk_button_new ();
     gtk_widget_show (btn_yes);
     gtk_dialog_add_action_widget (GTK_DIALOG (dlg_confirm_yes_no), btn_yes, GTK_RESPONSE_YES);
     GTK_WIDGET_SET_FLAGS (btn_yes, GTK_CAN_DEFAULT);

     alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
     gtk_widget_show (alignment);
     gtk_container_add (GTK_CONTAINER (btn_yes), alignment);

     btn_hbox = gtk_hbox_new (FALSE, 2);
     gtk_widget_show (btn_hbox);
     gtk_container_add (GTK_CONTAINER (alignment), btn_hbox);
     
     img = gtk_image_new_from_stock ("gtk-yes", GTK_ICON_SIZE_BUTTON);
     gtk_widget_show (img);
     gtk_box_pack_start (GTK_BOX (btn_hbox), img, FALSE, FALSE, 0);

     lbl = gtk_label_new_with_mnemonic (txt_btn_yes);
     gtk_widget_show (lbl);
     gtk_box_pack_start (GTK_BOX (btn_hbox), lbl, FALSE, FALSE, 0);
     gtk_label_set_justify (GTK_LABEL (lbl), GTK_JUSTIFY_LEFT);
  }
  ret = gtk_dialog_run (GTK_DIALOG (dlg_confirm_yes_no));
  gtk_widget_destroy (dlg_confirm_yes_no);
  return ret;
}
