/* Copyright (c) 1997-1999 Miller Puckette.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

#include "m_pd.h"
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>

/* T.Grill - windows compatibility for Matju's console */
#ifdef _MSC_VER
#define vsnprintf _vsnprintf
#endif

/* T.Grill - linked list structure */
typedef struct s_printhook t_printhook;

struct s_printhook {
    t_printhookfn hook;
    t_printhook *nxt;
};

static t_printhook *printlist = NULL;

/* T.Grill - this is the central printing function
   Strings are passed to registered hook functions and to the console
*/
static void vprintcon(char *fmt,va_list ap)
{
    /* eventually buffer stuff and pass strings?! */

    /* call print hook functions */
    t_printhook *hk = printlist;
    for(; hk; hk = hk->nxt) hk->hook(fmt,ap);

    /* print to console */
    vfprintf(stderr,fmt,ap);
}

static void printcon(char *fmt,...)
{
    va_list ap;
    va_start(ap,fmt);
    vprintcon(fmt,ap);
    va_end(ap);
}

/* T.Grill - add print hook function */
int sys_addprinthook(t_printhookfn fn)
{
    t_printhook *hkel = getbytes(sizeof(t_printhook));
    hkel->hook = fn;
    /* insert as first element */
    hkel->nxt = printlist;
    printlist = hkel;

    return 0;
}

/* T.Grill - remove print hook function */
int sys_rmprinthook(t_printhookfn fn)
{
    t_printhook *hk = printlist,*hkprv = NULL;

    /* iterate through list */
    for(; hk && hk->hook != fn; hkprv = hk,hk = hk->nxt);

    if(hk) {
        /* function was found -> detach from list */
        if(hkprv)
            hkprv->nxt = hk->nxt;
        else 
            printlist = hk->nxt;

        freebytes(hk,sizeof(t_printhook));
        return 0;
    }
    else
        return -1;
}


void post(char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    vprintcon(fmt, ap);
    va_end(ap);
    printcon("\n");
}

/* T.Grill: This is identical to printcon .... */
void startpost(char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    vprintcon(fmt, ap);
    va_end(ap);
}

void poststring(char *s)
{
    printcon(" %s",s);
}

void postatom(int argc, t_atom *argv)
{
    int i;
    for (i = 0; i < argc; i++)
    {
    	char buf[80];
    	atom_string(argv+i, buf, 80);
    	poststring(buf);
    }
}

void postfloat(float f)
{
    char buf[80];
    t_atom a;
    SETFLOAT(&a, f);
    postatom(1, &a);
}

void endpost(void)
{
    printcon("\n");
}

void error(char *fmt, ...)
{
    va_list ap;
    va_start(ap, fmt);
    printcon("error: ");
    vprintcon(fmt, ap);
    va_end(ap);
    printcon("\n");
}

    /* here's the good way to log errors -- keep a pointer to the
    offending or offended object around so the user can search for it
    later. */

static void *error_object;
static char error_string[256];
void canvas_finderror(void *object);

void pd_error(void *object, char *fmt, ...)
{
    va_list ap;
    static int saidit = 0;
    va_start(ap, fmt);
    vsprintf(error_string, fmt, ap);
    va_end(ap);
    printcon("error: %s\n", error_string);
    error_object = object;
    if (!saidit)
    {
    	post("... you might be able to track this down from the Find menu.");
    	saidit = 1;
    }
}

void glob_finderror(t_pd *dummy)
{
    if (!error_object)
    	post("no findable error yet.");
    else
    {
    	post("last trackable error:");
    	post("%s", error_string);
    	canvas_finderror(error_object);
    }
}

void bug(char *fmt, ...)
{
    va_list ap;
    printcon("Consistency check failed: ");
    va_start(ap, fmt);
    vprintcon(fmt,ap);
    va_end(ap);
    printcon("\n");
}

    /* this isn't worked out yet. */
static char *errobject;
static char *errstring;

void sys_logerror(char *object, char *s)
{
    errobject = object;
    errstring = s;
}

void sys_unixerror(char *object)
{
    errobject = object;
    errstring = strerror(errno);
}

void sys_ouch(void)
{
    if (*errobject) error("%s: %s", errobject, errstring);
    else error("%s", errstring);
}

/* ---------------------------------------------------------------- */
/* matju's gui console, take II */

extern int console_scrollback;
static const int console_ln=2048;
static int console_li=0;
static char console_line[2048]="";

void guiprint(char *fmt, va_list ap) {
    char *special = "{}[]\\\"\n";
    int console_li_old = console_li;
    int countbrace, i, n;
    console_li += vsnprintf(console_line+console_li, console_ln-console_li, fmt, ap);
    console_line[console_ln-1]=0;
    if (!strchr(&console_line[console_li_old],'\n')) return;
    countbrace=0;
	n = strlen(console_line);
    for (i=n-1; i>=0; i--) if (strchr(special,console_line[i])) countbrace++;
    while (n+countbrace>=(int)sizeof(console_line)) n--;
    console_line[n+countbrace]=0;
    for (i=n-1; i>=0; i--) {
	int c=console_line[i];
	if (c=='\n') c='n';
	console_line[i+countbrace]=c;
        if (strchr(special,console_line[i])) console_line[i+(--countbrace)]='\\';
    }
    sys_vgui("post_to_gui \"%s\"\n",console_line);
    console_li=0;
    console_line[0]=0;
}
