#-*- coding:utf-8 -*-
# cython: profile=False

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Ported from GNUbik
# Original filename: cube.c, drwBlock.c
# Original copyright and license:  1998, 2003, 2004  Dale Mellor, John Darrington,  GPL3+

# no unicode_literals for compiled modules, because it leads to compile errors in numpy
from __future__ import print_function, division

# This line makes cython happy
global __name__, __package__    # pylint: disable=W0604
#px/__compiled = True
__compiled = False

__all__ = ['draw_cube', 'pick_cube', 'draw_cube_debug', 'Matrix']

from .debug import debug, DEBUG_PICK  # pylint: disable=W0614,W0401

#pxd/from gl cimport * #:
from OpenGL.GL import *     # pylint: disable=W0614,W0401

#px/cimport numpy as np
import numpy as np


debug('Importing module:', __name__)
debug('  from package:', __package__)
debug('  compiled:', __compiled)


#px/cdef enum:
if True:
    MAX_BLOCKS = 1000
    MAX_FACES = 6
    MAX_FACES_PER_BLOCK = 6
    _TILED = 0
    _MOSAIC = 1

#pxd/ctypedef float Vector[4]
Vector = lambda: [0] * 4
#pxd/ctypedef Vector* p_Vector
p_Vector = None
#pxd/ctypedef Vector Matrix[4]
Matrix = lambda: [Vector() for unused_i in range(4)]

#px/cdef struct Block:
class Block:     # pylint: disable=R0903
#px-
    def __init__(self):
        # The position from the centre of the cube,  and the rotation from the
        # 'natural' position.
        #px/Matrix transformation
        self.transformation = Matrix()  # Matrix
        
        # only used when animation.running==True otherwise the value is meaningless
        #px/bint in_motion
        self.in_motion = None
        
        #px/int cnt_faces
        self.cnt_faces = None
        #px/int idx_label[MAX_FACES_PER_BLOCK]
        self.idx_label = [None]*MAX_FACES_PER_BLOCK
        #px/int cnt_label[MAX_FACES_PER_BLOCK]
        self.cnt_label = [None]*MAX_FACES_PER_BLOCK
        #px/int faceno[MAX_FACES_PER_BLOCK]
        self.faceno = [None]*MAX_FACES_PER_BLOCK
        
        #px/int idx_quads
        self.idx_quads = None
        #px/int cnt_quads
        self.cnt_quads = None
        #px/int idx_triangles
        self.idx_triangles = None
        #px/int cnt_triangles
        self.cnt_triangles = None
        
#px/cdef struct Cube:
class cube:     # pylint: disable=W0232, R0903
    #px+unsigned int size
    #px+unsigned int number_blocks
    #px/Block blocks[MAX_BLOCKS]
    blocks = [Block() for __block in xrange(MAX_BLOCKS)]
    
    #px+float *model_data_label
    #px+float *model_data_lnormal
    #px+float *model_data_pick
    #px+GLubyte *model_data_pickcolor
    #px+int cnt_pick
    
    #px+float *model_data_texpos_tiled
    #px+float *model_data_texpos_mosaic
#px+cdef Cube cube

#px/cdef float color_bevel[3]
color_bevel = [None] * 3

#px/cdef struct Face:
class Face: pass    # pylint: disable=W0232, C0321, R0903
    #px+float color[3]
    #px+int texName
    #px+int distr
#px/cdef Face face_rendering[MAX_FACES]
face_rendering = [Face() for __i in range(MAX_FACES)]

#px/cdef struct Animation_Struct:
class animation:    # pylint: disable=W0232, R0903
    #px+bint running
    #px+float angle, angle_max
    #px/float rotation[3]
    rotation = [None] * 3
#px+cdef Animation_Struct animation


def init_module():
    #px+cdef int i
    
    cube.size = cube.number_blocks = 0
    #cube.blocks
    
    #cube.model_data_label
    #cube.model_data_lnormal
    #cube.model_data_pick
    #cube.model_data_pickcolor
    cube.cnt_pick = 0
    
    #cube.model_data_texpos_tiled
    #cube.model_data_texpos_mosaic
    
    for i in xrange(3):
        color_bevel[i] = .1
        
    for i in xrange(MAX_FACES):
        #px-
        face_rendering[i].color = [0]*3
        set_face_rendering(i, red=1., green=1., blue=1., imagemode=_TILED)
        face_rendering[i].texName = -1
        
    animation.running = 0
    animation.angle = animation.angle_max = 0
    for i in xrange(3):
        animation.rotation[i] = 0.

#px/cpdef set_animation_block(int block_id, bint selected):
def set_animation_block(block_id, selected):
    cube.blocks[block_id].in_motion = selected

def set_transformations(blocks):
    #px+cdef int b,i,j
    for b in range(cube.number_blocks):
        for i in range(4):
            for j in range(4):
                cube.blocks[b].transformation[i][j] = float(blocks[b][i][j])

#px/cpdef start_animate(float angle_max, float axisx, float axisy, float axisz):
def start_animate(angle_max, axisx, axisy, axisz):
    animation.angle = 0.0
    animation.angle_max = angle_max
    animation.running = 1
    animation.rotation[0] = axisx
    animation.rotation[1] = axisy
    animation.rotation[2] = axisz
    
#px/cpdef step_animate(float increment):
def step_animate(increment):
    animation.angle -= increment
    return abs(animation.angle) < animation.angle_max

def end_animate():
    animation.running = 0

#pxd/cdef void draw_cube():
def draw_cube():
    if DEBUG_PICK:
        pick_cube()
        return
        
    #px+cdef int i
    #px+cdef p_Vector M
    
    for i in range(cube.number_blocks):
        glPushMatrix()
        if animation.running and cube.blocks[i].in_motion:
            glRotatef(animation.angle,
                    animation.rotation[0], animation.rotation[1], animation.rotation[2])
        #px/M = <p_Vector>cube.blocks[i].transformation
        M = cube.blocks[i].transformation
        #px/glMultMatrixf(<float*>M)
        glMultMatrixf(M)
        draw_block(i)
        glPopMatrix ()
        
#px/cdef void draw_block(int block_id):
def draw_block(block_id):
    #px+cdef int i
    
    glEnableClientState(GL_VERTEX_ARRAY)
    glEnableClientState(GL_NORMAL_ARRAY)
    glVertexPointer(3, GL_FLOAT, 0, cube.model_data_label)
    glNormalPointer(GL_FLOAT, 0, cube.model_data_lnormal)
    glColor3fv(color_bevel)
    glDrawArrays(GL_QUADS, cube.blocks[block_id].idx_quads, cube.blocks[block_id].cnt_quads)
    glDrawArrays(GL_TRIANGLES, cube.blocks[block_id].idx_triangles, cube.blocks[block_id].cnt_triangles)
    
    for i in range(cube.blocks[block_id].cnt_faces):
        faceno = cube.blocks[block_id].faceno[i]
        # render the colors (ie the little sticky labels)
        glColor3fv(face_rendering[faceno].color)
        if face_rendering[faceno].texName != -1:
            glEnableClientState(GL_TEXTURE_COORD_ARRAY)
            glEnable(GL_TEXTURE_2D)
            glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL)
            glBindTexture(GL_TEXTURE_2D, face_rendering[faceno].texName)
            if face_rendering[faceno].distr == _MOSAIC:
                glTexCoordPointer(2, GL_FLOAT, 0, cube.model_data_texpos_mosaic)
            else: # TILED 
                glTexCoordPointer(2, GL_FLOAT, 0, cube.model_data_texpos_tiled)
            glDrawArrays(GL_QUADS, cube.blocks[block_id].idx_label[i], cube.blocks[block_id].cnt_label[i])
            glDisableClientState(GL_TEXTURE_COORD_ARRAY)
            glDisable(GL_TEXTURE_2D)
        else:
            glDrawArrays(GL_QUADS, cube.blocks[block_id].idx_label[i], cube.blocks[block_id].cnt_label[i])
            
    glDisableClientState(GL_VERTEX_ARRAY)
    glDisableClientState(GL_NORMAL_ARRAY)
    
#pxd/cdef void pick_cube():
def pick_cube():
    #px+cdef int block_id, face
    
    glEnableClientState(GL_VERTEX_ARRAY)
    glEnableClientState(GL_COLOR_ARRAY)
    glVertexPointer(3, GL_FLOAT, 0, cube.model_data_pick)
    glColorPointer(3, GL_UNSIGNED_BYTE, 0, cube.model_data_pickcolor)
    glDrawArrays(GL_TRIANGLES, 0, cube.cnt_pick)
    glDisableClientState(GL_VERTEX_ARRAY)
    glDisableClientState(GL_COLOR_ARRAY)
    
def set_model(model):
    #px+cdef int block
    
    # The number of blocks per side of the cube.
    cube.size = model.sizes[0]
    cube.number_blocks = len(model.blocks)
    assert cube.number_blocks <= MAX_BLOCKS
    
    #### Create the raw GL data ####
    #px+cdef np.ndarray[float, mode="c"] glvertices
    #px+cdef np.ndarray[float, mode="c"] glnormals
    #px+cdef np.ndarray[GLubyte, mode="c"] glcolors
    #px+cdef np.ndarray[float, mode="c"] gltexpostiled, gltexposmosaic
    #px+cdef int idx_vbevelq, idx_vbevelt
    (   glvertices, glnormals, glcolors, gltexpostiled, gltexposmosaic,
        blockinfos,
        idx_vbevelq, idx_vbevelt
    ) = model.gl_data()
    
    #px/cube.model_data_label = &glvertices[0]
    cube.model_data_label = glvertices[0:]
    #px/cube.model_data_lnormal = &glnormals[0]
    cube.model_data_lnormal = glnormals[0:]
    #px/cube.model_data_texpos_tiled = &gltexpostiled[0]
    cube.model_data_texpos_tiled = gltexpostiled[0:]
    #px/cube.model_data_texpos_mosaic = &gltexposmosaic[0]
    cube.model_data_texpos_mosaic = gltexposmosaic[0:]
    
    #px+cdef int idx
    lidx = 0
    qidx = idx_vbevelq // 3
    tidx = idx_vbevelt // 3
    for block in range(cube.number_blocks):
        cube.blocks[block].cnt_faces = len(blockinfos[block][0])
        for i, (cnt, faceno) in enumerate(blockinfos[block][0]):
            cube.blocks[block].idx_label[i] = lidx
            cube.blocks[block].cnt_label[i] = cnt
            cube.blocks[block].faceno[i] = faceno
            lidx += cnt
        cube.blocks[block].idx_quads = qidx
        cube.blocks[block].cnt_quads = blockinfos[block][1]
        qidx += blockinfos[block][1]
        cube.blocks[block].idx_triangles = tidx
        cube.blocks[block].cnt_triangles = blockinfos[block][2]
        tidx += blockinfos[block][2]
        
#px/cpdef set_pick_model(np.ndarray[float, mode="c"] glvertices, np.ndarray[GLubyte, mode="c"] glcolors):
def set_pick_model(glvertices, glcolors):
    #px/cube.model_data_pick = &glvertices[0]
    cube.model_data_pick = glvertices
    #px/cube.model_data_pickcolor = &glcolors[0]
    cube.model_data_pickcolor = glcolors
    cube.cnt_pick = len(glvertices) // 3
    
#pxd/cdef void draw_cube_debug():
def draw_cube_debug():
    #px+cdef float offset
    offset = 1.4 * cube.size
    
    # X axis
    glPushMatrix()
    glTranslatef(-offset, -offset, -1.*cube.size)
    glBegin(GL_LINES)
    glColor3f(1., 0., 0.)
    glVertex3f(0, 0, 0)
    glVertex3f(2*cube.size, 0, 0)
    glEnd()
    
    glRasterPos3d(offset*1.1, 0, 0)
    glPopMatrix()
    
    # Y axis
    glPushMatrix()
    glTranslatef(-offset, -offset, -1.*cube.size)
    glBegin(GL_LINES)
    glColor3f(0., 1., 0.)
    glVertex3f(0, 0, 0)
    glVertex3f(0, 2*cube.size, 0)
    glEnd()
    
    glRasterPos3d(0.1*offset,  offset, 0)
    glPopMatrix()
    
    # Z axis
    glPushMatrix()
    glTranslatef(-offset, -offset, -1.*cube.size)
    glBegin(GL_LINES)
    glColor3f(0., 0., 1.)
    glVertex3f(0, 0, 0)
    glVertex3f(0, 0, 2*cube.size)
    glEnd()
    
    glRasterPos3d(0.1*offset, 0,  offset)
    glPopMatrix ()

def draw_dot(x, y):
    #px/cdef float *pixel = [1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,]
    pixel = [1.] * 16
    glWindowPos2i(x, y)
    glDrawPixels(4, 4, GL_GREEN, GL_FLOAT, pixel)
    
def draw_select_debug(modelview_points, viewport_points):
    glDisable(GL_DEPTH_TEST)
    glColor3f(1., 1., 1.)
    glBegin(GL_LINES)
    for point in modelview_points:
        glVertex3f(point[0], point[1], point[2])
    glEnd()
    for point in viewport_points:
        draw_dot(point[0], point[1])
    glEnable(GL_DEPTH_TEST)
    
    
def set_face_rendering(faceidx, red=-1, green=-1, blue=-1, imagemode=-1):
    # pylint: disable=C0321
    if red >= 0:    face_rendering[faceidx].color[0] = red
    if green >= 0:  face_rendering[faceidx].color[1] = green
    if blue >= 0:   face_rendering[faceidx].color[2] = blue
    if imagemode >= 0:  face_rendering[faceidx].distr = imagemode
    
#px/cpdef set_face_texture(int faceidx, int texname):
def set_face_texture(faceidx, texname):
    prevtexname = face_rendering[faceidx].texName
    face_rendering[faceidx].texName = texname
    return prevtexname
    

