/* gsm objects, a Python wrapper of libgsm. */

#include "Python.h"
#include "gsm.h"


typedef struct {
	PyObject_HEAD
	gsm ghandle;
	int bigendian;
} GsmObject;

static PyTypeObject Gsm_Type;

#define GsmObject_Check(v)	((v)->ob_type == &Gsm_Type)


static GsmObject *
newGsmObject(int bigendian)
{
	GsmObject *self;
	self = PyObject_New(GsmObject, &Gsm_Type);
	if (self == NULL) {
		return NULL;
	}
	self->ghandle = gsm_create();
	self->bigendian = bigendian;
	return self;
}


/* Gsm methods */

static void
Gsm_dealloc(GsmObject *self)
{
	gsm_destroy(self->ghandle);
	PyObject_Del(self);
}

static PyObject *
Gsm_encode(GsmObject *self, PyObject *args)
{
	char *s;
	int size, i;
	gsm_signal src[160];
	gsm_frame dst;

	if (!PyArg_ParseTuple(args, "s#:encode", &s, &size)) {
		return NULL;
	}
	if (size != 320) {
		PyErr_SetString(PyExc_ValueError, "passed string must have length of 320");
		return NULL;
	}
	/* convert bytes to 16 bit values. */
	for (i = 0; i < 160; i++) {
		if (self->bigendian) {
			src[i] = ((s[i * 2] & 0xff) << 8) | (s[(i*2)+1] & 0xff);
		} else {
			src[i] = ((s[(i * 2)+1] & 0xff) << 8) | (s[i*2] & 0xff);
		}
	}
	gsm_encode(self->ghandle, src, dst);
	return Py_BuildValue("s#", (char*) dst, 33);
}

static PyObject *
Gsm_decode(GsmObject *self, PyObject *args)
{
	int size, i;
	gsm_signal dst[160];
	gsm_byte *src;
	char result[320];
	
	if (!PyArg_ParseTuple(args, "s#:encode", (char **)&src, &size)) {
		return NULL;
	}
	if (size != 33) {
		PyErr_SetString(PyExc_ValueError, "passed string must have length of 33");
		return NULL;
	}
	if (gsm_decode(self->ghandle, src, dst) < 0) {
		PyErr_SetString(PyExc_ValueError, "input could not be decoded");
		return NULL;
	}
	/* convert bytes to 16 bit values. */
	for (i = 0; i < 160; i++) {
		if (self->bigendian) {
			result[i * 2] = (dst[i] >> 8) & 0xff;
			result[(i * 2)+1] = dst[i] & 0xff;
		} else {
			result[(i * 2)+1] = (dst[i] >> 8) & 0xff;
			result[i * 2] = dst[i] & 0xff;
		}
	}
	return Py_BuildValue("s#", (char*) result, 320);
}

static PyMethodDef GsmType_methods[] = {
	{"encode",	(PyCFunction)Gsm_encode,	METH_VARARGS,
	 PyDoc_STR("encode(s) -> encoded string")},
	{"decode",	(PyCFunction)Gsm_decode,	METH_VARARGS,
	 PyDoc_STR("decode(s) -> decoded string")},
	{NULL,		NULL}		/* sentinel */
};


static PyTypeObject Gsm_Type = {
	/* The ob_type field must be initialized in the module init function
	 * to be portable to Windows without using C++. */
	PyObject_HEAD_INIT(NULL)
	0,			/*ob_size*/
	"gsm.Gsm",		/*tp_name*/
	sizeof(GsmObject),	/*tp_basicsize*/
	0,			/*tp_itemsize*/
	/* methods */
	(destructor)Gsm_dealloc, /*tp_dealloc*/
	0,			/*tp_print*/
	0, /*tp_getattr*/
	0, /*tp_setattr*/
	0,			/*tp_compare*/
	0,			/*tp_repr*/
	0,			/*tp_as_number*/
	0,			/*tp_as_sequence*/
	0,			/*tp_as_mapping*/
	0,			/*tp_hash*/
        0,                      /*tp_call*/
        0,                      /*tp_str*/
        0,                      /*tp_getattro*/
        0,                      /*tp_setattro*/
        0,                      /*tp_as_buffer*/
        Py_TPFLAGS_DEFAULT,     /*tp_flags*/
        0,                      /*tp_doc*/
        0,                      /*tp_traverse*/
        0,                      /*tp_clear*/
        0,                      /*tp_richcompare*/
        0,                      /*tp_weaklistoffset*/
        0,                      /*tp_iter*/
        0,                      /*tp_iternext*/
        GsmType_methods,                      /*tp_methods*/
        0,                      /*tp_members*/
        0,                      /*tp_getset*/
        0,                      /*tp_base*/
        0,                      /*tp_dict*/
        0,                      /*tp_descr_get*/
        0,                      /*tp_descr_set*/
        0,                      /*tp_dictoffset*/
        0,                      /*tp_init*/
        0,                      /*tp_alloc*/
        0,                      /*tp_new*/
        0,                      /*tp_free*/
        0,                      /*tp_is_gc*/
};
/* --------------------------------------------------------------------- */

/* Function of no arguments returning new Gsm object */

static PyObject *
gsm_gsm(PyObject *self, PyObject *args)
{
	GsmObject *rv;
	int bigendian;

	if (!PyArg_ParseTuple(args, "i:gsm", &bigendian)) {
		return NULL;
	}
	if ((bigendian != 0) && (bigendian != 1)) {
		PyErr_SetString(PyExc_ValueError, "must pass gsm.BIG or gsm.LITTLE");
		return NULL;
	}
	rv = newGsmObject(bigendian);
	if (rv == NULL) {
		return NULL;
	}
	return (PyObject *)rv;
}





/* List of functions defined in the module */

static PyMethodDef gsm_methods[] = {
	{"gsm",		gsm_gsm,		METH_VARARGS,
	 PyDoc_STR("gsm(endianness) -> new gsm object.")},
	{NULL,		NULL}		/* sentinel */
};

PyDoc_STRVAR(module_doc,
"GSM encoding and decoding.");


/* Initialization function for the module (*must* be called initxx) */

PyMODINIT_FUNC
initgsm(void)
{
	PyObject *m;

	/* Finalize the type object including setting type of the new type
	 * object; doing it here is required for portability to Windows 
	 * without requiring C++. */
	if (PyType_Ready(&Gsm_Type) < 0)
		return;

	/* Create the module and add the functions */
	m = Py_InitModule3("gsm", gsm_methods, module_doc);
	PyModule_AddIntConstant(m, "BIG", 1);
	PyModule_AddIntConstant(m, "LITTLE", 0);
}
