#/*##########################################################################
# Copyright (C) 2004-2009 European Synchrotron Radiation Facility
#
# This file is part of the PyMCA X-ray Fluorescence Toolkit developed at
# the ESRF by the Beamline Instrumentation Software Support (BLISS) group.
#
# This toolkit is free software; you can redistribute it and/or modify it 
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option) 
# any later version.
#
# PyMCA is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# PyMCA; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
# Suite 330, Boston, MA 02111-1307, USA.
#
# PyMCA follows the dual licensing model of Trolltech's Qt and Riverbank's PyQt
# and cannot be used as a free plugin for a non-free program. 
#
# Please contact the ESRF industrial unit (industry@esrf.fr) if this license 
# is a problem for you.
#############################################################################*/
__author__ = "V.A. Sole - ESRF BLISS Group"
import sys
import os
import QtBlissGraph
qt = QtBlissGraph.qt
from PyMca_Icons import IconDict
import PyMcaPrintPreview
import PyMcaDirs

QTVERSION = qt.qVersion()
QWTVERSION4 = QtBlissGraph.QWTVERSION4
DEBUG = 0

class RGBCorrelatorGraph(qt.QWidget):
    def __init__(self, parent = None, selection=False, colormap=False,
                 imageicons=False, standalonesave=True, standalonezoom=True):
        qt.QWidget.__init__(self, parent)
        self.mainLayout = qt.QVBoxLayout(self)
        self.mainLayout.setMargin(0)
        self.mainLayout.setSpacing(0)
        self._buildToolBar(selection, colormap, imageicons,
                           standalonesave,
                           standalonezoom=standalonezoom)
        self.graph = QtBlissGraph.QtBlissGraph(self)
        self.graph.xlabel("Column")
        self.graph.ylabel("Row")
        self.graph.yAutoScale = 1
        self.graph.xAutoScale = 1
        self.saveDirectory = os.getcwd()
        self.mainLayout.addWidget(self.graph)
        self.printPreview = PyMcaPrintPreview.PyMcaPrintPreview(modal = 0)
        if DEBUG: print "printPreview id = ", id(self.printPreview)

    def sizeHint(self):
        return qt.QSize(1.5 * qt.QWidget.sizeHint(self).width(),
                        qt.QWidget.sizeHint(self).height())

    def _buildToolBar(self, selection=False, colormap=False,
                      imageicons=False, standalonesave=True,
                      standalonezoom=True):
        if QTVERSION < '4.0.0':
            if qt.qVersion() < '3.0':
                self.colormapIcon= qt.QIconSet(qt.QPixmap(IconDict["colormap16"]))
            else:
                self.colormapIcon= qt.QIconSet(qt.QPixmap(IconDict["colormap"]))
            self.selectionIcon	= qt.QIconSet(qt.QPixmap(IconDict["normal"]))
            self.zoomResetIcon	= qt.QIconSet(qt.QPixmap(IconDict["zoomreset"]))
            self.printIcon	= qt.QIconSet(qt.QPixmap(IconDict["fileprint"]))
            self.saveIcon	= qt.QIconSet(qt.QPixmap(IconDict["filesave"]))
            self.xAutoIcon	= qt.QIconSet(qt.QPixmap(IconDict["xauto"]))
            self.yAutoIcon	= qt.QIconSet(qt.QPixmap(IconDict["yauto"]))
            self.imageIcon      = qt.QIconSet(qt.QPixmap(IconDict["image"]))
            self.eraseSelectionIcon     = qt.QIconSet(qt.QPixmap(IconDict["eraseselect"]))
            self.rectSelectionIcon      = qt.QIconSet(qt.QPixmap(IconDict["boxselect"]))
            self.brushSelectionIcon     = qt.QIconSet(qt.QPixmap(IconDict["brushselect"]))
            self.brushIcon              = qt.QIconSet(qt.QPixmap(IconDict["brush"]))
            if not QWTVERSION4:
                self.hFlipIcon	= qt.QIconSet(qt.QPixmap(IconDict["gioconda16mirror"]))
        else:
            self.colormapIcon   = qt.QIcon(qt.QPixmap(IconDict["colormap"]))
            self.selectionIcon	= qt.QIcon(qt.QPixmap(IconDict["normal"]))
            self.zoomResetIcon	= qt.QIcon(qt.QPixmap(IconDict["zoomreset"]))
            self.printIcon	= qt.QIcon(qt.QPixmap(IconDict["fileprint"]))
            self.saveIcon	= qt.QIcon(qt.QPixmap(IconDict["filesave"]))            
            self.xAutoIcon	= qt.QIcon(qt.QPixmap(IconDict["xauto"]))
            self.yAutoIcon	= qt.QIcon(qt.QPixmap(IconDict["yauto"]))
            self.hFlipIcon	= qt.QIcon(qt.QPixmap(IconDict["gioconda16mirror"]))
            self.imageIcon     = qt.QIcon(qt.QPixmap(IconDict["image"]))
            self.eraseSelectionIcon = qt.QIcon(qt.QPixmap(IconDict["eraseselect"]))
            self.rectSelectionIcon  = qt.QIcon(qt.QPixmap(IconDict["boxselect"]))
            self.brushSelectionIcon = qt.QIcon(qt.QPixmap(IconDict["brushselect"]))
            self.brushIcon          = qt.QIcon(qt.QPixmap(IconDict["brush"]))
            self.additionalIcon     = qt.QIcon(qt.QPixmap(IconDict["additionalselect"]))

        self.toolBar = qt.QWidget(self)
        self.toolBarLayout = qt.QHBoxLayout(self.toolBar)
        self.toolBarLayout.setMargin(0)
        self.toolBarLayout.setSpacing(0)
        self.mainLayout.addWidget(self.toolBar)
        #Autoscale
        if standalonezoom:
            tb = self._addToolButton(self.zoomResetIcon,
                            self._zoomReset,
                            'Auto-Scale the Graph')
        else:
            tb = self._addToolButton(self.zoomResetIcon,
                            None,
                            'Auto-Scale the Graph')
        self.zoomResetToolButton = tb
        #y Autoscale
        tb = self._addToolButton(self.yAutoIcon,
                            self._yAutoScaleToggle,
                            'Toggle Autoscale Y Axis (On/Off)',
                            toggle = True, state=True)
        if qt.qVersion() < '4.0.0':
            tb.setState(qt.QButton.On)
        else:
            tb.setDown(True)
        self.yAutoScaleToolButton = tb
        tb.setDown(True)

        #x Autoscale
        tb = self._addToolButton(self.xAutoIcon,
                            self._xAutoScaleToggle,
                            'Toggle Autoscale X Axis (On/Off)',
                            toggle = True, state=True)
        self.xAutoScaleToolButton = tb
        tb.setDown(True)

        #colormap
        if colormap:
            tb = self._addToolButton(self.colormapIcon,
                                     None,
                                    'Change Colormap')
            self.colormapToolButton = tb
        #flip
        if not QWTVERSION4:
            tb = self._addToolButton(self.hFlipIcon,
                                     None,
                                     'Flip Horizontal')
            self.hFlipToolButton = tb


        #save
        if standalonesave:
            tb = self._addToolButton(self.saveIcon,
                                 self._saveIconSignal,
                                 'Save Graph')
        else:
            tb = self._addToolButton(self.saveIcon,
                                 None,
                                 'Save')
        self.saveToolButton = tb

        #Selection
        if selection:
            tb = self._addToolButton(self.selectionIcon,
                                None,
                                'Toggle Selection Mode',
                                toggle = True,
                                state = False)
            if qt.qVersion() < '4.0.0':
                tb.setState(qt.QButton.Off)
            else:
                tb.setDown(False)
            self.selectionToolButton = tb
        #image selection icons
        if imageicons:
            tb = self._addToolButton(self.imageIcon,
                                     None,
                                     'Reset')
            self.imageToolButton = tb

            tb = self._addToolButton(self.eraseSelectionIcon,
                                     None,
                                     'Erase Selection')
            self.eraseSelectionToolButton = tb

            tb = self._addToolButton(self.rectSelectionIcon,
                                     None,
                                     'Rectangular Selection')
            self.rectSelectionToolButton = tb

            tb = self._addToolButton(self.brushSelectionIcon,
                                     None,
                                     'Brush Selection')
            self.brushSelectionToolButton = tb

            tb = self._addToolButton(self.brushIcon,
                                     None,
                                     'Select Brush')
            self.brushToolButton = tb
            if QTVERSION > '4.0.0':
                tb = self._addToolButton(self.additionalIcon,
                                     None,
                                     'Additional Selections Menu')
                self.additionalSelectionToolButton = tb            
        else:
            self.imageToolButton = None

        self.infoWidget = qt.QWidget(self.toolBar)
        self.infoWidget.mainLayout = qt.QHBoxLayout(self.infoWidget)
        self.infoWidget.mainLayout.setMargin(0)
        self.infoWidget.mainLayout.setSpacing(0)
        self.infoWidget.label = qt.QLabel(self.infoWidget)
        self.infoWidget.label.setText("X = ???? Y = ???? Z = ????")
        self.infoWidget.mainLayout.addWidget(self.infoWidget.label)
        self.toolBarLayout.addWidget(self.infoWidget)
        self.infoWidget.hide()

        self.toolBarLayout.addWidget(HorizontalSpacer(self.toolBar))

        # ---print
        tb = self._addToolButton(self.printIcon,
                                 self.printGraph,
                                 'Prints the Graph')

    def showInfo(self):
        self.infoWidget.show()

    def hideInfo(self):
        self.infoWidget.hide()

    def setInfoText(self, text):
        self.infoWidget.label.setText(text)

    def infoText(self):
        return self.infoWidget.label.text()

    def hideImageIcons(self):
        if self.imageToolButton is None:return
        self.imageToolButton.hide()
        self.eraseSelectionToolButton.hide()
        self.rectSelectionToolButton.hide()
        self.brushSelectionToolButton.hide()
        self.brushToolButton.hide()

    def showImageIcons(self):
        if self.imageToolButton is None:return
        self.imageToolButton.show()
        self.eraseSelectionToolButton.show()
        self.rectSelectionToolButton.show()
        self.brushSelectionToolButton.show()
        self.brushToolButton.show()

    def _addToolButton(self, icon, action, tip, toggle=None, state=None, position=None):
        tb      = qt.QToolButton(self.toolBar)            
        if QTVERSION < '4.0.0':
            tb.setIconSet(icon)
            qt.QToolTip.add(tb,tip) 
            if toggle is not None:
                if toggle:
                    tb.setToggleButton(1)
                    if state is not None:
                        if state:
                            tb.setState(qt.QButton.On)
        else:
            tb.setIcon(icon)
            tb.setToolTip(tip)
            if toggle is not None:
                if toggle:
                    tb.setCheckable(1)
                    if state is not None:
                        if state:
                            tb.setChecked(state)
                    else:
                        tb.setChecked(False)
        if position is not None:
            self.toolBarLayout.insertWidget(position, tb)
        else:
            self.toolBarLayout.addWidget(tb)
        if action is not None:
            self.connect(tb,qt.SIGNAL('clicked()'), action)
        return tb

    def _zoomReset(self, replot=None):
        if DEBUG:print "_zoomReset"
        if replot is None:
            replot = True
        if self.graph is not None:
            self.graph.zoomReset()
            if self.graph.yAutoScale:
                if hasattr(self, '_y1Limit'):
                    self.graph.sety1axislimits(0, self._y1Limit)
            if self.graph.xAutoScale:
                if hasattr(self, '_x1Limit'):
                    self.graph.setx1axislimits(0, self._x1Limit)
            if replot:
                self.graph.replot()

    def _yAutoScaleToggle(self):
        if self.graph is not None:
            if self.graph.yAutoScale:
                self.graph.yAutoScale = False
                self.yAutoScaleToolButton.setDown(False)
            else:
                self.graph.yAutoScale = True
                self.yAutoScaleToolButton.setDown(True)
            
    def _xAutoScaleToggle(self):
        if self.graph is not None:
            if self.graph.xAutoScale:
                self.graph.xAutoScale = False
                self.xAutoScaleToolButton.setDown(False)
            else:
                self.graph.xAutoScale = True
                self.xAutoScaleToolButton.setDown(True)

    def _saveIconSignal(self):
        self.saveDirectory = PyMcaDirs.outputDir

        fileTypeList = ["Image *.png",
                        "Image *.jpg",
                        "ZoomedImage *.png",
                        "ZoomedImage *.jpg",
                        "Widget *.png",
                        "Widget *.jpg"]

        outfile = qt.QFileDialog(self)
        outfile.setModal(1)
        if QTVERSION < '4.0.0':
            outfile.setCaption("Output File Selection")
            filterlist = fileTypeList[0]
            for f in fileTypeList:
                filterlist += "\n%s" % f
            outfile.setFilters(filterlist)
            outfile.setMode(outfile.AnyFile)
            outfile.setDir(self.saveDirectory)
            ret = outfile.exec_loop()
        else:
            outfile.setWindowTitle("Output File Selection")
            strlist = qt.QStringList()
            for f in fileTypeList:
                strlist.append(f)
            outfile.setFilters(strlist)
            outfile.setFileMode(outfile.AnyFile)
            outfile.setAcceptMode(qt.QFileDialog.AcceptSave)
            outfile.setDirectory(self.saveDirectory)
            ret = outfile.exec_()

        if not ret: return
        filterused = str(outfile.selectedFilter()).split()
        filetype = filterused[0]
        extension = filterused[1]
        if QTVERSION < '4.0.0':
            outstr=str(outfile.selectedFile())
        else:
            outstr=str(outfile.selectedFiles()[0])
        try:            
            outputFile = os.path.basename(outstr)
        except:
            outputFile  = outstr
        outputDir  = os.path.dirname(outstr)
        self.saveDirectory = outputDir
        PyMcaDirs.outputDir = outputDir

        #always overwrite for the time being
        if len(outputFile) < len(extension[1:]):
            outputFile += extension[1:]
        elif outputFile[-4:] != extension[1:]:
            outputFile += extension[1:]
        outputFile = os.path.join(outputDir, outputFile)
        if os.path.exists(outputFile):
            try:
                os.remove(outputFile)
            except:
                qt.QMessageBox.critical(self, "Save Error",
                                        "Cannot overwrite existing file")
                return

        if filetype.upper() == "IMAGE":
            self.saveGraphImage(outputFile, original = True)
        elif filetype.upper() == "ZOOMEDIMAGE":
            self.saveGraphImage(outputFile, original = False)
        else:
            self.saveGraphWidget(outputFile)

    def saveGraphImage(self, filename, original = False):
        format_ = filename[-3:].upper()
        if original:
            #This is the whole image, not the zoomed one ...
            if QTVERSION < '4.0.0':
                pixmap = qt.QPixmap(self.graph.plotImage.image)
            else:
                pixmap = qt.QPixmap.fromImage(self.graph.plotImage.image)
        else:
            pixmap = qt.QPixmap.grabWidget(self.graph.canvas())
        if pixmap.save(filename, format_):
            return
        else:
            qt.QMessageBox.critical(self, "Save Error",
                                    "%s" % sys.exc_info()[1])
            return

    def saveGraphWidget(self, filename):
        format_ = filename[-3:].upper()
        pixmap = qt.QPixmap.grabWidget(self.graph)
        if pixmap.save(filename, format_):
            return
        else:
            qt.QMessageBox.critical(self, "Save Error", "%s" % sys.exc_info()[1])
            return

    def setSaveDirectory(self, wdir):
        if os.path.exists(wdir):
            self.saveDirectory = wdir
            return True
        else:
            return False

    def printGraph(self):
        pixmap = qt.QPixmap.grabWidget(self.graph.canvas())
        self.printPreview.addPixmap(pixmap)
        if self.printPreview.isHidden():
            self.printPreview.show()
        if QTVERSION < '4.0.0':
            self.printPreview.raiseW()
        else:
            self.printPreview.raise_()

    def selectColormap(self):
        qt.QMessageBox.information(self, "Open", "Not implemented (yet)")  

class HorizontalSpacer(qt.QWidget):
    def __init__(self, *args):
        qt.QWidget.__init__(self, *args)
      
        self.setSizePolicy(qt.QSizePolicy(qt.QSizePolicy.Expanding,
                           qt.QSizePolicy.Fixed))

    
class MyQLabel(qt.QLabel):
    def __init__(self,parent=None,name=None,fl=0,bold=True, color= qt.Qt.red):
        qt.QLabel.__init__(self,parent)
        if qt.qVersion() <'4.0.0':
            self.color = color
            self.bold  = bold
        else:
            palette = self.palette()
            role = self.foregroundRole()
            palette.setColor(role,color)
            self.setPalette(palette)
            self.font().setBold(bold)


    if qt.qVersion() < '4.0.0':
        def drawContents(self, painter):
            painter.font().setBold(self.bold)
            pal =self.palette()
            pal.setColor(qt.QColorGroup.Foreground,self.color)
            self.setPalette(pal)
            qt.QLabel.drawContents(self,painter)
            painter.font().setBold(0)

def test():
    app = qt.QApplication([])
    qt.QObject.connect(app,
                       qt.SIGNAL("lastWindowClosed()"),
                       app,
                       qt.SLOT('quit()'))

    container = RGBCorrelatorGraph()
    container.show()
    if QTVERSION < '4.0.0':
        app.setMainWidget(container)
        app.exec_loop()
    else:
        app.exec_()

if __name__ == "__main__":
    test()
        
