# nmblookup.py
# Copyright (C) 2008 Stefan J. Betz <stefan_betz@gmx.net>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import socket
import subprocess
import copy
import re
from pyneighborhood import config
from pyneighborhood.misc import print_debug

class nmblookup(object):
    """
    Parser for the nmblookup Application
    """
    ipre = re.compile(".*\n(.*), ([0-9\.]*) (.*)<00>")
    masterre = re.compile(".*\n([0-9\.]*) (.*)<1d>")
    masternetbiosre = re.compile(".*\n\t(.*)<00>.*\n")
    workgroupre = re.compile("\t(.*)<00>.*GROUP.*")
    hostre = re.compile(".*\n\t(.*)\<00\>")

    def __init__(self):
        """
        Constructor
        """
        self.iplookupcommand = [ config.get("Main", "nmblookup"), "-T" ]
        self.hostlookupcommand = [ config.get("Main", "nmblookup"), "-A" ]
        self.masterlookupcommand = [ config.get("Main", "nmblookup"), "-M" ]

    def retrieve_ip(self, host):
        """
        Fill out the ip from host
        """
        if host.name != "":
            print_debug("IP Lookup for: " + host.name)
            command = copy.copy(self.iplookupcommand)
            command.append(host.name)
            process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE)
            process.wait()
            output = process.stdout.read()
            print_debug(command, output)
            match = self.ipre.match(output)
            if match != None:
                host.dns_name = match.group(1)
                host.ip = match.group(2)
                print_debug("IP Lookup Result: " + str(match.groups()))
            else:
                print_debug("IP Lookup failed for: " + host.name)

    def retrieve_host(self, host):
        """
        Fill out the host from host by IP
        """
        if host.ip == "":
            return
        print_debug("NetBIOS Name Lookup for: " + host.ip)
        command = copy.copy(self.hostlookupcommand)
        command.append(host.ip)
        process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE)
        process.wait()
        output = process.stdout.read()
        print_debug(command, output)
        match = self.hostre.match(output)
        if match != None:
            host.name = match.group(1).strip()
        print_debug("NetBIOS Name Lookup Result: " + host.name)

    def retrieve_data(self, host):
        """
        Get hostname and/or ip via nmblookup
        """
        pass

    def retrieve_master(self, workgroup):
        """
        Get Workgroup Master Browser IP and/or DNS Name
        """
        if workgroup.name != "":
            print_debug("Master Lookup for: " + workgroup.name)
            command = copy.copy(self.masterlookupcommand)
            command.append("\"" + workgroup.name + "\"" )
            process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE)
            process.wait()
            output = process.stdout.read()
            print_debug(command, output)
            match = self.masterre.match(output)
            if match != None:
                print_debug("Master IP:" + match.group(1))
                workgroup.master_ip = match.group(1)
                workgroup.master = socket.getfqdn(match.group(1))
                print_debug("Master FQDN:" + workgroup.master)
                command = copy.copy(self.hostlookupcommand)
                command.append("\"" + workgroup.master_ip + "\"")
                process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE)
                process.wait()
                output = process.stdout.read()
                hostmatch = self.masternetbiosre.match(output)
                if hostmatch != None:
                    workgroup.master_netbios = hostmatch.group(1).strip()
                    print_debug("Master NetBIOS:" + workgroup.master_netbios)

    def retrieve_workgroup(self, host):
        """
        Get the Workgroup to host
        """
        if host.ip == "":
            return
        print_debug("Workgroup Lookup for " + host.ip)
        command = copy.copy(self.hostlookupcommand)
        command.append(host.ip)
        process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE)
        process.wait()
        output = process.stdout.read()
        print_debug(command, output)
        match = self.workgroupre.search(output)
        if match != None:
            host.workgroup = match.group(1).strip()
            print_debug("Workgroup found: " + host.workgroup)

    def groups(self):
        """
        Get the Workgroups
        """
        print_debug("Scanning for Workgroups using the Legacy Interface Emulation")
        command = [ config.get("Main", "nmblookup"), "-S", "-M" ]
        if config.get("Main", "workgroup") == "":
            command.append("--")
            command.append("-")
        else:
            command.append(config.get("Main", "workgroup"))
        process = subprocess.Popen(command, executable = command[0], stdout=subprocess.PIPE)
        process.wait()
        output = process.stdout.read()
        debug(command, output)
        match = self.workgroupre.search(output)
        while match != None:
            workgroup = match.group(1)
            print_debug("Found Workgroup:" + workgroup)
            match = self.workgroupre.search(output, match.end())
            yield workgroup
