/*
 *  Copyright (C) 2004 Alexandre Norman <norman@xael.org>
 *  Version 0.2.1
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* Tested with python 2.3 and clamav 0.80
 * Should work with any version of python from 2.1
 */

#include <Python.h>
#include <clamav.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>



/* ********************************************************* */

/* To be able to compile with 
   releases 0.75 of libclamav 

   Where cl_free was cl_freetrie
   and cl_build was cl_buildtrie
   CL_SCAN_STDOPT did not exist
*/
#ifndef CL_SCAN_STDOPT
#define CL_SCAN_STDOPT CL_RAW | CL_ARCHIVE | CL_MAIL | CL_DISABLERAR | CL_OLE2 | CL_ENCRYPTED
void cl_free(struct cl_node *rootnode) {
  cl_freetrie(rootnode);  
  return;    
} 

int cl_build(struct cl_node *rootnode) {  
  return cl_buildtrie(rootnode);    
} 
#endif

/* For python prior to 2.3 */
#ifndef PyMODINIT_FUNC
#define PyMODINIT_FUNC void
#endif

/* ********************************************************* */


/* Exception object for python */
static PyObject *PyclamavError;


/* Signature number */
int signumber = 0;
 
/* Structures for clamav */
struct cl_node *root = NULL;
struct cl_limits limits;
struct cl_stat dbstat;


/*
 * If the virus database has been changed, then
 * free the current tree and reload the new one
 */
int if_database_have_changed_then_reload(void)
{
  int ret = 0;

  /* Test if the database have changed */
  /* If yes : reload DB                */
  if (cl_statchkdir(&dbstat) == 1)
    {
      /* free the tree */
      cl_free(root); 
      signumber=0;
      root=NULL;

      /* Load DB */
      if((ret = cl_loaddbdir(cl_retdbdir(), &root, &signumber))) {
	/* Raise exception with error message */
	PyErr_SetString(PyclamavError,  cl_strerror(ret));
	return -2;
      }

      /* build the final tree */
      if((ret = cl_build(root))) {
	/* free the partial tree */
	cl_free(root); 
	/* Raise exception with error message */
	PyErr_SetString(PyclamavError, cl_strerror(ret));
	return -2;
      }

      /* Reinit db status update check */
      cl_statfree(&dbstat);
      cl_statinidir(cl_retdbdir(), &dbstat);

      return -1;
    }
  else
    {
      return 0;
    }

  return 0;
}


/*
 * Return clamav version
 */

static PyObject *pyclamav_get_version(PyObject *self, PyObject *args)
{
    const char *version;
    int daily_version = 0;
    int daily_date = 0;

    const char *dbdir;
    char *path = NULL;
    struct cl_cvd *daily;

    //Clamav version
    version = cl_retver();

    //Database version
    dbdir = cl_retdbdir();
    if((path = malloc(strlen(dbdir) + 11))){
        
        sprintf(path, "%sdaily.cvd", dbdir);

        if((daily = cl_cvdhead(path))){

            daily_date    = daily->stime;
            daily_version = daily->version;

            cl_cvdfree(daily);
        }
    }

    return Py_BuildValue("(s,i,i)", version, daily_version, daily_date);
}
    

/*
 * Return number of signature in the database as integer
 */
static PyObject *pyclamav_get_numsig(PyObject *self, PyObject *args)
{
  (void) if_database_have_changed_then_reload();

  return Py_BuildValue("i", signumber);
}




/*
 * Scan a file given as a parameter
 */
static PyObject *pyclamav_scanfile(PyObject *self, PyObject *args)
{
  char *file_to_scan;
  unsigned long int size = 0;
  const char *virname;
  int ret = 0;

  (void) if_database_have_changed_then_reload();

  if (!PyArg_ParseTuple(args, "s", &file_to_scan)) {
    /* Raise exception with error message */
    PyErr_SetString(PyclamavError,  "Pass filename to scan (string) as argument");
  }

  ret = cl_scanfile(file_to_scan, &virname, &size, root, &limits, CL_SCAN_STDOPT);

  /* Test return code */
  switch (ret) {
  case CL_VIRUS : /* File contains a virus */
    return Py_BuildValue("(i,s)",1, virname);
    break;
  case CL_CLEAN : /* File is clean */
    return Py_BuildValue("(i,s)",0, "");
    break;
  default: /* Error : return error message */
    return Py_BuildValue("(i,s)",ret, cl_strerror(ret));
  }

}




/*
 * Scan a buffer given as a parameter
 */
static PyObject *pyclamav_scanthis(PyObject *self, PyObject *args)
{
  const char *buffer_to_scan;
  unsigned int size = 0;
  const char *virname;

  int ret = 0;

  
  (void) if_database_have_changed_then_reload();


  if (!PyArg_ParseTuple(args, "s#", &buffer_to_scan, &size)) {
    /* Raise exception with error message */
    PyErr_SetString(PyclamavError,  "Pass buffer to scan (string) as argument");
  }

  /* Scan buffer */
  ret = cl_scanbuff(buffer_to_scan, size, &virname, root);

  /* Test return code */
  switch (ret) {
  case CL_VIRUS : /* File contains a virus */
    return Py_BuildValue("(i,s)",1, virname);
    break;
  case CL_CLEAN : /* File is clean */
    return Py_BuildValue("(i,s)",0, "");
    break;
  default: /* Error : return error message */
    return Py_BuildValue("(i,s)",ret, cl_strerror(ret));
  }

}


static PyMethodDef ClamavMethods[] = {
    {"scanfile",  pyclamav_scanfile, METH_VARARGS, "Scan a file for virus.\nArguments : filename (string)\n"},
    {"scanthis",  pyclamav_scanthis, METH_VARARGS, "Scan a buffer for virus.\nArguments : buffer (string)"},
    {"get_numsig",  pyclamav_get_numsig, METH_VARARGS, "Return the number of known signatures.\nArguments : None\n"},
    {"get_version",  pyclamav_get_version, METH_VARARGS, "Return the version of Clamav.\nArguments : None\n"},
    {NULL, NULL, 0, NULL}    
};




PyMODINIT_FUNC initpyclamav(void)
{
  int ret= 0;

  PyObject *module, *dict;
  module=Py_InitModule("pyclamav", ClamavMethods);
  dict = PyModule_GetDict(module);

  PyclamavError = PyErr_NewException("pyclamav.error", NULL, NULL);
  PyDict_SetItemString(dict, "error", PyclamavError);


  if((ret = cl_loaddbdir(cl_retdbdir(), &root, &signumber))) {
    /* Raise exception with error message */
    PyErr_SetString(PyclamavError,  cl_strerror(ret));
    return ;
  }

  /* build the final tree */
  if((ret = cl_build(root))) {
    /* free the partial tree */
    cl_free(root); 
    /* Raise exception with error message */
    PyErr_SetString(PyclamavError, cl_strerror(ret));
    return ;
  }


  /* Set dbstat to get notification of database changes */
  memset(&dbstat, 0, sizeof(struct cl_stat));
  cl_statinidir(cl_retdbdir(), &dbstat);



  /* set up archive limits */
  memset(&limits, 0, sizeof(struct cl_limits));
  limits.maxfiles = 1000; /* max files */
  limits.maxfilesize = 10 * 1048576; /* maximal archived file size == 10 Mb */
  limits.maxreclevel = 5; /* maximal recursion level */
  limits.maxratio = 200; /* maximal compression ratio */
  limits.archivememlim = 0; /* disable memory limit for bzip2 scanner */


  return ;
}


int main(int argc, char *argv[])
{
  /* Pass argv[0] to the Python interpreter */
  Py_SetProgramName(argv[0]);

  /* Initialize the Python interpreter.  Required. */
  Py_Initialize();
  
  /* Add a static module */
  initpyclamav();
  
  return 0;
}
