# DSR:2002.02.21-2002.02.24:
#   I reorganized and rewrote this module to a considerable extent in order to
# fix the numerous DB API incompatibilities that I documented and agreed to
# fix in SourceForge bug report #520793.
#   In addition to enhancing compliance with DB API spec, I added or updated
# doc strings throughout the module and made clear what API *guarantees*
# kinterbasdb makes.
#   Notably, the fact that users can only rely on the return values of certain
# functions/methods to be sequence or mapping types, not instances of a
# specific class or type.  This policy is still compliant with the DB API spec,
# and is far more future-proof than implying that all of the classes defined
# herein can be relied upon not to change.  Now, only module contents in
# "PUBLIC ..." sections can be expected to remain unchanged.

__version__ = (3, 0, 1, 'final', 0)

import string, sys, types

import _kinterbasdb

# DSR:2002.01.19:
# The core mx modules were restructured recently (2001?) and grouped under the
# title "eGenix mx Base Package".  Rather than standing alone, modules such as
# DateTime now belong to a package named "mx".
# On systems with the restructured mx packages, "import DateTime" will not
# work; "from mx import DateTime" should be used instead.
# The following code first tries the new import style, then the old style if
# the new style fails.
try:
    from mx import DateTime # new style
except ImportError:
    import DateTime # old style
    sys.stderr.write("WARNING:  Versions of DateTime earlier than 2.0.1 are"
        " not officially supported by kinterbasdb 3.0.")


##########################################
##     PUBLIC CONSTANTS: BEGIN          ##
##########################################

apilevel = '2.0'
threadsafety = 1
paramstyle = 'qmark'

# Named positional constants to be used as indices into the description
# attribute of a cursor (these positions are defined by the DB API spec).
# For example:
#   nameOfFirstField = cursor.description[0][kinterbasdb.DESCRIPTION_NAME]

DESCRIPTION_NAME            = 0
DESCRIPTION_TYPE_CODE       = 1
DESCRIPTION_DISPLAY_SIZE    = 2
DESCRIPTION_INTERNAL_SIZE   = 3
DESCRIPTION_PRECISION       = 4
DESCRIPTION_SCALE           = 5
DESCRIPTION_NULL_OK         = 6

# Header constants defined by the C API of the database:
# (This implementation actually draws the constants' values from the database
# header files rather than relying on brittle literals typed into
# kinterbasdb.py.)

# Transaction parameters constants:
from _kinterbasdb import \
    isc_tpb_consistency, \
    isc_tpb_concurrency, \
    isc_tpb_shared, \
    isc_tpb_protected, \
    isc_tpb_exclusive, \
    isc_tpb_wait, \
    isc_tpb_nowait, \
    isc_tpb_read, \
    isc_tpb_write, \
    isc_tpb_lock_read, \
    isc_tpb_lock_write, \
    isc_tpb_verb_time, \
    isc_tpb_commit_time, \
    isc_tpb_ignore_limbo, \
    isc_tpb_read_committed, \
    isc_tpb_autocommit, \
    isc_tpb_rec_version, \
    isc_tpb_no_rec_version, \
    isc_tpb_restart_requests, \
    isc_tpb_no_auto_undo

# default transaction parameters
default_tpb  =  ( isc_tpb_concurrency +
                isc_tpb_shared +
                isc_tpb_wait +
                isc_tpb_write +
                isc_tpb_read_committed +
                isc_tpb_rec_version )

# Database information constants:
from _kinterbasdb import \
    isc_info_db_id, \
    isc_info_reads, \
    isc_info_writes, \
    isc_info_fetches, \
    isc_info_marks, \
    isc_info_implementation, \
    isc_info_version, \
    isc_info_base_level, \
    isc_info_page_size, \
    isc_info_num_buffers, \
    isc_info_limbo, \
    isc_info_current_memory, \
    isc_info_max_memory, \
    isc_info_window_turns, \
    isc_info_license, \
    isc_info_allocation, \
    isc_info_attachment_id, \
    isc_info_read_seq_count, \
    isc_info_read_idx_count, \
    isc_info_insert_count, \
    isc_info_update_count, \
    isc_info_delete_count, \
    isc_info_backout_count, \
    isc_info_purge_count, \
    isc_info_expunge_count, \
    isc_info_sweep_interval, \
    isc_info_ods_version, \
    isc_info_ods_minor_version, \
    isc_info_no_reserve, \
    isc_info_logfile, \
    isc_info_cur_logfile_name, \
    isc_info_cur_log_part_offset, \
    isc_info_num_wal_buffers, \
    isc_info_wal_buffer_size, \
    isc_info_wal_ckpt_length, \
    isc_info_wal_cur_ckpt_interval, \
    isc_info_wal_prv_ckpt_fname, \
    isc_info_wal_prv_ckpt_poffset, \
    isc_info_wal_recv_ckpt_fname, \
    isc_info_wal_recv_ckpt_poffset, \
    isc_info_wal_grpc_wait_usecs, \
    isc_info_wal_num_io, \
    isc_info_wal_avg_io_size, \
    isc_info_wal_num_commits, \
    isc_info_wal_avg_grpc_size, \
    isc_info_forced_writes, \
    isc_info_user_names, \
    isc_info_page_errors, \
    isc_info_record_errors, \
    isc_info_bpage_errors, \
    isc_info_dpage_errors, \
    isc_info_ipage_errors, \
    isc_info_ppage_errors, \
    isc_info_tpage_errors, \
    isc_info_set_page_buffers

# DSR:2002.05.11:begin block
# There follows a group of constants that should only be imported if the C
# portion of kinterbasdb was compiled against Interbase 6 or later.
if hasattr(_kinterbasdb, 'isc_info_db_SQL_dialect'):
    from _kinterbasdb import \
        isc_info_db_SQL_dialect, \
        isc_info_db_read_only, \
        isc_info_db_size_in_pages
# DSR:2002.05.11:end block

##########################################
##     PUBLIC CONSTANTS: END            ##
##########################################

########################################################
## PUBLIC DB-API TYPE OBJECTS AND CONSTRUCTORS: BEGIN ##
########################################################

Warning = _kinterbasdb.Warning
Error = _kinterbasdb.Error
InterfaceError = _kinterbasdb.InterfaceError
DatabaseError = _kinterbasdb.DatabaseError
DataError = _kinterbasdb.DataError
OperationalError = _kinterbasdb.OperationalError
IntegrityError = _kinterbasdb.IntegrityError
InternalError = _kinterbasdb.InternalError
ProgrammingError = _kinterbasdb.ProgrammingError
NotSupportedError = _kinterbasdb.NotSupportedError

# DSR:2002.07.19:begin block
# Replaced the old constructors with validating versions that raise a DB API
# exception upon failure, rather than an exception from the underlying datetime
# module.
#
# NOTE:
# These new, validating constructors are designed to effect strict DB API
# compliance.  This goal takes priority over all others, but gives rise to
# have two important disadvantages when compared to the old approach:
# a) In some cases, the new constructors obscure the native features of the
#   underlying datetime module.
# b) The new constructors are slower.
#
# Neither of these disadvantages is insurmountable; client programmers can
# always use the underlying datetime module (mx DateTime) directly if they
# need its native features, or want to escape the validation cost present in
# the new constructors.

#OLD: Date = DateTime.DateTime
def Date(year, month, day):
    try:
        theDate = DateTime.DateTime(year, month, day)
    except DateTime.Error, e:
        raise DataError(str(e))

    return theDate

#OLD: Time = DateTime.TimeDelta
def Time(hour, minute, second):
    # mx DateTimeDeltas roll over when provided with an hour greater than
    # 23, a minute greater than 59, and so on.  That is not acceptable for our
    # purposes.
    if hour < 0 or hour > 23:
        raise DataError("hour must be between 0 and 23")
    if minute < 0 or minute > 59:
        raise DataError("minute must be between 0 and 59")
    if second < 0 or second > 59:
        raise DataError("second must be between 0 and 59")

    try:
        theTime = DateTime.TimeDelta(hour, minute, second)
    except DateTime.Error, e:
        raise DataError(str(e))

    return theTime

#OLD: Timestamp = DateTime.DateTime
def Timestamp(year, month, day, hour, minute, second):
    args = (year, month, day, hour, minute, second) # Yes, I know about the
      # *args syntactical shortcut, but it's not particularly readable.

    # mx DateTime's Timestamp constructor accepts negative values in the
    # spirit of Python's negative list indexes, but I see no reason to allow
    # that behavior in this DB API-compliance-obsessed module.
    if 0 < len(filter(lambda x: x < 0, args)):
        raise DataError("Values less than zero not allowed in Timestamp."
            " (Received arguments %s)"
            % repr(args)
          )

    try:
        theStamp = DateTime.DateTime(*args)
    except DateTime.Error, e:
        raise DataError(str(e))

    return theStamp
# DSR:2002.07.19:end block

DateFromTicks = DateTime.DateFromTicks
TimeFromTicks = DateTime.TimeFromTicks
# DSR:2002.07.18:begin block
# OLD (functionally equivalent, but slightly less readable):
# TimestampFromTicks = DateTime.localtime
TimestampFromTicks = DateTime.TimestampFromTicks
# DSR:2002.07.18:end block

Binary = buffer # XXX:DSR:This will change to str in 3.1, with the possible
  # addition of a lazy BLOB reader at some point in the future.

# DBAPITypeObject implementation is the DB API's suggested implementation.
class DBAPITypeObject:
    def __init__(self, *values):
        self.values = values
    def __cmp__(self, other):
        if other in self.values:
            return 0
        if other < self.values:
            return 1
        else:
            return -1

STRING = DBAPITypeObject(types.StringType)
BINARY = DBAPITypeObject(types.BufferType)
NUMBER = DBAPITypeObject(types.IntType, types.LongType, types.FloatType)
# DSR:2002.07.18:begin block
# DateTime.DateTimeDelta is a function, not a type!
#OLD: DATETIME = DBAPITypeObject(DateTime.DateTimeType, DateTime.DateTimeDelta)
DATETIME = DBAPITypeObject(DateTime.DateTimeType, DateTime.DateTimeDeltaType)
# DSR:2002.07.18:end block
ROWID = DBAPITypeObject()

########################################################
## PUBLIC DB-API TYPE OBJECTS AND CONSTRUCTORS: END   ##
########################################################

##########################################
##     PUBLIC FUNCTIONS: BEGIN          ##
##########################################

def connect(*args, **keywords_args):
    return apply(Connection, args, keywords_args)

def create_database(*args):
    """
    Creates a database according to the supplied CREATE DATABASE SQL statement.
    Returns an active connection to the newly created database.

    Arguments:
    - sql: string containing the CREATE DATABASE statement.  Note that you may
        need to specify a username and password as part of this statement (see
        IB/Firebird SQL Reference for syntax).
    - dialect: (optional) the dialect under which to execute the statement
    """
    # For a more general-purpose immediate execution facility (the non-"CREATE
    # TABLE" variant of isc_dsql_execute_immediate, for those who care), see
    # Connection.execute_immediate.

    C_conn = apply(_kinterbasdb.create_database, args)
    return Connection(_ConnectionObject=C_conn)


# DSR:2002.03.15:
# Utility function to be used with Connection.database_info.
# Python's built-in function ord would also serve this purpose, but its use
# for this purpose might blur the *conceptual* distinction between a raw byte
# and a Python char.
raw_byte_to_int = _kinterbasdb.raw_byte_to_int


##########################################
##     PUBLIC FUNCTIONS: END            ##
##########################################

##########################################
##     PUBLIC CLASSES: BEGIN            ##
##########################################

# DSR:2002.02.21:
# Renamed class from ConnectionObject to Connection, since the cursor
# class is named Cursor and not CursorObject.  As usual, I point out to
# the dubious that kinterbasdb 3.0 is a backwards-INcompatible release aimed
# at preparing for a long future period of API stability.
class Connection:
    """
    Represents a connection between the database client and the database
    server.

    The behavior of this class is further explained by the Python DB API
    Specification 2.0.

    Attributes:
    - dialect (read/write):
        The Interbase SQL dialect of the connection.  One of:
            1 for Interbase < 6.0
            2 for "migration mode"
            3 for Interbase >= 6.0 and Firebird

    - precision_mode (read/write):
          precision_mode 0 (the default) represents Interbase fixed point
        values (NUMERIC/DECIMAL fields) as Python floats, potentially losing
        precision.
          precision_mode 1 represents Interbase fixed point values as scaled
        Python integers, preserving precision.
          For more information, see the KInterbasDB Usage Guide.

    - server_version (read-only):
          The version string of the database server to which this connection
        is connected.

    - default_tpb (read/write):
          The transaction parameter buffer (TPB) that will be used by default
        for new transactions opened in the context of this connection.
          For more information, see the KInterbasDB Usage Guide.
    """

    def __init__(self, *args, **keywords_args):
        # self._C_conn is the instance of C type ConnectionType that represents
        # this cursor in _kinterbasdb.

        # Allow other code WITHIN THIS MODULE to obtain a connection some other
        # way and provide it to us instead of us getting one via attach_db.
        if keywords_args.has_key('_ConnectionObject'):
            self._C_conn = keywords_args['_ConnectionObject']
        else:
            self._C_conn = apply(_kinterbasdb.attach_db, args, keywords_args)

        self.default_tpb = default_tpb

    def drop_database(self):
        """
        Drops the database to which this connection is attached.
        """
        _kinterbasdb.drop_database(self._C_conn)

    def begin(self, tpb=None):
        """
        Starts a transaction explicitly.

        - tpb: Optional transaction parameter buffer (TPB) populated with
            kinterbasdb.isc_tpb_* constants.  See the Interbase API guide for
            these constants' meanings.
        """
        if tpb:
            _kinterbasdb.begin(self._C_conn, tpb)
        else:
            # DSR:2002.03.12:
            # Dropped support for Interbase 3 and earlier.
            _kinterbasdb.begin(self._C_conn, self.default_tpb)

    def commit(self):
        """
        Commits (permanently stores the results of the actions that have
        taken place as part of) the active transaction.
        """
        _kinterbasdb.commit(self._C_conn)

    def rollback(self):
        """
        Rolls back (cancels the actions that have taken place as part of) the
        active transaction.
        """
        _kinterbasdb.rollback(self._C_conn)

    def execute_immediate(self, sql):
        """
        Executes a statement without caching its prepared form.  The statement
        must NOT be of a type that returns a result set.

        Before this method is called, a transaction must have been explicitly
        started with the connection's begin() method.  In most cases
        (especially cases in which the same statement--perhaps a parameterized
        statement--is executed repeatedly), it is better to create a cursor
        using the connection's cursor() method, then execute the statement
        using one of the cursor's execute methods.
        """
        _kinterbasdb.execute_immediate_with_connection(self._C_conn, sql)

    def database_info(self, request, result_type):
        """
        Wraps the Interbase C API function isc_database_info.

        For documentation, see the IB 6 API Guide section entitled
        "Requesting information about an attachment" (p. 51).

        Note that this method is a VERY THIN wrapper around the IB C API
        function isc_database_info.  This method does NOT attempt to interpret
        its results except with regard to whether they are a string or an
        integer.
        For example, requesting isc_info_user_names will return a string
        containing a raw succession of length-name pairs.  A thicker wrapper
        might interpret those raw results and return a Python tuple, but it
        would need to handle a multitude of special cases in order to cover
        all possible isc_info_* items.

        Arguments:
        - result_type must be either
            's' if you expect a string result, or
            'i' if you expect an integer result.
        """
        return _kinterbasdb.database_info(self._C_conn, request, result_type)

    def cursor(self):
        """
        Creates and returns a new cursor that operates within the context of
        this connection.
        """
        return Cursor(self)

    def close(self):
        """
        Closes the connection to the database server.
        """
        # Previously, this method simply deleted our self._C_conn reference
        # and let it go at that.  That behavior was in defiance of the DB
        # API spec, as explained in _kinterbasdb.c in the comments for
        # function pyob_close_connection (which was added to address this
        # very problem).
        _kinterbasdb.close_connection(self._C_conn)
        self._C_conn = _kinterbasdb.null_connection

    def __getattr__(self, attr):
        if attr == "dialect":
            return _kinterbasdb.get_dialect(self._C_conn)
        # DSR:2002.02.28:bugfix #517093:begin block
        elif attr == "precision_mode":
            return _kinterbasdb.get_precision_mode(self._C_conn)
        # DSR:2002.02.28:bugfix #517093:end block
        #DSR:2002.03.15:begin block
        elif attr == "server_version":
            rawVersion = self.database_info(isc_info_version, 's')
            # As defined by the IB 6 API Guide, the first byte is the
            # constant 1; the second byte contains the size of the
            # server_version string in bytes.
            return rawVersion[2:2 + raw_byte_to_int(rawVersion[1])]
        #DSR:2002.03.15:end block
        else:
            try:
                return self.__dict__[attr]
            except KeyError:
                raise AttributeError(
                    "Connection instance has no attribute '%s'" % attr)

    def __setattr__(self, attr, value):
        if attr == "dialect":
            return _kinterbasdb.set_dialect(self._C_conn, value)
        # DSR:2002.02.28:bugfix #517093:begin block
        elif attr == "precision_mode":
            return _kinterbasdb.set_precision_mode(self._C_conn, value)
        # DSR:2002.02.28:bugfix #517093:end block
        elif attr == "server_version":
            raise AttributeError("server_version is a read-only attribute")
        else:
            self.__dict__[attr] = value


class Cursor:
    """
    Represents a database cursor operating within the context of a Connection.

    Cursors are typically used to send SQL statements to the database (via one
    of the 'execute*' methods) and to retrieve the results of those statements
    (via one of the 'fetch*' methods).

    The behavior of this class is further explained by the Python DB API
    Specification 2.0.

    Attribute and method notes:
    - description
    Note:  kinterbasdb makes ABSOLUTELY NO GUARANTEES about the description
    attribute of a cursor except those required by the Python Database API
    Specification 2.0 (that is, description is either None or a sequence of
    7-item sequences).
    Therefore, client programmers should NOT rely on description being an
    instance of a particular class or type.

    - rowcount
    The rowcount attribute is defined only in order to conform to the Python
    Database API Specification 2.0.
    The value of the rowcount attribute is initially -1, and it never changes,
    because the Interbase/Firebird C API does not support the determination of
    the number of rows affected by an executed statement.

    - fetch* methods
    Note:  kinterbasdb makes ABSOLUTELY NO GUARANTEES about the return value
    of the fetch(one|many|all) methods except that it is a sequence indexed
    by field position, and no guarantees about the return value of the
    fetch(one|many|all)map methods except that it is a mapping of
    field name to field value.
    Therefore, client programmers should NOT rely on the return value being
    an instance of a particular class or type.

    - arraysize
    The arraysize attribute is defined only in order to conform to the Python
    Database API Specification 2.0.
    In the current implementation, the value of the arraysize attribute does
    not affect perfomance at all, but the Python DB API specification still
    requires that it be present, and that method fetchmany take its value into
    account if fetchmany's optional parameter named 'size' is not specified.
    """

    def __init__(self, _conn):
        # The instance of Python class Connection that represents the database
        # connection within the context of which this cursor operates.
        self._conn = _conn

        # The instance of C type CursorType that represents this cursor in
        # _kinterbasdb.
        self._C_cursor = _kinterbasdb.cursor(self._conn._C_conn)

        # cursor attributes required by the Python Database API Specification
        # 2.0.
        self.description = None
        self.rowcount = -1
        self.arraysize = 1

    def setinputsizes(sizes):
        """
        Defined for compatibility with the DB API specification, but
        currently does nothing.
        """
        pass

    def setoutputsize(size, column=None):
        """
        Defined for compatibility with the DB API specification, but
        currently does nothing.
        """
        pass

    def execute(self, sql, params=()):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        res = _kinterbasdb.execute(self._C_cursor, sql, params)

        self.description = res[1]

    # DSR:2002.02.21:
    # Previously, executemany treated its -params argument as optional (and
    # would execute the statement once by default--without any parameters).
    # That behavior was both in defiance of the spec and illogical
    # (executemany(sql) executes once but executemany(sql, [1]) also executes
    # once--huh?).
    def executemany(self, sql, manySetsOfParameters):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        for setOfParameters in manySetsOfParameters:
            res = _kinterbasdb.execute(self._C_cursor, sql, setOfParameters)

        self.description = res[1]

    # The next two methods are used by the other fetch methods to fetch an
    # individual row as either a sequence (_fetch) or a mapping (_fetchmap).
    def _fetch(self):
        row = _kinterbasdb.fetch(self._C_cursor)
        return row or None

    def _fetchmap(self):
        row = self._fetch()
        if not row:
            return None
        return _RowMapping(self.description, row)


    # Generic fetch(one|many|all) methods parameterized by fetchMethod so that
    # their logical structure can be reused for both conventional sequence
    # fetch and mapping fetch.
    def _fetchone(self, fetchMethod):
        return fetchMethod()

    def _fetchmany(self, size, fetchMethod):
        if size is None:
            size = self.arraysize
        elif size <= 0:
            raise InterfaceError("The size parameter of the fetchmany or"
                " fetchmanymap method (which specifies the number of rows to"
                " fetch) must be greater than zero.  It is an optional"
                " parameter, so it can be left unspecifed, in which case it"
                " will default to the value of the cursor's arraysize"
                " attribute.")

        manyRows = []
        i = 0
        while i < size:
            row = fetchMethod()
            if row is None:
                break
            manyRows.append(row)
            i = i + 1
        return manyRows

    def _fetchall(self, fetchMethod):
        allRows = []
        while 1:
            row = fetchMethod()
            if row is None:
                break
            allRows.append(row)
        return allRows

    # The public fetch methods:
    def fetchone(self):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        return self._fetchone(self._fetch)

    def fetchonemap(self):
        """
        This method is just like fetchone, except that it returns a mapping
        of field name to field value, rather than a sequence.
        """
        return self._fetchone(self._fetchmap)

    def fetchmany(self, size=None):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        return self._fetchmany(size, self._fetch)

    def fetchmanymap(self, size=None):
        """
        This method is just like fetchmany, except that it returns a sequence
        of mappings of field name to field value, rather than a sequence of
        sequences.
        """
        return self._fetchmany(size, self._fetchmap)

    def fetchall(self):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        return self._fetchall(self._fetch)

    def fetchallmap(self):
        """
        This method is just like fetchall, except that it returns a sequence
        of mappings of field name to field value, rather than a sequence of
        sequences.
        """
        return self._fetchall(self._fetchmap)

    def close(self):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        _kinterbasdb.close_cursor(self._C_cursor)

        # DSR:2002.02.26:
        # _kinterbasdb often reopens its "closed" CursorType instances,
        # and the following line caused our self._C_cursor to be garbage
        # collected, so disable it and hang onto that reference!
        # self._C_cursor = _kinterbasdb.null_cursor

    def callproc(self, procname, params=()):
        """
        The behavior of this method is explained by the Python DB API
        Specification 2.0.
        """
        sql = 'EXECUTE PROCEDURE %s ' % procname
        if params:
            sql = sql + string.join(('?',) * len(params), ',')
        self.execute(sql, params)

##########################################
##     PUBLIC CLASSES: END              ##
##########################################

class _RowMapping:
    """
    An internal kinterbasdb class that wraps a row of results in order to map
    field name to field value.

    Note:  kinterbasdb makes ABSOLUTELY NO GUARANTEES about the return value
    of the fetch(one|many|all) methods except that it is a sequence indexed
    by field position, and no guarantees about the return value of the
    fetch(one|many|all)map methods except that it is a mapping of
    field name to field value.
    Therefore, client programmers should NOT rely on the return value being
    an instance of a particular class or type.
    """
    def __init__(self, description, row):
        if len(description) != len(row):
            raise InternalError("cursor description does not match the result"
                " set that it describes")

        # The sequence containing the values of this row's fields:
        self.row = row

        # The description of this result set, from which we can extract
        # field names and their corresponding positions in self.row.
        self.description = description

    def __getitem__(self, key):
        if type(key) is not types.StringType:
            columnNames = map(lambda x: x[DESCRIPTION_NAME], self.description)

            raise KeyError(
                    "Mapping of (column name -> column value) requires that"
                    " the column name be specified as a string.  In the case"
                    " of this particular result set, the column name must be"
                    " one of: "
                    + str(columnNames)
                )

        # Convert the key to the same case as the column names in
        # self.description so that clients can refer to columns in a
        # case-insensitive manner, just as they can in SQL.
        key = string.upper(key)

        # Of course this is an inefficient way to look up the field value,
        # but in most result set rows:
        # a) there are only a few fields
        # b) not every field is looked up before the client program moves
        #    to the next row
        #
        # For these reasons, it appears that the cost of constructing a
        # Python dictionary and then wrapping it in a case-insensitive
        # manner would be more costly on average than the current approach.
        #
        # If performance is the client programmer's main concern, he should
        # be using the fetch(one|many|all) methods rather than the
        # fetch(one|many|all)map methods.
        for i in range(len(self.description)):
            if key == self.description[i][DESCRIPTION_NAME]:
                return self.row[i]

    def __str__(self):
        # Return an easily readable dump of this row's field names and their
        # corresponding values.
        res = []

        for i in range(len(self.description)):
            fieldName = self.description[i][DESCRIPTION_NAME] or '[unknown]'
            res.append('%s = %s' %
                    ( fieldName, str(self.row[i]) )
                )

        return '<result set row with ' + string.join(res, ', ') + '>'
