# Original implementation by Evgeny Cherkashin during 2001.
# Totally rewritten by David Rushby (DSR) beginning on 2002.02.12.

# XXX:DSR:2002.02.17:
# Someone with more RPM sophistication than DSR needs to ensure that
# python setup.py bdist_rpm
# is working prorperly.  It goes through the motions, but...?

import ConfigParser, os, string, sys

DEBUG = int(os.environ.get('KINTERBASDB_DEBUG', 0))

VERSION_FILE = 'version.txt'
CONFIG_FILE = 'setup.cfg'

DISTUTILS_URL = "http://www.python.org/sigs/distutils-sig/distutils.html"
DATETIME_URL = "http://www.lemburg.com/files/python/eGenix-mx-Extensions.html"

# Be careful about changing these message; the docs refer to them.
PYTHON_SYSTEM_ERROR_HEADER = "PYTHON SYSTEM ERROR:"
# DSR:2002.03.03:begin BCPP fix block
COMPILER_CONFIGURATION_ERROR_HEADER = "COMPILER CONFIGURATION ERROR:"
# DSR:2002.03.03:end BCPP fix block
KIDB_DISTRIBUTION_ERROR_HEADER = "KINTERBASDB DISTRIBUTION ERROR:"
AUTODETECTION_ERROR_HEADER = "LIBRARY AUTODETECTION ERROR:"
MANUAL_SPECIFICATION_ERROR_HEADER = "LIBRARY MANUAL SPECIFICATION ERROR:"

try:
    import distutils
except ImportError:
    sys.stderr.write(
            "%s\n"
            "  Cannot import the standard package 'distutils'.\n"
            "  distutils did not join the standard library until Python 1.6,\n"
            "  and some some nonstandard Python distributions do not\n"
            "  include the package.\n"
            "  You must either upgrade to a Python version with distutils\n"
            "  integrated, or download the package from:\n"
            "    %s"
            % (PYTHON_SYSTEM_ERROR_HEADER, DISTUTILS_URL,)
        )
    sys.exit(1)

import distutils.core
import distutils.ccompiler
import distutils.sysconfig

# Module name and version number:
kinterbasdb_name = "kinterbasdb"
# Retrive the kinterbasdb version number from a standard text file for the sake
# of maintainability:
try:
    kinterbasdb_version = string.strip(open(VERSION_FILE).read())
except IOError:
    sys.stderr.write(
            "%s\n"
            " Missing file 'version.txt'; cannot determine kinterbasdb"
            " version."
            % KIDB_DISTRIBUTION_ERROR_HEADER
        )
    sys.exit(1)

# These config parameters are user-specifiable via setup.cfg:
DATETIME_INCLUDE_DIR = None

DATABASE_IS_FIREBIRD = None
DATABASE_HOME_DIR = None
DATABASE_INCLUDE_DIR = None
DATABASE_LIB_DIR = None
DATABASE_LIB_NAME = None

# These config parameters are not drawn from setup.cfg:
CUSTOM_PREPROCESSOR_DEFS = []
PLATFORM_SPECIFIC_INCLUDE_DIRS = []
PLATFORM_SPECIFIC_LIB_DIRS = []
# DSR:2002.03.07:begin block
PLATFORM_SPECIFIC_LIB_NAMES = []
# DSR:2002.03.07:end block
# DSR:2002.03.03:begin BCPP fix block
PLATFORM_SPECIFIC_EXTRA_COMPILER_ARGS = []
PLATFORM_SPECIFIC_EXTRA_LINKER_ARGS = []
# DSR:2002.03.03:end BCPP fix block

# See if the user manually specified various build options in the setup config
# file.  If so, skip autodetection for the options that the user has specified.
config = ConfigParser.ConfigParser()
config.read(CONFIG_FILE)

if config.has_section('manual_config'):
    # It would be better to test against sys.version_info rather than the
    # string sys.version, but it's not available on older Python versions.
    if sys.version < '1.6':
        # Compensate for 1.5.2's lack of ConfigParser.ConfigParser.has_option:
        def config_has_option(section, option, config=config):
            try:
                return config.get(section, option)
            except ConfigParser.NoOptionError:
                return None
        config.has_option = config_has_option
    if config.has_option('manual_config', 'datetime_include_dir'):
        DATETIME_INCLUDE_DIR = config.get('manual_config', 'datetime_include_dir')

    if config.has_option('manual_config', 'database_is_firebird'):
        DATABASE_IS_FIREBIRD = config.get('manual_config', 'database_is_firebird')
    if config.has_option('manual_config', 'database_home_dir'):
        DATABASE_HOME_DIR = config.get('manual_config', 'database_home_dir')
    if config.has_option('manual_config', 'database_include_dir'):
        DATABASE_INCLUDE_DIR = config.get('manual_config', 'database_include_dir')
    if config.has_option('manual_config', 'database_lib_dir'):
        DATABASE_LIB_DIR = config.get('manual_config', 'database_lib_dir')
    if config.has_option('manual_config', 'database_lib_name'):
        DATABASE_LIB_NAME = config.get('manual_config', 'database_lib_name')

if DEBUG:
    print "*** CONFIG OPTIONS SPECIFIED IN %s SECTION 'manual_config' ***" % CONFIG_FILE
    for key in config.options('manual_config'):
        print '%s:' % (key)
        print '    %s' % (config.get('manual_config', key))

ALL_AUTODETECT_OPTIONS_MANUALLY_SPECIFIED = (
            DATETIME_INCLUDE_DIR

        and DATABASE_IS_FIREBIRD is not None
        and DATABASE_HOME_DIR
        and DATABASE_INCLUDE_DIR
        and DATABASE_LIB_DIR
        and DATABASE_LIB_NAME
    )

# Utility functions:

def addMXIncludedir(d):
    return os.path.join(d, os.path.join('mx', 'DateTime', 'mxDateTime'))

def verifyAutodetectedDatetimeIncludeDir():
    if not os.path.exists(DATETIME_INCLUDE_DIR):
            sys.stderr.write(
                    "%s\n"
                    "  Cannot find the eGenix.com mx Base Extensions for Python.\n"
                    "  (Tried %s)\n"
                    "  Try specifying the 'datetime_include_dir' option in\n"
                    "  the 'manual_config' section of the setup config file (%s).\n"
                    % (AUTODETECTION_ERROR_HEADER, DATETIME_INCLUDE_DIR, CONFIG_FILE)
                )
            sys.exit(1)

def verifyUserSpecifiedDatetimeIncludeDir():
    if not os.path.exists(DATETIME_INCLUDE_DIR):
        sys.stderr.write(
                "%s\n"
                "  The user-specified header file directory of the\n"
                "  eGenix.com mx Base Extensions for Python\n"
                "    (%s)\n"
                "  does not exist.\n"
                "  Try modifying the 'datetime_include_dir' option in\n"
                "  the 'manual_config' section of the setup config file (%s),\n"
                "  or comment out that option to force this script to\n"
                "  to autodetect it.\n"
                % (MANUAL_SPECIFICATION_ERROR_HEADER, DATETIME_INCLUDE_DIR, CONFIG_FILE)
            )
        sys.exit(1)

def verifyAutodetectedDatabaseIncludeDir():
    if not os.path.exists(DATABASE_INCLUDE_DIR):
        sys.stderr.write(
                "%s\n"
                "  Cannot autodetect the database header file directory.\n"
                "  (Tried %s)\n"
                "  Try specifying the 'database_include_dir' option in\n"
                "  the 'manual_config' section of the setup config file (%s).\n"
                % (AUTODETECTION_ERROR_HEADER, DATABASE_INCLUDE_DIR, CONFIG_FILE)
            )
        sys.exit(1)

def verifyUserSpecifiedDatabaseIncludeDir():
    if not os.path.exists(DATABASE_INCLUDE_DIR):
        sys.stderr.write(
                "%s\n"
                "  The user-specified database header file directory\n"
                "    %s\n"
                "  does not exist.\n"
                "  Try modifying the 'database_include_dir' option in\n"
                "  the 'manual_config' section of the setup config file (%s),\n"
                "  or comment out that option to force this script to\n"
                "  to autodetect it.\n"
                % (MANUAL_SPECIFICATION_ERROR_HEADER, DATABASE_INCLUDE_DIR, CONFIG_FILE)
            )
        sys.exit(1)

def verifyAutodetectedDatabaseLibraryDir():
    if not os.path.exists(DATABASE_LIB_DIR):
        sys.stderr.write(
                "%s\n"
                "  Cannot autodetect the database lib directory.\n"
                "  (Tried %s)\n"
                "  Try specifying the 'database_include_dir' option in\n"
                "  the 'manual_config' section of the setup config file (%s).\n"
                % (AUTODETECTION_ERROR_HEADER, DATABASE_LIB_DIR, CONFIG_FILE)
            )
        sys.exit(1)

def verifyUserSpecifiedDatabaseLibraryDir():
    if not os.path.exists(DATABASE_LIB_DIR):
        sys.stderr.write(
                "%s\n"
                "  The user-specified database lib directory\n"
                "    %s\n"
                "  does not exist.\n"
                "  Try modifying the 'database_lib_dir' option in\n"
                "  the 'manual_config' section of the setup config file (%s),\n"
                "  or comment out that option to force this script to\n"
                "  to autodetect it.\n"
                % (MANUAL_SPECIFICATION_ERROR_HEADER, DATABASE_LIB_DIR, CONFIG_FILE)
            )
        sys.exit(1)

def findSpacelessDirName(d):
    # On Windows, try to find the spaceless version of the provided directory
    # name.
    # This function was added on 2002.03.14 as part of an ugly hack to
    # surmount a bug in the distutils package.

    # Sometime distutils causes None to be fed to this function.
    if not d:
        return d

    d = os.path.normpath(os.path.abspath(d))

    if ' ' not in d:
        return d

    # If d doesn't exist, its short equivalent can't be determined.
    # However, for the purposes of this program (which is solely for
    # convenience anyway) it's better just to risk feeding the
    # compiler/linker a path with a space in it than it is to raise
    # an exception when there's still a *chance* of success.
    if not os.path.isdir(d):
        return d

    try:
        import win32api

        return os.path.normcase(win32api.GetShortPathName(d))
    except ImportError:
        # Since win32api is not available, we'll revert to a lame,
        # manual approximation of GetShortPathName.
        pass

    ds = d.split(os.sep) # Split into components.

    if DEBUG:
        print 'ds is', ds

    ds[0] = ds[0] + '\\' # Add slash back onto drive designation so that
                         # it's like c:\ rather than just c:

    dsNoSpaces = [] # Will contain a version of the directory components
                    # with all spaces removed.
    for x in range(len(ds)):
        dir = ds[x]
        if DEBUG:
            print 'dir is', dir

        if dir.find(' ') == -1:
            shortDir = dir
        else:
            fullDir = apply(os.path.join, ds[:x + 1])

            if DEBUG:
                print 'fullDir is', fullDir

            # Must deal with names like 'abc de' that have their space
            # before the sixth character.
            dirNoSpaces = dir.replace(' ', '')
            if len(dirNoSpaces) < 6:
                shortDirBase = dirNoSpaces
            else:
                shortDirBase = dirNoSpaces[:6]

            # Search for shortDirBase~i until we find it.
            shortDir = None
            i = 1
            while i < 9: # This code doesn't handle situations where there are
                         # more than nine directories with the same prefix.
                maybeShortDir = '%s~%d' % (shortDirBase, i)
                fullMaybeShortDir = os.path.join(
                    os.path.dirname(fullDir), maybeShortDir)
                if not os.path.isdir(fullMaybeShortDir):
                    continue

                # There follows a *really* lame approximation of
                # os.path.samefile, which is not available on Windows.
                if os.listdir(fullMaybeShortDir) == os.listdir(fullDir):
                    shortDir = maybeShortDir
                    break

                i = i + 1

            if shortDir is None:
                raise Exception('Unable to find shortened version of'
                    ' directory named %s' % d)

        dsNoSpaces.append(shortDir)

    if DEBUG:
        print 'dsNoSpaces is', dsNoSpaces

    return os.path.normcase(apply(os.path.join, dsNoSpaces))


# Perform generic compilation parameter setup, then switch to platform-
# specific.

curdirAbs = os.path.abspath(os.curdir)

# Autodetect Python directory info.
if DEBUG:
    print '*** PYTHON SETTINGS ***'

# XXX:DSR:  Does the pythonHomeDir contain the expected files and directories
# on UNIX-like systems as well?
pythonHomeDir = sys.exec_prefix
pythonPkgDir = distutils.sysconfig.get_python_lib()

if DEBUG:
    print '\tPython home dir:', pythonHomeDir
    print '\tPython package dir:', pythonPkgDir

# Autodetect mx DateTime directory info if the user did not specify it.
if DEBUG:
    print '*** MX DATETIME SETTINGS ***'

if DATETIME_INCLUDE_DIR:
    verifyUserSpecifiedDatetimeIncludeDir()
else: # try to autodetect DATETIME_INCLUDE_DIR
    DATETIME_INCLUDE_DIR = addMXIncludedir(pythonPkgDir)
    if DEBUG:
        print '\tDATETIME_INCLUDE_DIR exists at\n\t  %s: %d' \
            % (DATETIME_INCLUDE_DIR, os.path.exists(DATETIME_INCLUDE_DIR))

    if not os.path.exists(DATETIME_INCLUDE_DIR):
        # If we can successfully import the DateTime package, then it must
        # be available to the system somewhere; search for it.
        try:
            from mx import DateTime

            # Ok, it's here somewhere.
            for d in sys.path:
                hypotheticalMXDir = os.path.join(d, 'mx')
                if DEBUG:
                    print '\tSearching for mx extensions in:', hypotheticalMXDir
                if os.path.exists(hypotheticalMXDir):
                    hypotheticalMXIncludeDir = addMXIncludedir(hypotheticalMXDir)
                    if os.path.exists(hypotheticalMXIncludeDir):
                        DATETIME_INCLUDE_DIR = hypotheticalMXIncludeDir
                        break
        except ImportError:
            pass

    verifyAutodetectedDatetimeIncludeDir()

# Begin platform-specific compilation parameter setup:
if sys.platform == 'win32':
    ALL_AUTODETECT_WINDOWS_REGISTRY_OPTIONS_MANUALLY_SPECIFIED = (
                DATABASE_HOME_DIR
            and DATABASE_INCLUDE_DIR
            and DATABASE_LIB_DIR
            and DATABASE_LIB_NAME
        )

    CUSTOM_PREPROCESSOR_DEFS.append( ('WIN32', None) )

    # If this is a source distribution, add a couple of necessary include and
    # lib directories.
    pcbuildDir = os.path.join(
        os.path.split(distutils.sysconfig.get_python_inc())[0], 'PCBuild')
    if os.path.exists(pcbuildDir):
        PLATFORM_SPECIFIC_LIB_DIRS.append(pcbuildDir)

        pySrcDistExtraIncludeDir = os.path.join(
            os.path.split(distutils.sysconfig.get_python_inc())[0], 'PC')
        PLATFORM_SPECIFIC_INCLUDE_DIRS.append(pySrcDistExtraIncludeDir)

    # Verify the various directories (such as include and library dirs) that
    # will be used during compilation.

    # Open the registry in preparation for reading various installation
    # directories from it.
    try:
        import _winreg
    except ImportError:
        # If the user has manually specified all of the options that would
        # require registry access to autodetect, we can proceed despite the
        # lack of _winreg.
        if not ALL_AUTODETECT_WINDOWS_REGISTRY_OPTIONS_MANUALLY_SPECIFIED:
            sys.stderr.write(
                    "%s\n"
                    "  Cannot import the standard package '_winreg'.\n"
                    "  _winreg did not join the standard library until\n"
                    "  Python 2.0.  If you are using a source distribution\n"
                    "  of Python 2.0 or later, you may have simply forgotten\n"
                    "  to compile the _winreg C extension.\n"
                    "  You can get around the lack of _winreg by manually\n"
                    "  specifying all of the configuration options in the\n"
                    "  'manual_config' section of the setup config file (%s)."
                    % (AUTODETECTION_ERROR_HEADER, CONFIG_FILE)
                )
            sys.exit(1)

    if not ALL_AUTODETECT_WINDOWS_REGISTRY_OPTIONS_MANUALLY_SPECIFIED:
        try:
            r = _winreg.ConnectRegistry(None, _winreg.HKEY_LOCAL_MACHINE)
        except WindowsError, e:
            sys.stderr.write(
                    "%s\n"
                    "  Unable to connect to the HKEY_LOCAL_MACHINE section of\n"
                    "  the Windows registry.\n"
                    "  The specific error encountered is:\n"
                    "  %s"
                    % (AUTODETECTION_ERROR_HEADER, str(e))
                )
            sys.exit(1)

    if DEBUG:
        print '*** DATABASE SETTINGS ***'

    # Autodetect database directory info if the user did not specify it.
    if not DATABASE_HOME_DIR:
        def findDatabaseHomeDir(databaseInfoKey, databaseHomeValueName):
            databaseCurrentVersionKey = _winreg.OpenKey(r, databaseInfoKey)
            try:
                return _winreg.QueryValueEx(
                        databaseCurrentVersionKey,
                        databaseHomeValueName
                    )[0]
            finally:
                _winreg.CloseKey(databaseCurrentVersionKey)

        # Try to find Firebird first; revert to Interbase only if necessary.
        try:
            try:
                DATABASE_HOME_DIR = findDatabaseHomeDir(
                        r'SOFTWARE\FirebirdSQL\Firebird\CurrentVersion',
                        'RootDirectory'
                    )
                DATABASE_IS_FIREBIRD = 1

                if DEBUG:
                    print '    Autodetected database is Firebird.'
            except WindowsError:
                # Revert to Interbase.
                DATABASE_IS_FIREBIRD = 0
                DATABASE_HOME_DIR = findDatabaseHomeDir(
                        r'SOFTWARE\Borland\InterBase\CurrentVersion',
                        'RootDirectory'
                    )

                if DEBUG:
                    print '    Autodetected database is Interbase.'
        except WindowsError, e:
            sys.stderr.write(
                    "%s\n"
                    "  Unable to retrieve database directory from the\n"
                    "  Windows registry.\n"
                    "  Try specifying the 'database_home_dir' option in the\n"
                    "  'manual_config' section of the setup config file (%s).\n"
                    "  The specific error encountered is:\n"
                    "  %s"
                    % (AUTODETECTION_ERROR_HEADER, CONFIG_FILE, str(e))
                )
            sys.exit(1)

    if DATABASE_INCLUDE_DIR:
        verifyUserSpecifiedDatabaseIncludeDir()
    else: # user-specified DATABASE_INCLUDE_DIR
        if DATABASE_IS_FIREBIRD:
            databaseSDKDir = DATABASE_HOME_DIR
        else:
            databaseSDKDir = os.path.join(DATABASE_HOME_DIR, 'SDK')
        DATABASE_INCLUDE_DIR = os.path.join(databaseSDKDir, 'include')

        if DEBUG:
            print '\tDATABASE_INCLUDE_DIR exists at\n\t  %s: %d' \
                % (DATABASE_INCLUDE_DIR, os.path.exists(DATABASE_INCLUDE_DIR))

        verifyAutodetectedDatabaseIncludeDir()

    if DATABASE_LIB_DIR:
        verifyUserSpecifiedDatabaseLibraryDir()
    else: # user-specified DATABASE_LIB_DIR
        DATABASE_LIB_DIR = os.path.join(databaseSDKDir, 'lib')

        verifyAutodetectedDatabaseLibraryDir()

    # Perform compiler-specific setup.
    if not DATABASE_LIB_NAME:
        DATABASE_LIB_NAME = 'gds32'

    # I should discover a way to ask distutils what compiler it's
    # configured to use--the current code would only detect a compiler
    # change via the command line.  I've looked at the compiler subpackage
    # of distutils, and can't find any such facility (though it can be
    # queried for the system's default compiler).

    # DSR:2002.03.03:begin BCPP fix block
    customCompilerName = 'msvc'
    import re
    argJam = string.lower(string.join(sys.argv[1:]))

    match = re.search(r'--compiler=(?P<cname>\S+)', argJam)
    if match:
        customCompilerName = match.group('cname')
    else:
        match = re.search(r'-c\s+(?P<cname>\S+)', argJam)
        if match:
            customCompilerName = match.group('cname')

    if customCompilerName == 'msvc':
        if not DATABASE_IS_FIREBIRD:
            DATABASE_LIB_DIR = os.path.join(databaseSDKDir, r'lib_ms')
        DATABASE_LIB_NAME = 'gds32_ms'
    elif customCompilerName == 'bcpp':
        COMPILER_EXE_NAME = 'bcc32.exe'

        # Try to find the home directory of the Borland compiler by searching
        # each directory listed in the PATH.
        osPath = string.split(os.environ['PATH'], os.pathsep)
        bccHome = None
        for dir in osPath:
            if os.path.exists(os.path.join(dir, COMPILER_EXE_NAME)):
                bccHome = os.path.split(dir)[0]
                break
        else:
            # Couldn't find it.
            sys.stderr.write(
                    "%s\n"
                    "  Unable to find the home directory of the Borland"
                    " compiler.\n"
                    "  You must add the 'bin' subdirectory of the"
                    " compiler's\n"
                    "  home directory to your PATH.\n"
                    "  One way to do this is to type a command of the"
                    " format\n"
                    "    SET PATH=%%PATH%%;c:\\EXAMPLE_DIR\\bin\n"
                    "  in the same command prompt you use to run the"
                    " kinterbasdb setup script."
                    % (COMPILER_CONFIGURATION_ERROR_HEADER,)
                )
            sys.exit(1)

        # Override the default behavior of distutils.bcppcompiler.BCPPCompiler
        # in order to force it to issue the correct command.
        #
        # I hate to even admit to having written this code, but I assure the
        # reader that I exhausted all other known options before I resorted
        # to this junk.
        from distutils.bcppcompiler import BCPPCompiler
        class BCPPUglyHack(BCPPCompiler):
            # Too bad Python 1.5.2 doesn't support *args, **kwargs syntax.
            def compile (self,
                 sources,
                 output_dir=None,
                 macros=None,
                 include_dirs=None,
                 debug=0,
                 extra_preargs=None,
                 extra_postargs=None
             ):
                bccIncludePreargDir = findSpacelessDirName('%s\include' % bccHome)
                bccLibPreargDir = findSpacelessDirName('%s\lib' % bccHome)
                bccLibPSDKPreargDir = findSpacelessDirName('%s\lib\psdk' % bccHome)

                extra_preargs = (
                    [
                        r'-I"%s"' % bccIncludePreargDir,
                        r'-L"%s"' % bccLibPreargDir,
                        r'-L"%s"' % bccLibPSDKPreargDir
                     ]
                    + (extra_preargs or [])
                )

                return BCPPCompiler.compile(self,
                    sources,
                    findSpacelessDirName(output_dir),
                    macros,
                    map(findSpacelessDirName, include_dirs),
                    debug,
                    extra_preargs,
                    extra_postargs
                )

            def link (self,
                target_desc,
                objects,
                output_filename,
                output_dir=None,
                libraries=None,
                library_dirs=None,
                runtime_library_dirs=None,
                export_symbols=None,
                debug=0,
                extra_preargs=None,
                extra_postargs=None,
                build_temp=None
            ):
                ilinkLibPreargDir = findSpacelessDirName('%s\lib' % bccHome)
                ilinkLibPSDKPreargDir = findSpacelessDirName('%s\lib\psdk' % bccHome)

                extra_preargs = (
                        [
                            r'/L"%s"' % ilinkLibPreargDir,
                            r'/L"%s"' % ilinkLibPSDKPreargDir
                        ]
                        + (extra_preargs or [])
                    )

                return BCPPCompiler.link(self,
                    target_desc,
                    objects,
                    output_filename,
                    findSpacelessDirName(output_dir),
                    libraries,
                    map(findSpacelessDirName, library_dirs),
                    map(findSpacelessDirName, runtime_library_dirs),
                    export_symbols,
                    debug,
                    extra_preargs,
                    extra_postargs,
                    findSpacelessDirName(build_temp)
                )

        compilerSetupTuple = distutils.ccompiler.compiler_class['bcpp']
        import distutils.bcppcompiler
        distutils.bcppcompiler.BCPPUglyHack = BCPPUglyHack
        distutils.ccompiler.compiler_class['bcpp'] = (
            compilerSetupTuple[0], 'BCPPUglyHack', compilerSetupTuple[2])

        # Use coff2omf to create a Borland-compiler-compatible library file,
        # "pythonVV_bcpp.lib", from the standard "pythonVV.lib".

        # It would be better to use sys.version_info rather than the
        # string sys.version, but it's not available on older Python
        # versions.
        pyLibVersionSuffix = sys.version[0] + sys.version[2]
        newLibFilename = r'%s\libs\python%s_bcpp.lib' % (pythonHomeDir,
               pyLibVersionSuffix)
        print 'setup.py is trying to create %s' % newLibFilename
        if not os.path.exists(os.path.join(pythonHomeDir, 'libs',
            'python%s_bcpp.lib' % pyLibVersionSuffix)
          ):
            oldLibFilename = '%s\libs\python%s.lib' % (pythonHomeDir,
               pyLibVersionSuffix)
            coff2omfCommand = ('coff2omf %s %s'
                % (oldLibFilename, newLibFilename))

            os.system(coff2omfCommand)
            # Do this test instead of checking the return value of
            # os.system, which will not reliably indicate an error
            # condition on Win9x.
            if not os.path.exists(newLibFilename):
                sys.stderr.write(
                        "%s\n"
                        "  Unable to create a Borland-compatible Python"
                        " library file using the\n"
                        "  coff2omf utility.\n"
                        "  Tried command:\n"
                        "    %s"
                        % ( COMPILER_CONFIGURATION_ERROR_HEADER,
                            coff2omfCommand)
                    )
                sys.exit(1)
        assert os.path.exists(newLibFilename)
    # DSR:2002.03.03:end BCPP fix block

    if DEBUG:
        print '\tDATABASE_LIB_DIR exists at\n\t  %s: %d' \
            % (DATABASE_LIB_DIR, os.path.exists(DATABASE_LIB_DIR))
        print '\tDATABASE_LIB_NAME is\n\t  %s' % DATABASE_LIB_NAME
else: # not Windows
    # If the platform isn't Linux, issue a warning.
    if len(sys.platform) < 5 or not string.lower(sys.platform)[:5] == 'linux':
        sys.stderr.write("Warning:  The kinterbasdb setup code was not"
            " specifically written to support your platform (%s), and"
            " may not work properly."
            % sys.platform)
    else:
        # The platform is Linux.
        pass

    # DSR:2002.03.14:
    # Is libcrypt necessary on all POSIX OSes, or just Linux?
    # Until someone informs me otherwise, I'll assume all.
    if os.name == 'posix':
        PLATFORM_SPECIFIC_LIB_NAMES.append('crypt')

    # Verify the various directories (such as include and library dirs) that
    # will be used during compilation.

    # Assumption:
    # This is a Unix-like OS, where a proper installation routine would have
    # placed the database [header, library] files in a system-wide dirs.
    # We have no way of knowing beyond the shadow of a doubt whether that
    # has happened (as opposed to the situation on Windows, where we can
    # consult the registry to determine where a binary installer placed its
    # files), so we'll just let the compiler complain if it can't find the
    # [header, library] files.
    # If, on the other hand, the user manually specified the directories, we
    # may verify that they exist.

    if DATABASE_INCLUDE_DIR: # the user manually specified it
        verifyUserSpecifiedDatabaseIncludeDir()

    if DATABASE_LIB_DIR: # the user manually specified it
        verifyUserSpecifiedDatabaseLibraryDir()

    DATABASE_LIB_NAME = 'gds'

# Now finished with platform-specific compilation parameter setup.

# Create a list of all INCLUDE dirs to be passed to setup():
allIncludeDirs = []

# Add Python include directory:
allIncludeDirs.append(distutils.sysconfig.get_python_inc())

if len(PLATFORM_SPECIFIC_INCLUDE_DIRS) > 0:
    allIncludeDirs.extend(PLATFORM_SPECIFIC_INCLUDE_DIRS)

if DATETIME_INCLUDE_DIR:
    allIncludeDirs.append(DATETIME_INCLUDE_DIR)

if DATABASE_INCLUDE_DIR:
    allIncludeDirs.append(DATABASE_INCLUDE_DIR)

# Create a list of all LIB names to be passed to setup():
allLibNames = []
if DATABASE_LIB_NAME:
    allLibNames.append(DATABASE_LIB_NAME)
# DSR:2002.03.07:begin block
allLibNames.extend(PLATFORM_SPECIFIC_LIB_NAMES)
# DSR:2002.03.07:end block

# Create a list of all LIB directories to be passed to setup():
allLibDirs = []
if len(PLATFORM_SPECIFIC_LIB_DIRS) > 0:
    allLibDirs.extend(PLATFORM_SPECIFIC_LIB_DIRS)

if DATABASE_LIB_DIR:
    allLibDirs.append(DATABASE_LIB_DIR)

# Create a list of all macro definitions that must be passed to distutils.
allMacroDefs = []

if len(CUSTOM_PREPROCESSOR_DEFS) > 0:
    allMacroDefs.extend(CUSTOM_PREPROCESSOR_DEFS)

# DSR:2002.03.03:begin BCPP fix block
# Create a list of all extra options to pass to the compiler, linker.
allExtraCompilerArgs = []
if len(PLATFORM_SPECIFIC_EXTRA_COMPILER_ARGS) > 0:
    allExtraCompilerArgs.extend(PLATFORM_SPECIFIC_EXTRA_COMPILER_ARGS)

allExtraLinkerArgs = []
if len(PLATFORM_SPECIFIC_EXTRA_LINKER_ARGS) > 0:
    allExtraLinkerArgs.extend(PLATFORM_SPECIFIC_EXTRA_LINKER_ARGS)
# DSR:2002.03.03:end BCPP fix block

# Now we've detected and verified all compilation parameters, and are ready to
# compile.

if DEBUG:
    print '*** SETTINGS DETECTION PHASE COMPLETE; READY FOR BUILD ***'
    print ("\tThe DEBUG flag is enabled, so the setup script stops\n"
           "\t  before actually invoking the distutils setup procedure."
        )
    sys.exit(0)

# The MEAT:
distutils.core.setup(
    name=kinterbasdb_name,
    version=kinterbasdb_version,
    author="""Originally by Alexander Kuznetsov ;
                   now maintained and enhanced by others
                   (see docs/license.txt for details).""",
    url="http://kinterbasdb.sourceforge.net",
    description="Python DB-API 2.0-compliant extension for Interbase/Firebird",
    long_description=
        "kinterbasdb allows Python to access the Interbase/Firebird\n"
        "SQL server according to the interface defined by the\n"
        "Python Database API Specification version 2.0.",
    license="see docs/license.txt",
    packages=["kinterbasdb"],
    package_dir={"kinterbasdb": os.curdir},
    ext_modules=[
        distutils.core.Extension( "kinterbasdb._kinterbasdb",
            sources=["_kinterbasdb.c"],
            libraries=allLibNames,
            include_dirs=allIncludeDirs,
            library_dirs=allLibDirs,
            define_macros=allMacroDefs,
            # DSR:2002.03.03:begin BCPP fix block
            extra_compile_args=allExtraCompilerArgs,
            extra_link_args=allExtraLinkerArgs
            # DSR:2002.03.03:end BCPP fix block
        )
    ],
    data_files = [
        # documentation:
        ("kinterbasdb/docs",
            [
                "docs/index.html",
                "docs/installation-source.html",
                "docs/installation-binary.html",
                "docs/usage.html",
                "docs/changelog.txt",
                "docs/license.txt",
                "docs/Python-DB-API-2.0.html",
                "docs/links.html",
                "docs/global.css"
            ]
        )
    ]
)