#!/usr/bin/env python
# $Id: pane_utilities.py,v 1.6 2007/07/14 07:38:13 mwm Exp $
#
# pane_utilities.py -- Utility clases and functions for use with panes
#
#    Copyright (C) 2005  Mike Meyer <mwm@mired.org>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from os import environ
import cfilter
from webbrowser import open_new
from urllib import quote

class appmenu:
    "Creates a menu of applications to run in a pane."

    def __init__(my, pane, apps):
        "Create and run the applications menu from the keys."

        labels = apps.keys()
        labels.sort(key=lambda x: (x[0].lower(), x))
        width, height = pane.screen.menu_make(labels)
        my.system = pane.screen.system
        my.apps = apps
        pane.screen.menu_run((pane.width - width) / 2 + pane.x,
                         (pane.height - height) / 2 + pane.y,
                         my)

    def __call__(my, choice):
        "Call the system function on the value of the given choice."

        my.system(my.apps[choice] + " &")

                                                               
class codemenu:
    "Create a menu of Python actions to run."

    def __init__(my, pane, actions=None):
        "Create the menu and run it."

        if actions:
            my.actions = actions
        else:
            my._make(pane)
        labels = my.actions.keys()
        labels.sort(key=lambda c: (c[0].lower(), c))
        width, height = pane.screen.menu_make(labels, align="left")
        pane.screen.menu_run((pane.width - width) / 2 + pane.x,
                         (pane.height - height) / 2 + pane.y,
                         my)

    def __call__(my, choice):
        "Run the selection for the user."

	apply(apply, my.actions[choice])

    def _make(my, pane):
        """Make a dictionary from my methods and docstrings:

        For all method names that are one character long, the key will
        be the method name, a ': ', then the first word of the docstring. The value
        will be a the method."""

        my.actions = dict()
        for name in dir(my):
            if len(name) == 1:
                meth = getattr(my, name)
                name = '%s: %s' % (name, meth.__doc__)
                my.actions[name] = meth, (pane,)

class view_menu(codemenu):
    "The view for use when actions is an MVC controller."

    def __call__(my, choice):
        my.actions.choose(choice)
        

class windowmenu:
    "Create a menu of windows to add to a pane."

    def __init__(my, pane, filter = cfilter.true, startlist = None):
        labels = []
        clients = {}
        clientlist = startlist or pane.screen.query_clients(filter, 1)
        clientlist.sort(key=lambda c: (c.get_title()[0].lower(), c.get_title()))
        i = 'a'
        # We really need to deal with window lists longer than 26.
        for c in clientlist:
            l = "%c: %s" % (i, c.get_title())
            labels.append(l)
            clients[l] = c
            i = chr(ord(i) + 1)
        if labels:
            width, height = pane.screen.menu_make(labels, align = 'left')
            my.add = pane.add_window
            my.clients = clients
            pane.screen.menu_run((pane.width - width) / 2 + pane.x,
                                 (pane.height - height) / 2 + pane.y,
                                 my)
        else:
            width, height = pane.screen.message_make("No windows")
            pane.screen.message_display((pane.width - width) / 2 + pane.x,
                                        (pane.height - height) / 2 + pane.y)

    def __call__(my, choice):
        "Add the selected window to the pane."

        my.add(my.clients[choice])


class panesmenu:
    "Create a menu of all the panes."

    def __init__(my, screen):
        wm = screen.wm
        labels = []
        panes = {}
        for i in range(len(wm.panes_list)):
            w = wm.panes_list[i].window
            if w: l = "%d: %s" % (i, w.get_title())
            else: l = "%d: <EMPTY>" % i
            labels.append(l)
            panes[l] = i
        width, height = screen.menu_make(labels, align = 'left')
        my.goto = wm.panes_goto
        my.panes = panes
        screen.menu_run(screen.root_x + (screen.root_width - width) / 2,
                        screen.root_y + (screen.root_height - height) / 2, my)

    def __call__(my, choice):
        "Activate the selected pane."

        my.goto(my.panes[choice])
            

class websearch:
    "Launch a browser with a web search from the user."

    def __init__(my, pane, name, winclass, format, browser = None):
        my.format = format
        my.browser = browser
        my.pane = pane
        window = winclass("Search %s: " % name, pane.screen, length=50)
        window.read(my, window.editHandler, pane.x, pane.y)

    def __call__(my, string):
        query = my.format % quote(string)
        if my.browser:
            my.browser(query)
        else:
            environ['DISPLAY'] = my.pane.screen.displaystring
            open_new(query)
            

class runcommand:
    "Read a string from the user, and run it."

    def __init__(my, pane, winclass, prompt = "Command: "):
        my.system = pane.screen.system
        window = winclass(prompt, pane.screen, length = 50)
        window.read(my, window.editHandler, pane.x, pane.y)

    def __call__(my, string):
        my.system(string + " &")


class splitpane:
    "Read in a fraction, and split that much off the current pane."

    def __init__(my, pane, splitter, winclass, prompt = "Fraction: "):
        my.pane, my.splitter = pane, splitter
        window = winclass(prompt, pane.screen, length = 50)
        window.read(my, window.editHandler, pane.x, pane.y)

    def __call__(my, fraction):
        try:
            f = string.atof(fraction)
            my.splitter(f)
        except ValueError:
            pass


class numberpane:
    "Read a new number for the current pane."

    def __init__(my, pane, winclass, prompt = "New number: "):
        my.wm = pane.wm
        window = winclass(prompt, pane.screen)
        window.read(my, window.editHandler, pane.x, pane.y)

    def __call__(my, name):
        my.wm.panes_number(int(name))


class pullwindow:
    "Read a window's name, and pull it to the current pane."

    def __init__(my, pane, winclass, prompt = "Pull: "):
        my.pane = pane
        window = winclass(prompt, pane.screen)
        window.read(my, window.editHandler, pane.x, pane.y)

    def __call__(my, name):
        clients = my.pane.screen.query_clients(cfilter.re_title(name + ".*"), 1)
        if len(clients) == 1: my.pane.add_window(clients[0])
        elif clients: windowmenu(my.pane, startlist = clients)
        else:
            width, height = my.pane.screen.message_make("No windows")
            my.pane.screen.message_display(my.pane.x, my.pane.y)


class gotowindow:
    "Emulate the rp 'goto' command."

    def __init__(my, pane, winclass, prompt = "Goto: "):
        my.pane = pane
        window = winclass(prompt, pane.screen)
        window.read(my, window.editHandler)

    def __call__(my, name):
        clients = my.pane.screen.query_clients(cfilter.re_title(name + ".*"), 1)
        if clients:
            window = clients[0]
            if window.panes_pane.window and window.panes_pane.window == window:
                my.pane.wm.panes_activate(window.panes_pane)
            else:
                my.pane.add_window(window)
        else:
            width, height = my.pane.screen.message_make("No windows")
            my.pane.screen.message_display(0, 0)


def split_pane(count, splitter):
    """Invoke a pane splitter to divide a pane up into count pieces."""

    while count > 1:
        splitter(1. / count)
        count -= 1


def getapp(pane, name, command = None):
    "Find a window starting with name, and run command if it doesn't exist."

    clients = pane.screen.query_clients(cfilter.re_title(".*%s.*" % name), 1)
    if len(clients) > 1: windowmenu(pane, startlist = clients)
    elif clients: pane.add_window(clients[0])
    else: pane.screen.system((command or name) + " &")
    

