from __future__ import generators
from SimPy.Simulation  import *
from random import Random
from SimGUI import *

__doc__="""
Modified bank11.py (from Bank Tutorial) with GUI.

Model: Simulate customers arriving
at random, using a Source, requesting service
from two counters each with their own queue
random servicetime.

Uses Monitor objects to record waiting times
and total service times."""


class Bank11(SimGUI):
    def __init__(self,win,**p):
        SimGUI.__init__(self,win,**p)
        self.help.add_command(label="Author(s)",
                              command=self.showAuthors,underline=0)
        self.run.add_command(label="Run simulation",
                              command=self.model,underline=0)
        self.view.add_command(label="Statistics",
                              command=self.statistics,underline=0)
    
    def NoInSystem(self,R):
        """ The number of customers in the resource R
        in waitQ and active Q"""
        return (len(R.waitQ)+len(R.activeQ))

    def model(self):
        #global counter,waitMonitor,serviceMonitor,trace,lastLeave,noRunYet,Nc
        self.counterRV = Random(self.params.counterseed)
        self.sourceseed = self.params.sourceseed
        nrRuns=self.params.nrRuns
        self.lastLeave=0
        self.noRunYet=True
        for runNr in range(nrRuns):
            self.noRunYet=False
            self.trace=self.params.trace
            if self.trace:
                self.writeConsole(text='\n** Run %s'%(runNr+1))
            self.Nc = 2
            self.counter = [Resource(name="Clerk0"),Resource(name="Clerk1")]
            self.waitMonitor = Monitor(name='Waiting Times')
            self.waitMonitor.xlab='Time'
            self.waitMonitor.ylab='Customer waiting time'
            self.serviceMonitor = Monitor(name='Service Times')
            self.serviceMonitor.xlab='Time'
            self.serviceMonitor.ylab='Total service time = wait+service'
            initialize()
            source = Source(self,seed = self.sourceseed)
            activate(source,source.generate(self.params.numberCustomers,self.params.interval),0.0)
            simulate(until=self.params.endtime)
            self.writeStatusLine("Time = %s"%now())
            self.lastLeave+=now()
        self.writeConsole("%s simulation run(s) completed\n"%nrRuns)
        self.writeConsole("Parameters:\n%s"%self.params)
        
    def statistics(self):
        if self.noRunYet:
            showwarning(title='SimPy warning',message="Run simulation first -- no data available.")
            return
        aver=self.lastLeave/self.params.nrRuns
        self.writeConsole(text="Average time for %s customers to get through bank: %.1f\n(%s runs)\n"\
                          %(self.params.numberCustomers,aver,self.params.nrRuns))
        
    def showAuthors(self):
        self.showTextBox(text="Tony Vignaux\nKlaus Muller",title="Author information")

class Source(Process):
    """ Source generates customers randomly"""
    def __init__(self,sim,seed=333):
        Process.__init__(self)
        self.sim=sim
        self.SEED = seed

    def generate(self,number,interval):       
        rv = Random(self.SEED)
        for i in range(number):
            c = Customer(self.sim,name = "Customer%02d"%(i,))
            activate(c,c.visit(timeInBank=12.0))
            t = rv.expovariate(1.0/interval)
            yield hold,self,t

class Customer(Process):
    """ Customer arrives, is served and leaves """
    def __init__(self,sim,**p):
        Process.__init__(self,**p)
        self.sim=sim
        
    def visit(self,timeInBank=0):       
        arrive=now()
        Qlength = [self.sim.NoInSystem(self.sim.counter[i]) for i in range(self.sim.Nc)]
        ##print "%7.4f %s: Here I am. %s   "%(now(),self.name,Qlength)
        for i in range(self.sim.Nc):
            if Qlength[i] ==0 or Qlength[i]==min(Qlength): join =i ; break
        yield request,self,self.sim.counter[join]
        wait=now()-arrive
        self.sim.waitMonitor.observe(wait,t=now())                                 #Lmond
        ##print "%7.4f %s: Waited %6.3f"%(now(),self.name,wait)
        tib = self.sim.counterRV.expovariate(1.0/timeInBank)
        yield hold,self,tib
        yield release,self,self.sim.counter[join]
        self.sim.serviceMonitor.observe(now()-arrive,t=now())
        if self.sim.trace:
            self.sim.writeConsole("Customer leaves at %.1d"%now())
        ##print "%7.4f %s: Finished    "%(now(),self.name)            

root=Tk()
gui=Bank11(root,title="SimPy GUI example",doc=__doc__)
gui.params=Parameters(endtime=2000,
        sourceseed=1133,
        counterseed=3939393,
        numberCustomers=50,
        interval=10.0,
        trace=0,
        nrRuns=1)
gui.mainloop()