#ifndef VISUAL_NUM_UTIL_H
#define VISUAL_NUM_UTIL_H

// num_util.h and num_util.cpp were obtained from: 
// http://www.eos.ubc.ca/research/clouds/num_util.html on 2003-12-17 under the 
// terms and conditions of the Boost Software License, version 1.0.  num_util 
// was written by Rhys Goldstein, Chris Seymour and Phil Austin.  

// Questions or comments about num_util should be directed to Phil Austin at 
// paustin@eos.ubs.ca.

/*
Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

// Changes from the original num_util:
// - Moved into the visual namespace to prevent clashes with other projects 
// using num_util.  2003-12-18
// - Changed header inclusion guards to follow the same conventions as the rest 
// of Visual.  2003-12-18
// - Changed the definition of PY_ARRAY_UNIQUE_SYMBOL to prevent clashes with
// other projects using num_util.  2003-12-18
// - Changed header #includes to reduce compile times.  2003-12-19
// - Bring boost::python::numeric::array into the visual namespace for 
// convienience.  2003-12-19
// - data(), shape(), rank(): Modified to ensure that no calls are made into the
// Python interpreter, espcially reference counting operations and PyArray_Check()
// For the reference counting operations, we can guarantee that we do not need
// them base on the fact that these functions are never called without owning
// at least one real boost::python::numeric::array.  For the PyArray_Check(),
// we can guarantee that we *never* have an array on our hands that is not a 
// genuine array.  These functions needed to be changed to ensure that we do not
// call any functions in the interpreter or Numeric from within the rendering
// loop since they cause the interpreter to crash sporadically on multiprocessor
// machines. 2004-01-12


#include <boost/python/numeric.hpp>
#include <vector>

namespace visual {
	enum array_types
	{
		char_t,
		uchar_t,
		schar_t,
		short_t,
		ushort_t,
		int_t,
		uint_t,
		long_t,
		float_t,
		double_t,
		cfloat_t,
		cdouble_t,
		object_t,
		notype_t
	};
	
  using boost::python::numeric::array;

  /** 
   *Creates a n-dimensional Numeric array with dimensions dimens and Numeric 
   *type t. The elements of the array are initialized to zero.
   *@param dimens a vector of interger specifies the dimensions of the array.
   *@param t elements' Numeric type. Default is double.
   *@return a numeric array of shape dimens with elements initialized to zero.
   */
  // USED.
  array makeNum(std::vector<int> dimens, array_types t = double_t);

  /** 
   *A free function that retrieves the Numeric type of a Numeric array.
   *@param arr a Boost/Python numeric array.
   *@return the Numeric type of the array's elements 
   */
  // USED
  array_types type( array arr);


  /** 
   *Returns the dimensions in a vector.
   *@param arr a Boost/Python numeric array.
   *@return a vector with integer values that indicates the shape of the array.
  */
  // USED.
  std::vector<int> shape(const array& arr);
  
  /** 
   *Returns true if the array is contiguous.
   *@param arr a Boost/Python numeric array.
   *@return true if the array is contiguous, false otherwise.
  */
  bool iscontiguous( array arr);

  /** 
   *Returns a pointer to the data in the array.
   *@param arr a Boost/Python numeric array.
   *@return a char pointer pointing at the first element of the array.
   */
  // USED
  char* data(const array& arr);
  
	/** 
	 *Returns a clone of this array with a new type.
	 *@param arr a Boost/Python numeric array.
	 *@param t PyArray_TYPES of the output array.
	 *@return a replicate of 'arr' with type set to 't'.
	 */
	// USED
	array astype( array arr, array_types t);
	
	// These functions initialize the PyArray_API function pointers       
    void init_numeric_impl();
    void init_numarray_impl();
    // While these functions set the chosen API.
    void use_numeric_impl();
    void use_numarray_impl();

} //  namespace visual

#endif // !defined VISUAL_NUM_UTIL_H
