# -*- coding: ISO-8859-1 -*-

# Copyright (C) 2003 Jrg Lehmann <joerg@luga.de>
#
# This file is part of PyTone (http://www.luga.de/pytone/)
#
# PyTone is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2
# as published by the Free Software Foundation.
#
# PyTone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PyX; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os.path
import random
import time
import threading
import pickle

import config
import events, hub, requests
import dbitem, item
import log

_counter = 0

class playlistitem:
    """ wrapped song with the two additional attributes played and id
    
    - played: has playlist item already been played
    - id:     unique id for playlist item (note that the same song
              can be present more than once in the playlist)"""
    
    def __init__(self, song):
        self.song = song
        self.played = 0
        global _counter
        self.id = _counter
        _counter += 1

    def __repr__(self):
        return "playlistitem: id=%s" % `self.id`

    def __cmp__(self, other):
        try:
            return cmp(self.id, other.id)
        except:
            return 1

    def __getattr__(self, attr):
        # Python tries to call __setstate__ upon unpickling -- prevent this
        if attr=="__setstate__":
            raise AttributeError
        return getattr(self.song, attr)

    def markplayed(self):
        self.played = 1
        self.song.play()

    def markunplayed(self):
        self.played = 0

    def hasbeenplayed(self):
        return self.played


def initplaylist(id, playerid, songdbid):
    """initialize playlist service corresponding to player with playerid
    """
    playlist(id, playerid, songdbid).start()


class playlist(threading.Thread):
    """manage playlist for a single player, which can be accessed 
    by multipled users"""

    def __init__(self, id, playerid, songdbid):
        threading.Thread.__init__(self)
        self.setName("playlist thread")
        self.id = id
        # each playlist service is identified by the corresponding player
        self.playerid = playerid
        self.songdbid = songdbid
        self.songs = []
        self.ttime = 0
        self.ptime = 0
        self.playingsong = None
        self.logfilename = config.general.logfile
        self.autoplaymode = config.general.autoplaymode

        # as independent thread, we want our own event and request channel
        self.channel = hub.hub.newchannel()

        self.done = 0
        
        self.channel.subscribe(events.playbackinfochanged, self.playbackinfochanged)
        self.channel.subscribe(events.playerstop, self.playerstop)
        self.channel.subscribe(events.playlistaddsongs, self.playlistaddsongs)
        self.channel.subscribe(events.playlistaddsongtop, self.playlistaddsongtop)
        self.channel.subscribe(events.playlistdeletesong, self.playlistdeletesong)
        self.channel.subscribe(events.playlistmovesongup, self.playlistmovesongup)
        self.channel.subscribe(events.playlistmovesongdown, self.playlistmovesongdown)
        self.channel.subscribe(events.playlistclear, self.playlistclear)
        self.channel.subscribe(events.playlistdeleteplayedsongs,
                               self.playlistdeleteplayedsongs)
        self.channel.subscribe(events.playlistreplay, self.playlistreplay)
        self.channel.subscribe(events.playlistload, self.playlistload)
        self.channel.subscribe(events.playlistsave, self.playlistsave)
        self.channel.subscribe(events.playlistshuffle, self.playlistshuffle)
        self.channel.subscribe(events.playlisttoggleautoplaymode, self.playlisttoggleautoplaymode)
        self.channel.subscribe(events.quit, self.quit)

        self.channel.supply(requests.requestnextsong, self.requestnextsong)
        self.channel.supply(requests.playlistgetcontents, self.playlistgetcontents)
        
        # try to load dump from prior crash, if existent
        if config.general.dumpfile:
            try:
                if os.path.isfile(config.general.dumpfile):
                    self.load()
                    os.unlink(config.general.dumpfile)
            except:
                pass

    def run(self):
        """main loop"""
        while not self.done:
            self.channel.process(blocking=1)

    def append(self, item):
        self.ttime += item.length
        if item.hasbeenplayed():
            self.ptime += item.length
        self.songs.append(item)

    def insert(self, index, item):
        self.ttime += item.length
        if item.hasbeenplayed():
            self.ptime += item.length
        self.songs.insert(index, item)

    def __delitem__(self, index):
        item = self.songs[index]
        self.ttime -= item.length
        if item.hasbeenplayed():
            self.ptime -= item.length
        self.songs.__delitem__(index)

    def _notifyplaylistchanged(self):
        hub.hub.notify(events.playlistchanged(self.songs, self.ptime, self.ttime, self.autoplaymode, self.playingsong))

    def searchnextsong(self):
        """return playlistitem which has to be played next or None"""
        for i in range(len(self.songs)):
            if not self.songs[i].hasbeenplayed():
                return self.songs[i]
        return None

    def _playnext(self):
        """ mark next item from playlist as played & currently playing and return
        corresponding song"""
        nextsong = self.searchnextsong()
        if nextsong:
            self.ptime += nextsong.length
            nextsong.markplayed()
            self._notifyplaylistchanged()
            if self.logfilename:
                logfile = open(self.logfilename, "a")
                logfile.write("%s: %s\n" % (time.asctime(), nextsong.path))
                logfile.close()
            return nextsong
        else:
            return None

    def _clear(self):
        self.songs = []
        self.ptime = 0
        self.ttime = 0
        self.playingsong = None

    def _deleteplayedsongs(self):
        for i in range(len(self.songs)-1,-1,-1):
            # XXX should we delete only if the song is not currently being played
            if self.songs[i].hasbeenplayed():
                del self[i]

    def _checksong(self, song):
        # it is ok if the song is contained in a local song database, so we first
        # check whether this is the case.
        # XXX make this behaviour configurable?
        if isinstance(song, item.song):
            dbtype, location = hub.hub.request(requests.getdatabaseinfo(song.songdbid))
            if dbtype == "local":
                return song
        if os.path.isfile(song.path):
            # first we try to access the song via its filesystem path
            asong = hub.hub.request(requests.queryregistersong(self.songdbid, song.path))
            return item.song(self.songdbid, song=asong)

        if song.artist!=dbitem.UNKNOWN and song.album!=dbitem.UNKNOWN:
            # otherwise we use the artist and album tags and try to obtain the song via
            # the database
            songs = hub.hub.request(requests.getsongs(self.songdbid,
                                                      artist=song.artist, album=song.album))
            for asong in songs:
                if asong.title==song.title:
                    return item.song(self.songdbid, song=asong)

        # song not found
        # XXX start transmitting song
        return

    def _addsongs(self, songs):
        """add songs to end of playlist"""
        for song in songs:
            song = self._checksong(song)
            if song:
                self.append(playlistitem(song))
        self._notifyplaylistchanged()

    def _markallunplayed(self):
        """ mark all songs in playlist as not having been played """
        for song in self.songs:
            if song.hasbeenplayed():
                song.markunplayed()
                self.ptime -= song.length
        self._notifyplaylistchanged()

    # statusbar input handler

    def saveplaylisthandler(self, name, key):
        name = os.path.join(config.general.playlistdir, name.strip())
        if key==ord("\n") and name!="":
            if name[-4:]!=".m3u":
                name = name + ".m3u"
            try:
                file = open(name, "w")
                for song in self.songs:
                    file.write("%s\n" % song.path)
                file.close()
                hub.hub.notify(events.registerplaylists(self.songdbid, [name]))
            except (IOError, OSError):
                pass

    def loadplaylisthandler(self, name, key):
        if key==ord("\n"):
            if name[-4:]!=".m3u":
                name = name + ".m3u"
            try:
                path = os.path.join(config.general.playlistdir, name)
                file = open(path, "r")
                self._clear()
                for line in file.xreadlines():
                    if not line.startswith("#"):
                        dbsong = hub.hub.request(requests.queryregistersong(self.songdbid,
                                                                          line.strip()))
                        if dbsong:
                            song = item.song(self.songdbid, song=dbsong)
                            self.append(playlistitem(song))

                file.close()
                self._notifyplaylistchanged()
            except (IOError, OSError):
                pass

    def _locatesong(self, id):
        """ locate position of song in playlist by id """
        for song, i in zip(self.songs, range(len(self.songs))):
            if song.id==id:
                return i
        else:
            return None

    # event handlers

    def playbackinfochanged(self, event):
        if event.playbackinfo.playerid == self.playerid:
            newplayingsong = event.playbackinfo.song

            if self.playingsong!=newplayingsong:
                self.playingsong = newplayingsong
                self._notifyplaylistchanged()

    def playerstop(self, event):
        if event.playerid == self.playerid:
            if self.playingsong:
                self.playingsong.markunplayed()
                self._notifyplaylistchanged()


    def playlistaddsongs(self, event):
        self._addsongs(event.songs)

    def playlistaddsongtop(self, event):
        song = self._checksong(event.song)
        if song:
            newsong = playlistitem(song)
            for i in range(len(self.songs)):
                if not self.songs[i].hasbeenplayed():
                    self.insert(i, newsong)
                    break
            else:
                self.append(newsong)
            self._notifyplaylistchanged()
            hub.hub.notify(events.nextsong(self.playerid, self._playnext()))

    def playlistdeletesong(self, event):
        i = self._locatesong(event.id)
        if i is not None:
            del self[i]
            self._notifyplaylistchanged()

    def playlistmovesongup(self, event):
        i = self._locatesong(event.id)
        if i is not None and i>0:
            self.songs[i-1], self.songs[i] = self.songs[i], self.songs[i-1]
            self._notifyplaylistchanged()

    def playlistmovesongdown(self, event):
        i = self._locatesong(event.id)
        if i is not None and i<len(self.songs)-1:
            self.songs[i], self.songs[i+1] = self.songs[i+1], self.songs[i]
            self._notifyplaylistchanged()

    def playlistclear(self, event):
        self._clear()
        self._notifyplaylistchanged()
        
    def playlistdeleteplayedsongs(self, event):
        self._deleteplayedsongs()
        self._notifyplaylistchanged()

    def playlistreplay(self, event):
        self._markallunplayed()

    def playlistsave(self, event):
        hub.hub.notify(events.requestinput(_("Save playlist"),
                                              _("Name:"),
                                             self.saveplaylisthandler))

    def playlistload(self, event):
        hub.hub.notify(events.requestinput(_("Load playlist"),
                                              _("Name:"),
                                             self.loadplaylisthandler))

    def playlistshuffle(self, event):
        random.shuffle(self.songs)
        self._notifyplaylistchanged()

    def playlisttoggleautoplaymode(self, event):
        if self.autoplaymode == "off":
            self.autoplaymode = "repeat"
        elif self.autoplaymode == "repeat":
            self.autoplaymode = "random"
        else:
            self.autoplaymode = "off"
        self._notifyplaylistchanged()

    def quit(self, event):
        self.dump()
        self.done = 1
        
    def dump(self):
        """ write playlist to dump file """
        if self.playingsong:
            self.playingsong.markunplayed()
        self._deleteplayedsongs()
        self._notifyplaylistchanged()
        dumpfile = open(config.general.dumpfile, "w")
        pickle.dump(self.songs, dumpfile)

    def load(self):
        """ load playlist from file """
        dumpfile = open(config.general.dumpfile, "r")
        self._clear()
        for song in pickle.load(dumpfile):
            self.append(song)
        self._notifyplaylistchanged()
    
    #
    # request handler
    #

    def requestnextsong(self, request):
        if request.playerid == self.playerid:
            nextsong = self._playnext()
            if not nextsong:
                if self.autoplaymode == "random":
                    # add some randomly selected song to the end of the playlist
                    randomsongs = hub.hub.request(requests.getsongs(self.songdbid, random=1))
                    if randomsongs:
                        self._addsongs(randomsongs[0:1])
                        nextsong = self._playnext()
                elif self.autoplaymode == "repeat":
                    self._markallunplayed()
                    nextsong = self._playnext()
            return nextsong
        else:
            return None

    def playlistgetcontents(self, request):
        return self.songs, self.ptime, self.ttime, self.autoplaymode, self.playingsong
